package org.eaglei.ui.gwt.security;

import java.util.ArrayList;

import org.eaglei.model.EIURI;
import org.eaglei.security.Session;
import org.eaglei.ui.gwt.security.rpc.SecurityServiceRemote;
import org.eaglei.ui.gwt.security.rpc.SecurityServiceRemoteAsync;

import com.google.gwt.core.client.GWT;
import com.google.gwt.user.client.Cookies;
import com.google.gwt.user.client.Window;
import com.google.gwt.user.client.rpc.AsyncCallback;

public class SessionContext {

    public interface SessionListener {
        void onLogIn();
        void onLogOut();
    }

    private static final SecurityServiceRemoteAsync securityService = GWT
            .create(SecurityServiceRemote.class);
    public static final SessionContext INSTANCE = new SessionContext();

    public static final String EAGLEI_SESSION_ID = "eaglei_session";
    public static final String EAGLEI_USER_URI_ID = "eaglei_userUri";
    public static final String EAGLEI_USER_COOKIE_ID = "eaglei_username";
    // Duration for remembering login
    public static final long DURATION = 1000 * 60 * 60 * 24 * 14;

    public static boolean hasSession() {
        return Cookies.getCookieNames().contains(EAGLEI_SESSION_ID);
    }

    public static boolean hasUserUri() {
        return Cookies.getCookieNames().contains(EAGLEI_USER_URI_ID);
    }

    public static boolean hasUserName() {
        return Cookies.getCookieNames().contains(EAGLEI_USER_COOKIE_ID);
    }

    public static String getSessionId() {
        return hasSession() ? Cookies.getCookie(EAGLEI_SESSION_ID) : null;
    }

    public static String getUserUri() {
        return hasUserUri() ? Cookies.getCookie(EAGLEI_USER_URI_ID) : null;
    }

    public static String getUserName() {
        return hasUserName() ? Cookies.getCookie(EAGLEI_USER_COOKIE_ID) : null;
    }

    private static void setCookies(String username, String userUri, String sessionId) {
        Cookies.setCookie(EAGLEI_USER_COOKIE_ID, username);
        Cookies.setCookie(EAGLEI_USER_URI_ID, userUri);
        Cookies.setCookie(EAGLEI_SESSION_ID, sessionId);
    }

    private static void removeCookies() {
        Cookies.removeCookie(EAGLEI_SESSION_ID);
        Cookies.removeCookie(EAGLEI_USER_COOKIE_ID);
        Cookies.removeCookie(EAGLEI_USER_URI_ID);

    }

    private String pendingUser = null;
    private ArrayList<SessionListener> listeners = new ArrayList<SessionListener>();

    private SessionContext() {
        String sessionId = getSessionId();
        if (sessionId != null) {
            securityService.getSession(sessionId, new AsyncCallback<Session>() {

                @Override
                public void onFailure(Throwable caught) {
                    logOut();
                }

                @Override
                public void onSuccess(Session result) {
                    if (result == null) {
                        logOut();
                    }
                }
            });
        }
    }

    public void addListener(SessionListener listener) {
        listeners.add(listener);
    }

    public String getPendingUser() {
        return pendingUser;
    }

    public void logIn(String institutionId, String user, String password) {
        this.pendingUser = user;
        securityService.logIn(institutionId, user, password, new AsyncCallback<Session>() {

            public void onFailure(Throwable caught) {
                Window.alert("Invalid login.  Please check username and password.");
            }

            public void onSuccess(Session result) {
                if (result == null) {
                    Window.alert("Invalid login.  Please check username and password.");
                    return;
                }
                setCookies(result.getUserName(), result.getUserName(), result.getSessionId());
                for (SessionListener listener : listeners) {
                    listener.onLogIn();
                }
            }

        });
    }

    public void logOut() {
        this.pendingUser = null;
        String sessionId = getSessionId();
        removeCookies();
        if (sessionId == null) {
            // Assume listeners have already been notified.
            return;
        }
        for (SessionListener listener : listeners) {
            listener.onLogOut();
        }
        securityService.logOut(sessionId, new AsyncCallback<Void>() {

            public void onFailure(Throwable caught) {
            }

            public void onSuccess(Void result) {
            }

        });
    }
}
