/*
 * Copyright (c) 2015-2016, President and Fellows of Harvard College
 *
 * Redistribution and use in source and binary forms, with or without
 * modification, are permitted provided that the following conditions are met:
 *
 * 1. Redistributions of source code must retain the above copyright
 * notice, this list of conditions and the following disclaimer.
 *
 * 2. Redistributions in binary form must reproduce the above copyright
 * notice, this list of conditions and the following disclaimer in the
 * documentation and/or other materials provided with the distribution.
 *
 * 3. The name of the author may not be used to endorse or promote products
 * derived from this software without specific prior written permission.
 *
 * THIS SOFTWARE IS PROVIDED BY THE AUTHOR "AS IS" AND ANY EXPRESS OR IMPLIED
 * WARRANTIES, INCLUDING, BUT NOT LIMITED TO, THE IMPLIED WARRANTIES OF
 * MERCHANTABILITY AND FITNESS FOR A PARTICULAR PURPOSE ARE DISCLAIMED. IN NO
 * EVENT SHALL THE AUTHOR BE LIABLE FOR ANY DIRECT, INDIRECT, INCIDENTAL,
 * SPECIAL, EXEMPLARY, OR CONSEQUENTIAL DAMAGES (INCLUDING, BUT NOT LIMITED
 * TO, PROCUREMENT OF SUBSTITUTE GOODS OR SERVICES; LOSS OF USE, DATA, OR
 * PROFITS; OR BUSINESS INTERRUPTION) HOWEVER CAUSED AND ON ANY THEORY OF
 * LIABILITY, WHETHER IN CONTRACT, STRICT LIABILITY, OR TORT (INCLUDING
 * NEGLIGENCE OR OTHERWISE) ARISING IN ANY WAY OUT OF THE USE OF THIS
 * SOFTWARE, EVEN IF ADVISED OF THE POSSIBILITY OF SUCH DAMAGE.
 */
var selectedCheckboxes =  new Map();
var SchedulerTables = function(options) {
    this.tableId = options.tableId;
    this.arrayColumns = options.columns;
    this.reloadFn = options.reloadFn;
    this.paginationWidget = null;
    this.filterWidget = null;
    this.sortInfo = new SchedulerTables.SortInfo();
    this.totalCount = 0;
    this.selectedRows = [];
    this.expandFn = options.expandFn;
    this.doubleClickRowIdCallback = options.doubleClickRowIdCallback;
    this.tableSpecificGroupRowClass = null;
    this.tableSpecificExpandRowClass = null;
    this.tableSpecificGroupRowClass = this.tableId + "-tableGroupHeader";
    this.groupBy = options.groupBy;

    this.getId = function() {
        return this.tableId;
    };

    this.getColumns = function() {
        return this.arrayColumns;
    };

    this._addGroupHeaderRow = function(tableRow, value, backgroundClass) {
        backgroundClass = backgroundClass || "";
        var groupHeaderRow = $("<tr></tr>", {
            class: this.tableSpecificGroupRowClass + " tableGroupHeaderRow " + backgroundClass
        });

        var groupHeaderRowTd = $("<td></td>", {
            colspan: this.arrayColumns.length
        });

        if(this.groupBy.render) {
            value = this.groupBy.render(value);
        }

        groupHeaderRowTd.append(value);
        groupHeaderRow.append(groupHeaderRowTd);
        groupHeaderRow.insertBefore(tableRow);
    };

    this._createTableHeader = function() {
        var table = this;
        var tableBody = $("<tbody></tbody>");
        $('#' + table.tableId).append(tableBody);

        var tableRow = $('<tr></tr>', {"id": table.tableId + "-columnLabelRow"});
        tableBody.append(tableRow);

        $.each(table.arrayColumns, function (indexOfThis, column) {
            column.generateTableHeadElement(tableRow, function(column) {
                table._doSort(column, true);
            });

            // in addition to setting onSort above, set the order if this
            // column is 'special', i.e., is the one with defaultSortOrder
            if (column.defaultSortOrder) {

                table.sortInfo.setSortOrder(column.defaultSortOrder);
                table.sortInfo.setSortColumn(column.dbColumn);
            }
        });

        this.refreshTableColumnsSorting();
    };

    this._toggleValue = function(currentValue, option1, option2) {
        return currentValue === option1 ? option2 : option1;
    };

    this._createTableBody = function(data, currentSelected) {
        var table = this;
        var tableBody = $('#' + table.tableId + ' tbody');

        var rowHighlightClass = "expandableTable-rowHighlight";
        var noRowHighlightClass = "expandableTable-noRowHighlight";

        var backgroundClass = "";
        table.totalCount = 0;
        $.each(data, function (index, val) {
            if (index === 0) {
                currentPage = val.navigatePage + 1;
                table.setCurrentPage(currentPage);
            }

            table.totalCount = val.totalCount;

            var tableRow = $("<tr></tr>");
            tableRow.addClass(table.tableId + "-tableRow");
            tableBody.append(tableRow);

            if (table.doubleClickRowIdCallback) {
                tableRow.bind("dblclick", function() {
                    table.doubleClickRowIdCallback(val.id);
                });
            }

            $.each(table.arrayColumns, function (indexOfThis, column) {
                if(table.groupBy && table.groupBy.columnName === column.columnName) {
                    var currentValue = column.extractDataFunction(val);
                    //if current value != previous value
                    if(index === 0 || table.groupBy.isNewGroup(currentValue, column.extractDataFunction(data[index - 1]))) {
                        backgroundClass = table._toggleValue(backgroundClass, noRowHighlightClass, rowHighlightClass);
                        table._addGroupHeaderRow(tableRow, currentValue, backgroundClass);
                    }
                }
                column.generateTableElement(val, currentSelected, tableRow);
            });

            if(table.expandFn) {
                backgroundClass = table._toggleValue(backgroundClass, noRowHighlightClass, rowHighlightClass);
                tableRow.addClass(backgroundClass);
            }
        });
    };

    this._generateColumnId = function(index) {
        return this.tableId + "_" + index;
    };

    this._setupColumns = function() {
        var table = this;

        if(table.groupBy && !table.groupBy.isNewGroup) {
            table.groupBy.isNewGroup = function(oldValue, newValue){ return oldValue !== newValue};
        }

        var expandRowClass = "expandRow";
        table.tableSpecificExpandRowClass = expandRowClass + "-" + table.tableId;
        $.each(table.arrayColumns, function(index, column) {
            column.id = table._generateColumnId(index);

            column.columnType.setOptions({
                onSelect : function(value, checked) {
                    if(checked) {
                        table.selectedRows.push(value);
                    }
                    else {
                        table.unSelectRow(value);
                    }
                },
                onExpand: function(tdElement, value)
                {
                    var parentRow = tdElement.parent("tr");
                    var expandRow = $("<tr></tr>").addClass(expandRowClass)
                        .addClass(table.tableSpecificExpandRowClass)
                        .insertAfter(parentRow);

                    var parentRowBg = parentRow.css("background-color");
                    expandRow.css({"background-color": parentRowBg});

                    var expandRowTd = $("<td></td>", {colspan: table.arrayColumns.length}).appendTo(expandRow);
                    var containerDiv = $("<div></div>").appendTo(expandRowTd);
                    table.expandFn(containerDiv, value);
                },
                onCollapse: function(tdElement)
                {
                    var parentRow = tdElement.parent("tr");
                    parentRow.next("." + expandRowClass).remove();
                }
            });
        });
    };

    this.unSelectRow = function (value) {
        var index = $.inArray(value);
        this.selectedRows.splice(index, 1);
    };

    this.clearAllSelectedRows = function () {
        this.selectedRows = [];
    };

    this._setup = function() {
        $('#' + this.tableId).children().remove();

        if(this.expandFn)
        {
            $('#' + this.tableId).addClass("expandableTable");
        }
        this._setupColumns();
    };

    this.generateTable = function(data, maxResults, paginationContainerId, paginationReload, displayRecordsReload) {

        this._setup();
        this._createTableHeader();

        var currentSelected = [];
        this._createTableBody(data, currentSelected);
        SchedulerTables.reselectCheckboxes(this.tableId, this.totalCount, currentSelected);

        var table = this;
        this.paginationWidget = PaginationHelper.initPagination(
            paginationContainerId,
            table.totalCount,
            maxResults,
            paginationReload,
            displayRecordsReload
        );

        this.filterWidget = new Filter();
        this.filterWidget.initFilter(this);
    };

    this.refreshTableColumnsSorting = function() {
        this.arrayColumns.map(function (column) {
            column.refreshClass();
        });
    };

    this.refreshTableBody = function(data) {

        $('.' + this.tableId + '-tableRow').each(function() {
            $(this).remove();
        });

        $('.' + this.tableSpecificGroupRowClass).each(function() {
            $(this).remove();
        });

        $('.' + this.tableSpecificExpandRowClass).each(function() {
            $(this).remove();
        });

        this._createTableBody(data, null);
        PaginationHelper.updatePagination(this.paginationWidget, this.totalCount);
    };

    this.getSelectedRows = function(){
        return this.selectedRows;
    };

    this.getRecordsPerPage = function() {
        var recordsPerPage;
        if(this.paginationWidget != undefined) {
            recordsPerPage = this.paginationWidget.getRecordsPerPage();
        }

        return recordsPerPage;
    };

    this.getCurrentPage = function() {
        return this.paginationWidget ? this.paginationWidget.getCurrentPageNumber() : 1;
    };

    this.setCurrentPage = function(curr) {
        if (this.paginationWidget && curr) {
            this.paginationWidget.setCurrentPageNumber(curr);
        }
    };

    this.getAllFilterKeyValuePairs = function() {
        return this.filterWidget.getAllFilterKeyValuePairs();
    };

    this.clearAllFilterKeyValuePairs = function() {
        return this.filterWidget.clearAllFilterKeyValuePairs();
    };

    this._doSort = function(column){
        if (! column.dbColumn) {
            // not sortable
            return;
        }

        this.sortInfo.setSortOrder(column.toggleOrInitOrderBy());
        this.sortInfo.setSortColumn(column.dbColumn);

        this.resetSortOrderBy();
        column.setCurrentOrderBy(this.getSortOrder());

        this.refreshTableColumnsSorting();
        this.reloadFn();
    };
    
    this.getSortOrder = function() {
        return this.sortInfo.getSortOrder();
    };

    this.resetSortOrderBy = function() {
        $.each(this.arrayColumns, function(index, column) {
            column.setCurrentOrderBy("");
        })
    };

    this.getSortColumn = function() {
        return this.sortInfo.getSortColumn();
    };
};

SchedulerTables.SortInfo = function(sortColumn, sortOrder){
    this.sortColumn = sortColumn;
    this.sortOrder = sortOrder;

    this.setSortColumn = function(sortColumn){
        this.sortColumn = sortColumn;
    };

    this.getSortColumn = function() {
        return this.sortColumn;
    };

    this.setSortOrder = function(sortOrder){
        this.sortOrder = sortOrder;
    };

    this.getSortOrder = function(){
        return this.sortOrder;
    };
};


SchedulerTables.getSelectedCheckboxes = function (tableId) {
    return selectedCheckboxes.get(tableId);
};

SchedulerTables.clearSelectedCheckboxes =  function (tableId) {
    selectedCheckboxes.set(tableId, []);
};

SchedulerTables.addSelectedCheckbox = function(tableId, value) {
    if (selectedCheckboxes.get(tableId)) {
        selectedCheckboxes.get(tableId).push(value);
    }
    else {
        selectedCheckboxes.set(tableId, []);
        selectedCheckboxes.get(tableId).push(value);
    }
};

SchedulerTables.setSelectedCheckboxes = function (tableId, arrayOfSelected) {
    selectedCheckboxes.set(tableId, arrayOfSelected);
};

SchedulerTables.reselectCheckboxes = function(tableId, totalData, currentSelected) {
    SchedulerTables.setSelectedCheckboxes(tableId, currentSelected);

    if (totalData === SchedulerTables.getSelectedCheckboxes(tableId).length && totalData !== 0) {
        $('#' + tableId + '-selectAll').prop('checked', true);
    }
    else {
        $('#' + tableId + '-selectAll').prop('checked', false);
    }
};

SchedulerTables.selectAllCheckboxes = function(tableId) {
    var checkboxes = $('.' + tableId +'-checkbox');
    for (var i = 0; i < checkboxes.length; i++) {
        var checkbox = $(checkboxes[i]);
        if (($.inArray(checkbox.val(), SchedulerTables.getSelectedCheckboxes(tableId)) == -1)) {
            if ($.isNumeric(checkbox.val())) {
                this.addSelectedCheckbox(tableId, parseInt(checkbox.val()));
            }
            checkbox.prop('checked', true);
        }
        else if (checkbox.checked != true) {
            checkbox.prop('checked', true);
        }
    }
    $('#' + tableId + '-selectAll').prop('checked', true);
};

SchedulerTables.deselectAllCheckboxes = function(tableId) {
    var checkboxes = $('.' + tableId +'-checkbox');
    this.clearSelectedCheckboxes(tableId);
    checkboxes.prop('checked', false);
    $('#' + tableId + '-selectAll').prop('checked', false);
};
