/*
 * Copyright (c) 2015-2016, President and Fellows of Harvard College
 *
 * Redistribution and use in source and binary forms, with or without
 * modification, are permitted provided that the following conditions are met:
 *
 * 1. Redistributions of source code must retain the above copyright
 * notice, this list of conditions and the following disclaimer.
 *
 * 2. Redistributions in binary form must reproduce the above copyright
 * notice, this list of conditions and the following disclaimer in the
 * documentation and/or other materials provided with the distribution.
 *
 * 3. The name of the author may not be used to endorse or promote products
 * derived from this software without specific prior written permission.
 *
 * THIS SOFTWARE IS PROVIDED BY THE AUTHOR "AS IS" AND ANY EXPRESS OR IMPLIED
 * WARRANTIES, INCLUDING, BUT NOT LIMITED TO, THE IMPLIED WARRANTIES OF
 * MERCHANTABILITY AND FITNESS FOR A PARTICULAR PURPOSE ARE DISCLAIMED. IN NO
 * EVENT SHALL THE AUTHOR BE LIABLE FOR ANY DIRECT, INDIRECT, INCIDENTAL,
 * SPECIAL, EXEMPLARY, OR CONSEQUENTIAL DAMAGES (INCLUDING, BUT NOT LIMITED
 * TO, PROCUREMENT OF SUBSTITUTE GOODS OR SERVICES; LOSS OF USE, DATA, OR
 * PROFITS; OR BUSINESS INTERRUPTION) HOWEVER CAUSED AND ON ANY THEORY OF
 * LIABILITY, WHETHER IN CONTRACT, STRICT LIABILITY, OR TORT (INCLUDING
 * NEGLIGENCE OR OTHERWISE) ARISING IN ANY WAY OUT OF THE USE OF THIS
 * SOFTWARE, EVEN IF ADVISED OF THE POSSIBILITY OF SUCH DAMAGE.
 */


var minuteMillis = (60 * 1000);
var resourceNames;

var resourcePage = (function () {
    var resourceTable;
    var bulkActivateMode = false;

    var initFn = function () {
        resetUI();
        commonInit();
        getResourceStaticLists();
        commonResourceScreenData();
        app_runIdleTimer();
    };

    var initResourceTable = function(){
        var tableId = "resourceTable";

        var columns = [];

        if (bulkActivateMode) {
            columns.push(new Column({
                width: 5,
                extractDataFunction: function (val) {return val.id;},
                columnType: Column.Checkbox
            }));
        }

        columns.push(new Column({
            dbColumn: 'r.name',
            columnName: "Resource Name",
            width: 36,
            extractDataFunction: function (val) {return val.resource;},
            defaultSortOrder: Column.ASC,
            filter: {
                name: "resourceName",
                onFilter: function (value) {
                    reloadResourcesData(1);
                }
            }
        }));

        columns.push(new Column({
            dbColumn: 'r.resourceType',
            columnName: "Resource Type",
            width: 20,
            extractDataFunction: function (val) {return val.resourceType;},
            filter: {
                name: "resourceType",
                onFilter: function (value) {
                    reloadResourcesData(1);
                }
            }
        }));

        columns.push(new Column({
            dbColumn: 'rs.sublocation.name',
            columnName: "Sub-Location",
            width: 20,
            extractDataFunction: function (val) {return val.sublocation;},
            filter: {
                name: "resourceSublocation",
                onFilter: function (value) {
                    reloadResourcesData(1);
                }
            }
        }));

        columns.push(new Column({
            dbColumn: 'rs.active',
            columnName: "Status",
            width: 12,
            extractDataFunction: function (val) {return val.active ? "Active" : "Inactive";}
        }));

        if (!bulkActivateMode) {
            columns.push(new Column({
                width: 12,
                extractDataFunction: function () {return "More Actions";},
                columnType: Column.Hyperlink,
                rowElementHyperLink: function (val) {return "resource_handleResourceSelection(" + val.id + ")";}}));
        }

        resourceTable = new SchedulerTables({
            tableId: tableId,
            columns: columns,
            reloadFn: reloadResourcesData
        });
    };

    var setBulkActivateMode = function(bAMode)
    {
        bulkActivateMode = bAMode;
    };

    var isBulkActivateMode = function()
    {
        return bulkActivateMode;
    };

    var commonInit = function () {

        loadMetaHeaders();
        initFooter();
        eraseLicense();
    };

    var initDetailsFn = function () {
        resetUI();
        commonInit();
        commonData();
        resourceModuleRoles();
        app_selectedResource = JSON.parse(sessionStorage.getItem("resourceData"));
    };

    function commonResourceScreenData() {
        commonData();
        resourceModuleRoles();
        renderBreadcrumbs('resource_screen');
        resource_checkUnassignedResources();
        initial_load = true;
        loadResourcesData({currentPage: 1, maxResults:100});
    }

    function initSubLocationAddition() {

        commonInit();
        getResourceStaticLists(createSublocationClosureData);

        app_runIdleTimer();
    }

    // TODO-XH: Is this for a standalone page or for resource_form.html?
    // if for resource_form.html then remove call to getResourceStaticLists()
    function initSublocationClosure() {
        getResourceStaticLists();
        commonInit();
        app_runIdleTimer();
        sublocationClosureScreenData();
    }

    function getResourceTable()
    {
        return resourceTable;
    }

    return {
        init: initFn,
        initDetails: initDetailsFn,
        initSublocationClosure: initSublocationClosure,
        initSublocationAddition: initSubLocationAddition,
        initResourceTable: initResourceTable,
        getResourceTable: getResourceTable,
        setBulkActivateMode: setBulkActivateMode,
        isBulkActivateMode: isBulkActivateMode
    };

}());


function resourceModuleRoles() {
    user = JSON.parse(sessionStorage.getItem("userData"));
    var roleId = user.institutionRole.id;
    var visibility = {};
    if (UserRoleUtil.userIsSuperAdminOrResourceManager()) {
        visibility = {
            "resource_bulkActivate": "visible",
            "resource_addNewResource": "visible",
            "resource_sublocationClosure": "visible",
            "resource_editResource": "visible",
            "resource_addDefault": "visible",
            "resource_addAlternate": "visible",
            "resource_addTemporary": "visible"
        };
    } else if (UserRoleUtil.userIsScheduler()) {
        visibility = {
            "resource_bulkActivate": "hidden",
            "resource_addNewResource": "hidden",
            "resource_sublocationClosure": "visible",
            "resource_editResource": "hidden",
            "resource_addDefault": "hidden",
            "resource_addAlternate": "hidden",
            "resource_addTemporary": "hidden"
        };
    } else {
        // i.e. STUDY_STAFF, FRONT_DESK, GENERAL_VIEW, or anything else
        visibility = {
            "resource_bulkActivate": "hidden",
            "resource_addNewResource": "hidden",
            "resource_sublocationClosure": "hidden",
            "resource_editResource": "hidden",
            "resource_addDefault": "hidden",
            "resource_addAlternate": "hidden",
            "resource_addTemporary": "hidden"
        };
    }

    // TODO: stop using 'visibility' completely for the following 3 classes:
    // resource_bulkActivate
    // resource_addNewResource
    // resource_sublocationClosure
    $('.resource_bulkActivate')
        .css({
            visibility: visibility.resource_bulkActivate,
            display: visibility.resource_bulkActivate == 'visible' ? 'inline' : 'none'
        });
    $('.resource_addNewResource')
        .css({
            visibility: visibility.resource_addNewResource,
            display: visibility.resource_addNewResource == 'visible' ? 'inline' : 'none'
        });
    $('.resource_sublocationClosure')
        .css({
            visibility: visibility.resource_sublocationClosure,
            display: visibility.resource_sublocationClosure == 'visible' ? 'inline' : 'none'
        });
    // TODO: get rid of this -- .resource_editResource no long used, except in test
    $('.resource_editResource')
        .css({
            visibility: visibility.resource_editResource
        });
    // TODO: get rid of this -- .resource_editResource no long used, except in test
    $('.resource_addDefault')
        .css({
            visibility: visibility.resource_addDefault
        });
    // TODO: get rid of this -- .resource_editResource no long used, except in test
    $('.resource_addAlternate')
        .css({
            visibility: visibility.resource_addAlternate
        });
    // TODO: get rid of this -- .resource_editResource no long used, except in test
    $('.resource_addTemporary')
        .css({
            visibility: visibility.resource_addTemporary
        });

}

function sublocationClosureScreenData() {
    commonData();
    renderBreadcrumbs('resource_close_sub_list');
    setParentLocationHash("SublocationClosureInterval");
    resetUI();
    initial_load = true;
    loadSublocationClosureData(currentPage);
}

function createSublocationClosureData() {
    commonData();
    renderBreadcrumbs('resource_close_sub_form');
    sublocationClosureInterval_createWidgets();
    sublocation_clearForm();
}

function alternateResourceData(curriedDataLoadingFunction) {
    commonData();
    resetUI();
    initial_load = true;
    curriedDataLoadingFunction();
}

function cancelAddAlternateResource() {
    clearAlternateResourceSelection();
}

function clearAlternateResourceSelection() {
    $('#addAlternateResourcesData').multipleSelect("uncheckAll");
    clearAlternativesError();
    return false;
}

function editResource() {
    sessionStorage.setItem("mode", JSON.stringify('edit'));
    setLocationHref("resource_form.html");
}

function cancelResourceClick() {
    clearSelectedResource();
    resourceModule();
}

function addNewResourceClick() {
    if (NEW_RESOURCE_LINK_ENABLED == true) {
        sessionStorage.setItem("mode", JSON.stringify('new'));
        clearSelectedResource();
        setLocationHref("resource_form.html");
    }
}

function resource_checkUnassignedResources() {
    $.getJSON("rest/resource/getUnassignedResources", function (data) {
        var check = data.resources;
        if (check[0] != undefined) {
            $('#resource_newResourceLink').css({color: '#0A507D'});
            NEW_RESOURCE_LINK_ENABLED = true;
        }
        else {
            $('#resource_newResourceLink').css({color: '#AAAAAA'});
            NEW_RESOURCE_LINK_ENABLED = false;
        }
    });
}

function addSublocationClosureClick() {
    setLocationHref("add_sublocation_closure.html");
}


function sublocationClosureClick() {
    setLocationHref("sublocation_closure_screen.html");
}

function getResourceStaticLists(callback) {
    $.get("rest/app/getStaticLists", {}, function (data) {
        var parsedData = $.parseJSON(data);

        sublocations = parsedData.sublocations;
        sublocations.sort(function (a, b) {
            if (a.name > b.name) {
                return 1;
            }
            else if (b.name > a.name) {
                return -1;
            }
            return 0;
        });
        sublocationSelectOptions = buildSelectOptions(sublocations, 'name');

        $("#sublocation_sublocation").html(sublocationSelectOptions);

        WidgetUtil.createCombobox("#sublocation_sublocation");

        $('#resourceSublocationSelect').html(buildSubLocCheckBoxes());
        createResourceCombobox("#resourceSublocationSelect", {onSelect: updateResourceNameSuffix});

        var resourceTypes = parsedData.resourceTypes;

        resourceNames = parsedData.resourceNames;

        if (!app_selectedResource) {
            $('#existingResourceSelect').html(buildSelectOptionsFromStringList(resourceNames));
            createResourceCombobox("#existingResourceSelect", {onSelect: updateResourceNameFromSelect});
            $('#resourceNameInput').keyup(clearExistingResourceName);
            resourceTypes.sort(function (a, b) {
                if (a > b) {
                    return 1;
                }
                else if (b > a) {
                    return -1;
                }
                return 0;
            });
            resourceTypeSelectOptions = buildSelectOptionsFromStringList(resourceTypes);
            $("#resourceTypeSelect").html(resourceTypeSelectOptions);
            createResourceCombobox("#resourceTypeSelect");
        }

        resources = parsedData.resources;
        var nonSharedResources = resources.filter(
            function (resource) {
                return resource.sharedResource === undefined || resource.sharedResource == 0 || resource.sharedResource == null
            });
        $('#sharedResourceSelect').html(buildIdAndNameMenu(nonSharedResources));
        WidgetUtil.createCombobox("#sharedResourceSelect", {
            width: 200
        });
        WidgetUtil.createCombobox("#restrictionTypeSelect", {
            width: 200,
            value: 'Gender Restriction',
            prompt: 'Gender Restriction'
        });

        var tmpl = $.templates("#nurseAnnotationsListTemplate");
        var renderedTemplate = tmpl.render({annotations: parsedData.nurseAnnotations});
        $('#nurseAnnotationList').html(renderedTemplate);

        var tmpl2 = $.templates("#nutritionAnnotationsListTemplate");
        var renderedTemplate2 = tmpl2.render({annotations: parsedData.nutritionAnnotations});
        $('#nutritionAnnotationList').html(renderedTemplate2);

        var tmpl3 = $.templates("#equipmentAnnotationsListTemplate");
        var renderedTemplate3 = tmpl3.render({annotations: parsedData.equipmentAnnotations});
        $('#equipmentAnnotationList').html(renderedTemplate3);

        var tmpl4 = $.templates("#roomAnnotationsListTemplate");
        var renderedTemplate4 = tmpl4.render({annotations: parsedData.roomAnnotations});
        $('#roomAnnotationList').html(renderedTemplate4);

        var tmpl5 = $.templates("#labAnnotationsListTemplate");
        var renderedTemplate5 = tmpl5.render({annotations: parsedData.labAnnotations});
        $('#labAnnotationList').html(renderedTemplate5);

        if (callback) {
            callback();
        }
    });
}

var sublocationClosureToDelete;
var sublocationId = [];
var resource_da_days = [];
var resource_daToRemove;
var resource_taToRemove;
var resource_raToRemove;
var resource_bulkActivate;
var resourceTableFilter;
var selected;


function resource_handleResourceSelection(id) {
    $.getJSON("rest/resource/getResourceDetail?resourceId=" + id, function (data) {
        sessionStorage.setItem("resourceData", JSON.stringify(data));
        app_selectedResource = JSON.parse(sessionStorage.getItem("resourceData"));
        sessionStorage.setItem("mode", JSON.stringify('view'));
        $.blockUI();
        setLocationHref("resource_form.html");
    });
}

function addAlternateDialog(alternateResourcesId, curriedDataLoadingFunction) {
    var finalList = alternateResourcesId.toString().split(',').map(Number);
    DialogsUtil.showConfirmationDialog("#add-alternate-dialog-confirm", {
        buttons: {
            "Yes": function () {
                var jsonData = JSON.stringify({resourceId: app_selectedResource.resourceId, alternatives: finalList});
                $.post("rest/resource/assignResourceAlternatives", {data: jsonData}, function (data) {
                    $('#resource_ra_responseLoading').css({visibility: "hidden"});
                    alternateResourceData(curriedDataLoadingFunction);
                    clearAlternateResourceSelection();
                });
                $(this).dialog("close");
            },
            "No": function () {
                $(this).dialog("close");
                return;
            }
        }
    });
}

function deleteTempAjustmentDialog(id) {
    DialogsUtil.showConfirmationDialog("#delete-adjustment-dialog-confirm", {
        buttons: {
            "Yes": function () {
                resource_taToRemove = id;
                $.getJSON("rest/resource/deleteTemporaryAdjustment?id=" + resource_taToRemove, function (data) {
                    if (data.result == true) {
                        $('#resource_deleteTempAdjLoading').css({visibility: "hidden"});
                        var confirmationMessage = 'Temporary Adjustment Deleted.';
                        util_showMainMessage(confirmationMessage);
                        currentPage = 1;
                        TemporaryAdjustment.loadTemporaryAdjustmentsData();
                    }
                    else {
                        var confirmationMessage = 'There was an error in deleting temporary adjustment.';
                        util_showMainMessage(confirmationMessage);
                        return;
                    }
                });
                $(this).dialog("close");
            },
            "No": function () {
                $(this).dialog("close");
                return;
            }
        }
    });
}

function deleteDefaultDialog(id) {
    var departureWarning = '';
    if (!isCleanOfEditsOrAdd()) {
        departureWarning = "You will lose unsaved edits and additions."
    }
    var message = "Are you sure you want to delete this Default Availability? " + departureWarning;

    DialogsUtil.showMessageDialogWithOkFn(message,
        function () {
            resource_daToRemove = id;
            $.getJSON("rest/resource/deleteDefaultSchedule?id=" + resource_daToRemove, function (data) {
                if (data.result == true) {
                    $('#resource_deleteDefaultScheduleLoading').css({visibility: "hidden"});
                    defaultAvailabilityPagingWidget.decrRecordsOnCurrentPage();
                    defaultAvailabilityPagingWidget.applyCurriedDataLoader(loadDefaultAvailabilityData);
                }
                else {
                    var confirmationMessage = 'There was an error in deleting default availability.';
                    util_showMainMessage(confirmationMessage);
                    return;
                }
            });
        },
        "Yes",
        "No",
        ""
    );
}

function deleteAlternateDialog(id, curriedDataLoadingFunction) {
    DialogsUtil.showConfirmationDialog("#delete-alternate-dialog-confirm", {
        buttons: {
            "Yes": function () {
                resource_raToRemove = id;
                $.getJSON("rest/resource/removeResourceAlternative?resourceId=" + app_selectedResource.resourceId + "&alternateResourceId=" + resource_raToRemove, function (data) {
                    if (data.result == true) {
                        $('#resource_removeAlternativeLoading').css({visibility: "hidden"});
                        curriedDataLoadingFunction();
                    }
                    else {
                        $(this).dialog("close");
                        var confirmationMessage = 'There was an error in removing the alternate resource.';
                        util_showMainMessage(confirmationMessage);
                    }
                });
                $(this).dialog("close");
            },
            "No": function () {
                $(this).dialog("close");
            }
        }
    });
}

function deleteRestrictionDialog(resourceId, dataLoadingFunction, postRenderRestrictionsTable) {
    DialogsUtil.showConfirmationDialog("#delete-restriction-dialog-confirm", {
        buttons: {
            "Yes": function () {
                $(this).dialog("close");
                $.getJSON("rest/resource/removeRestriction?resourceId=" + resourceId, function (data) {
                    if (data.result) {
                        // reload resource from server
                        $.getJSON("rest/resource/getResourceDetail?resourceId=" + resourceId, function (data) {
                            sessionStorage.setItem("resourceData", JSON.stringify(data));
                            app_selectedResource = JSON.parse(sessionStorage.getItem("resourceData"));
                            dataLoadingFunction(app_selectedResource, postRenderRestrictionsTable);
                        });
                    }
                    else {
                        alert("An internal error occurred");
                    }
                });
            },
            "No": function () {
                $(this).dialog("close");
            }
        }
    });
}

function deleteSublocationDialog(id) {
    DialogsUtil.showConfirmationDialog("#delete-sublocation-dialog-confirm", {
        buttons: {
            "Yes": function () {
                sublocationClosureToDelete = id;
                jsonData = JSON.stringify({
                    sublocationClosureIntervalId: sublocationClosureToDelete,
                    userId: user.id
                });

                $.post("rest/resource/deleteSublocationClosureInterval", {data: jsonData}, function (data) {
                    $('#sublocationLoading').css({visibility: "hidden"});
                    var confirmationMessage = 'Sublocation Closure Deleted.';
                    util_showMainMessage(confirmationMessage);
                    currentPage = 1;
                    loadSublocationClosureData(currentPage);
                });
                $(this).dialog("close");
            },
            "No": function () {
                $(this).dialog("close");
                return;
            }
        }
    });
}

function resource_assignAlternatives(curriedDataLoadingFunction) {
    // TODO: convert to a list of integers, not strings
    var selectedAlternateResourceValues = $("#addAlternateResourcesData").multipleSelect('getSelects');
    if (selectedAlternateResourceValues.length === 0) {
        showAlternativesError();
    }
    else {
        addAlternateDialog(selectedAlternateResourceValues, curriedDataLoadingFunction);
        clearAlternativesError();
    }
}

function showAlternativesError() {
    $('#addAlternateResourceDataValidationDiv').css({display: "inline-block"});
    showErrorInValidationSpan('#addAlternateResourceDataValidation',
        "Please choose a resource.");
}
function clearAlternativesError() {
    $('#addAlternateResourceDataValidationDiv').css({display: "none"});
}

//////////////////////Resource directory/////////////////////////////////////////////////////

var sortByResourceName = "r.name";
var sortByResourceType = "r.resourceType";
var sortByResourceSublocation = "rs.sublocation.name";
var sortByResourceSublocationActive = "rs.active";
var columns;

function loadResourcesData(args) {
    args.callback = displayResourcesData;
    getResourcesData(args);
}

function displayResourcesData(data, maxResults) {

    resourcePage.initResourceTable();

    //TODO: remove resourceTableControls as an argument
    resourcePage.getResourceTable().generateTable(data, maxResults, "resourceTableControls",
                                                    function(currentPage, recordsPerPage)
                                                    {
                                                        reloadResourcesData(currentPage);
                                                    });
}

function reloadResourcesData(currentPage) {

    var table = resourcePage.getResourceTable();

    var resourceStatus = $("#stateSelector").val();
    currentPage = currentPage || table.getCurrentPage();

    var args = {
        currentPage: currentPage,
        maxResults: table.getRecordsPerPage(),
        searchFilter: table.getAllFilterKeyValuePairs(),
        isBulkActivate: false,
        status: resourceStatus,
        orderBy: table.getSortOrder(),
        sortColumn: table.getSortColumn(),
        callback: function(data) {
            table.refreshTableBody(data);
        }
    };

    getResourcesData(args);
}

/*
 params: {
    currentPage
    maxResults,
    searchFilter,
    status,
    orderBy,
    sortColumn,
    callback
 }*/
function getResourcesData(args)
{
    if(!args) {
        args = {};
    }
    var searchQuery = {};

    args.searchFilter = args.searchFilter || [];
    args.maxResults = args.maxResults || 100;
    args.status = args.status || "active";
    args.orderBy = args.orderBy || "ASC";
    args.sortColumn = args.sortColumn || sortByResourceName;

    initial_load = false;

    searchQuery.searchItems = args.searchFilter;

    searchQuery = JSON.stringify(searchQuery);

    var getResourcesDataUrl = "rest/resource/getResourcesData?page=" + (args.currentPage != undefined ? args.currentPage : 1)
        + "&maxResults=" + args.maxResults
        + "&orderBy=" + args.orderBy
        + "&sortBy=" + args.sortColumn
        + "&status=" + args.status
        + "&search=" + encodeURIComponent(searchQuery);

    $.getJSON(getResourcesDataUrl, function (data) {
        if(args.callback) {
            args.callback(data, args.maxResults, args.sortColumn, args.orderBy);
        }
    });
}

function bulkActivateResourcesCall() {
    var anArray = resourcePage.getResourceTable().getSelectedRows();
    if (anArray.length !== 0) {
        DialogsUtil.showConfirmationDialog("#bulkActivateDialog", {
            buttons: {
                "Yes": function () {
                    jsonData = JSON.stringify({resourcesIds: anArray});
                    $.post("rest/resource/activateResources", {data: jsonData}, function (data) {
                        var parsedData = $.parseJSON(data);
                        var confirmationMessage = "";
                        if (parsedData.result == true) {
                            confirmationMessage = "Success. Your resources have been activated.";
                            util_showMainMessage(confirmationMessage);
                            resourcePage.getResourceTable().clearAllSelectedRows();
                            updateResourceView("inactive", true);
                        }
                        else {
                            confirmationMessage = parsedData.errorMsg;
                            util_showMainMessage(confirmationMessage);
                        }
                    });
                    $(this).dialog("close");
                    clearBulkActivateErrors();
                },
                "No": function () {
                    clearBulkActivateErrors();
                    $(this).dialog("close");
                }
            }
        });
    }
    else {
        $('#bulkActivateResourceValidation').text('Please select at least one resource.');
        $('#bulkActivateResourceValidation').css({opacity: 0.0, visibility: "visible"}).animate({opacity: 1.0});
    }
}

function clearResourceFilters() {
    resourcePage.getResourceTable().clearAllFilterKeyValuePairs();
}

function updateResourceView(activeFilter, isBulkActivate) {
    if (isBulkActivate === true) {
        clearResourceFilters();

        $("#resourceState").css({visibility: "hidden"});
        $("#resourceBulkActivate").css({display: "none"});
        $("#resourceCancelBulkActivate").css({display: "inline-block"});
        $("#resourceSaveBulkActivate").css({display: "inline-block"});
        SchedulerTables.clearSelectedCheckboxes("resourceTable");
    }
    else {
        $("#resourceState").css({visibility: "visible"});
        $("#resourceBulkActivate").attr("onclick", "javascript:bulkActivateResources();");
        $("#resourceBulkActivate span").html("Bulk Activate");
        $("#resourceBulkActivate").css({display: "inline-block"});
        $("#resourceCancelBulkActivate").css({display: "none"});
        $("#resourceSaveBulkActivate").css({display: "none"});
        clearBulkActivateErrors();
        SchedulerTables.clearSelectedCheckboxes("resourceTable");
    }
    $("#stateSelector").val(activeFilter);
    var currentPage = 1;
    reloadResourcesData(currentPage);
}

function bulkActivateResources() {
    clearResourceFilters();

    $("#resourceState").css({visibility: "hidden"});
    $("#resourceBulkActivate").css({display: "none"});
    $("#resourceCancelBulkActivate").css({display: "inline-block"});
    $("#resourceSaveBulkActivate").css({display: "inline-block"});
    SchedulerTables.clearSelectedCheckboxes("resourceTable");

    $("#stateSelector").val("inactive");
    var status = $("#stateSelector").val();

    resourcePage.setBulkActivateMode(true);

    loadResourcesData({
        currentPage:1,
        maxResults: 100,
        status: status
    });
}

function clearBulkActivateErrors() {
    $('#bulkActivateResourceValidation').text('');
    $('#bulkActivateResourceValidation').css({visibility: "hidden"});
}

function cancelBulkActivateMode() {
    $("#resourceState").css({visibility: "visible"});
    $("#resourceBulkActivate span").html("Bulk Activate");
    $("#resourceBulkActivate").css({display: "inline-block"});
    $("#resourceCancelBulkActivate").css({display: "none"});
    $("#resourceSaveBulkActivate").css({display: "none"});
    clearBulkActivateErrors();
    SchedulerTables.clearSelectedCheckboxes("resourceTable");

    $("#stateSelector").val("active");
    var resourceStatus = $("#stateSelector").val();

    resourcePage.setBulkActivateMode(false);

    loadResourcesData({
        currentPage:1,
        maxResults: 100,
        status: resourceStatus
    });
}

function saveBulkActivateResources() {
    bulkActivateResourcesCall();
}

function changeResourceStatus(resourceData) {
    var resourceId = resourceData.id;
    var sublocationId = resourceData.sublocationId;
    var active = resourceData.active;
    $.getJSON("rest/resource/changeResourceStatus?resourceId=" + resourceId + "&sublocationId=" + sublocationId + "&active=" + active, function (parsedData) {
        var state = active ? "Active" : "Inactive";
        if (parsedData.result) {
            util_showMainMessage("Changed status of resource " + resourceData.resource + " located at " + resourceData.sublocation + " to " + state);
        }
        else {
            util_showMainMessage("Error: Failed to change status of resource " + resourceData.resource + " located at " + resourceData.sublocation);
        }
    });
}

///////////////////////////Default Availability/////////////////////////////////////////////////////////////////////

var sortByDayOfWeek = "da.dayOfWeek";
var sortByResourceStartTime = "da.startTime";
var sortByResourceEndTime = "da.endTime";
var sortByQuantity = "da.quantity";

var defaultAvailabilityCurrentPage;
var defaultAvailabilityRecordsPerPage;

var defaultAvailabilityPagingWidget;

function loadDefaultAvailabilityData(currentPage, recordsPerPage, paginationCallback) {

    defaultAvailabilityCurrentPage = currentPage;
    defaultAvailabilityRecordsPerPage = recordsPerPage;
    defaultAvailabilityEditingCount = 0;

    if (app_selectedResource) {
        loadDefaultAvailabilityDataForCurrentResource(paginationCallback);
    }
}

function loadDefaultAvailabilityDataForCurrentResource(paginationCallback) {
    $.getJSON("rest/resource/getResourceSchedules?resourceId=" +
        app_selectedResource.resourceId + "&page=" +
        (defaultAvailabilityCurrentPage != undefined ? defaultAvailabilityCurrentPage : 1) + "&maxResults=" +
        (defaultAvailabilityRecordsPerPage != undefined ? defaultAvailabilityRecordsPerPage : 25) +
        "&orderBy=" + (orderBy != undefined ? orderBy : "ASC") + "&sortBy=" +
        (sortBy != undefined ? sortBy : sortByDayOfWeek),
        function (data) {

            var editColumns = " ";

            var sort_header;
            var day_header1 = 'sortable';
            var start_header2 = 'sortable';
            var end_header3 = 'sortable';
            var quantity_header4 = 'sortable';

            if (orderBy == 'ASC') {
                sort_header = 'sorting_asc';
            }
            else if (orderBy == 'DESC') {
                sort_header = 'sorting_desc';
            }

            if (sortBy == undefined || sortBy == sortByDayOfWeek) {
                day_header1 = sort_header;
            } else if (sortBy == sortByResourceStartTime) {
                start_header2 = sort_header;
            } else if (sortBy == sortByResourceEndTime) {
                end_header3 = sort_header;
            } else if (sortBy == sortByQuantity) {
                quantity_header4 = sort_header;
            }

            var out =
                " <table id='defaultSchedule'>" +
                "  <tr>" +
                "   <td class= " + day_header1 + " onclick='defaultAvailabilitySortingFunction(sortByDayOfWeek)'><strong>Day Of Week</strong></td> " +
                "   <td class= " + start_header2 + " onclick='defaultAvailabilitySortingFunction(sortByResourceStartTime)'><strong>Start Time</strong></td>" +
                "   <td class= " + end_header3 + " onclick='defaultAvailabilitySortingFunction(sortByResourceEndTime)'><strong>End Time</strong></td>" +
                "   <td class= " + quantity_header4 + " onclick='defaultAvailabilitySortingFunction(sortByQuantity)'><strong>Quantity</strong></td>";

            if (UserRoleUtil.userIsSuperAdminOrResourceManager()) {
                out +=
                    "   <td class='not_sortable'><strong></strong></td>" +
                    "   <td class='not_sortable'><strong></strong></td>";
            }

            out += "</tr> ";

            var defaultAvailabilityTotalRows = 0;
            var daRowsThisPage = 0;

            $.each(data, function (key, val) {
                defaultAvailabilityTotalRows = val.totalCount;
                out += '<tr id="da_' + val.id + '">';
                out += htmlForReadOnlyDefaultAvailabilityRow(val);
                out += "</tr>";
                daRowsThisPage++;
            });
            defaultAvailabilityPagingWidget.setRecordsOnCurrentPage(daRowsThisPage);

            // row for adding a new default availability
            if (UserRoleUtil.userIsSuperAdminOrResourceManager()) {
                out += htmlForAddDefaultAvailabilityRow();

                // row for validation complaints
                out += '<tr id="addDaValidation">' +
                    '<td><span id="resource_da_dayValidation_add" class="formElementRequired"></span></td>' +
                    '<td><span id="resource_da_startTimeValidation_add" class="formElementRequired"></span></td>' +
                    '<td><span id="resource_da_endTimeValidation_add" class="formElementRequired"></span></td>' +
                    '<td><span id="resource_da_quantityValidation_add" class="formElementRequired"></span></td>' +
                    '<td/>' +
                    '<td/>' +
                    '</tr>';
            }

            out += " </table>";

            $('#defaultResourcesData').html(out);

            if (UserRoleUtil.userIsSuperAdminOrResourceManager()) {
                setupAddDefaultAvailabilityWidgets();
            }

            var remainder = defaultAvailabilityTotalRows % defaultAvailabilityRecordsPerPage;
            var defaultAvailabilityPageCount = 1 + Math.floor(defaultAvailabilityTotalRows / defaultAvailabilityRecordsPerPage);

            if (defaultAvailabilityCurrentPage > defaultAvailabilityPageCount) {
                defaultAvailabilityCurrentPage = defaultAvailabilityPageCount;
            }

            if (paginationCallback) {
                paginationCallback({
                    totalNumberOfPages: defaultAvailabilityPageCount
                });
            }
        });
}

function trForAddDefaultAvailabilityRow() {

    var dayPickerHtml = '<td><div id="rdda-div"></div>' +
        '<select id="resource_defaultAvailability_dayPicker_add" multiple="multiple"></select>' +
        '</div>' +
        '</td>';

    var startTimePickerHtml = '<td>' +
        '<input type="text" ' +
        'id="resource_default_availability_start_time_add" ' +
        'style="background:white;" ' +
        '/>' +
        '</td>\n';

    var endTimePickerHtml = '<td>' +
        '<input type="text" ' +
        'id="resource_default_availability_end_time_add" ' +
        'style="background:white;" ' +
        '/>' +
        '</td>\n';

    var editColumnsHtml =
        ' <td class="editColumn"> <a href="javascript:addDefaultAvailability()"> Add New </a> </td>\n' +
        ' <td class="editColumn"> <a href="javascript:cancelAddDefaultAvailability()"> Cancel </a> </td>\n';

    var quantityPickerHtml = "<td>" +
        '<input id="resource_default_availability_number_add" ' +
        'class="numberUpTo999" type="text" value="1" style="width: 30px;" ' +
        'autocomplete="off" role="spinbutton">' +
        '<span id="resource_da_quantityValidation" class="formElementRequired"></span>' +
        '</td>\n';

    var out = "";
    out += dayPickerHtml;
    out += startTimePickerHtml;
    out += endTimePickerHtml;
    out += quantityPickerHtml;
    out += editColumnsHtml;

    return out;
}

function htmlForAddDefaultAvailabilityRow() {

    var out = "<tr id='addDefaultAvailabilityRow'>";

    out += trForAddDefaultAvailabilityRow();

    out += "</tr>";

    return out;
}

function setupDefaultAvailabiltyQuantitySpinner(id) {
    var inputElement = $("#" + id);
    inputElement.spinner({
        "min": 0,
        "max": 999,
        "disabled": false
    });
}

function setupAddDefaultAvailabilityWidgets() {

    var dayPickerOptions = [];
    for (var i = 1; i <= 7; i++) {
        dayPickerOptions.push(day_option_for_multi_select(i));
    }

    setMultiSelectValues(
        "resource_defaultAvailability_dayPicker_add",
        dayPickerOptions,
        1,
        "178px", "-4px");

    setupDefaultAvailabiltyQuantitySpinner("resource_default_availability_number_add");

    WidgetUtil.createTimepicker('#resource_default_availability_start_time_add', {
        display59: true,
        disabled: false
    });
    $("#resource_default_availability_start_time_add").timepicker('setTime', '00:00');

    WidgetUtil.createTimepicker('#resource_default_availability_end_time_add', {
        display59: true,
        disabled: false
    });
    $("#resource_default_availability_end_time_add").timepicker('setTime', '00:00');

    clearAddErrors();
}

function htmlForReadOnlyDefaultAvailabilityRow(val) {
    var out = " 	   <td>" + util_day(val.dayOfWeek) + "</td>" +
        "          <td>" + val.startTime + "</td>" +
        "          <td>" + val.endTime + "</td>" +
        "          <td>" + val.quantity + "</td>";

    if (UserRoleUtil.userIsSuperAdminOrResourceManager()) {
        out +=
            " <td class='editColumn'> <a href='javascript:makeDefaultAvailabilityRowEditable(" + val.id + ")'> Edit </a> </td> " +
            " <td class='editColumn'> <a href='javascript:deleteDefaultConfirmation(" + val.id + ")'> Delete </a> </td> ";
    }
    return out;
}

// track how many rows in the default-availability table are waiting for save/cancel
var defaultAvailabilityEditingCount = 0;
function incrementDefaultAvailabilityEditingCount() {
    defaultAvailabilityEditingCount++;
}

function decdecrementDefaultAvailabilityEditingCount() {
    if (defaultAvailabilityEditingCount > 0) {
        defaultAvailabilityEditingCount--;
    }
}

function isDefaultAvailabilityAddRowPristine() {
    var addRow = $("#addDefaultAvailabilityRow");
    if (addRow[0] === undefined) {
        return true;
    }

    var selectedDays = $("#resource_defaultAvailability_dayPicker_add").multipleSelect('getSelects');

    var resource_da_startTimeVal = $('#resource_default_availability_start_time_add').timepicker().val();
    var resource_da_endTimeVal = $('#resource_default_availability_end_time_add').timepicker().val();
    var resource_da_quantity = $("#resource_default_availability_number_add").spinner("value");

    return (        selectedDays.length === 0
        && resource_da_quantity == 1
        && resource_da_endTimeVal == "00:00"
        && resource_da_startTimeVal == "00:00"
    );
}
function isCleanOfEditsOrAdd() {
    return isDefaultAvailabilityAddRowPristine() && defaultAvailabilityEditingCount == 0;
}

function undefinedIFFclean() {
    return isCleanOfEditsOrAdd() ? undefined : "There are unsaved changes!";
}

function makeDefaultAvailabilityRowEditable(id) {
    $.getJSON("rest/resource/getDefaultSchedule?id=" + id, function (val) {

        app_selectedResourceSchedule = val;

        var dayPickerOptions = "";
        for (var i = 1; i <= 7; i++) {
            dayPickerOptions += day_option_maybe_selected(i, val.dayOfWeek);
        }

        var elementId = "resource_defaultAvailability_dayPicker_" + id;
        var dayPickerHtml = '<td>' +
            '<select id="' + elementId + '">' +
            dayPickerOptions +
            '</select>' +
            '<span id="resource_da_dayValidation' + id + '" class="formElementRequired"></span>' +
            '</td>\n';

        var startTimePickerHtml = '<td>' +
            '<input value="' + showStandardTime(val.startTime) + '"\n' +
            'type="text" ' +
            'id="resource_default_availability_start_time_' + id + '" ' +
            'style="background:white;" ' +
            'onclick="this.select()" />' +
            '<span id="resource_da_startTimeValidation_' + id + '" class="formElementRequired"></span>' +

            '</td>\n';

        var endTimePickerHtml = '<td>' +
            '<input value="' + showStandardTime(val.endTime) + '"\n' +
            'type="text" ' +
            'id="resource_default_availability_end_time_' + id + '" ' +
            'style="background:white;" ' +
            'onclick="this.select()" />' +
            '<span id="resource_da_endTimeValidation_' + id + '"" class="formElementRequired"></span>' +
            '</td>\n';

        var editColumnsHtml =
            " <td class='editColumn'> <a href='javascript:saveDefaultAvailability(" + val.id + ", \"edit\")'> Save </a> </td> " +
            " <td class='editColumn'> <a href='javascript:cancelDefaultAvailabilityEdit(" + val.id + ")'> Cancel </a> </td> ";

        var quantityPickerHtml = "<td>" +
            '<input id="resource_default_availability_number_' + id + '" value="' + val.quantity +
            '" class="numberUpTo999" type="text" value="1" style="width: 30px;" autocomplete="off" role="spinbutton">' +
            '<span id="resource_da_quantityValidation_' + id + '" class="formElementRequired"></span>' +
            '</td>\n';

        out = "";
        out += dayPickerHtml;
        out += startTimePickerHtml;
        out += endTimePickerHtml;
        out += quantityPickerHtml;
        out += editColumnsHtml;

        $('#da_' + id).html(out);

        setupDefaultAvailabiltyQuantitySpinner("resource_default_availability_number_" + id);
        WidgetUtil.createTimepicker('#resource_default_availability_start_time_' + id, {
            display59: true,
            disabled: false
        });
        $("#resource_default_availability_start_time_" + id).timepicker('setTime', val.startTime);

        WidgetUtil.createTimepicker('#resource_default_availability_end_time_' + id, {
            display59: true,
            disabled: false
        });
        $("#resource_default_availability_end_time_" + id).timepicker('setTime', val.endTime);
    });

    incrementDefaultAvailabilityEditingCount();
}

// make the corresponding row in the Default Availability back to 'read-only' mode,
// with an edit button if the user has edit privilege
function cancelDefaultAvailabilityEdit(id) {
    makeDefaultAvailabilityReadOnlyRow(id);
    decdecrementDefaultAvailabilityEditingCount();
}

function makeDefaultAvailabilityReadOnlyRow(id) {
    $.getJSON("rest/resource/getDefaultSchedule?id=" + id, function (val) {
        $('#da_' + id).html(htmlForReadOnlyDefaultAvailabilityRow(val));
    });
}

function cancelAddDefaultAvailability() {
    $('#addDefaultAvailabilityRow').html(trForAddDefaultAvailabilityRow());
    setupAddDefaultAvailabilityWidgets();
}

function clearAddErrors() {
    $('#resource_da_startTimeValidation_add').css({visibility: "hidden"});
    $('#resource_da_endTimeValidation_add').css({visibility: "hidden"});
    $('#resource_da_dayValidation_add').css({visibility: "hidden"});
    $('#resource_da_quantityValidation_add').css({visibility: "hidden"});
    $('#addDaValidation').hide();
}

function addDefaultAvailability() {

    saveDefaultAvailability('add', 'add');
}

// saveMode: add or edit
function saveDefaultAvailability(id, saveMode) {

    clearAddErrors();

    var elementId = "#resource_defaultAvailability_dayPicker_" + id;
    var selectedDay = $(elementId).val();

    var isValid = true;

    resource_da_days = [];
    if (saveMode == 'edit') {
        resource_da_days.push(selectedDay);
    }
    else {
        var selectedDays = $("#resource_defaultAvailability_dayPicker_add").multipleSelect('getSelects');
        if (selectedDays.length === 0) {
            validationComplain('#resource_da_dayValidation_add', 'at least one day');
            isValid = false;
        }
        else {
            resource_da_days = selectedDays;
        }
    }

    var resource_da_startTimeVal = $('#resource_default_availability_start_time_' + id).timepicker().val();
    var resource_da_endTimeVal = $('#resource_default_availability_end_time_' + id).timepicker().val();
    var resource_da_quantity = $("#resource_default_availability_number_" + id).spinner("value");

    if (resource_da_startTimeVal == null) {
        validationComplain('#resource_da_startTimeValidation_' + id, 'a valid start time');
        isValid = false;
    }

    if (resource_da_endTimeVal == null) {
        validationComplain('#resource_da_endTimeValidation_' + id, 'a valid end time');
        isValid = false;
    }

    var resourceStart = (new Date(new Date(2008, 0, 1).toDateString() + ' ' + resource_da_startTimeVal));
    var resourceEnd = (new Date(new Date(2008, 0, 1).toDateString() + ' ' + resource_da_endTimeVal));

    if (resourceStart >= resourceEnd) {
        validationComplain('#resource_da_startTimeValidation_' + id, 'a valid time range');
        validationComplain('#resource_da_endTimeValidation_' + id, 'a valid time range');
        isValid = false;
    }

    // the spinner widget returns null if the field's text is blank
    if (resource_da_quantity === null) {
        validationComplain('#resource_da_quantityValidation_' + id, 'a valid quantity');
        isValid = false;
    }

    if (isValid == false) {
        return;
    }

    var saveAndReloadContinuation = function () {
        defaultAvailabilitySaveAndReload(saveMode, resourceStart, resourceEnd, resource_da_quantity, id);
    }

    if (saveMode == 'add' && defaultAvailabilityEditingCount > 0) {
        DialogsUtil.showMessageDialogWithOkCancelFns("Adding this item will cancel any " +
            "Default Availability edits in progress. OK to continue?",
            saveAndReloadContinuation);
    }
    else {
        saveAndReloadContinuation();
    }
}

function defaultAvailabilitySaveAndReload(saveMode, resourceStart, resourceEnd, resource_da_quantity, id) {

    if (id == 'add') {
        id = 0;
    }

    $('#resource_da_responseLoading').css({visibility: "visible"});
    jsonData = JSON.stringify({
        id: id,
        resourceId: app_selectedResource.resourceId,
        days: resource_da_days,
        startDate: resourceStart.valueOf(),
        endDate: resourceEnd.valueOf(),
        quantity: resource_da_quantity,
        override: false
    });

    var url = saveMode == 'add' ? "rest/resource/addDefaultAvailability" : "rest/resource/updateDefaultAvailability";

    $.post(url, {data: jsonData}, function (data) {
        $('#resource_da_responseLoading').css({visibility: "hidden"});
        var confirmationMessage = 'Default Availability ' + (saveMode == 'add' ? 'Added.' : 'Updated.');

        var onOk;
        if (saveMode == 'add') {
            onOk = function () {
                defaultAvailabilityPagingWidget.applyCurriedDataLoader(loadDefaultAvailabilityData);
            }
        }
        else {
            onOk = function () {
                makeDefaultAvailabilityReadOnlyRow(id);
                decdecrementDefaultAvailabilityEditingCount();
            }
        }
        DialogsUtil.showMessageDialog(confirmationMessage, onOk);
    });
}


function deleteDefaultConfirmation(id) {
    deleteDefaultDialog(id);
}

function validationComplain(elementId, type) {
    $('#addDaValidation').show();

    $(elementId).css({display: 'block'});
    $(elementId).text('Please enter ' + type + '.');
    $(elementId).css({opacity: 0.0, visibility: "visible"}).animate({opacity: 1.0});
}

function defaultAvailabilitySortingFunction(sort_by) {
    basicSortBy(sort_by);

    defaultAvailabilityPagingWidget.applyCurriedDataLoader(loadDefaultAvailabilityData);
}

///////////////////////////Alternate Resources/////////////////////////////////////////////////////////////////////

var sortByAltResourceName = "ra.alternateResource.name";
var sortByAltResourceType = "ra.alternateResource.resourceType";
var sortByAltResourceSublocation = "rs.sublocation.name";

/**
 * Load the list of resources that are *currently* set as alternative resource
 * and ALSO populate the combo box for selecting new alternative resources
 */
function loadAlternateResourcesData(alternateResourcesCurrentPage, recordsPerPage, updatePaginationCallback) {

    var tableTemplate = $.templates("#alternateResourceTabTemplate");

    var url = "rest/resource/getResourceAlternates?resourceId=" +
        app_selectedResource.resourceId +
        "&page=" + (alternateResourcesCurrentPage != undefined ? alternateResourcesCurrentPage : 1) +
        "&maxResults=" + (recordsPerPage != undefined ? recordsPerPage : 5555555) +
        "&orderBy=" + (orderBy != undefined ? orderBy : "ASC") +
        "&sortBy=" + (sortBy != undefined ? sortBy : sortByAltResourceName);

    $.getJSON(url, function (dataWithCount) {

        // we played a trick there, to pass a list and an int along the wire.
        // see ResourceResource.getResourceAlternates()
        var totalResourceCount = dataWithCount.count;
        var data = JSON.parse(dataWithCount.data);

        var sort_header;
        var altResource_header1 = 'sortable';
        var altResourceType_header2 = 'sortable';
        var altSublocation_header3 = 'sortable';

        if (orderBy == 'ASC') {
            sort_header = 'sorting_asc';
        }
        else if (orderBy == 'DESC') {
            sort_header = 'sorting_desc';
        }

        if (sortBy == undefined || sortBy == sortByAltResourceName) {
            altResource_header1 = sort_header;
        } else if (sortBy == sortByAltResourceType) {
            altResourceType_header2 = sort_header;
        } else if (sortBy == sortByAltResourceSublocation) {
            altSublocation_header3 = sort_header;
        }

        var templateData = {
            resources: data,
            new: false,
            editable: (UserRoleUtil.userIsSuperAdminOrResourceManager()),

            altResource_header1: altResource_header1,
            altResourceType_header2: altResourceType_header2,
            altSublocation_header3: altSublocation_header3
        };
        var tableContent = tableTemplate.render(templateData);
        $('#alternateResourcesData').html(tableContent);

        var remainder = totalResourceCount % recordsPerPage;

        var resourcesPageCount = Math.floor(totalResourceCount / recordsPerPage);
        if (data.length == 0) {
            remainder = 0;
            resourcesPageCount = 1;
        }
        if (remainder > 0) {
            resourcesPageCount = resourcesPageCount + 1;
        }

        if (alternateResourcesCurrentPage > resourcesPageCount) {
            alternateResourcesCurrentPage = resourcesPageCount;
        }

        createAddAlternateResourcesComboBox();

        if (updatePaginationCallback) {
            updatePaginationCallback({
                totalNumberOfPages: resourcesPageCount
            });
        }
    });
}

function alternateResourcesSortingFunction(sort_by) {
    basicSortBy(sort_by);

    alternateResourcePagingWidget.applyCurriedDataLoader(loadAlternateResourcesData);
}

// This function is called from HTML code generated by javascript
function toggleAll(source) {
    var checkboxes = document.getElementsByName('case');
    for (var i = 0, n = checkboxes.length; i < n; i++) {
        checkboxes[i].checked = source.checked;
    }
}

///////////////////////////Sublocation Closure/////////////////////////////////////////////////////////////////////

var sortBySublocationClosure = "sci.sublocation.name";
var sortBySublocationStartTime = "sci.startTime";
var sortBySublocationEndTime = "sci.endTime";
var sortBySublocationReason = "sci.reason";

function loadSublocationClosureData() {

    SublocationClosureTable.create();
    SublocationClosureTable.populate();

}

function sublocationClosureInterval_createWidgets() {
    WidgetUtil.createDatetimepicker("#sublocation_startDate", {
        onSelect: function (selectedDate) {
            var endDate = new Date(selectedDate);
            endDate.setTime(endDate.getTime() + (15 * minuteMillis));

            $("#sublocation_endDate").datetimepicker('setDate', endDate);
        }
    });
    WidgetUtil.createDatetimepicker("#sublocation_endDate");

    sublocation_clearForm();
}

function createResourceCombobox(element, config) {
    if (!config) {
        config = {}
    }
    config.width = 350;
    WidgetUtil.createCombobox(element, config);
}

function clearSelectedResource() {
    sessionStorage.setItem("resourceData", null);
    app_selectedResource = null;
}

function modifyResource(jsonData, activate, callBack) {
    var parsedJsonData = $.parseJSON(jsonData);
    var resourceName = parsedJsonData.resourceName;
    var sublocationId = parsedJsonData.sublocationId;

    $.post("rest/resource/modifyResource", {data: jsonData}, function (data) {
        var parsedData = $.parseJSON(data);
        var confirmationMessage = "";
        if (parsedData.successful === true) {
            if (app_selectedResource.active == false && activate === true) {
                confirmationMessage = 'Success! This resource has been saved and activated.';
                $("#theActiveResource").css({display: "none"});
            }
            else {
                confirmationMessage = 'Success! This resource has been saved.';
            }
            resource_clearErrors();

            var sublocationData = lookUpComboboxDataByValue("#resourceSublocationSelect", sublocationId);
            var updatedResourceName = resourceName;
            updatedResourceName += " - " + sublocationData.text;

            var index = $.inArray(app_selectedResource.resource, resourceNames);
            resourceNames.splice(index, 1);
            app_selectedResource.resource = updatedResourceName;
            resourceNames.push(app_selectedResource.resource);

            sessionStorage.setItem("resourceData", JSON.stringify(app_selectedResource));
            $("#resource_selected_resource_name").text(updatedResourceName);

            util_showMainMessage(confirmationMessage);
        }
        else {
            confirmationMessage = parsedData.message;
            util_showMainMessage(confirmationMessage);
        }

        if (callBack !== undefined) {
            callBack();
        }
    });
}

/**
 *
 * @param params is expected to be of the form {active: true|false} when calling to create
 * a new subject. Otherwise left unspecified.
 */
function saveResourceClick(params) {
    resource_clearErrors();
    var isValid = true;
    var subLocationName = $("#resourceSublocationSelect").combobox("getText");
    var resourceName = $("#resourceNameInput").val();
    var fullResourceName = resourceName + " - " + subLocationName;
    if (resourceName.length < 1) {
        $('#resourceNameInputValidation').text('Please enter a Resource Name.');
        $('#resourceNameInputValidation').css({opacity: 0.0, visibility: "visible"}).animate({opacity: 1.0});
        isValid = false;
    }
    else if ($.inArray(fullResourceName, resourceNames) != -1) {
        if (app_selectedResource) {
            if (!(fullResourceName === app_selectedResource.resource)) {
                $('#resourceNameInputValidation').text('This resource name already exists.');
                $('#resourceNameInputValidation').css({opacity: 0.0, visibility: "visible"}).animate({opacity: 1.0});
                isValid = false;
            }
        }
        else {
            $('#resourceNameInputValidation').text('This resource name already exists.');
            $('#resourceNameInputValidation').css({opacity: 0.0, visibility: "visible"}).animate({opacity: 1.0});
            isValid = false;
        }
    }
    else if (/^[a-zA-Z0-9()/ ,&-]*$/.test(resourceName) == false) {
        $('#resourceNameInputValidation').text('This resource name contains illegal characters.');
        $('#resourceNameInputValidation').css({opacity: 0.0, visibility: "visible"}).animate({opacity: 1.0});
        isValid = false;
    }

    var resourceType;
    if (!app_selectedResource) {
        resourceType = $("#resourceTypeSelect").combobox("getValue");
        if (!resourceType) {
            $('#resourceTypeSelectValidation').text('Please select a Resource Type');
            $('#resourceTypeSelectValidation').css({opacity: 0.0, visibility: "visible"}).animate({opacity: 1.0});
            isValid = false;
        }
    }

    var sublocationId = $("#resourceSublocationSelect").combobox("getValue");
    if (!sublocationId) {
        $('#resourceSublocationSelectValidation').text('Please select a Sub-Location');
        $('#resourceSublocationSelectValidation').css({opacity: 0.0, visibility: "visible"}).animate({opacity: 1.0});
        isValid = false;
    }

    if (isValid == false) {
        return;
    }

    var activate = $("#activateResourceCheckbox").prop('checked');

    // FIXME: using string resourceName for new resource, and resourceId for existing resource
    var resourceId = '';

    // if app_selectedResource is populated then we are in edit mode
    if (app_selectedResource) {
        resourceId = app_selectedResource.resourceId;
        $('#resourceLoading').css({visibility: "visible"});
        jsonData = JSON.stringify({
            resourceId: resourceId,
            sublocationId: parseInt(sublocationId),
            resourceName: resourceName,
            activate: activate
        });

        if (activate === true) {
            DialogsUtil.showConfirmationDialog("#activateDialog", {
                buttons: {
                    "Yes": function () {
                        modifyResource(jsonData, activate);
                        $(this).dialog("close");
                    },
                    "No": function () {
                        $("#activateResourceCheckbox").prop('checked', false);
                        $(this).dialog("close");
                    }
                }
            });
        }
        else {
            modifyResource(jsonData, activate);
        }
    }
    // we are saving a new resource
    else {
        $('#resourceLoading').css({visibility: "visible"});
        jsonData = JSON.stringify({
            resourceName: resourceName,
            resourceType: resourceType,
            sublocationId: sublocationId,
            active: params.active
        });

        if (params.active === true) {
            DialogsUtil.showConfirmationDialog("#activateDialog", {
                buttons: {
                    "Yes": function () {
                        $.post("rest/resource/createResource", {data: jsonData}, function (data) {
                            var parsedData = $.parseJSON(data);
                            var confirmationMessage = "";
                            if (parsedData.result == true) {
                                var newResourceName = parsedData.name;
                                var newResourceId = parsedData.resourceId;
                                confirmationMessage = "Success! This resource has been created.";
                                // after creating a resource we go back to the form, this time in edit mode
                                // This makes it possible for the user to start adding default availabilities,
                                // alternate resources, and temp adjustments right away
                                resource_handleResourceSelection(newResourceId);
                                util_showMainMessage(confirmationMessage);
                            }
                            else {
                                confirmationMessage = parsedData.errorMsg;
                                util_showMainMessage(confirmationMessage);
                            }
                        });
                        $(this).dialog("close");
                    },
                    "No": function () {
                        $(this).dialog("close");
                    }
                }
            });
        }
        else {
            $.post("rest/resource/createResource", {data: jsonData}, function (data) {
                var parsedData = $.parseJSON(data);
                var confirmationMessage = "";
                if (parsedData.result == true) {
                    var newResourceName = parsedData.name;
                    var newResourceId = parsedData.resourceId;
                    confirmationMessage = "Success! This resource has been created.";
                    // after creating a resource we go back to the form, this time in edit mode
                    // This makes it possible for the user to start adding default availabilities,
                    // alternate resources, and temp adjustments right away
                    resource_handleResourceSelection(newResourceId);
                    util_showMainMessage(confirmationMessage);
                }
                else {
                    confirmationMessage = parsedData.errorMsg;
                    util_showMainMessage(confirmationMessage);
                }
            });
        }
    }

}

function cancelSublocationClick() {
    resourceModule();
}

function crudSublocationClick() {
    sublocation_clearErrors();
    var isValid = true;

    var sublocation_startDateVal = new Date($("#sublocation_startDate").val());
    var sublocation_endDateVal = new Date($("#sublocation_endDate").val());

    if ($("#sublocation_startDate").val() == '') {
        $('#sublocation_startDateValidation').css({display: 'inline'});
        $('#sublocation_startDateValidation').text('Please enter valid start date');
        $('#sublocation_startDateValidation').css({opacity: 0.0, visibility: "visible"}).animate({opacity: 1.0});
        isValid = false;
    }

    if ($("#sublocation_endDate").val() == '') {
        $('#sublocation_endDateValidation').css({display: 'inline'});
        $('#sublocation_endDateValidation').text('Please enter valid end date');
        $('#sublocation_endDateValidation').css({opacity: 0.0, visibility: "visible"}).animate({opacity: 1.0});
        isValid = false;
    }

    if (sublocation_startDateVal >= sublocation_endDateVal) {
        $('#sublocation_startDateValidation').css({display: 'inline'});
        $('#sublocation_startDateValidation').text('Please enter valid start date range');
        $('#sublocation_startDateValidation').css({opacity: 0.0, visibility: "visible"}).animate({opacity: 1.0});
        $('#sublocation_endDateValidation').css({display: 'inline'});
        $('#sublocation_endDateValidation').text('Please enter valid end date range');
        $('#sublocation_endDateValidation').css({opacity: 0.0, visibility: "visible"}).animate({opacity: 1.0});
        isValid = false;
    }

    if ($("#sublocation_sublocation").combobox("getValue").length < 1) {
        $('#sublocation_sublocationValidation').css({display: 'inline'});
        $('#sublocation_sublocationValidation').text('Required Field');
        $('#sublocation_sublocationValidation').css({opacity: 0.0, visibility: "visible"}).animate({opacity: 1.0});
        isValid = false;
    }

    if (isValid == false) {
        return;
    }


    $('#sublocationLoading').css({visibility: "visible"});

    jsonData = JSON.stringify({
        sublocationId: $('#sublocation_sublocation').combobox("getValue"),
        startTime: sublocation_startDateVal.valueOf(),
        endTime: sublocation_endDateVal.valueOf(),
        reason: $('#sublocation_reason').val(),
        userId: user.id
    });

    $.post("rest/resource/createSublocationClosureInterval", {data: jsonData}, function (data) {
        $('#sublocationLoading').css({visibility: "hidden"});
        var confirmationMessage = "Sublocation Closure Scheduled";
        util_showMainMessage(confirmationMessage);
        alert(confirmationMessage);
        window.location.href = "sublocation_closure_screen.html";
    });
}

function resource_clearErrors() {
    $('#resourceNameInputValidation').css({visibility: "hidden"});
    $('#resourceTypeSelectValidation').css({visibility: "hidden"});
    $('#resourceSublocationSelectValidation').css({visibility: "hidden"});
}

function resource_clearForm() {
    $('#resourceNameInput').text("");
    $('#resource_resourceType').combobox("clear");
    $('#resourceSublocationSelect').combobox("clear");
    sublocationId = [];
    resource_clearErrors();
}

function sublocation_clearErrors() {
    $('.formElementRequired').css({visibility: "hidden"});
}

function sublocation_clearForm() {
    $('#sublocation_sublocation').combobox("clear");
    var date = new Date();
    date.setHours(0, 0, 0, 0);
    var later15 = new Date(date.getTime() + (15 * minuteMillis));

    $("#sublocation_startDate").val(showDateTime(date));
    $("#sublocation_endDate").val(showDateTime(later15));
    $('#sublocation_reason').val('');
    sublocation_clearErrors();
}

function buildSubLocCheckBoxes() {
    var html = [];
    for (var i = 0, len = sublocations.length; i < len; i++) {
        html[html.length] = "<option value='";
        html[html.length] = sublocations[i].id;
        html[html.length] = "'>";
        html[html.length] = sublocations[i].name;
        html[html.length] = "</option>";
    }
    return html.join('');
}

function buildResourcesMenu() {
    var html = [];
    for (var i = 0, len = resources.length; i < len; i++) {
        html[html.length] = "<option value='";
        html[html.length] = resources[i].id;
        html[html.length] = "'>";
        html[html.length] = resources[i].name;
        html[html.length] = "</option>";
    }
    return html.join('');
}

function buildIdAndNameMenu(items) {
    var html = [];
    for (var i = 0, len = items.length; i < len; i++) {
        html[html.length] = "<option value='";
        html[html.length] = items[i].id;
        html[html.length] = "'>";
        html[html.length] = items[i].name;
        html[html.length] = "</option>";
    }
    return html.join('');
}

var sortByAlternateResourceName = "r.name";
// Get the list of *allowed* alternative resources for the resource with the specified ID (if specified)
function createAddAlternateResourcesComboBox() {
    // if a resource is currently selected (i.e. edit mode) then look for alternate resources
    // allowed for this resource
    if (app_selectedResource) {
        var url = "rest/resource/getAlternateResources?" +
            "resourceId=" + app_selectedResource.resourceId +
            "&page=1" +
            "&maxResults=1000000" +
            "&orderBy=ASC" +
            "&sortBy=" +
            sortByAlternateResourceName;
    }
    // otherwise, if no resource is currently selected (i.e. 'new' mode) then get the list of all active resources
    // since any of them can be set as alternative
    else {
        var url = "rest/resource/getResourcesData?" +
            "&page=1" +
            "&maxResults=1000000" +
            "&orderBy=ASC" +
            "&sortBy=" +
            sortByAlternateResourceName +
            "&status=active";
    }
    $.getJSON(url, function (data) {
        createAlternateResourceMultiSelect(data, "addAlternateResourcesData", 4);
    });
}

function createAlternateResourceMultiSelect(alternateResources, selectElementId, minimumCount) {
    var html = [];
    for (var i = 0, len = alternateResources.length; i < len; i++) {
        html[html.length] = "<option value='";
        html[html.length] = alternateResources[i].id;
        html[html.length] = "'>";
        html[html.length] = alternateResources[i].resource + " (" + alternateResources[i].resourceType + ")";
        html[html.length] = "</option>";
    }

    setMultiSelectValues(selectElementId, html, minimumCount, "268px");
}

function setMultiSelectValues(selectElementId, html, minimumCount, width) {
    $('#' + selectElementId).html(html.join('')).multipleSelect({
        placeholder: "choose",
        width: width,
        allSelected: "All Selected",
        maxHeight: 200,
        minimumCountSelected: minimumCount,
    });
}

/*
 * an onchange callback triggered when the sub-location selection changes in the resource form.
 * It modifies the #resourceNameSuffix span to display the currently selected value of the sublocation combo box
 */
function updateResourceNameSuffix(record) {

    var resourceNameSuffix = " - " + record.text;
    $('#resourceNameSuffix').text(resourceNameSuffix);

}

/**
 * an onchange callback triggered when the existing resource selection changes in the resource form.
 * It modifies the resource name text input field
 */
function updateResourceNameFromSelect(record) {

    var existingResourceName = "" + record.text;
    for (var i = 0; i < sublocations.length; i++) {
        if (strEndsWith(existingResourceName, ' - ' + sublocations[i].name) === true) {
            existingResourceName = existingResourceName.substring(0, existingResourceName.lastIndexOf(' - ' + sublocations[i].name));
            break;
        }
    }
    if (existingResourceName) {
        $('#resourceNameInput').val(existingResourceName);
    }

}

/**
 * An onchange callback which resets the existing-resource-name select based on the
 * manually entered resource name
 */
function clearExistingResourceName() {
    var newResourceName = $('#resourceNameInput').val();
    var existingResourceName = $('#existingResourceSelect').combobox('getValue');
    if (newResourceName != existingResourceName) {
        $('#existingResourceSelect').combobox('setValue', null);
    }
}

/**
 * This function does one thing only: populate the #mainResourceForm div with the top-part
 * of the resource form (type, sublocation, name)
 */
function createMainResourceForm(existingResourceNames) {
    var tableTemplate = $.templates("#mainFormTemplate");
    var templateData = {
        new: app_selectedResource == null,
        inactive: app_selectedResource && !app_selectedResource.active,
        existingResourceNames: existingResourceNames,
        editable: (UserRoleUtil.userIsSuperAdminOrResourceManager())
    };
    var tableContent = tableTemplate.render(templateData);

    $('#mainResourceForm').html(tableContent);
}


/**
 * Expand/Collapse function custom-made for the resource form. Inspired by
 * expandOrCollapseSubject() in subject.js
 */

function expandOrCollapseResourceFormSection(headingId, imageId, sectionId) {
    if ($('#' + headingId).attr('title') == 'Expand') {
        displayResourceFormSection(headingId, imageId, sectionId);
    } else {
        hideResourceFormSection(headingId, imageId, sectionId);
    }

    if (checkIfAllBlocksCollapsed() === true) {
        collapseAllHelper();
    }
    else {
        expandAllHelper();
    }
}

function displayResourceFormSection(headingId, imageId, sectionId) {
    $('#' + headingId).attr('title', 'Collapse').css({cursor: 'pointer'});
    $('#' + imageId).attr('src', 'css/images/sm_circle_minus.png');
    $('#' + sectionId).css({display: 'block', cursor: 'pointer'});
}

function hideResourceFormSection(headingId, imageId, sectionId) {
    $('#' + headingId).attr('title', 'Expand').css({cursor: 'pointer'});
    $('#' + imageId).attr('src', 'css/images/sm_circle_plus.png');
    $('#' + sectionId).css({display: 'none', cursor: 'default'});
}

function collapseAllHelper() {
    document.getElementById('toggleAllResourceDetails').title = "Expand";
    document.getElementById("toggleText").innerHTML = "Expand All";
    $('#toggleResourceDetailsImg').attr('src', 'css/images/arrows_expand.png');
}

function expandAllHelper() {
    document.getElementById('toggleAllResourceDetails').title = "Collapse";
    document.getElementById("toggleText").innerHTML = "Collapse All";
    $('#toggleResourceDetailsImg').attr('src', 'css/images/arrows_collapse.png');
}

function toggleAllResourceDetails(event, obj) {
    preventDefaultAction(event);

    if (obj.title == "Collapse") {
        collapseAllHelper();
        hideAllResourceForms();
    }
    else {
        expandAllHelper();
        displayAllResourceForms();
    }
}

function hideAllResourceForms() {
    for (var i = 0; i <= 5; i++) {

        var headerId = 'resourceTabHeading' + i;
        var imageId = 'resourceTabExpandIcon' + i;
        var sectionId = ((i !== 3) ? 'resourceTab' + i : 'tempAdjustmentContent');

        hideResourceFormSection(headerId, imageId, sectionId);
    }
}

function displayAllResourceForms() {
    for (var i = 0; i <= 5; i++) {

        var headerId = 'resourceTabHeading' + i;
        var imageId = 'resourceTabExpandIcon' + i;
        var sectionId = ((i !== 3) ? 'resourceTab' + i : 'tempAdjustmentContent');

        displayResourceFormSection(headerId, imageId, sectionId);
    }
}

function checkIfAllBlocksCollapsed() {
    var isAllCollapsed = true;
    for (var i = 1; i <= 5; i++) {
        var headerId = 'resourceTabHeading' + i;
        if ($('#' + headerId).attr('title') == 'Collapse') {
            isAllCollapsed = false;
        }
    }
    return isAllCollapsed;
}

function clearRestrictionForm() {

    // Because there is currently only one kind of restriction, pre-set the restriction type menu to that unique gender restriction ("Gender Restriction")
    $("#restrictionTypeSelect").combobox('setValue', "Gender Restriction");
    $("#sharedResourceSelect").combobox('setValue', null);
    WidgetUtil.commentBox.clearValue("#restrictionNotesInput");
    $("#restrictionNotesInput").attr('style', '');

    clearRestrictionValidation();

}

function clearRestrictionValidation() {

    $('#restrictionTypeSelectValidation').val('');
    $('#sharedResourceSelectValidation').val('');

    $('#restrictionTypeSelectValidation').css({visibility: "hidden"});
    $('#sharedResourceSelectValidation').css({visibility: "hidden"});
}


function validateResourceRestriction() {
    var isValid = true;

    var sharedResourceId = $("#sharedResourceSelect").combobox('getValue');
    if (!sharedResourceId) {
        $('#sharedResourceSelectValidation').text('Please select associated resource');
        $('#sharedResourceSelectValidation').css({opacity: 0.0, visibility: "visible"}).animate({opacity: 1.0});
        isValid = false;
    }

    return isValid;
}

function addResourceRestriction(loadRestrictionsData, postRenderRestrictionsTable) {

    clearRestrictionValidation();

    var restrictionTypeId = $("#restrictionTypeSelect").combobox('getValue');
    if (!restrictionTypeId) {
        $("#restrictionTypeSelect").combobox('setValue', "Gender Restriction");
        restrictionTypeId = "Gender Restriction";
    }

    if (!validateResourceRestriction()) {
        return;
    }

    var restrictionNotes = WidgetUtil.commentBox.getValue("#restrictionNotesInput");
    var sharedResourceId = $("#sharedResourceSelect").combobox('getValue');

    var jsonData = JSON.stringify({
        resourceId: app_selectedResource.resourceId,
        sharedResourceId: sharedResourceId,
        notes: restrictionNotes
    });

    $.post("rest/resource/addRestriction", {data: jsonData}, function (data) {
        var parsedData = $.parseJSON(data);
        handleUpdateRestriction(parsedData, loadRestrictionsData, postRenderRestrictionsTable)
    });
}

function handleUpdateRestriction(parsedData, loadRestrictionsData, postRenderRestrictionsTable) {
    var confirmationMessage = "";
    if (parsedData.result == true) {
        confirmationMessage = app_selectedResource.resource + ' has been updated.';
    }
    else {
        confirmationMessage = "An internal error prevented " + app_selectedResource.resource + " from being updated";
    }
    util_showMainMessage(confirmationMessage);
    $.getJSON("rest/resource/getResourceDetail?resourceId=" + app_selectedResource.resourceId, function (data) {
        sessionStorage.setItem("resourceData", JSON.stringify(data));
        app_selectedResource = JSON.parse(sessionStorage.getItem("resourceData"));
        loadRestrictionsData(app_selectedResource, postRenderRestrictionsTable);
    });
}

function modifyResourceRestriction(loadRestrictionsData, postRenderRestrictionsTable) {

    clearRestrictionValidation();

    var restrictionNotes = WidgetUtil.commentBox.getValue("#restrictionNotesInput");

    var jsonData = JSON.stringify({
        resourceId: app_selectedResource.resourceId,
        notes: restrictionNotes
    });

    $.post("rest/resource/modifyRestriction", {data: jsonData}, function (data) {
        var parsedData = $.parseJSON(data);
        handleUpdateRestriction(parsedData, loadRestrictionsData, postRenderRestrictionsTable);
    });
}

function loadRestrictionsData(resource, callback) {

    // This ajax call is common to both scenarios (with or without shared resource)
    $.getJSON("rest/resource/getResourcesAvailableForGenderBlockRestriction?resourceId=" + resource.resourceId, function (data) {
        resource.sharableResources = data.resources;
        // If there is a shared resource, load it and add its data to the resource object, then render the restrictions table
        if (resource.sharedResourceId) {
            $.getJSON("rest/resource/getResourceDetail?resourceId=" + resource.sharedResourceId, function (data) {
                resource.sharedResource = data;
                renderRestrictionsTable(resource, callback);
            });
        }
        // otherwise, just render the restriction table
        else {
            resource.sharedResource = null;
            renderRestrictionsTable(resource, callback);
        }
    });


}

function renderRestrictionsTable(resource, callback, editing) {

    var tableTemplate = $.templates("#restrictionsTabTemplate");
    var templateData;
    if (resource.sharedResource) {
        templateData = {
            restricted: true,
            resourceId: resource.resourceId,
            sharedResourceId: resource.sharedResourceId,
            editable: (UserRoleUtil.userIsSuperAdminOrResourceManager()),
            editing: editing,
            sharableResources: resource.sharableResources,
            associatedResourceName: resource.sharedResource.resource,
            notes: resource.sharedResource.sharedResourceNotes
        };
    }
    else if (resource.resourceType == 'Room') {
        templateData = {
            restricted: false,
            restrictable: true,
            editable: (UserRoleUtil.userIsSuperAdminOrResourceManager()),
            sharableResources: resource.sharableResources
        };
    }
    else {
        templateData = {
            restricted: false,
            restrictable: false
        }
    }

    var tableContent = tableTemplate.render(templateData);
    $('#restrictionsData').html(tableContent);

    if (callback) {
        callback();
    }
}
