/*
 * Decompiled with CFR 0.152.
 */
package org.spin.tools.crypto;

import java.io.ByteArrayInputStream;
import java.io.IOException;
import java.io.InputStream;
import java.math.BigInteger;
import java.net.URI;
import java.security.InvalidKeyException;
import java.security.KeyStore;
import java.security.KeyStoreException;
import java.security.Principal;
import java.security.PrivateKey;
import java.security.PublicKey;
import java.security.Signature;
import java.security.SignatureException;
import java.security.cert.Certificate;
import java.security.cert.CertificateEncodingException;
import java.security.cert.CertificateException;
import java.security.cert.CertificateFactory;
import java.security.cert.X509Certificate;
import java.util.Enumeration;
import java.util.List;
import java.util.Map;
import java.util.Set;
import javax.crypto.Cipher;
import org.apache.log4j.Logger;
import org.spin.tools.Base64Codec;
import org.spin.tools.Util;
import org.spin.tools.config.ConfigException;
import org.spin.tools.config.ConfigTool;
import org.spin.tools.config.KeyStoreConfig;
import org.spin.tools.config.KeyStoreType;
import org.spin.tools.crypto.BadSignatureException;
import org.spin.tools.crypto.CryptoException;
import org.spin.tools.crypto.PKCryptor;
import org.spin.tools.crypto.signature.CertData;
import org.spin.tools.crypto.signature.CertID;

public final class PKITool {
    public static final String certificateFormat = "X.509";
    public static final String publicKeyAlgorithm = "RSA";
    public static final String signatureAlgorithm = "SHA1withRSA";
    static final Logger log = Logger.getLogger(PKITool.class);
    static final boolean DEBUG = log.isDebugEnabled();
    static final boolean INFO = log.isInfoEnabled();
    private static PKITool instance;
    private final KeyStoreConfig config;
    private KeyStoreType keystoreType;
    private final KeyStore keystore;
    private final CertificateFactory certFactory;
    private final Map<BigInteger, CertData> x509KeyInfoMap = Util.makeHashMap();
    private final CertData myX509;
    private final PrivateKey myPrivateKey;
    private final PublicKey myPublicKey;
    private final Map<Principal, X509Certificate> caCertificates = Util.makeHashMap();

    public static synchronized PKITool getInstance() {
        if (instance == null) {
            if (DEBUG) {
                log.debug("instance == null");
            }
            instance = PKITool.makeInstance(ConfigTool.loadKeyStoreConfig());
        }
        return instance;
    }

    public static PKITool makeInstance(KeyStoreConfig keyStoreConfig) {
        return new PKITool(keyStoreConfig);
    }

    static final URI getKeyStoreURI(KeyStoreConfig keyStoreConfig) {
        if (keyStoreConfig.getFile() == null) {
            throw new ConfigException("KeyStoreConfig has null File property");
        }
        URI result2 = ConfigTool.getConfigFileAsURI(keyStoreConfig.getFile().getPath());
        if (INFO) {
            log.info("KeyStore URI: " + result2);
        }
        return result2;
    }

    private static final InputStream getKeyStoreStream(KeyStoreConfig keyStoreConfig) {
        try {
            return PKITool.getKeyStoreURI(keyStoreConfig).toURL().openStream();
        }
        catch (IOException e2) {
            throw new ConfigException("Error getting keystore stream", e2);
        }
    }

    private PKITool(KeyStoreConfig config) {
        if (DEBUG) {
            log.debug("Initializing Keystore");
        }
        Util.guardNotNull(config, "Null KeyStoreConfig");
        Util.guardNotNull((Object)config.getKeystoreType(), "Null KeyStoreType");
        Util.guardNotNull(config.getPasswordAsCharArray(), "Null keystore password");
        try {
            this.config = config;
            this.keystoreType = config.getKeystoreType();
            this.certFactory = CertificateFactory.getInstance(certificateFormat);
            this.keystore = KeyStore.getInstance(this.keystoreType.name());
            this.keystore.load(PKITool.getKeyStoreStream(config), config.getPasswordAsCharArray());
            KeyPair keyPair = PKITool.determineKeyPair(this.keystore, config);
            this.myX509 = keyPair.x509Data;
            this.myPrivateKey = keyPair.privateKey;
            this.myPublicKey = keyPair.publicKey;
            this.fillInCACertMap(config);
            this.fillKeyInfoMap();
            this.debug();
        }
        catch (Exception e2) {
            throw new ConfigException("Error initializing PKITool", e2);
        }
    }

    private void fillInCACertMap(KeyStoreConfig keystoreConfig) {
        if (keystoreConfig.getCaPublicKeyAliases().size() > 0) {
            for (String certAlias : keystoreConfig.getCaPublicKeyAliases()) {
                X509Certificate tempCert = this.getCertificate(certAlias);
                this.caCertificates.put(tempCert.getIssuerDN(), tempCert);
            }
        }
    }

    public KeyStoreType getKeystoreType() {
        return this.keystoreType;
    }

    public KeyStore getKeystore() {
        return this.keystore;
    }

    private void fillKeyInfoMap() {
        this.x509KeyInfoMap.clear();
        try {
            for (String alias : Util.iterable(this.keystore.aliases())) {
                X509Certificate cert = this.getCertificate(alias);
                CertData x509Data = PKITool.getX509Data(cert);
                this.x509KeyInfoMap.put(x509Data.getCertID().getSerial(), x509Data);
            }
        }
        catch (CryptoException e2) {
            throw e2;
        }
        catch (Exception e3) {
            throw new CryptoException(e3);
        }
    }

    private X509Certificate getCertificate(String alias) {
        try {
            return (X509Certificate)this.keystore.getCertificate(alias);
        }
        catch (Exception e2) {
            throw new CryptoException(e2);
        }
    }

    static final KeyPair determineKeyPair(KeyStore keystore, KeyStoreConfig config) {
        Util.guardNotNull(keystore);
        Util.guardNotNull(config);
        int numKeyPairs = PKITool.countKeyPairs(keystore);
        if (numKeyPairs == 0) {
            throw new CryptoException("At least one public/private key pair is required.");
        }
        if (numKeyPairs == 1) {
            return PKITool.findKeyPair(keystore, config);
        }
        return PKITool.getKeyPairByKeyAlias(keystore, config);
    }

    static KeyPair findKeyPair(KeyStore keystore, KeyStoreConfig config) {
        try {
            for (String alias : Util.iterable(keystore.aliases())) {
                if (!keystore.isKeyEntry(alias)) continue;
                return PKITool.getKeyPair(keystore, config, alias);
            }
        }
        catch (CryptoException e2) {
            throw e2;
        }
        catch (Exception e3) {
            throw new CryptoException(e3);
        }
        throw new CryptoException("At least one public/private key pair is required.");
    }

    static KeyPair getKeyPairByKeyAlias(KeyStore keystore, KeyStoreConfig config) {
        if (config.getKeyAlias() == null) {
            throw new CryptoException("There are " + PKITool.countKeyPairs(keystore) + " keypairs in " + config.getFile() + ".  Set the keyAlias property in keystore.xml to specify which keypair to use");
        }
        return PKITool.getKeyPair(keystore, config, config.getKeyAlias());
    }

    private static final int countKeyPairs(KeyStore keystore) {
        int count2 = 0;
        try {
            for (String alias : Util.iterable(keystore.aliases())) {
                if (!keystore.isKeyEntry(alias)) continue;
                ++count2;
            }
        }
        catch (Exception e2) {
            throw new CryptoException(e2);
        }
        return count2;
    }

    public KeyPair getKeyPair(String alias) {
        return PKITool.getKeyPair(this.keystore, this.config, alias);
    }

    private static final KeyPair getKeyPair(KeyStore keystore, KeyStoreConfig config, String alias) {
        try {
            X509Certificate cert = (X509Certificate)keystore.getCertificate(alias);
            CertData myX509 = PKITool.getX509Data(cert);
            PrivateKey myPrivateKey = (PrivateKey)keystore.getKey(alias, config.getPasswordAsCharArray());
            PublicKey myPublicKey = cert.getPublicKey();
            return new KeyPair(myX509, myPrivateKey, myPublicKey);
        }
        catch (CryptoException e2) {
            throw e2;
        }
        catch (Exception e3) {
            throw new CryptoException(e3);
        }
    }

    public KeyStoreConfig getConfig() {
        return this.config;
    }

    public PrivateKey getMyPrivateKey() {
        return this.myPrivateKey;
    }

    public PublicKey getMyPublicKey() {
        return this.myPublicKey;
    }

    public CertData getMyX509() {
        return this.myX509;
    }

    public CertID getMyCertID() {
        return this.myX509.getCertID();
    }

    public static CertData getX509Data(X509Certificate cert) {
        try {
            return new CertData(new CertID(cert), cert.getEncoded());
        }
        catch (CertificateEncodingException e2) {
            throw new CryptoException("Could not read encoded bytes for x509 cert", e2);
        }
    }

    public Set<BigInteger> getSerialNumbersCopy() {
        return Util.asSet(this.x509KeyInfoMap.keySet());
    }

    public Set<CertData> getImportedCertsCopy() {
        return Util.asSet(this.x509KeyInfoMap.values());
    }

    public boolean containsX509Certificate(CertID serial) {
        return this.containsX509Certificate(serial.getSerial());
    }

    public boolean containsX509Certificate(BigInteger serial) {
        return this.x509KeyInfoMap.containsKey(serial);
    }

    /*
     * WARNING - Removed try catching itself - possible behaviour change.
     */
    public void addX509Certificate(X509Certificate cert) {
        if (cert == null) {
            throw new CryptoException("Null certificate passed in");
        }
        Map<BigInteger, CertData> map3 = this.x509KeyInfoMap;
        synchronized (map3) {
            CertData x509 = PKITool.getX509Data(cert);
            this.x509KeyInfoMap.put(x509.getCertID().getSerial(), x509);
        }
    }

    public void addX509Certificate(byte[] encodedInput) {
        this.addX509Certificate(this.getX509Certificate(encodedInput));
    }

    public void addX509Certificate(InputStream encodedInput) {
        if (encodedInput == null) {
            throw new CryptoException("Null InputStream passed in");
        }
        try {
            this.addX509Certificate((X509Certificate)this.certFactory.generateCertificate(encodedInput));
        }
        catch (Exception e2) {
            throw new CryptoException("Error decoding certificate.  Is it an X.509 cert in the DER encoding?", e2);
        }
    }

    public X509Certificate getX509Certificate(CertData x509) {
        Util.guardNotNull(x509);
        if (x509.getX509Certificate() != null) {
            return this.getX509Certificate(x509.getX509Certificate());
        }
        return this.getX509Certificate(x509.getCertID());
    }

    public X509Certificate getX509Certificate(CertID x509Serial) {
        Util.guardNotNull(x509Serial);
        return this.getX509Certificate(x509Serial.getSerial());
    }

    public X509Certificate getX509Certificate(BigInteger serial) {
        Util.guardNotNull(serial);
        if (this.containsX509Certificate(serial)) {
            return this.getX509Certificate(this.x509KeyInfoMap.get(serial).getX509Certificate());
        }
        return null;
    }

    public X509Certificate getX509Certificate(byte[] encoded) {
        Util.guardNotNull(encoded);
        try {
            return (X509Certificate)this.certFactory.generateCertificate(new ByteArrayInputStream(encoded));
        }
        catch (CertificateException e2) {
            throw new CryptoException("Could not get x509 cert from encoded bytes[]", e2);
        }
    }

    public List<X509Certificate> getAllCACerts() {
        return Util.makeArrayList(this.caCertificates.values());
    }

    public X509Certificate removeX509Certificate(CertData x509) {
        Util.guardNotNull(x509);
        return this.removeX509Certificate(x509.getCertID());
    }

    public X509Certificate removeX509Certificate(CertID x509serial) {
        Util.guardNotNull(x509serial);
        return this.removeX509Certificate(x509serial.getSerial());
    }

    /*
     * WARNING - Removed try catching itself - possible behaviour change.
     */
    public X509Certificate removeX509Certificate(BigInteger x509serial) {
        CertData removed;
        if (x509serial == null) {
            throw new CryptoException("Null x509Serial passed in.");
        }
        Map<BigInteger, CertData> map3 = this.x509KeyInfoMap;
        synchronized (map3) {
            removed = this.x509KeyInfoMap.remove(x509serial);
        }
        if (removed != null) {
            return this.getX509Certificate(removed.getX509Certificate());
        }
        return null;
    }

    public PublicKey getPublicKey(BigInteger x509Serial) {
        if (this.isUnknownCert(x509Serial)) {
            throw new CryptoException("No entry found for x509Serial " + x509Serial);
        }
        return this.getX509Certificate(this.x509KeyInfoMap.get(x509Serial)).getPublicKey();
    }

    private static final Cipher getPublicKeyCipher() {
        try {
            return Cipher.getInstance(publicKeyAlgorithm);
        }
        catch (Exception e2) {
            throw new CryptoException(e2);
        }
    }

    private static final Signature getSignerVerifier() {
        try {
            return Signature.getInstance(signatureAlgorithm);
        }
        catch (Exception e2) {
            throw new CryptoException(e2);
        }
    }

    public byte[] sign(String challenge) {
        return this.sign(challenge.getBytes());
    }

    public byte[] sign(byte[] challenge) {
        try {
            Signature signerVerifier = PKITool.getSignerVerifier();
            signerVerifier.initSign(this.myPrivateKey);
            signerVerifier.update(challenge);
            return signerVerifier.sign();
        }
        catch (CryptoException e2) {
            throw e2;
        }
        catch (Exception e3) {
            throw new CryptoException("Invalid key, failed to sign challenge " + new String(challenge), e3);
        }
    }

    public boolean verifySignature(String challenge, byte[] signature2) {
        return this.verifySignature(challenge.getBytes(), signature2);
    }

    public boolean verifySignature(String challenge, byte[] signature2, CertData signerKeyInfo) {
        return this.verifySignature(challenge, signature2, signerKeyInfo.getCertID());
    }

    public boolean verifySignature(String challenge, byte[] signature2, CertID x509Serial) {
        return this.verifySignature(challenge, signature2, x509Serial.getSerial());
    }

    public boolean verifySignature(String challenge, byte[] signature2, BigInteger signerSerial) {
        return this.verifySignature(challenge.getBytes(), signature2, signerSerial);
    }

    public boolean verifySignature(byte[] challenge, byte[] signature2) {
        return this.verifySignature(challenge, signature2, this.myX509.getCertID().getSerial());
    }

    public boolean verifySignature(byte[] challenge, byte[] signature2, CertData signerX509) {
        return this.verifySignature(challenge, signature2, signerX509.getCertID().getSerial());
    }

    public boolean verifySignature(byte[] challenge, byte[] signature2, CertID signedBy) {
        return this.verifySignature(challenge, signature2, signedBy.getSerial());
    }

    public boolean verifySignature(byte[] challenge, byte[] signature2, BigInteger signerSerial) {
        this.guardNoNulls(challenge, signature2, signerSerial);
        try {
            if (DEBUG) {
                log.debug("Verifying signature, signer: " + signerSerial + " challenge: '" + Base64Codec.toString(challenge) + "' signature: '" + Base64Codec.toString(signature2) + "'");
            }
            if (this.isUnknownCert(signerSerial)) {
                log.error("No certificate found for signer key ID " + signerSerial);
                return false;
            }
            X509Certificate signerCert = this.getX509Certificate(signerSerial);
            if (DEBUG) {
                log.debug("Signer keyID: " + signerSerial);
                log.debug("Signer cert: " + signerCert.toString());
            }
            return this.verifyWith(signature2, this.makeSignerVerifier(challenge, signerCert));
        }
        catch (SignatureException e2) {
            throw new BadSignatureException(e2);
        }
        catch (CryptoException e3) {
            throw e3;
        }
        catch (Exception e4) {
            throw new CryptoException("Invalid key, failed to verify challenge " + Base64Codec.toString(challenge), e4);
        }
    }

    private boolean verifyWith(byte[] signature2, Signature signerVerifier) throws SignatureException {
        boolean verified = signerVerifier.verify(signature2);
        if (DEBUG) {
            log.debug("signature verified: " + verified);
        }
        return verified;
    }

    private Signature makeSignerVerifier(byte[] challenge, Certificate signerCert) throws InvalidKeyException, SignatureException {
        Signature signerVerifier = PKITool.getSignerVerifier();
        signerVerifier.initVerify(signerCert);
        signerVerifier.update(challenge);
        return signerVerifier;
    }

    private boolean isUnknownCert(BigInteger signerSerial) {
        return !this.containsX509Certificate(signerSerial);
    }

    private void guardNoNulls(byte[] challenge, byte[] signature2, BigInteger signerSerial) {
        if (challenge == null || signature2 == null || signerSerial == null) {
            throw new CryptoException("Invalid input:  Verifying signature, signer: " + signerSerial + " challenge: '" + Base64Codec.toString(challenge) + "' signature: '" + Base64Codec.toString(signature2) + "'");
        }
    }

    public byte[] encrypt(byte[] plainText, CertData x509) {
        X509Certificate cert = this.getX509Certificate(x509.getX509Certificate());
        return this.encrypt(plainText, cert.getPublicKey());
    }

    public byte[] encrypt(byte[] plainText, CertID x509Serial) {
        return this.encrypt(plainText, x509Serial.getSerial());
    }

    public byte[] encrypt(byte[] plainText, BigInteger x509Serial) {
        try {
            return this.encrypt(plainText, this.getPublicKey(x509Serial));
        }
        catch (CryptoException e2) {
            throw new CryptoException("Error encrypting using the provided key ", e2);
        }
    }

    public byte[] encrypt(byte[] inpBytes, PublicKey key) {
        try {
            Cipher myCipher = Cipher.getInstance(publicKeyAlgorithm);
            myCipher.init(1, key);
            return myCipher.doFinal(inpBytes);
        }
        catch (Exception e2) {
            throw new CryptoException("Error encrypting: ", e2);
        }
    }

    public byte[] decrypt(byte[] cipherText) {
        try {
            Cipher myCipher = PKITool.getPublicKeyCipher();
            myCipher.init(2, this.myPrivateKey);
            return myCipher.doFinal(cipherText);
        }
        catch (Exception e2) {
            throw new CryptoException("Error decrypting: ", e2);
        }
    }

    public final PKCryptor cryptor() {
        return new PKCryptor(this);
    }

    public final void debug() throws CryptoException {
        if (INFO) {
            Enumeration<String> aliases2;
            log.info("Keystore Type: " + this.keystore.getType());
            log.info("Public-key crypto Algorithm: RSA");
            log.info("Signature Algorithm: SHA1withRSA");
            log.info("Config: " + this.config);
            log.info("Key Aliases:");
            try {
                aliases2 = this.keystore.aliases();
            }
            catch (KeyStoreException e2) {
                throw new CryptoException(e2);
            }
            while (aliases2.hasMoreElements()) {
                log.info("\t" + aliases2.nextElement());
            }
            log.info("Key IDs:");
            for (CertData x509Serial : this.x509KeyInfoMap.values()) {
                log.info(x509Serial.getCertID());
            }
            if (INFO) {
                log.info("Private Key ID: " + this.myX509.getCertID());
            }
        }
    }

    public X509Certificate getCaCertificate(Principal p) {
        return this.caCertificates.get(p);
    }

    public static final class KeyPair {
        public final CertData x509Data;
        public final PrivateKey privateKey;
        public final PublicKey publicKey;

        public KeyPair(CertData myX509, PrivateKey myPrivateKey, PublicKey myPublicKey) {
            this.x509Data = myX509;
            this.privateKey = myPrivateKey;
            this.publicKey = myPublicKey;
        }
    }
}

