/*
 * Decompiled with CFR 0.152.
 */
package org.spin.tools.crypto;

import java.io.ByteArrayInputStream;
import java.io.IOException;
import java.io.InputStream;
import java.math.BigInteger;
import java.net.URI;
import java.security.InvalidKeyException;
import java.security.KeyStore;
import java.security.KeyStoreException;
import java.security.Principal;
import java.security.PrivateKey;
import java.security.PublicKey;
import java.security.Signature;
import java.security.SignatureException;
import java.security.cert.Certificate;
import java.security.cert.CertificateEncodingException;
import java.security.cert.CertificateException;
import java.security.cert.CertificateFactory;
import java.security.cert.X509Certificate;
import java.util.Enumeration;
import java.util.List;
import java.util.Map;
import java.util.Set;
import javax.crypto.Cipher;
import org.apache.log4j.Logger;
import org.spin.tools.Base64Codec;
import org.spin.tools.Util;
import org.spin.tools.config.ConfigException;
import org.spin.tools.config.ConfigTool;
import org.spin.tools.config.KeyStoreConfig;
import org.spin.tools.config.KeyStoreType;
import org.spin.tools.crypto.BadSignatureException;
import org.spin.tools.crypto.CryptoException;
import org.spin.tools.crypto.PKCryptor;
import org.spin.tools.crypto.signature.CertData;
import org.spin.tools.crypto.signature.CertID;

public final class PKITool {
    public static final String certificateFormat = "X.509";
    public static final String publicKeyAlgorithm = "RSA";
    public static final String signatureAlgorithm = "SHA1withRSA";
    static final Logger log = Logger.getLogger(PKITool.class);
    static final boolean DEBUG = log.isDebugEnabled();
    static final boolean INFO = log.isInfoEnabled();
    private static PKITool instance;
    private final KeyStoreConfig config;
    private KeyStoreType keystoreType;
    private final KeyStore keystore;
    private final CertificateFactory certFactory;
    private final Map<BigInteger, CertData> x509KeyInfoMap = Util.makeHashMap();
    private final CertData myX509;
    private final PrivateKey myPrivateKey;
    private final PublicKey myPublicKey;
    private final Map<Principal, X509Certificate> caCertificates = Util.makeHashMap();

    public static synchronized PKITool getInstance() {
        if (instance == null) {
            if (DEBUG) {
                log.debug((Object)"instance == null");
            }
            instance = PKITool.makeInstance(ConfigTool.loadKeyStoreConfig());
        }
        return instance;
    }

    public static PKITool makeInstance(KeyStoreConfig keyStoreConfig) {
        return new PKITool(keyStoreConfig);
    }

    static final URI getKeyStoreURI(KeyStoreConfig keyStoreConfig) {
        if (keyStoreConfig.getFile() == null) {
            throw new ConfigException("KeyStoreConfig has null File property");
        }
        URI result = ConfigTool.getConfigFileAsURI(keyStoreConfig.getFile().getPath());
        if (INFO) {
            log.info((Object)("KeyStore URI: " + result));
        }
        return result;
    }

    private static final InputStream getKeyStoreStream(KeyStoreConfig keyStoreConfig) {
        try {
            return PKITool.getKeyStoreURI(keyStoreConfig).toURL().openStream();
        }
        catch (IOException e) {
            throw new ConfigException("Error getting keystore stream", e);
        }
    }

    private PKITool(KeyStoreConfig config) {
        if (DEBUG) {
            log.debug((Object)"Initializing Keystore");
        }
        Util.guardNotNull((Object)config, (String)"Null KeyStoreConfig");
        Util.guardNotNull((Object)((Object)config.getKeystoreType()), (String)"Null KeyStoreType");
        Util.guardNotNull((Object)config.getPasswordAsCharArray(), (String)"Null keystore password");
        try {
            this.config = config;
            this.keystoreType = config.getKeystoreType();
            this.certFactory = CertificateFactory.getInstance(certificateFormat);
            this.keystore = KeyStore.getInstance(this.keystoreType.name());
            this.keystore.load(PKITool.getKeyStoreStream(config), config.getPasswordAsCharArray());
            KeyPair keyPair = PKITool.determineKeyPair(this.keystore, config);
            this.myX509 = keyPair.x509Data;
            this.myPrivateKey = keyPair.privateKey;
            this.myPublicKey = keyPair.publicKey;
            this.fillInCACertMap(config);
            this.fillKeyInfoMap();
            this.debug();
        }
        catch (Exception e) {
            throw new ConfigException("Error initializing PKITool", e);
        }
    }

    private void fillInCACertMap(KeyStoreConfig keystoreConfig) {
        if (keystoreConfig.getCaPublicKeyAliases().size() > 0) {
            for (String certAlias : keystoreConfig.getCaPublicKeyAliases()) {
                X509Certificate tempCert = this.getCertificate(certAlias);
                this.caCertificates.put(tempCert.getIssuerDN(), tempCert);
            }
        }
    }

    public KeyStoreType getKeystoreType() {
        return this.keystoreType;
    }

    public KeyStore getKeystore() {
        return this.keystore;
    }

    private void fillKeyInfoMap() {
        this.x509KeyInfoMap.clear();
        try {
            for (String alias : Util.iterable(this.keystore.aliases())) {
                X509Certificate cert = this.getCertificate(alias);
                CertData x509Data = PKITool.getX509Data(cert);
                this.x509KeyInfoMap.put(x509Data.getCertID().getSerial(), x509Data);
            }
        }
        catch (CryptoException e) {
            throw e;
        }
        catch (Exception e) {
            throw new CryptoException((Throwable)e);
        }
    }

    private X509Certificate getCertificate(String alias) {
        try {
            return (X509Certificate)this.keystore.getCertificate(alias);
        }
        catch (Exception e) {
            throw new CryptoException((Throwable)e);
        }
    }

    static final KeyPair determineKeyPair(KeyStore keystore, KeyStoreConfig config) {
        Util.guardNotNull((Object)keystore);
        Util.guardNotNull((Object)config);
        int numKeyPairs = PKITool.countKeyPairs(keystore);
        if (numKeyPairs == 0) {
            throw new CryptoException("At least one public/private key pair is required.");
        }
        if (numKeyPairs == 1) {
            return PKITool.findKeyPair(keystore, config);
        }
        return PKITool.getKeyPairByKeyAlias(keystore, config);
    }

    static KeyPair findKeyPair(KeyStore keystore, KeyStoreConfig config) {
        try {
            for (String alias : Util.iterable(keystore.aliases())) {
                if (!keystore.isKeyEntry(alias)) continue;
                return PKITool.getKeyPair(keystore, config, alias);
            }
        }
        catch (CryptoException e) {
            throw e;
        }
        catch (Exception e) {
            throw new CryptoException((Throwable)e);
        }
        throw new CryptoException("At least one public/private key pair is required.");
    }

    static KeyPair getKeyPairByKeyAlias(KeyStore keystore, KeyStoreConfig config) {
        if (config.getKeyAlias() == null) {
            throw new CryptoException("There are " + PKITool.countKeyPairs(keystore) + " keypairs in " + config.getFile() + ".  Set the keyAlias property in keystore.xml to specify which keypair to use");
        }
        return PKITool.getKeyPair(keystore, config, config.getKeyAlias());
    }

    private static final int countKeyPairs(KeyStore keystore) {
        int count = 0;
        try {
            for (String alias : Util.iterable(keystore.aliases())) {
                if (!keystore.isKeyEntry(alias)) continue;
                ++count;
            }
        }
        catch (Exception e) {
            throw new CryptoException((Throwable)e);
        }
        return count;
    }

    public KeyPair getKeyPair(String alias) {
        return PKITool.getKeyPair(this.keystore, this.config, alias);
    }

    private static final KeyPair getKeyPair(KeyStore keystore, KeyStoreConfig config, String alias) {
        try {
            X509Certificate cert = (X509Certificate)keystore.getCertificate(alias);
            CertData myX509 = PKITool.getX509Data(cert);
            PrivateKey myPrivateKey = (PrivateKey)keystore.getKey(alias, config.getPasswordAsCharArray());
            PublicKey myPublicKey = cert.getPublicKey();
            return new KeyPair(myX509, myPrivateKey, myPublicKey);
        }
        catch (CryptoException e) {
            throw e;
        }
        catch (Exception e) {
            throw new CryptoException((Throwable)e);
        }
    }

    public KeyStoreConfig getConfig() {
        return this.config;
    }

    public PrivateKey getMyPrivateKey() {
        return this.myPrivateKey;
    }

    public PublicKey getMyPublicKey() {
        return this.myPublicKey;
    }

    public CertData getMyX509() {
        return this.myX509;
    }

    public CertID getMyCertID() {
        return this.myX509.getCertID();
    }

    public static CertData getX509Data(X509Certificate cert) {
        try {
            return new CertData(new CertID(cert), cert.getEncoded());
        }
        catch (CertificateEncodingException e) {
            throw new CryptoException("Could not read encoded bytes for x509 cert", (Throwable)e);
        }
    }

    public Set<BigInteger> getSerialNumbersCopy() {
        return Util.asSet(this.x509KeyInfoMap.keySet());
    }

    public Set<CertData> getImportedCertsCopy() {
        return Util.asSet(this.x509KeyInfoMap.values());
    }

    public boolean containsX509Certificate(CertID serial) {
        return this.containsX509Certificate(serial.getSerial());
    }

    public boolean containsX509Certificate(BigInteger serial) {
        return this.x509KeyInfoMap.containsKey(serial);
    }

    /*
     * WARNING - Removed try catching itself - possible behaviour change.
     */
    public void addX509Certificate(X509Certificate cert) {
        if (cert == null) {
            throw new CryptoException("Null certificate passed in");
        }
        Map<BigInteger, CertData> map = this.x509KeyInfoMap;
        synchronized (map) {
            CertData x509 = PKITool.getX509Data(cert);
            this.x509KeyInfoMap.put(x509.getCertID().getSerial(), x509);
        }
    }

    public void addX509Certificate(byte[] encodedInput) {
        this.addX509Certificate(this.getX509Certificate(encodedInput));
    }

    public void addX509Certificate(InputStream encodedInput) {
        if (encodedInput == null) {
            throw new CryptoException("Null InputStream passed in");
        }
        try {
            this.addX509Certificate((X509Certificate)this.certFactory.generateCertificate(encodedInput));
        }
        catch (Exception e) {
            throw new CryptoException("Error decoding certificate.  Is it an X.509 cert in the DER encoding?", (Throwable)e);
        }
    }

    public X509Certificate getX509Certificate(CertData x509) {
        Util.guardNotNull((Object)x509);
        if (x509.getX509Certificate() != null) {
            return this.getX509Certificate(x509.getX509Certificate());
        }
        return this.getX509Certificate(x509.getCertID());
    }

    public X509Certificate getX509Certificate(CertID x509Serial) {
        Util.guardNotNull((Object)x509Serial);
        return this.getX509Certificate(x509Serial.getSerial());
    }

    public X509Certificate getX509Certificate(BigInteger serial) {
        Util.guardNotNull((Object)serial);
        if (this.containsX509Certificate(serial)) {
            return this.getX509Certificate(this.x509KeyInfoMap.get(serial).getX509Certificate());
        }
        return null;
    }

    public X509Certificate getX509Certificate(byte[] encoded) {
        Util.guardNotNull((Object)encoded);
        try {
            return (X509Certificate)this.certFactory.generateCertificate(new ByteArrayInputStream(encoded));
        }
        catch (CertificateException e) {
            throw new CryptoException("Could not get x509 cert from encoded bytes[]", (Throwable)e);
        }
    }

    public List<X509Certificate> getAllCACerts() {
        return Util.makeArrayList(this.caCertificates.values());
    }

    public X509Certificate removeX509Certificate(CertData x509) {
        Util.guardNotNull((Object)x509);
        return this.removeX509Certificate(x509.getCertID());
    }

    public X509Certificate removeX509Certificate(CertID x509serial) {
        Util.guardNotNull((Object)x509serial);
        return this.removeX509Certificate(x509serial.getSerial());
    }

    /*
     * WARNING - Removed try catching itself - possible behaviour change.
     */
    public X509Certificate removeX509Certificate(BigInteger x509serial) {
        CertData removed;
        if (x509serial == null) {
            throw new CryptoException("Null x509Serial passed in.");
        }
        Map<BigInteger, CertData> map = this.x509KeyInfoMap;
        synchronized (map) {
            removed = this.x509KeyInfoMap.remove(x509serial);
        }
        if (removed != null) {
            return this.getX509Certificate(removed.getX509Certificate());
        }
        return null;
    }

    public PublicKey getPublicKey(BigInteger x509Serial) {
        if (this.isUnknownCert(x509Serial)) {
            throw new CryptoException("No entry found for x509Serial " + x509Serial);
        }
        return this.getX509Certificate(this.x509KeyInfoMap.get(x509Serial)).getPublicKey();
    }

    private static final Cipher getPublicKeyCipher() {
        try {
            return Cipher.getInstance(publicKeyAlgorithm);
        }
        catch (Exception e) {
            throw new CryptoException((Throwable)e);
        }
    }

    private static final Signature getSignerVerifier() {
        try {
            return Signature.getInstance(signatureAlgorithm);
        }
        catch (Exception e) {
            throw new CryptoException((Throwable)e);
        }
    }

    public byte[] sign(String challenge) {
        return this.sign(challenge.getBytes());
    }

    public byte[] sign(byte[] challenge) {
        try {
            Signature signerVerifier = PKITool.getSignerVerifier();
            signerVerifier.initSign(this.myPrivateKey);
            signerVerifier.update(challenge);
            return signerVerifier.sign();
        }
        catch (CryptoException e) {
            throw e;
        }
        catch (Exception e) {
            throw new CryptoException("Invalid key, failed to sign challenge " + new String(challenge), (Throwable)e);
        }
    }

    public boolean verifySignature(String challenge, byte[] signature) {
        return this.verifySignature(challenge.getBytes(), signature);
    }

    public boolean verifySignature(String challenge, byte[] signature, CertData signerKeyInfo) {
        return this.verifySignature(challenge, signature, signerKeyInfo.getCertID());
    }

    public boolean verifySignature(String challenge, byte[] signature, CertID x509Serial) {
        return this.verifySignature(challenge, signature, x509Serial.getSerial());
    }

    public boolean verifySignature(String challenge, byte[] signature, BigInteger signerSerial) {
        return this.verifySignature(challenge.getBytes(), signature, signerSerial);
    }

    public boolean verifySignature(byte[] challenge, byte[] signature) {
        return this.verifySignature(challenge, signature, this.myX509.getCertID().getSerial());
    }

    public boolean verifySignature(byte[] challenge, byte[] signature, CertData signerX509) {
        return this.verifySignature(challenge, signature, signerX509.getCertID().getSerial());
    }

    public boolean verifySignature(byte[] challenge, byte[] signature, CertID signedBy) {
        return this.verifySignature(challenge, signature, signedBy.getSerial());
    }

    public boolean verifySignature(byte[] challenge, byte[] signature, BigInteger signerSerial) {
        this.guardNoNulls(challenge, signature, signerSerial);
        try {
            if (DEBUG) {
                log.debug((Object)("Verifying signature, signer: " + signerSerial + " challenge: '" + Base64Codec.toString((byte[])challenge) + "' signature: '" + Base64Codec.toString((byte[])signature) + "'"));
            }
            if (this.isUnknownCert(signerSerial)) {
                log.error((Object)("No certificate found for signer key ID " + signerSerial));
                return false;
            }
            X509Certificate signerCert = this.getX509Certificate(signerSerial);
            if (DEBUG) {
                log.debug((Object)("Signer keyID: " + signerSerial));
                log.debug((Object)("Signer cert: " + signerCert.toString()));
            }
            return this.verifyWith(signature, this.makeSignerVerifier(challenge, signerCert));
        }
        catch (SignatureException e) {
            throw new BadSignatureException((Throwable)e);
        }
        catch (CryptoException e) {
            throw e;
        }
        catch (Exception e) {
            throw new CryptoException("Invalid key, failed to verify challenge " + Base64Codec.toString((byte[])challenge), (Throwable)e);
        }
    }

    private boolean verifyWith(byte[] signature, Signature signerVerifier) throws SignatureException {
        boolean verified = signerVerifier.verify(signature);
        if (DEBUG) {
            log.debug((Object)("signature verified: " + verified));
        }
        return verified;
    }

    private Signature makeSignerVerifier(byte[] challenge, Certificate signerCert) throws InvalidKeyException, SignatureException {
        Signature signerVerifier = PKITool.getSignerVerifier();
        signerVerifier.initVerify(signerCert);
        signerVerifier.update(challenge);
        return signerVerifier;
    }

    private boolean isUnknownCert(BigInteger signerSerial) {
        return !this.containsX509Certificate(signerSerial);
    }

    private void guardNoNulls(byte[] challenge, byte[] signature, BigInteger signerSerial) {
        if (challenge == null || signature == null || signerSerial == null) {
            throw new CryptoException("Invalid input:  Verifying signature, signer: " + signerSerial + " challenge: '" + Base64Codec.toString((byte[])challenge) + "' signature: '" + Base64Codec.toString((byte[])signature) + "'");
        }
    }

    public byte[] encrypt(byte[] plainText, CertData x509) {
        X509Certificate cert = this.getX509Certificate(x509.getX509Certificate());
        return this.encrypt(plainText, cert.getPublicKey());
    }

    public byte[] encrypt(byte[] plainText, CertID x509Serial) {
        return this.encrypt(plainText, x509Serial.getSerial());
    }

    public byte[] encrypt(byte[] plainText, BigInteger x509Serial) {
        try {
            return this.encrypt(plainText, this.getPublicKey(x509Serial));
        }
        catch (CryptoException e) {
            throw new CryptoException("Error encrypting using the provided key ", (Throwable)e);
        }
    }

    public byte[] encrypt(byte[] inpBytes, PublicKey key) {
        try {
            Cipher myCipher = Cipher.getInstance(publicKeyAlgorithm);
            myCipher.init(1, key);
            return myCipher.doFinal(inpBytes);
        }
        catch (Exception e) {
            throw new CryptoException("Error encrypting: ", (Throwable)e);
        }
    }

    public byte[] decrypt(byte[] cipherText) {
        try {
            Cipher myCipher = PKITool.getPublicKeyCipher();
            myCipher.init(2, this.myPrivateKey);
            return myCipher.doFinal(cipherText);
        }
        catch (Exception e) {
            throw new CryptoException("Error decrypting: ", (Throwable)e);
        }
    }

    public final PKCryptor cryptor() {
        return new PKCryptor(this);
    }

    public final void debug() throws CryptoException {
        if (INFO) {
            Enumeration<String> aliases;
            log.info((Object)("Keystore Type: " + this.keystore.getType()));
            log.info((Object)"Public-key crypto Algorithm: RSA");
            log.info((Object)"Signature Algorithm: SHA1withRSA");
            log.info((Object)("Config: " + this.config));
            log.info((Object)"Key Aliases:");
            try {
                aliases = this.keystore.aliases();
            }
            catch (KeyStoreException e) {
                throw new CryptoException((Throwable)e);
            }
            while (aliases.hasMoreElements()) {
                log.info((Object)("\t" + aliases.nextElement()));
            }
            log.info((Object)"Key IDs:");
            for (CertData x509Serial : this.x509KeyInfoMap.values()) {
                log.info((Object)x509Serial.getCertID());
            }
            if (INFO) {
                log.info((Object)("Private Key ID: " + this.myX509.getCertID()));
            }
        }
    }

    public X509Certificate getCaCertificate(Principal p) {
        return this.caCertificates.get(p);
    }

    public static final class KeyPair {
        public final CertData x509Data;
        public final PrivateKey privateKey;
        public final PublicKey publicKey;

        public KeyPair(CertData myX509, PrivateKey myPrivateKey, PublicKey myPublicKey) {
            this.x509Data = myX509;
            this.privateKey = myPrivateKey;
            this.publicKey = myPublicKey;
        }
    }
}

