package org.eaglei.suggest.server;

import java.io.IOException;
import java.io.PrintWriter;
import java.net.URLDecoder;
import java.util.List;
import java.util.Map;

import javax.servlet.ServletException;
import javax.servlet.http.HttpServlet;
import javax.servlet.http.HttpServletRequest;
import javax.servlet.http.HttpServletResponse;

import org.apache.commons.logging.Log;
import org.apache.commons.logging.LogFactory;
import org.eaglei.suggest.provider.Suggestion;
import org.eaglei.suggest.provider.SuggestionProvider;
import org.eaglei.suggest.provider.model.lucene.EIModelSuggestionProviderFactory;

/**
 * Auto-suggest servlet implementation.
 * 
 * TODO Use Spring to configure the SearchProviders
 * 
 * @author tbashor Original implementation.
 * @author rfrost Refactored out SearchProvider logic.
 */
public class SuggestionServlet extends HttpServlet {

	private static final long serialVersionUID = 1L;
	
    protected static final Log logger = LogFactory.getLog(SuggestionServlet.class);
    protected static final boolean DEBUG = logger.isDebugEnabled(); 

    /*
     * Flag that indicates whether the servlet is intialized.
     */
	private boolean isInitialized = false;
	/*
	 * Main suggestion provider.
	 */
	private SuggestionProvider mainProvider; 
	/*
	 * Maps from class URI to SuggestionProvider
	 */
	private Map<String, SuggestionProvider> mapIdToProvider;

    /**
     * Default constructor. 
     */
    public SuggestionServlet() {
        // TODO Auto-generated constructor stub
    }
    
    @Override
    public void init() throws ServletException {
        Thread logThread = new Thread(new SuggestionIndexer(), "SuggestionIndexer");
        logThread.setPriority(Thread.MIN_PRIORITY);
        logThread.setDaemon(true);
        logThread.start();
    }
    
    /**
     * Thread that initializes the indices
     */
    private class SuggestionIndexer implements Runnable {
    	public void run() {
    		try {
    			mapIdToProvider = EIModelSuggestionProviderFactory.createProviderForEachTopLevel();
    			mainProvider = EIModelSuggestionProviderFactory.createProviderForAllRoots();
    			if (DEBUG) {
    				logger.debug("Indexing complete");
    			}
    			isInitialized = true;
    		} catch (Throwable t) {
    			logger.error(t);
    		}
    	}    
    }

	/**
	 * @see HttpServlet#doGet(HttpServletRequest request, HttpServletResponse response)
	 */
	protected void doGet(HttpServletRequest request, HttpServletResponse response) throws ServletException, IOException {
	    doPost(request, response);
	}

	/**
	 * @see HttpServlet#doPost(HttpServletRequest request, HttpServletResponse response)
	 */
	protected void doPost(HttpServletRequest request, HttpServletResponse response) throws ServletException, IOException {
        String id = request.getParameter("id");
        String classid = request.getParameter("classid");
        String q = request.getParameter("q");
        // Needed to bracket the query with some delimiter, otherwise trailing whitespace is lost.
        // Strip the brackets here, we don't need them for the json response.
        q = q.substring(1, q.length() - 1);
        // getParameter does the decoding for us
        //System.out.println("encoded: "+q);
        //q = URLDecoder.decode(q, "UTF-8");
        //System.out.println("decoded: "+q);
        String asyncCallback = request.getParameter("callback");
        
        // For suggest list, don't throw an error, just return nothing.
        List<Suggestion> suggestions = null;
        if (isInitialized) {
            if (classid == null) {
                suggestions = mainProvider.getSuggestions(q, 10);
            } else {
                String decodedClassId = URLDecoder.decode(classid);
                SuggestionProvider provider = mapIdToProvider.get(decodedClassId);
                if (provider != null) {
                    suggestions = provider.getSuggestions(q, 10);
                } else {
                    logger.warn("SuggestionServlet: no provider found for [" + decodedClassId
                            + "]");
                }
            }
        }
        
        StringBuilder sb = new StringBuilder();
        
        sb.append(asyncCallback);
        sb.append("({");  // function wrap start
        writeJsonNameValueString(sb, "id", id);
        sb.append(',');
        /*  Commenting out return of the user query input.
         *  It is only used for debugging, and putting user input
         *  into json is a potential security risk.
        writeJsonNameValueString(sb, "q", q);
        sb.append(',');
        */
        sb.append("suggestions");
        sb.append(":[");
        if (suggestions != null && suggestions.size() > 0) {
            int i=0;
            while (true) {
            	Suggestion suggestion = suggestions.get(i++);
                sb.append("\"");
                sb.append(suggestion.getDisplayText());
                sb.append("\"");
                sb.append(',');
                writeJsonValueString(sb, suggestion.getReplacementText());
                sb.append(',');
                writeJsonValueString(sb, suggestion.getURI());
                if (i == suggestions.size()) {
                    break;
                } else {
                    sb.append(',');
                }
            }
        }
        sb.append("]");
        sb.append("})");  // function wrap end
        String output = sb.toString();

        response.setContentType("text/javascript");
        response.addHeader("Pragma", "no-cache");
        response.setStatus(200);
        
        PrintWriter out = response.getWriter();

        out.println(output);
	}
	
    private static void writeJsonNameValueString(StringBuilder sb, String name, String value) {
        sb.append(name);
        sb.append(":");
        writeJsonValueString(sb, value);
    }

    private static void writeJsonValueString(StringBuilder sb, String value) {
        sb.append("\"");
        sb.append(value);
        sb.append("\"");
    }

}
