package org.eaglei.datatools.jena;

import static org.eaglei.datatools.jena.SPARQLConstants.CREATION_DATE_VARIABLE;
import static org.eaglei.datatools.jena.SPARQLConstants.LABEL_VARIABLE;
import static org.eaglei.datatools.jena.SPARQLConstants.LAB_NAME_VARIABLE;
import static org.eaglei.datatools.jena.SPARQLConstants.LAB_VARIABLE;
import static org.eaglei.datatools.jena.SPARQLConstants.MODIFIED_DATE_VARIABLE;
import static org.eaglei.datatools.jena.SPARQLConstants.OWNER_NAME_VARIABLE;
import static org.eaglei.datatools.jena.SPARQLConstants.OWNER_VARIABLE;
import static org.eaglei.datatools.jena.SPARQLConstants.STATE_LABEL_VARIABLE;
import static org.eaglei.datatools.jena.SPARQLConstants.STATE_VARIABLE;
import static org.eaglei.datatools.jena.SPARQLConstants.SUBJECT_VARIABLE;
import static org.eaglei.datatools.jena.SPARQLConstants.TYPE_LABEL_VARIABLE;
import static org.eaglei.datatools.jena.SPARQLConstants.TYPE_VARIABLE;
import static org.eaglei.datatools.jena.SPARQLConstants.resultSetVariables;
import static org.eaglei.datatools.model.DataToolsOntConstants.DC_URI;
import static org.eaglei.datatools.model.DataToolsOntConstants.EAGLE_I_REPO_URI;
import static org.eaglei.datatools.model.DataToolsOntConstants.FOAF_PERSON;
import static org.eaglei.datatools.model.DataToolsOntConstants.OBI_ORGANIZATION;
import static org.eaglei.datatools.model.DataToolsOntConstants.RDFS_URI;
import static org.eaglei.datatools.model.DataToolsOntConstants.RDF_URI;

import static org.eaglei.datatools.model.DataToolsOntConstants.EAGLE_I_DT_URI;
import static org.eaglei.datatools.model.DataToolsOntConstants.EAGLE_I_DT_PREFIX;

import java.util.List;

import org.apache.commons.logging.Log;
import org.apache.commons.logging.LogFactory;
import org.eaglei.datatools.model.DataToolsOntConstants;
import org.eaglei.model.EIEntity;
import org.eaglei.model.EIURI;

/**
 * 
 * @author Ricardo De Lima
 * @author Daniela Bourges
 * 
 *         May 12, 2010
 * 
 *         Center for Biomedical Informatics (CBMI)
 * @link https://cbmi.med.harvard.edu/
 * 
 * 
 */


public class SPARQLQueryUtil {

	private static final Log		logger	= LogFactory.getLog(SPARQLQueryUtil.class);

	private static SPARQLQueryUtil	instance;

	private SPARQLQueryUtil() {

	}

	public static SPARQLQueryUtil getInstance() {
		if (instance == null) {
			instance = new SPARQLQueryUtil();
		}
		return instance;
	}

	// ETL-specific queries
	/**
	 * Returns a sparql query that asks the URI of a resource of type rootURI
	 * with a label
	 * 
	 * @param rootURI
	 * @param label
	 * @return the sparql query string
	 */

	public String getEIResourcesByLabelQuery(final EIURI rootURI, final String label) {
		final StringBuilder sparql = new StringBuilder();
		sparql.append("PREFIX rdf: <");
		sparql.append(RDF_URI);
		sparql.append("> ");
		sparql.append("PREFIX rdfs: <");
		sparql.append(RDFS_URI);
		sparql.append("> ");
		sparql.append("SELECT ?resource WHERE {?resource a <");
		sparql.append(rootURI.toString());
		sparql.append("> . ");
		sparql.append("?resource <http://www.w3.org/2000/01/rdf-schema#label> ?o ");
		// Ignore case
		sparql.append("FILTER regex(str(?o), \"" + label + "\", \"i\").}");
		return sparql.toString();

		// The query should be (substitute type and label but keep <> and "" )
		// SELECT ?resource WHERE { ?resource a <type> . ?resource <http://www.w3.org/2000/01/rdf-schema#label> ?o
		// FILTER regex(?o, "label", "i").}

	}

	// Web-tool queries
	@Deprecated
	public String getWFStatesQuery(final String user) {

		final StringBuilder sparql = new StringBuilder();
		sparql.append("PREFIX repo: <");
		sparql.append(EAGLE_I_REPO_URI);
		sparql.append(">");
		sparql.append("SELECT DISTINCT ?");
		sparql.append(STATE_VARIABLE);
		sparql.append(" WHERE {?");
		sparql.append(STATE_VARIABLE);
		sparql.append(" repo:hasWriter ?role . <");
		sparql.append(user);
		sparql.append("> repo:hasRole ?role .}");
		return sparql.toString();

		// select distinct ?s where {?s :hasWriter ?role . <user> :hasRole ?role}
	}

	public String getRetrieveLabelQuery(final EIURI uri) {
		final StringBuilder sparql = new StringBuilder();
		sparql.append("PREFIX rdfs: <");
		sparql.append(RDFS_URI);
		sparql.append("> ");
		sparql.append("SELECT DISTINCT ?");
		sparql.append(LABEL_VARIABLE);
		sparql.append(" WHERE {<");
		sparql.append(uri.toString());
		sparql.append("> <http://www.w3.org/2000/01/rdf-schema#label> ?");
		sparql.append(LABEL_VARIABLE);
		sparql.append("}");
		return sparql.toString();
	}

	/**
	 * Returns a sparql query that gets all the resources with owner = user or
	 * no owner
	 * 
	 * @param user
	 * @return the sparql query string
	 */
	@Deprecated
	public String getAllResourcesQuery(final String user, final boolean strictOwnerFilter) {
		// final StringBuilder graphPattern = new StringBuilder();
		// graphPattern.append("?s repo:hasWorkflowState ?state . ");
		// graphPattern.append(allTypesPattern());
		// .return assembleQuery(user, graphPattern.toString());
		return getFilterQuery(user, EIURI.NULL_EIURI, EIURI.NULL_EIURI, EIURI.NULL_EIURI, strictOwnerFilter);
	}

	@Deprecated
	public String getReferencedByQuery(final String user, final EIURI resourceUri, final boolean strictOwnerFilter) {
		final StringBuilder sparql = new StringBuilder();
		sparql.append(namespaces());
		sparql.append(selectClause());
		sparql.append(commonPattern());
		sparql.append(typePattern());
		sparql.append(referencedByPattern(resourceUri));
		sparql.append(typeRestrictionPattern(EIURI.NULL_EIURI, false));
		sparql.append(stateRestrictionPattern(EIURI.NULL_EIURI));
		sparql.append(labRestrictionPattern(EIURI.NULL_EIURI));
		sparql.append(ownerRestrictionPattern(EIURI.create(user), strictOwnerFilter));
		sparql.append(closingClause());
		return sparql.toString();
	}

	public String referencedByPattern(final EIURI resourceUri) {
		final StringBuilder sparql = new StringBuilder();
		sparql.append("?");
		sparql.append(SUBJECT_VARIABLE);
		sparql.append(" ?anyPredicate <");
		sparql.append(resourceUri.toString());
		sparql.append("> . ");
		return sparql.toString();
	}


	@Deprecated
	public String getFilterQuery(final String user, final EIURI classUri, final EIURI state, final EIURI lab, final boolean strictOwnerFilter) {
		final StringBuilder sparql = new StringBuilder();
		sparql.append(namespaces());
		sparql.append(selectClause());
		sparql.append(commonPattern());
		sparql.append(typePattern());
		sparql.append(typeRestrictionPattern(classUri, true));
		sparql.append(stateRestrictionPattern(state));
		sparql.append(labRestrictionPattern(lab));
		sparql.append(ownerRestrictionPattern(EIURI.create(user), strictOwnerFilter));
		sparql.append(closingClause());
		return sparql.toString();

	}

	@Deprecated
	private String namespaces() {
		final StringBuilder namespaces = new StringBuilder();
		namespaces.append("PREFIX dcterms: <");
		namespaces.append(DC_URI);
		namespaces.append("> ");
		namespaces.append("PREFIX rdf: <");
		namespaces.append(RDF_URI);
		namespaces.append("> ");
		namespaces.append("PREFIX rdfs: <");
		namespaces.append(RDFS_URI);
		namespaces.append("> ");
		namespaces.append("PREFIX repo: <");
		namespaces.append(EAGLE_I_REPO_URI);
		namespaces.append("> ");
		return namespaces.toString();
	}

	@Deprecated
	private String selectClause() {
		final StringBuilder sparql = new StringBuilder();
		sparql.append("SELECT DISTINCT ");
		for (final String variable : resultSetVariables) {
			sparql.append("?");
			sparql.append(variable);
			sparql.append(" ");
		}
		sparql.append("WHERE {");
		return sparql.toString();
	}

	@Deprecated
	private String commonPattern() {
		final StringBuilder sparql = new StringBuilder();
		sparql.append("?");
		sparql.append(SUBJECT_VARIABLE);
		sparql.append(" <http://www.w3.org/2000/01/rdf-schema#label> ?");
		sparql.append(LABEL_VARIABLE);
		sparql.append(" . ?");
		sparql.append(SUBJECT_VARIABLE);
		sparql.append(" dcterms:created ?");
		sparql.append(CREATION_DATE_VARIABLE);
		sparql.append(" . ");
		return sparql.toString();
	}

	@Deprecated
	private String typePattern() {
		return "graph ?g{?" + SUBJECT_VARIABLE + " a ?" + TYPE_VARIABLE + "} filter(?g != repo:NG_Inferred) . ";
	}

	public String allTypesPattern(final boolean excludeSomeTypes) {
		// Need to use a variable different than ?t,
		// because that gets only the triples that are not inferred types
		final StringBuilder sparql = new StringBuilder();
		sparql.append("?");
		sparql.append(SUBJECT_VARIABLE);
		sparql.append(" a ?rt . ");
		sparql.append("?rt <http://eagle-i.org/ont/app/1.0/inClassGroup> <http://eagle-i.org/ont/app/1.0/ClassGroup/resourceRoot> ");
		// FIXME manually exclude Organizations and Persons as resources
		// should be done by annotations in ontology
		if (excludeSomeTypes) {
			sparql.append("filter(?rt != <");
			sparql.append(OBI_ORGANIZATION);
			sparql.append("> && ?rt != <");
			sparql.append(FOAF_PERSON);
			sparql.append(">) .");
		}
		return sparql.toString();
	}

	public String includeOwnerLabel() {
		final StringBuilder sparql = new StringBuilder();
		sparql.append("OPTIONAL{?").append(SUBJECT_VARIABLE);
		sparql.append(" <http://eagle-i.org/ont/repo/1.0/hasWorkflowOwner> ?").append(OWNER_VARIABLE);
		sparql.append(". ?").append(OWNER_VARIABLE);
		sparql.append(" <http://www.w3.org/2000/01/rdf-schema#label> ?");
		sparql.append(OWNER_NAME_VARIABLE).append("} . ");
		return sparql.toString();
	}

	public String getAllPossibleOwners() {
		final StringBuilder sparql = new StringBuilder();
		sparql.append("SELECT DISTINCT ?").append(OWNER_VARIABLE);
		sparql.append(" ?").append(OWNER_NAME_VARIABLE);
		sparql.append(" WHERE { ?").append(SUBJECT_VARIABLE);
		sparql.append(" <http://eagle-i.org/ont/repo/1.0/hasWorkflowOwner> ?").append(OWNER_VARIABLE);
		sparql.append(". ?").append(OWNER_VARIABLE);
		sparql.append(" <http://www.w3.org/2000/01/rdf-schema#label> ?");
		sparql.append(OWNER_NAME_VARIABLE).append("}");
		return sparql.toString();
	}

	public String ownerRestrictionPattern(final EIURI user, final boolean strictOwnerFilter) {
		final StringBuilder sparql = new StringBuilder();
		if (strictOwnerFilter) {
			sparql.append("?");
			sparql.append(SUBJECT_VARIABLE);
			sparql.append(" <http://eagle-i.org/ont/repo/1.0/hasWorkflowOwner> ?");
			sparql.append(OWNER_VARIABLE);
			sparql.append(". FILTER(?");
			sparql.append(OWNER_VARIABLE);
			sparql.append("= <");
		} else {
			sparql.append("OPTIONAL {?");
			sparql.append(SUBJECT_VARIABLE);
			sparql.append(" <http://eagle-i.org/ont/repo/1.0/hasWorkflowOwner> ?");
			sparql.append(OWNER_VARIABLE);
			sparql.append("} . FILTER(!bound(?");
			sparql.append(OWNER_VARIABLE);
			sparql.append(") || ?");
			sparql.append(OWNER_VARIABLE);
			sparql.append(" = <");
		}
		sparql.append(user.toString());
		sparql.append(">) . ");
		return sparql.toString();
	}

	private String typeRestrictionPattern(final EIURI classUri, final boolean excludeSomeTypes) {
		final StringBuilder sparql = new StringBuilder();
		if (!isNull(classUri)) {
			sparql.append("?");
			sparql.append(SUBJECT_VARIABLE);
			sparql.append(" a <");
			sparql.append(classUri.toString());
			sparql.append("> .");
		} else {
			sparql.append(allTypesPattern(excludeSomeTypes));
		}
		return sparql.toString();
	}

	private String stateRestrictionPattern(final EIURI state) {
		final StringBuilder sparql = new StringBuilder();
		if (!isNull(state)) {
			sparql.append("?");
			sparql.append(SUBJECT_VARIABLE);
			sparql.append(" repo:hasWorkflowState <");
			sparql.append(state.toString());
			sparql.append("> . ?");
			sparql.append(SUBJECT_VARIABLE);
			sparql.append(" repo:hasWorkflowState ?");
			sparql.append(STATE_VARIABLE);
			sparql.append(" . ");
		} else {
			sparql.append("OPTIONAL {?");
			sparql.append(SUBJECT_VARIABLE);
			sparql.append(" repo:hasWorkflowState ?");
			sparql.append(STATE_VARIABLE);
			sparql.append("} . ");
		}
		return sparql.toString();
	}

	public String labRestrictionPattern(final EIURI lab) {
		final StringBuilder sparql = new StringBuilder();
		if (!isNull(lab)) {
			sparql.append("?");
			sparql.append(SUBJECT_VARIABLE);
			sparql.append(" ?labProperty ?");
			sparql.append(LAB_VARIABLE);
			sparql.append(" . ?");
			sparql.append(LAB_VARIABLE);
			sparql.append(" <http://www.w3.org/2000/01/rdf-schema#label> ?");
			sparql.append(LAB_NAME_VARIABLE);
			sparql.append(" . ?labProperty <http://eagle-i.org/ont/app/1.0/inPropertyGroup> <http://eagle-i.org/ont/app/1.0/PropertyGroup/relatedLab> . filter(?");
			sparql.append(LAB_VARIABLE);
			sparql.append(" = <");
			sparql.append(lab.toString());
			sparql.append(">) . ");
		} else {
			sparql.append("OPTIONAL {?");
			sparql.append(SUBJECT_VARIABLE);
			sparql.append(" ?labProperty ?");
			sparql.append(LAB_VARIABLE);
			sparql.append(" . ?labProperty <http://eagle-i.org/ont/app/1.0/inPropertyGroup> <http://eagle-i.org/ont/app/1.0/PropertyGroup/relatedLab> . ?");
			sparql.append(LAB_VARIABLE);
			sparql.append(" a <");
			sparql.append(DataToolsOntConstants.EI_LAB);
			sparql.append("> . ?");
			sparql.append(LAB_VARIABLE);
			sparql.append(" <http://www.w3.org/2000/01/rdf-schema#label> ?");
			sparql.append(LAB_NAME_VARIABLE);
			sparql.append("} . ");
		}
		return sparql.toString();
	}

	public String getResourcesETLedFromFileQuery(String fileName) {
		final StringBuilder sparql = new StringBuilder();
		sparql.append("PREFIX rdfs: <");
		sparql.append(RDFS_URI);
		sparql.append("> ");
		sparql.append("PREFIX rdf: <");
		sparql.append(RDF_URI);
		sparql.append("> ");
		sparql.append("PREFIX " + EAGLE_I_DT_PREFIX + ": <");
		sparql.append(EAGLE_I_DT_URI);
		sparql.append("> ");
		sparql.append("SELECT ?resource WHERE {?resource  " + EAGLE_I_DT_PREFIX + ":source_file  \"" + fileName + "\".}");
		return sparql.toString();
	}

	private String closingClause() {
		return "} order by repo:upperCaseStr(?" + LABEL_VARIABLE + ")";
	}

	private boolean isNull(final EIURI uri) {
		return uri == null || uri.equals(EIURI.NULL_EIURI) || "".equals(uri.toString());
	}

	public String modifiedDatePattern() {
		return " OPTIONAL {?" + SPARQLConstants.SUBJECT_VARIABLE + " <http://purl.org/dc/terms/modified> ?" + MODIFIED_DATE_VARIABLE + "} .";
	}
	
	public String getModifiedDateQuery(EIURI uri) {
		return "SELECT ?" + MODIFIED_DATE_VARIABLE + " WHERE { <" + uri + ">  <http://purl.org/dc/terms/modified> ?" + MODIFIED_DATE_VARIABLE + " }";
	}
	
	public String additionalLabelsPattern() {
		StringBuilder sparql = new StringBuilder();
		sparql.append( "OPTIONAL {?").append( STATE_VARIABLE ).append( " <http://www.w3.org/2000/01/rdf-schema#label> ?").append( STATE_LABEL_VARIABLE ).append( "} . " );
		sparql.append( "OPTIONAL {?").append( TYPE_VARIABLE ).append( " <http://www.w3.org/2000/01/rdf-schema#label> ?").append( TYPE_LABEL_VARIABLE ).append( "} . " );
		return sparql.toString();
	}
}
