/**
 * eagle-i Project
 * Harvard University
 * Apr 23, 2010
 */
package org.eaglei.datatools.jena;

import static org.eaglei.datatools.jena.SPARQLConstants.DATE_VARIABLE;
import static org.eaglei.datatools.jena.SPARQLConstants.LABEL_VARIABLE;
import static org.eaglei.datatools.jena.SPARQLConstants.OWNER_VARIABLE;
import static org.eaglei.datatools.jena.SPARQLConstants.STATE_VARIABLE;
import static org.eaglei.datatools.jena.SPARQLConstants.SUBJECT_VARIABLE;
import static org.eaglei.datatools.jena.SPARQLConstants.TYPE_VARIABLE;

import java.io.IOException;
import java.io.StringReader;
import java.io.StringWriter;
import java.util.ArrayList;
import java.util.Collections;
import java.util.HashMap;
import java.util.List;
import java.util.Map;
import java.util.Set;

import org.apache.commons.logging.Log;
import org.apache.commons.logging.LogFactory;
import org.eaglei.datatools.model.DataToolsEIOntModel;
import org.eaglei.model.EIClass;
import org.eaglei.model.EIDatatypeProperty;
import org.eaglei.model.EIEntity;
import org.eaglei.model.EIInstance;
import org.eaglei.model.EIObjectProperty;
import org.eaglei.model.EIProperty;
import org.eaglei.model.EIURI;

import com.hp.hpl.jena.query.QuerySolution;
import com.hp.hpl.jena.query.ResultSet;
import com.hp.hpl.jena.rdf.model.Model;
import com.hp.hpl.jena.rdf.model.ModelFactory;
import com.hp.hpl.jena.rdf.model.Property;
import com.hp.hpl.jena.rdf.model.RDFNode;
import com.hp.hpl.jena.rdf.model.Resource;
import com.hp.hpl.jena.rdf.model.SimpleSelector;
import com.hp.hpl.jena.rdf.model.Statement;
import com.hp.hpl.jena.rdf.model.StmtIterator;
import com.hp.hpl.jena.vocabulary.RDF;
import com.hp.hpl.jena.vocabulary.RDFS;
import static org.eaglei.datatools.jena.MetadataConstants.*;

/**
 * @author Daniela Bourges-Waldegg
 * @author Ricardo De Lima
 * 
 *         April 11, 2010
 * 
 *         Center for Biomedical Informatics (CBMI)
 * @link https://cbmi.med.harvard.edu/
 * 
 * 
 */
 
 
public class EIInstanceFactory {
	private static final Log logger = LogFactory
			.getLog(EIInstanceFactory.class);
	
	private static final boolean isDebugEnabled = logger.isDebugEnabled();
	
	public static final EIInstanceFactory INSTANCE = new EIInstanceFactory();
	
    public static EIInstanceFactory getInstance()
    {
        return INSTANCE;
    }


    private  EIInstanceFactory()
    {
    }

    
    /**
     * Serialize an EInstance into an RDF language
     * @param instance
     * @param lang - one of the RDF languages supported by Jena; predefined values are 
     * "RDF/XML", "RDF/XML-ABBREV", "N-TRIPLE", "TURTLE", (and "TTL") and "N3". The default value, represented by null is "RDF/XML".
     * @return - the serialized EIInstance as a STring
     */
    
    public String serialize(final EIInstance instance, final String lang)   {
    	if(instance == null)
    		return null;
    	Model model = convertToJenaModel(instance);
        //Serialize model to String in specified format
        StringWriter sw = new StringWriter();
        try {
        	//TODO validate format
        	model.write(sw, lang);
        	String s = sw.toString();
        	sw.flush();
        	return s;
        } finally {
        	try {if(sw != null) sw.close(); } catch (IOException e){/*not much to do*/}
        }      
    	
    }
    public Model convertToJenaModel(final EIInstance instance)   {
    	if(instance == null)
    		return null;
        Model model = ModelFactory.createDefaultModel();
        //Create the resource that will be the subject of all statements
        Resource resourceInstance = model.createResource(instance.getInstanceURI().toString());

        //Set its type and label
        Resource eiType = model.createResource(instance.getInstanceType().getURI().toString());
        model.add(model.createStatement(resourceInstance, RDF.type, eiType));
        model.add(model.createStatement(resourceInstance, RDFS.label, instance.getInstanceLabel()));

        //Set other types
        List<EIEntity> otherEITypes = instance.getOtherEITypes();
        if(otherEITypes != null) {
        	for(EIEntity t : otherEITypes) {
        		final Resource typeResource = model.createResource(t.getURI().toString());
        		model.add(model.createStatement(resourceInstance, RDF.type, typeResource));
        	}
        }
        //Set all other properties
        Map<EIEntity, Set<String>> dataProps = instance.getDatatypeProperties();
        Map<EIEntity, Set<EIURI>> objectProps = instance.getObjectProperties();
        Map<EIEntity, Set<String>> literalProps = instance.getNonOntologyLiteralProperties();
        Map<EIEntity, Set<EIURI>> resourceProps = instance.getNonOntologyResourceProperties();
        Map<EIEntity, EIURI> roResourceProps = instance.getReadOnlyResourceProperties();
        Map<EIEntity, String> roLiteralProps = instance.getReadOnlyLiteralProperties();
        
        for(Map.Entry<EIEntity, Set<String>> entry : dataProps.entrySet())
        {
            final Property p = model.createProperty(entry.getKey().getURI().toString());
            final Set<String> values = entry.getValue();
            for(String value : values)
            	model.add(model.createStatement(resourceInstance, p, value));
        }

        for(Map.Entry<EIEntity, Set<EIURI>> entry : objectProps.entrySet())
        {
            final Property p = model.createProperty(entry.getKey().getURI().toString());
            final Set<EIURI> values = entry.getValue();
            for(EIURI value : values) {
            	Resource valueResource = model.createResource(value.toString());
            	model.add(model.createStatement(resourceInstance, p, valueResource));
            }
        }
        
        for(Map.Entry<EIEntity, Set<String>> entry : literalProps.entrySet())
        {
            final Property p = model.createProperty(entry.getKey().getURI().toString());
            final Set<String> values = entry.getValue();
            for(String value : values)
            	model.add(model.createStatement(resourceInstance, p, value));
        }

        for(Map.Entry<EIEntity, Set<EIURI>> entry : resourceProps.entrySet())
        {
            final Property p = model.createProperty(entry.getKey().getURI().toString());
            final Set<EIURI> values = entry.getValue();
            for(EIURI value : values) {
            	Resource valueResource = model.createResource(value.toString());
            	model.add(model.createStatement(resourceInstance, p, valueResource));         	
            }
        }
        
        for(Map.Entry<EIEntity, String> entry : roLiteralProps.entrySet()) {
        	final Property p = model.createProperty(entry.getKey().getURI().toString());
        	model.add(model.createStatement(resourceInstance, p, entry.getValue()));
        
        }
        
        for(Map.Entry<EIEntity, EIURI> entry : roResourceProps.entrySet()) {
        	final Property p = model.createProperty(entry.getKey().getURI().toString());
        	final Resource value = model.createResource(entry.getValue().toString());
        	model.add(model.createStatement(resourceInstance, p, value));
        }
        
        return model;
    }
  
    

    /**
     * Create an EIInstance from an RDF String serialization in one of the RDF languages supported by Jena
     * @param instanceUri - the instance uri (all the triples in the rdf parameter should have it as subject)
     * @param rdf 
     * @param lang
     * @return
     */  
    //TODO deal with cases where the RDF does not represent an EIInstance
    
    public EIInstance create(final EIURI instanceUri, final String rdf, final String lang) {
    	if(instanceUri == null || instanceUri.toString().length() ==0 || rdf == null || rdf.length() == 0) {
    		return null;
    	}
    	Model model = ModelFactory.createDefaultModel();
    	model.read(new StringReader(rdf), null, lang);
    	return create(instanceUri, model);
    }
    
    /**
     * Create an EIInstance with no properties other than an eiType (at least one type must be set)
     * @param typeUri - type of the EInstance; if more than one, add can be used
     * @param instanceEntity - entity containing the instance Uri and its label
     * @return null if the typeUri is not an EI Class
     */
    public EIInstance createEmpty(final EIURI typeUri, final EIEntity instanceEntity) {
    	//type may be null
    	if(instanceEntity == null)  {
    		return null;
    	}     	
    	EIClass instanceClass = DataToolsEIOntModel.INSTANCE.getClass(typeUri);
    	if(instanceClass == null)
    		return null;
    	String label = instanceClass.getEntity().getLabel();
    	//FIXME temporary fix to lack of capitalization
    	instanceClass.getEntity().setLabel(capitalizeFirst(label));
    	return EIInstance.create(instanceClass.getEntity(), instanceEntity);
    }

    
    /**
     * Create an EIInstance from a Jena model, with a given instanceUri
     * @param instanceUri - the instance uri (all the triples in the model should have it as subject)
     * @param model
     * @return
     */
    public EIInstance create(final EIURI instanceUri, final Model model)
    {
    	if(instanceUri == null || model == null) {
    		if(isDebugEnabled) logger.debug("null parameters; no EIInstance was created");
    		return null;
    	}
    	final Resource subject = model.getResource(instanceUri.toString());
    	//Get resource will create a resource if one doens't exist already, so need to check
    	if(!model.contains(subject, null, (RDFNode)null)) {
    		if(isDebugEnabled) logger.debug("model doesn't contain subject; no EIInstance was created");
    		return null;
    	}
    	//Instances can have multiple types
    	List<EIClass> eiClasses = getEIClasses(model, subject);

    	//Nothing in the received model matches an EIClass
    	if(eiClasses == null) {
    		if(isDebugEnabled)logger.debug("No eagle-i types found in the model");
    		return null;
    	}
    	
    	//FIXME for now set instance type to first in list; 
    	//The most common case is only one ei type; we can't really know what to put there otherwise
		//Ideally the most specific type but that's hard to know
    	final EIClass instanceClass = eiClasses.get(0);
    	eiClasses.remove(0);
    	
    	final String instanceClassUri = instanceClass.getEntity().getURI().toString();
    	final Statement labelStatement = subject.getProperty(RDFS.label);
    	if(labelStatement == null) {
    		if(isDebugEnabled)logger.debug("RDFS Label is not set for instance");
    		return null;
    	}
        final String instanceLabel = labelStatement.getString();
        if(isDebugEnabled) logger.debug("Creating an instance of class: " + instanceClass.getEntity().toString() + " with URI: " + instanceUri + " and label: " + instanceLabel);

        // The entity of the new EIInstance
        final EIEntity instanceEntity = EIEntity.create(instanceUri, instanceLabel);

        // create IEInstance
        final EIInstance ei = EIInstance.create(instanceClass.getEntity(), instanceEntity);
        //If there were more than one ei types found previously
        if(!eiClasses.isEmpty()) {
        	final List<EIEntity> eiTypes = new ArrayList<EIEntity>(eiClasses.size());
        	for(EIClass c : eiClasses)
        		eiTypes.add(c.getEntity());
        	ei.setOtherEITypes(eiTypes);
        }
        // query the model for all statements about this subject
        final StmtIterator iter = model.listStatements(subject, null, (RDFNode) null);
        if(isDebugEnabled) logger.debug("Searching for statements with subject: " + instanceUri.toString() + " of class: " + instanceClass.getEntity() + " and label: " + instanceLabel);
       
        //Iterate to add properties in the different maps
        while(iter.hasNext())
        {
            final Statement statement = iter.nextStatement();
            final Property predicate = statement.getPredicate();
            //No need to process label, as it was done previously
            if(predicate.equals(RDFS.label))
            	continue;
            
            final EIURI propertyUri = EIURI.create(predicate.getURI());
            final RDFNode o = statement.getObject();
            
            //No need to process tyep if it is instanceClass
            if(predicate.equals(RDF.type) && o.isResource() && instanceClassUri.equals(((Resource)o).getURI())) {
            	continue;
            }
            
            //No need to process types already in eiClasses
            if(predicate.equals(RDF.type) && o.isResource() && containsUri(eiClasses, ((Resource)o).getURI())) {
            	continue;
            }
            

            // Check if property is in ontology and add accordingly
            final EIProperty p = getEIOntProperty(instanceClass.getEntity().getURI(), propertyUri);
            if(p != null)
            	addEIProperty(ei, p, propertyUri, o);
            else
            	addNonEIProperty(ei, predicate, propertyUri, o);
        }
        return ei;
    }
    
    
    
    
    /**
     * Batch creation of a list of EIInstances from a model
     * @param model
     * @return
     */
    public List<EIInstance> create(final Model model) {
    	if(model == null)
    		return Collections.EMPTY_LIST;
    	final List<EIInstance> instances = new ArrayList<EIInstance>();
    	final Set<Resource> subjects = model.listSubjects().toSet();
		//create an EIInstance per subject
		for(Resource r : subjects) {
			final Model results = model.query(new SimpleSelector(r, null, (RDFNode)null));			
			EIInstance ei = create(EIURI.create(r.getURI()), results);
			if(ei != null)
				instances.add(ei);
		}
		return instances;
    }
    
    
    //FIXME this is largely duplicating SearchResult; to review
    /**
     * Create method for abridged EIInstances (label, type, owner, WFState, creation date)
     * @param resultSet
     * @return
     */
    
    public List<EIInstance> create(final ResultSet resultSet) {
    	if(resultSet == null)
    		return Collections.EMPTY_LIST;
    	final Map<EIURI, EIInstance> instances = new HashMap<EIURI, EIInstance>();
    	//Need to preserve ordering to avoid a sort operation; this list will contain live elements
    	final List<EIInstance> instanceList = new ArrayList<EIInstance>();
    	while (resultSet.hasNext()) {
    		final QuerySolution qs = resultSet.next();
    		//logger.debug("QuerySolution: " + qs);
    		if(qs.contains(SUBJECT_VARIABLE) && qs.contains(TYPE_VARIABLE) && qs.contains(LABEL_VARIABLE)) {
    			final EIURI instanceUri = EIURI.create(qs.getResource(SUBJECT_VARIABLE).getURI());
    			final EIURI instanceType = EIURI.create(qs.getResource(TYPE_VARIABLE).getURI());
    			final String label = qs.getLiteral(LABEL_VARIABLE).getString();
    			//type is not an EIClass ; we can skip solution
    			if (DataToolsEIOntModel.INSTANCE.getClass(instanceType)== null)
    				continue;
    			
    			EIInstance ei;
    			if(!instances.containsKey(instanceUri)) {
    				ei = createEmpty(instanceType, EIEntity.create(instanceUri, label));
    				instances.put(instanceUri, ei);
    				instanceList.add(ei);
    				if(isDebugEnabled) logger.debug("Creating new EIInstance: " + ei);
    				
    			} else {
    				ei = instances.get(instanceUri);
    				ei.addEIType(EIEntity.create(instanceType, ""));
    				if(isDebugEnabled) logger.debug("Adding type: " + instanceType + " to existing EIInstance :" +ei);
    			}
    			if(qs.contains(OWNER_VARIABLE)) {
    				final EIURI owner = EIURI.create(qs.getResource(OWNER_VARIABLE).getURI());
    				ei.setWFOwner(owner);
    			}
    			if(qs.contains(STATE_VARIABLE)) {
    				final EIURI state = EIURI.create(qs.getResource(STATE_VARIABLE).getURI());
    				ei.setWFState(state);
    			}
    			if(qs.contains(DATE_VARIABLE)) {
    				final String date = qs.getLiteral(DATE_VARIABLE).getString();
    				ei.setCreationDate(date);
    			}
    		} else {
    			continue;
    		}
    	}
    	//This adds way too much time
    	//Collections.sort(instanceList);
    	return instanceList;
    }
    
    
    /**
     * Find a property for a class in the eagle-i ontology.  
     * @param instanceClassUri
     * @param propertyUri
     * @return the property if found, null otherwise
     */
    private EIProperty getEIOntProperty(final EIURI instanceClassUri, final EIURI propertyUri) {
    	if(instanceClassUri == null || propertyUri == null) 
    		return null;

    	final List<EIProperty> properties = DataToolsEIOntModel.INSTANCE.getProperties(instanceClassUri);
    	if(properties == null || properties.isEmpty())
    		return null;
    	for (EIProperty p : properties) {
    		final EIURI propUri = p.getEntity().getURI();
    		if(propUri.equals(propertyUri))
    			return p;   		
    	}
    	//If property wasn't found in domain
    	return null;
    }
    
    
    /**
     * 
     * @param model
     * @param subject
     * @return a list of EIEntities if the model has EITypes
			   null if EITypes were found in model
     */
    
    private List<EIClass> getEIClasses(Model model, Resource subject) {
    	final List<Statement> typeStatements = model.listStatements(subject, RDF.type, (Resource)null).toList();
    	if(typeStatements == null || typeStatements.isEmpty())
    		return null;
    	List<EIClass> eiClasses = new ArrayList<EIClass>();
    	for(Statement st : typeStatements) {
    		final Resource type = (Resource)st.getObject();
    		final EIURI typeUri = EIURI.create(type.getURI());
    		//logger.debug("Trying to match URI to EICLass: " + typeUri);
    		final EIClass eiClass = DataToolsEIOntModel.INSTANCE.getClass(typeUri);
    		if(eiClass!=null) {
    			eiClasses.add(eiClass);
    		}
    	}
    	return (eiClasses.isEmpty()? null : eiClasses);
    }
    
    
    private void addEIProperty(final EIInstance ei, final EIProperty p, final EIURI propertyUri, final RDFNode o) {

        	final String preferredLabel = DataToolsEIOntModel.INSTANCE.getPreferredLabel(propertyUri);
        	final EIEntity propEntity = EIEntity.create(propertyUri, capitalizeFirst(preferredLabel));
        	if(p instanceof EIDatatypeProperty) {
        		if(o.isLiteral()) {
        			final String value = o.toString();
        			if(isDebugEnabled)logger.debug("adding property: [" + propertyUri + "] with literal value : [" + value + "] ");
        			ei.addDatattypePropertyToInstance(propEntity, value);
        		} else {
        			if(isDebugEnabled)logger.debug("property: [" + propertyUri + "] found in ontology but value is not literal; adding as non-ontology property ");
        			ei.addNonOntologyResourceProperty(propEntity, EIURI.create(((Resource)o).getURI()));
        		}
        	} else if (p instanceof EIObjectProperty){
        		if(o.isResource()) {
        			final EIURI objectUri = EIURI.create(((Resource)o).getURI());
        			if(isDebugEnabled)logger.debug("adding property: [" + propertyUri + "] with object value : [" + objectUri + "] ");
        			ei.addObjectPropertyToInstance(propEntity, objectUri);	
        		} else {
        			if(isDebugEnabled)logger.debug("property: [" + propertyUri + "] found in ontology but value is not resource; adding as non-ontology property ");
        			ei.addNonOntologyLiteralProperty(propEntity, o.toString());
        		}
        	}
        }
    
        private void addNonEIProperty(final EIInstance ei, final Property predicate, final EIURI propertyUri, final RDFNode o) {
        	if(isDebugEnabled) logger.debug("Property not found in ontology; we'll still add it: " + propertyUri);
        	EIEntity propEntity;
        	if(readOnlyResources.containsKey(propertyUri))
        		propEntity = readOnlyResources.get(propertyUri);
        	else if (readOnlyLiterals.containsKey(propertyUri))
        		propEntity = readOnlyLiterals.get(propertyUri);
        	else {
        		final String predicateLabel = predicate.getLocalName();
        		propEntity = EIEntity.create(propertyUri, predicateLabel);
        	}
        	if(o.isLiteral()) {
        		if(readOnlyLiterals.containsKey(propertyUri))
        			ei.setReadOnlyLiteralProperty(propEntity, o.toString());
        		else
        			ei.addNonOntologyLiteralProperty(propEntity, o.toString());
        	} else if (o.isResource())
        		if(readOnlyResources.containsKey(propertyUri))
        			ei.setReadOnlyResourceProperty(propEntity, EIURI.create(((Resource)o).getURI()));
        		else
        			ei.addNonOntologyResourceProperty(propEntity, EIURI.create(((Resource)o).getURI()));
        	else
        		if(isDebugEnabled) logger.debug("Could not add property " + propertyUri);
        }
    
        
        private boolean containsUri(List<EIClass> eiClasses, String uri) {
        	if(eiClasses == null || uri == null)
        		return false;
        	for(EIClass eiClass : eiClasses) {
        		if(uri.equalsIgnoreCase(eiClass.getEntity().getURI().toString()))
        			return true;
        	}
        	return false;
        }

        private String capitalizeFirst(String s) {
        	char first = s.charAt(0);
        	String S =Character.toUpperCase(first) + s.substring(1);
        	return S;
        }
}
