package org.eaglei.datatools.server;

import java.io.File;
import java.net.URL;
import java.util.ArrayList;
import java.util.List;
import java.util.Map;
import java.util.Set;

import javax.servlet.ServletException;

import org.apache.commons.logging.Log;
import org.apache.commons.logging.LogFactory;
import org.eaglei.datatools.client.rpc.RepositoryToolsModelService;
import org.eaglei.datatools.config.DatatoolsConfiguration;
import org.eaglei.datatools.jena.EIInstanceFactory;
import org.eaglei.datatools.jena.RESTRepositoryProviderFactory;
import org.eaglei.datatools.model.DataToolsOntConstants;
import org.eaglei.datatools.provider.RepositoryProvider;
import org.eaglei.datatools.provider.RepositoryProviderFactory;
import org.eaglei.model.EIClass;
import org.eaglei.model.EIEntity;
import org.eaglei.model.EIEquivalentClass;
import org.eaglei.model.EIInstance;
import org.eaglei.model.EIInstanceMinimal;
import org.eaglei.model.EIOntModel;
import org.eaglei.model.EIProperty;
import org.eaglei.model.EIURI;
import org.eaglei.model.gwt.rpc.LoggedException;
import org.eaglei.model.gwt.rpc.ModelService;
import org.springframework.context.ApplicationContext;
import org.springframework.web.context.WebApplicationContext;
import org.springframework.web.context.support.WebApplicationContextUtils;

import com.google.gwt.user.server.rpc.RemoteServiceServlet;

public class RepositoryToolsModelServlet extends RemoteServiceServlet implements RepositoryToolsModelService
{

    /**
     * 
     */
    private static final long serialVersionUID = 1L;
    private static final Log log = LogFactory.getLog(RepositoryToolsModelServlet.class);
    private RepositoryProvider provider = null;
    private EIOntModel ontModel;
    private DatatoolsConfiguration config;

    @Override
    public void init() throws ServletException
    {

        config = getConfiguration();
        try
        {
            if(config != null)
            {
            	if (log.isDebugEnabled()) log.debug("about to create provider");
                provider = createProvider();
            }
            else
            {
            	if (log.isDebugEnabled()) log.debug("about to load defaults (null config)");
                loadDefaults();
            }

        }
        catch(Exception e)
        {
            log.error("Error initializing RepositoryToolsModelServlet: " + e);
            throw new ServletException(e);
        }
        
        initializeOntology();

        lazyLoadFactories();

        log.info("init succesful");

    }

	protected void initializeOntology() {
		WebApplicationContext ctx = 
            WebApplicationContextUtils.getWebApplicationContext(getServletContext());
		
       doDependencyInjection(ctx);
	}
	
	/*
     * Allow test subclasses to override servlet init and invoke this
     * with their own ApplicationContext.
     */
    protected void doDependencyInjection(ApplicationContext springAppContext) {
        this.ontModel = springAppContext.getBean(EIOntModel.class);
    }

    private RepositoryProvider createProvider() throws Exception
    {

        final String providerFactoryClass = config.getDatatoolsRepositoryProviderFactory();
        final Class factoryClass = Class.forName(providerFactoryClass);
        final RepositoryProviderFactory factory = (RepositoryProviderFactory) factoryClass.newInstance();
        return factory.createRepositoryProvider(config);

    }

    private void lazyLoadFactories()
    {
        EIInstanceFactory.getInstance();
    }

    private void loadDefaults() throws Exception
    {
        // default provider is the REST provider
        log.info("Loading default provider, RESTProvider: ");

        final RepositoryProviderFactory factory = new RESTRepositoryProviderFactory();
        provider = factory.createRepositoryProvider(config);
        log.info("Loading default provider, RESTProvider: " + factory.toString());

    }
    
	@Override
	public void createInstance(String session, EIInstance instance)
			throws Exception {
		try {
			if (instance == null) {
				log.error("trying to create a null instance");
			}
			provider.createInstance(session, instance);
		} catch (Exception e) {
			log.error(e.getMessage());
			for (StackTraceElement element : e.getStackTrace())
			log.error(element.toString());
			throw new LoggedException(e.getMessage());
		}
		
	}

	@Override
	public void deleteInstance(String session, EIURI instanceUri)
			throws Exception {
		try {
			provider.deleteInstance(session, instanceUri);
		} catch (Exception e) {
			throw new LoggedException(e.getMessage());
		}
		
	}

    @Override
	public String updateInstance(String session, EIInstance instance, String token)
			throws Exception {
		try {
			return provider.updateInstance(session, instance, token);
		} catch (Exception e) {
			throw new LoggedException(e.getMessage());
		}
		
	}

    @Override
    public String[] login(final String user, final String password) throws Exception
    {
		try {
			return provider.login(user, password);
		} catch (Exception e) {
			throw new LoggedException("Repository error:" + e.getMessage());
		}
    }

    @Override
    public void logout(String session) throws Exception
    {
		try {
			provider.logout(session);
		} catch (Exception e) {
			throw new LoggedException(e.getMessage());
		}
    }

    @Override
    public String[] whoami(String session) throws Exception
    {
    	try {
    		return provider.whoami(session);
    	} catch (Exception e) {
    		log.error("whoami exception " + e);
			throw new LoggedException(e.getMessage());
		}
    }


    /*
     * Loads the DatatoolsConfiguration.
     */
    private DatatoolsConfiguration getConfiguration() throws ServletException
    {
        // check if the system property is specified
        String propFile = System.getProperty(DataToolsOntConstants.DATATOOLS_CONFIG);

        if(propFile == null)
        {
            // use the default
            propFile = DataToolsOntConstants.DEFAULT_DATATOOLS_CONFIG;
        }
        log.debug("Using properties file " + propFile);

        final URL configURL = this.getClass().getClassLoader().getResource(propFile);
        if(configURL == null)
        {
            log.error("Could not locate " + propFile + " on classpath ");
            throw new ServletException("Could not locate " + propFile + " on classpath ");
        }
        try
        {
            final File configFile = new File(configURL.toURI());
            return new DatatoolsConfiguration(configFile);
        }
        catch(Exception e)
        {
            log.error("Error loading configuration from " + configURL + " " + e);
            throw new ServletException("Error loading configuration from " + configURL + " " + e);
        }
    }

    @Override
    public boolean isOnline()
    {
        
        return provider.isOnline();
    }
	
    @Override
    public List<EIInstanceMinimal> EIQuery(final String session, final String query) throws Exception{
		try {
			return provider.EIQuery(session, query);
		} catch (Exception e) {
			throw new LoggedException(e.getMessage());
		}
    }
    
	@Override
	@Deprecated
	public void uploadInstances(String session, String rdf) throws Exception {
		provider.uploadInstances(session, rdf);
		
	}

	@Override
	public EIInstance getEmptyEIInstance(String session, EIURI classUri,
			EIEntity instanceEntity) throws Exception {
		try {
			return provider.getEmptyEIInstance(session, classUri,
					instanceEntity);
		} catch (Exception e) {
			throw new LoggedException(e.getMessage());
		}
	}
	
	@Override
	public EIInstance getEmptyEIInstance(String session, EIURI classUri) throws Exception {
		try {
			EIInstance instance = provider.getEmptyEIInstance(session, classUri);
			if (instance.getInstanceClass() == null) {
				instance.setInstanceClass(ontModel.getClass(classUri));
			}
			return instance;
		} catch (Exception e) {
			throw new LoggedException(e.getMessage());
		}
	}
	
	@Override
	public EIInstance deepCopy(String session, EIURI originalUri) throws Exception {
		return provider.deepCopy(session, originalUri);
	} 

	@Override
	public List<EIURI> getNewInstanceID(String session, int count)
			throws Exception {
		try {
			return provider.getNewInstanceID(session, count);
		} catch (Exception e) {
			throw new LoggedException(e.getMessage());
		}
	}

	@Override
	public String query(String session, String sparql) throws Exception {
		try {
			return provider.query(session, sparql);
		} catch (Exception e) {
			throw new LoggedException(e.getMessage());
		}
	}

	@Override
	public EIInstance getInstance(String session, EIURI instanceID)
			throws Exception {
		return provider.getInstance(session, instanceID);
	}

	@Override
	public String claim(String session, EIURI uri, String claimant)
			throws Exception {
		try {
			return provider.claim(session, uri, claimant);
		} catch (Exception e) {
			throw new LoggedException(e.getMessage());
		}
	}


	public String[] claim(String session, EIURI[] uriArray, String claimant)
			throws Exception {
		try {
			String[] responses = new String[uriArray.length];
			int i=0;
			for(EIURI uri : uriArray)
			{
				 responses[i++]= provider.claim(session, uri, claimant);
			}
			return responses;
		} catch (Exception e) {
			throw new LoggedException(e.getMessage());
		}
	}

	
	
	@Override
	public String promote(String session, EIURI uri, EIURI newState)
			throws Exception {
		try
		{
			return provider.promote(session, uri, newState);
		}
		catch(Exception e)
		{
			throw new LoggedException(e.getMessage());
		}
	}
	
	

	public String[] bulkPromote(String session, List<EIURI> urisToPromote, EIURI newState)
			throws Exception {
		try
		{
			String[] responses = new String[urisToPromote.size()];
			int i=0;
			for(EIURI uri : urisToPromote)
			{
				 responses[i++]= provider.promote(session, uri, newState);
			}
			return responses;
			
		}
		catch(Exception e)
		{
			throw new LoggedException(e.getMessage());
		}
	}
	
	
	

	/* (non-Javadoc)
	 * @see org.eaglei.datatools.provider.RepositoryProvider#getWFStates(java.lang.String)
	 */
	@Override
	public String[] getWFStates(String session, String user) throws Exception {
		try
		{
			return provider.getWFStates(session, user);
		}
		catch(Exception e)
		{
			throw new LoggedException(e.getMessage());
		}
	}

	@Override
	public List<EIInstanceMinimal> getFilterQuery(final String session, String user, EIURI classUri,
			EIURI state, EIURI lab) throws Exception {
		try
		{
			return provider.getFilterQuery(session, user, classUri, state, lab);
		}
		catch(Exception e)
		{
			throw new LoggedException(e.getMessage());
		}	
	}
	
	@Override
	public List<EIInstanceMinimal> getFilterQuery(final String session, String user, EIURI classUri,
			EIURI state, EIURI lab, boolean strictOwnerFilter) throws Exception {
		try
		{
			return provider.getFilterQuery(session, user, classUri, state, lab, strictOwnerFilter);
		}
		catch(Exception e)
		{
			throw new LoggedException(e.getMessage());
		}	
	}
	
	
	@Override
	public String retrieveLabel(String session, EIURI uri) throws Exception {
		try
		{
			return provider.retrieveLabel(session, uri);
		}
		catch(Exception e)
		{
			throw new LoggedException(e.getMessage());
		}
	}

	@Override
	public Map<EIEntity, String> retrieveLabels(String session,	List<EIEntity> entities) throws Exception {
		try
		{
			return provider.retrieveLabels(session, entities);
		}
		catch(Exception e)
		{
			throw new LoggedException(e.getMessage());
		}
	}
	
	@Override
	public Map<EIURI, String> retrieveUriLabels(String session, List<EIURI> uris)
			throws Exception {
		return provider.retrieveUriLabels(session, uris);
	}
	
	@Override
	public EIClass getRootSuperclassForInstanceUri(String session, EIURI instanceUri) throws Exception {
		EIInstance instance = provider.getInstance(session, instanceUri);
		if (instance == null)
			return null;
		return getRootSuperClass(instance.getInstanceClass());
	}
	
	@Override
	public List<EIClass> getClassAndSuperclassesForInstanceUri(String session, EIURI instanceUri) throws Exception {
		EIInstance instance = provider.getInstance(session, instanceUri);
		if (instance == null)
			return null;
		List<EIClass> classes =  ontModel.getSuperClasses(instance.getInstanceType().getURI());
		if (classes == null || classes.isEmpty())
			classes = new ArrayList<EIClass>(1);
		classes.add(ontModel.getClass(instance.getInstanceType().getURI()));
		return classes;
	}


	@Override
	public EIClass getRootSuperClass(EIClass eclass) throws LoggedException {
		while(eclass.hasSuperClass()) { 
			eclass = ontModel.getSuperClass(eclass);
		}
		return eclass;
	}
	
	@Override
	public EIClass getSuperClass(EIClass eclass) throws LoggedException {
		eclass = ontModel.getSuperClass(eclass);
		return eclass;
	}

	
	
	public boolean isModelClassURI(String uri) {
		return ontModel.isModelClassURI(uri);
	}

}
