package org.eaglei.institution.server;

import java.io.IOException;
import java.util.HashMap;
import java.util.Map;

import org.eaglei.model.EIEntity;
import org.eaglei.model.EIOntModel;
import org.eaglei.search.config.SearchProperties;
import org.eaglei.search.datagen.DataGenParams;
import org.eaglei.search.provider.MultiDataSetSearchProvider;
import org.eaglei.search.provider.SearchProvider;
import org.eaglei.search.provider.SearchRequest;
import org.eaglei.search.provider.SearchRequest.DataSet;
import org.eaglei.search.provider.lucene.CompositeLuceneProvider;
import org.eaglei.search.provider.lucene.LuceneSearchProviderIndexer;
import org.eaglei.search.provider.lucene.RepositoryHarvester;
import org.eaglei.search.provider.lucene.search.ClassUsageCache;
import org.eaglei.search.provider.lucene.suggest.LuceneAutoSuggestConfig;
import org.eaglei.search.provider.ncbi.gene.NCBIGeneSearchProvider;
import org.eaglei.search.provider.ncbi.pubmed.NCBIPubMedSearchProvider;
import org.eaglei.search.provider.nif.NIFSearchProvider;
import org.eaglei.search.provider.rdf.RDFModelProvider;
import org.eaglei.services.InstitutionRegistry;
import org.eaglei.services.repository.RepositoryConfig;
import org.springframework.beans.factory.annotation.Autowired;
import org.springframework.context.annotation.Bean;
import org.springframework.context.annotation.Configuration;
import org.springframework.context.annotation.Import;
import org.springframework.context.annotation.ImportResource;

/**
 * Allows programatic specification of bean implementations, while leveraging
 * spring to provide DI and ensure singleton scope.
 */
@Configuration
@ImportResource("classpath:search-common-config.xml")
@Import({RepositoryConfig.class, LuceneAutoSuggestConfig.class})
public class InstitutionSearchConfig {
    
    @Autowired
    private EIOntModel eagleiOntModel;

    @Autowired
    private RepositoryHarvester harvester;
    
    @Autowired
    private InstitutionRegistry registry; 
    
    @Autowired
    private LuceneSearchProviderIndexer indexer;
    
    @Autowired
    private ClassUsageCache classUsageCache;
    
    @Bean
    public SearchProvider rootSearchProvider() {
        try {

            final boolean test = isTest();
            
            // TODO constructors probably shouldn't be throwing checked exceptions

            SearchProvider nestedProvider;
            if (test) {
                final EIEntity institution = registry.getInstitution();
                final DataGenParams params = DataGenParams.createForOneInstitution(institution.getURI().toString(), institution.getLabel());            
                params.setMaterializeHierarchy(true);        
                nestedProvider = new RDFModelProvider(eagleiOntModel, params, institution);
            } else {
                nestedProvider = harvester;
            }
            Map<SearchRequest.DataSet, SearchProvider> mapDSToProvider = new HashMap<SearchRequest.DataSet, SearchProvider>();
            CompositeLuceneProvider resourceSearchProvider = new CompositeLuceneProvider(eagleiOntModel, nestedProvider, indexer);
            resourceSearchProvider.setModelUsageCache(classUsageCache);
            mapDSToProvider.put(DataSet.EI_RESOURCES, resourceSearchProvider);
            NIFSearchProvider nifSearchProvider = new NIFSearchProvider(eagleiOntModel, registry);
            mapDSToProvider.put(DataSet.NIF_RESOURCES, nifSearchProvider);
            NCBIPubMedSearchProvider pubSearchProvider = new NCBIPubMedSearchProvider(eagleiOntModel, registry);
            mapDSToProvider.put(DataSet.PUBS, pubSearchProvider);
            NCBIGeneSearchProvider geneSearchProvider = new NCBIGeneSearchProvider(eagleiOntModel, registry);
            mapDSToProvider.put(DataSet.GENES, geneSearchProvider);
            MultiDataSetSearchProvider searchProvider = new MultiDataSetSearchProvider(mapDSToProvider);
            return searchProvider;
        } catch (IOException e) {
            throw new RuntimeException(e);
        }
    }

    /*
     * Checks if the test property is enabled.
     */
    private static boolean isTest() {
        boolean test = false;
        final String testMode = System.getProperty(SearchProperties.SEARCH_TEST);
        if (testMode != null && Boolean.parseBoolean(testMode)) {
            test = true;
        }
        return test;
    }

}
