package org.eaglei.lexical.lucene;

import org.apache.lucene.queryParser.ParseException;
import org.apache.lucene.queryParser.QueryParser;
import org.apache.lucene.search.BooleanClause;
import org.apache.lucene.search.BooleanQuery;
import org.apache.lucene.search.FuzzyQuery;
import org.apache.lucene.search.PhraseQuery;
import org.apache.lucene.search.Query;
import org.apache.lucene.search.TermQuery;

/**
 * Lucene utility code.
 * 
 * @author rfrost
  */
public class LuceneUtils {
    
    /**
     * Attempts to parse the specified query string using the specified QueryParser. 
     * If a parsing error is encountered, invalid syntax is escaped and the parse is attempted again.
     * @param parser The QueryParser.
     * @param query The query string
     * @return Query
     * @throws ParseException Parse exception generated after escaping special syntax.
     */
    public static Query escapeIfInvalid(final QueryParser parser, final String query)
        throws ParseException {
        assert parser != null;
        assert query != null;
        try {
            return parser.parse(query);
        } catch (ParseException pe) {
            return parser.parse(QueryParser.escape(query));
        }
    }
    
    /**
     * Rewrites the specified Query to a form where all contained Terms have the "fuzzy" modifier (Levenstein edit distance)
     * Only rewrites queries that are: 
     * <ul>
     * <li>BooleanQueries
     *   <ul>
     *   <li>if any clause is more complex than a term or phrase, it is skipped
     *   <li>if any clause has the NOT modifier, skip
     *   </ul> 
     * <li>TermQueries
     * </ul>
     * @param originalQuery
     * @return Query with child TermQueries rewritten as FuzzyQueries.
     */
    public static Query rewriteToFuzzy(final Query originalQuery) {
        if (originalQuery instanceof BooleanQuery) {
            BooleanQuery bq = (BooleanQuery) originalQuery;
            BooleanClause[] clauses = bq.getClauses();
            for (BooleanClause clause: clauses) {
                Query q = clause.getQuery();
                    
                if (!(clause.getQuery() instanceof TermQuery || clause.getQuery() instanceof PhraseQuery)) { 
                    // if any clause is more complex than a Lucene term or phrase expression, skip
                    continue;
                }

                if (clause.getOccur().equals(BooleanClause.Occur.MUST_NOT)) {
                    // if any clause is NOT, skip
                    continue;
                }
                if (clause.getQuery() instanceof TermQuery) {
                    TermQuery tq = (TermQuery) q;
                    FuzzyQuery fq = new FuzzyQuery(tq.getTerm());
                    clause.setQuery(fq);                        
                }
            }
            return bq;
        } else if (originalQuery instanceof TermQuery ) {
            TermQuery tq = (TermQuery) originalQuery;
            FuzzyQuery fq = new FuzzyQuery(tq.getTerm());
            return fq;
        }
        return originalQuery; 
    }    
}
