package org.eaglei.model;

import java.util.Deque;
import java.util.List;

/**
 * Provides access to the eagle-i ontology model, a.k.a the data model.
 *
 * Note that this interface is currently intended for use by server-side
 * components only.
 */
public interface EIOntModel {
    
    /**
     * Interface passed into a call to traverseDataModel.
     * A Visitor implementation will receive a callback for
     * every class and property in the eagle-i data model.
     * The stack parameter describes the semantic
     * relationship of the class or property to a root resource.
     */
    interface Visitor {
        void visit(EIClass c, Deque<String> stack);
        void visit(EIProperty p, Deque<String> stack);
    }
    
    /**
     * Walks the entire eagle-i data model by
     * starting with the set of "root resource" (top-level resource)
     * classes and traversing through object property relationships
     * and subclass relationships until every class and property
     * in the data model has been visited.
     * 
     * Each of the Visitor implementations provided will be called as
     * data model classes and properties are visited.
     * 
     * @param visitors List of Visitor implementations
     */
    void traverseDataModel(List<Visitor> visitors);

    /**
     * Produces a default display string from the semantic relationship
     * stack generated by data model traversal.
     * 
     * @param stack Artifact produced by calling traverseDataModel with a 
     *              Visitor implementation.
     * @return a default display string
     */
    String generateStackTrace(Deque<String> stack);
    
    /**
     * Gets the version of the eagle-i ontology.
     * 
     * @return version string
     */
    String getVersion();
    
    /**
     * 
     * @return
     * @deprecated use the InstitutionRegistry bean instead
     */
    List<EIEntity> getInstitutions();
    
    /**
     * Determines if the given URI is recognized as being an eagle-i 
     * data model URI.  The eagle-i data model is defined as being
     * the resource classses (the "top-level" classes), the non-resource classes, 
     * and the subtypes of all of the resource and non-resource classes.
     * 
     * Non-resource classes are those classes which are related to 
     * a resource class via object property, but are not a subclass
     * of a resource class.  A non-resource class related to a resource
     * class may in turn have a relation to another non-resource class
     * bringing that class into the data model.
     * 
     * Note that the backing eagle-i ontology files may define superclasses
     * of resource and non-resource classes.  These are NOT considered
     * part of the data model from the EIClass api perspective.
     * 
     * @param uri
     * @return true if the uri is a data model class uri, otherwise false.
     */
    boolean isModelClassURI(String uri);

    List<EIClass> getTopLevelClasses();

    List<EIClass> getNonResourceBaseClasses();

    /**
     * Returns an EIClass for the given URI.  This method should
     * only be called with uri's that are known to be part of the 
     * data model.  If a given URI is not definitively known to 
     * part of the data model, isModelClassURI() should be used to 
     * verify the uri string.
     * 
     * @param classId
     * @return EIClass, undefined if uri is not a valid member of the
     * data model as defined by isModelClassURI()
     */
    EIClass getClass(EIURI classId);

    EIClass getSuperClass(EIClass childClass);

    List<EIClass> getSuperClasses(EIURI classId);

    List<EIClass> getSubClasses(EIURI classId);

    List<EIProperty> getProperties(EIURI classId);

    List<EIEquivalentClass> getEquivalentClasses(EIURI classId);

    List<String> getClassDefinitions(List<EIURI> classURIs);
    
    String getClassDefinition(EIURI uri);
    
    List<String> getPropertyDefinitions(List<EIURI> propertyURIs);
    
    String getPropertyDefinition(EIURI uri);
    
    /**
     * Retrieves the preferred label for the specified resource. Computes the preferred label using
     * the following priority order of properties (if multiple values for any property are specified, an arbitrary
     * value is returned):
     * <ul>
     * <li>http://eagle-i.org/ont/app/1.0/preferredLabel
     * <li>http://purl.obolibrary.org/obo/IAO_0000111 (IAO preferred term) 
     * <li>rdfs:label
     * </ul> 
     * 
     * If a value does not exist for any of the above properties, a preferred label is computed from 
     * the local name portion of the URI.
     * 
     * @param uri URI of the resource. Cannot be null.
     * @return Preferred label or null if the specified URI does not have a corresponding resource in the ontology.
     */
    String getPreferredLabel(EIURI uri);
    
    /**
     * Retrieves all unique labels for the specified resource using the
     * the following priority order
     * <ul>
     * <li>http://eagle-i.org/ont/app/1.0/preferredLabel
     * <li>http://purl.obolibrary.org/obo/IAO_0000111 (IAO preferred term) 
     * <li>rdfs:label
     * <li>http://purl.obolibrary.org/obo/IAO_0000118 (IAO alternate term) 
     * </ul> 
     * 
     * @param uri URI of the resource. Cannot be null.
     * @return Ordered list of labels. Will be empty if the specified URI does not have a corresponding resource in the ontology.
     */
    List<String> getLabels(EIURI uri);
}
