package org.eaglei.model.jena;

import java.io.BufferedReader;
import java.io.IOException;
import java.io.InputStream;
import java.io.InputStreamReader;
import java.util.ArrayDeque;
import java.util.ArrayList;
import java.util.Arrays;
import java.util.Collections;
import java.util.Comparator;
import java.util.Deque;
import java.util.HashMap;
import java.util.HashSet;
import java.util.List;
import java.util.Map;
import java.util.Set;
import java.util.TreeMap;
import java.util.TreeSet;

import org.apache.commons.logging.Log;
import org.apache.commons.logging.LogFactory;
import org.eaglei.model.EIClass;
import org.eaglei.model.EIDatatypeProperty;
import org.eaglei.model.EIEntity;
import org.eaglei.model.EIEquivalentClass;
import org.eaglei.model.EIObjectProperty;
import org.eaglei.model.EIOntModel;
import org.eaglei.model.EIOntResource;
import org.eaglei.model.EIProperty;
import org.eaglei.model.EIURI;
import org.eaglei.model.EIValueRestriction;
import org.eaglei.model.EIOntConstants;

import com.hp.hpl.jena.ontology.AllValuesFromRestriction;
import com.hp.hpl.jena.ontology.AnnotationProperty;
import com.hp.hpl.jena.ontology.ConversionException;
import com.hp.hpl.jena.ontology.HasValueRestriction;
import com.hp.hpl.jena.ontology.IntersectionClass;
import com.hp.hpl.jena.ontology.OntClass;
import com.hp.hpl.jena.ontology.OntModel;
import com.hp.hpl.jena.ontology.OntProperty;
import com.hp.hpl.jena.ontology.OntResource;
import com.hp.hpl.jena.ontology.Restriction;
import com.hp.hpl.jena.ontology.SomeValuesFromRestriction;
import com.hp.hpl.jena.ontology.UnionClass;
import com.hp.hpl.jena.rdf.model.Literal;
import com.hp.hpl.jena.rdf.model.Property;
import com.hp.hpl.jena.rdf.model.RDFList;
import com.hp.hpl.jena.rdf.model.RDFNode;
import com.hp.hpl.jena.rdf.model.Resource;
import com.hp.hpl.jena.rdf.model.Statement;
import com.hp.hpl.jena.rdf.model.StmtIterator;
import com.hp.hpl.jena.rdf.model.impl.StatementImpl;
import com.hp.hpl.jena.util.iterator.ExtendedIterator;
import com.hp.hpl.jena.vocabulary.RDFS;
import com.hp.hpl.jena.vocabulary.OWL;

/**
 * Exposes the eagle-i ontology using GWT friendly model classes.
 */
public class JenaEIOntModel implements EIOntModel {
    
    private class EIPropertyComparator implements Comparator<EIProperty> {

        @Override
        public int compare(EIProperty o1, EIProperty o2) {
            if (o1.getValueRestriction() != null && o2.getValueRestriction() == null) {
                return -1;
            } else if (o1.getValueRestriction() == null && o2.getValueRestriction() != null) {
                return 1;
            } else {
                int result = o1.getEntity().getLabel().toLowerCase().compareTo(o2.getEntity().getLabel().toLowerCase());
                if (result == 0) {
                    result = o1.getEntity().getURI().toString().compareTo(o2.getEntity().getURI().toString());
                }
                return result;
            }
        }
        
    }

    protected static final Log logger = LogFactory.getLog(JenaEIOntModel.class);
    
    // Jena model of eagle-i ontology
    private final OntModel jenaOntModel;
    private final String version;
    private final AnnotationProperty inPropertyGroup;
    private final AnnotationProperty inClassGroup;
    private final Resource dataModelExclude;
    private final Resource noAssertedInstances;
    private final Map<String, List<String>> mapPropURIToDomainConstraints;
    private final Map<String, List<String>> mapPropURIToRangeConstraints;
    private final AnnotationProperty eiPreferredDefinition;
    private final AnnotationProperty obiDefinition;
    
    // Priority ordered list of properties that are used to compute preferred label
    private final List<Property> prefLabelProperties = new ArrayList<Property>();
    // Property used to specify synonyms 
    private final Property iaoAltTerm;
    // List of subtype labels of given type
    private final HashMap<EIURI, List<EIEntity>> mapClassIdToTypeLabels = new HashMap<EIURI, List<EIEntity>>();
    
    // Should not be hardcoded here
    private final Property functionProperty;
    
    // Cache of EIClasses
    // TODO Put non-base EIClasses in a weak reference cache.
    // It's kind of tricky to do so, because the base classes have hard
    // references to them, and they have hard refs to their subclass lists.
    // So nothing in your not-so-weak cache will actually ever be GC'd.
    private final HashMap<String, EIClass> mapURIStrToEIClass = new HashMap<String, EIClass>();
    
    private final HashMap<String, Set<String>> mapGroupURIToClassURIs = new HashMap<String, Set<String>>();
    private final HashMap<String, List<OntClass>> mapGroupURIToOntClasses = new HashMap<String, List<OntClass>>();
    private final HashMap<String, List<EIClass>> mapGroupURIToEIClasses = new HashMap<String, List<EIClass>>();
    
    // Cache of property value restrictions per class
    private final HashMap<String, Map<String,EIValueRestriction>> mapURIStrToRestrictionMap = new HashMap<String, Map<String,EIValueRestriction>>();
    
    // Computed on first access
    private Set<EIURI> resourceProviderPropertyURIs;
    private Map<EIURI, EIURI> mapResourceToResourceProviderProperty;

    /*
     * Should generally be using Spring config to construct
     */
    protected JenaEIOntModel(OntModel jenaOntModel) {
        this.jenaOntModel = jenaOntModel;
        
        // read the version number from a the root ontology file
        version = jenaOntModel.getOntology(EIOntConstants.EAGLE_I_APP_EXT_URI).getVersionInfo();
        logger.info("eagle-i ontology version: " + version);
                
        inPropertyGroup = jenaOntModel.getAnnotationProperty(EIOntConstants.IN_PROPERTY_GROUP);
        inClassGroup = jenaOntModel.getAnnotationProperty(EIOntConstants.IN_CLASS_GROUP);
        dataModelExclude = jenaOntModel.getResource(EIOntConstants.PG_DATA_MODEL_EXCLUDE);
        noAssertedInstances = jenaOntModel.getResource(EIOntConstants.CG_NO_ASSERTED_INSTANCES);
        eiPreferredDefinition = jenaOntModel.getAnnotationProperty(EIOntConstants.EI_PREFERRED_DEFINITION_URI);
        obiDefinition = jenaOntModel.getAnnotationProperty(EIOntConstants.OBI_DEFINITION_URI);
        
        // Create a map of properties which have eagle-i specialized
        // domain constraints
        mapPropURIToDomainConstraints = new HashMap<String, List<String>>();
        Property domainConstraint = jenaOntModel.getProperty(EIOntConstants.DOMAIN_CONSTRAINT);
        List<Statement> domainConstrainedStatements = jenaOntModel.listStatements((Resource) null, domainConstraint, (RDFNode) null).toList();
        for (Statement stmt : domainConstrainedStatements) {
            Resource constraintSubject = stmt.getSubject();
            String propURI = constraintSubject.getURI();
            List<String> constraintList = mapPropURIToDomainConstraints.get(propURI);
            if (constraintList == null) {
                constraintList = new ArrayList<String>();
                mapPropURIToDomainConstraints.put(propURI, constraintList);
            }
            String domainURI = stmt.getString();
            constraintList.add(domainURI);
        }

        // Create a map of properties which have eagle-i specialized
        // range constraints
        mapPropURIToRangeConstraints = new HashMap<String, List<String>>();
        Property rangeConstraint = jenaOntModel.getProperty(EIOntConstants.RANGE_CONSTRAINT);
        List<Statement> rangeConstrainedStatements = jenaOntModel.listStatements((Resource) null, rangeConstraint, (RDFNode) null).toList();
        for (Statement stmt : rangeConstrainedStatements) {
            Resource constraintSubject = stmt.getSubject();
            String propURI = constraintSubject.getURI();
            List<String> constraintList = mapPropURIToRangeConstraints.get(propURI);
            if (constraintList == null) {
                constraintList = new ArrayList<String>();
                mapPropURIToRangeConstraints.put(propURI, constraintList);
            }
            String rangeURI = stmt.getString();
            constraintList.add(rangeURI);
        }
        
        functionProperty = jenaOntModel.getProperty("http://purl.obolibrary.org/obo/OBI_0000306");

        // define the ordered lists of label properties
        Property eiPrefLabel = jenaOntModel.getProperty(EIOntConstants.EI_PREFERRED_LABEL);
        Property iaoPrefTerm = jenaOntModel.getProperty(EIOntConstants.IAO_PREFERRED_TERM_URI);

        prefLabelProperties.add(eiPrefLabel);
        prefLabelProperties.add(iaoPrefTerm);
        prefLabelProperties.add(RDFS.label);
        
        iaoAltTerm = jenaOntModel.getProperty(EIOntConstants.IAO_ALTERNATE_TERM_URI);        

        logger.info("initializing class pre-cache...");
        long start = System.currentTimeMillis();
        
        List<String> preCacheGroupURIs = 
            Arrays.asList(EIOntConstants.CG_RESOURCE_ROOT, 
                    EIOntConstants.CG_NON_RESOURCE_ROOT,
                    EIOntConstants.CG_EMBEDDED_CLASS,
                    EIOntConstants.CG_DATA_MODEL_CREATE,
                    EIOntConstants.CG_RESOURCE_PROVIDER);
        initPreCache(preCacheGroupURIs);
        
        logger.info("class pre-cache initialized: " + (System.currentTimeMillis()-start) + " msecs");
    }
    
    @Override
    public String getVersion() {
        return this.version;
    }
    
    /**
     * Retrieves the underlying Jena OntModel.
     * @return The underlying eagle-i OntModel.
     */
    public OntModel getOntModel() {
        return this.jenaOntModel;
    }
    
    private void initPreCache(List<String> listGroupURIs) {
        List<List<OntClass>> listClassesInGroup = EagleIOntUtils.getClassesInGroup(jenaOntModel, listGroupURIs);
        int i = 0;
        for (String groupURI : listGroupURIs) {
            List<OntClass> listOntClasses = listClassesInGroup.get(i);
            cacheGroupOntClasses(groupURI, listOntClasses);
            i++;
        }
    }
    
    private void cacheGroupOntClasses(String groupURI, List<OntClass> listOntClasses) {
        mapGroupURIToOntClasses.put(groupURI, listOntClasses);
        Set<String> setURIs = new HashSet<String>(listOntClasses.size());
        mapGroupURIToClassURIs.put(groupURI, setURIs);
        // Pre-compute an alpha sorted list of EIClasses and cache them.
        TreeMap<String, EIClass> tmClasses = new TreeMap<String, EIClass>();
        for (OntClass ontClass: listOntClasses) {
            // This is a little sketchy, but it's required that the URI be added
            // to the set *before* calling createClass().
            // Otherwise, the hasSuperClass() computation may not work properly.
            setURIs.add(ontClass.getURI());
            final EIClass eiClass = createClass(ontClass); 
            tmClasses.put(eiClass.getEntity().getLabel().toLowerCase(), eiClass);
        }
        List<EIClass> listEIClasses = new ArrayList<EIClass>(tmClasses.values());
        mapGroupURIToEIClasses.put(groupURI, listEIClasses);
    }
    
    public Set<EIURI> getResourceProviderProperties() {
        if (resourceProviderPropertyURIs != null) {
            return resourceProviderPropertyURIs;
        }
        computeResourceProviderProperties();
        return resourceProviderPropertyURIs;
    }

    public EIURI getResourceProviderProperty(EIURI classURI) {
        Map<EIURI, EIURI> mapResourceToResourceProviderProperty = getResourceProviderPropertyMap();
        EIURI propertyURI = mapResourceToResourceProviderProperty.get(classURI);
        if (propertyURI == null) {
            List<EIClass> superclasses = getSuperClasses(classURI);
            for (EIClass superclass : superclasses) {
            	propertyURI = mapResourceToResourceProviderProperty.get(superclass.getEntity().getURI());
            	if (propertyURI != null) {
            		return propertyURI;
            	}
            }
        }
        return propertyURI;
    }

    public Map<EIURI, EIURI> getResourceProviderPropertyMap() {
        if (mapResourceToResourceProviderProperty != null) {
            return mapResourceToResourceProviderProperty;
        }
        computeResourceProviderProperties();
        return mapResourceToResourceProviderProperty;
    }

    private void computeResourceProviderProperties() {
        HashMap<EIURI, EIURI> result = new HashMap<EIURI, EIURI>();
        // For now, assume we only have to examine the DATA_MODEL_CREATE classes
        List<EIClass> eiResourceRoots = getClassesInGroup(EIOntConstants.CG_DATA_MODEL_CREATE);
        for (EIClass eiResourceRoot : eiResourceRoots) {
            // For now, assume it is a single property per class
            boolean foundIt = false;
            for (EIProperty prop : getProperties(eiResourceRoot.getEntity().getURI())) {
                for (String anno :prop.getAnnotations()) {
                    if (anno.equals(EIOntConstants.PG_RELATED_RESOURCE_PROVIDER)) {
                        result.put(eiResourceRoot.getEntity().getURI(), prop.getEntity().getURI());
                        foundIt = true;
                    }
                    if (foundIt) break;
                }
                if (foundIt) break;
            }
        }   
        mapResourceToResourceProviderProperty = result;
        resourceProviderPropertyURIs = new HashSet<EIURI>(mapResourceToResourceProviderProperty.values());
    }
    
    public List<EIClass> getClassesInGroup(String groupId) {
        List<EIClass> resultList = mapGroupURIToEIClasses.get(groupId);
        if (resultList != null) {
            return resultList;
        }
        
        // If list is not already in cache, add it.
        List<OntClass> listOntClasses = EagleIOntUtils.getClassesInGroup(jenaOntModel, groupId);
        for (OntClass ontClass : listOntClasses) {
            // Check that this uri is a valid member of the
            // data model.
            // Could silently prune instead...
            assert(isResource(ontClass) || isNonResource(ontClass)) : "getClassesInGroup found a non-datamodel class with annotation.  class: "+ontClass.getURI()+ "  annotation: "+groupId;
        }
        cacheGroupOntClasses(groupId, listOntClasses);
        return mapGroupURIToEIClasses.get(groupId);
    }
    
    private List<OntClass> getResourceListFromCache(String path) {
        //final URL cacheFileURL = this.getClass().getClassLoader().getResource(path);
        final InputStream is = this.getClass().getResourceAsStream(path);
        if (is == null) {
            return null;
        }
        BufferedReader br = null;
        try {
            br = new BufferedReader(new InputStreamReader(is));
            String list = br.readLine();
            if (list == null) {
                return null;
            }
            String[] uriStrings = list.split(",");
            List<OntClass> listOntClass = new ArrayList<OntClass>(uriStrings.length);
            for (String uri : uriStrings) {
                listOntClass.add(getOntologyClass(uri));
            }
            return listOntClass;
        } catch (Exception e) {
            logger.error(e);
            return null;
        } finally {
            try {
                if (br != null) br.close();
                if (is != null) is.close();
            } catch (IOException e) {
            }
        }
    }

    public List<EIEntity> getTypeEntities(EIURI classId) {
        assert classId != null;
        List<EIEntity> results = mapClassIdToTypeLabels.get(classId);
        if (results != null) {
            return results;
        }
        results = new ArrayList<EIEntity>();
        EIClass root = getClass(classId);
        getTypeLabelsInternal(root, results);
        mapClassIdToTypeLabels.put(classId, results);
        return results;
    }

    private void getTypeLabelsInternal(EIClass c, List<EIEntity> results) {
        results.add(c.getEntity());
        if (c.hasSubClass()) {
            List<EIClass> subclasses = getSubClasses(c.getEntity().getURI());
            for (EIClass subclass : subclasses) {
                getTypeLabelsInternal(subclass, results);
            }
        }
    }
    

    @Override
    public boolean isModelClassURI(String uri) {
        OntClass c = getOntologyClass(uri);
        if (c == null) {
            return false;
        }
        if (isResource(c)) {
            return true;
        }
        return isNonResource(c);
    }
    
    @Override
    public List<EIClass> getTopLevelClasses() {
        return getClassesInGroup(EIOntConstants.CG_RESOURCE_ROOT);
    }

    @Override
    public List<EIClass> getNonResourceBaseClasses() {
        return getClassesInGroup(EIOntConstants.CG_NON_RESOURCE_ROOT);
    }
    
    /**
     * Retrieves the classes in the specified group. See EagleIOntUtils.getClassesInGroup().
     * 
     * @param groupURI URI of the group.
     * 
     * @return Alpha sorted (by label) list of classes in the group.
     */
    /*
    private List<EIClass> getClassesInGroup(final String groupURI) {
        // TreeMap for alpha sort
        TreeMap<String, EIClass> tmClasses = new TreeMap<String, EIClass>();
        List<EIClass> eiClasses = new ArrayList<EIClass>();
        for (OntClass ontClass: EagleIOntUtils.getClassesInGroup(groupURI)) {
            final EIClass eiClass = createClass(ontClass); 
            tmClasses.put(eiClass.getEntity().getLabel(), eiClass);
        }
        return new ArrayList<EIClass>(tmClasses.values());
    }
    */
    
    public EIClass getClass(EIURI uri) {
        assert (uri != null);
        return getClass(uri.toString());
    }
    
    protected EIClass getClass(String uri) {
        OntClass ontClass = getOntologyClass(uri);
        if (ontClass == null) {
            // could change this to an assert.
            return null;
        }
        return createClass(ontClass);
    }

    /*
     * Only call with a non-null ont class
     * that is known to be a data model resource or non-resource class.
     */
    private EIClass createClass(OntClass ontClass) {
        assert (ontClass != null);
        EIClass result = mapURIStrToEIClass.get(ontClass.getURI());
        if (result != null) {
            return result;
        }
        final EIURI uri = EIURI.create(ontClass.getURI());
        final String label = getPreferredLabel(uri);
        // All resource base classes are flagged as having no super
        boolean hasSuperClass = hasSuperClass(ontClass);
        boolean hasSubClass = hasSubClass(ontClass);
        /* TODO should figure out how to get this into sync
         *      with the getProperties implementation.
         *      Note that simply calling getProperties as it is
         *      currently implemented is bad because it currently
         *      creates an EIClass, so circular instantiation in the
         *      case where this method is called from createClass.
         */
        boolean hasProperties = true; //hasProperty(ontClass);
        boolean isEagleIResource = isResource(ontClass);
        if (!isEagleIResource) {
            // Check that this uri is a valid member of the
            // data model.
            // Can uncomment when http://jira.eagle-i.net:8080/browse/DTOOLS-215 is resolved.
            //assert(isNonResource(ontClass));
        }
        EIEntity entity = EIEntity.create(uri, label);

        // Process equivalent classes.
        // This is definitely not general purpose handling yet!
        // This is custom quick hack code to handle our one current
        // equivalent class case.
        // There are a bunch of asserts that will try to detect
        // the addition to the ontology of any new equivalentClass 
        // constructs that this code doesn't handle.
        List<OntClass> listEquivOntClasses = ontClass.listEquivalentClasses().toList();
        List<EIEquivalentClass> listEquivalentClasses = null;
        boolean isInferred = ontClass.hasProperty(inClassGroup, noAssertedInstances);
        // If list is size 1, then assume it is just the original URI.
        // TODO handle more than 2
        if (listEquivOntClasses.size() > 1) {
            listEquivalentClasses = new ArrayList<EIEquivalentClass>(listEquivOntClasses.size()-1);
            for (OntClass equivOntClass : listEquivOntClasses) {
                if (ontClass.getURI().equals(equivOntClass.getURI())) {
                    // the class itself will always appear in this list
                    continue;
                } else if (equivOntClass.isIntersectionClass()) {
                    boolean hasFunctionRestriction = false;
                    String equivClassURI = null;
                    List<EIClass> listEquivalentToClasses = new ArrayList<EIClass>();
                    IntersectionClass intersectionClass = equivOntClass.asIntersectionClass();
                    for (OntClass intersectionOperand : intersectionClass.listOperands().toList()) {
                        if (intersectionOperand.isRestriction()) {
                            Restriction r = intersectionOperand.asRestriction();
                            hasFunctionRestriction = r.onProperty(functionProperty);
                            //logger.info(entity + " :  Function restriction property: " + rProp);
                            if (!hasFunctionRestriction) {
                                logger.warn(entity + " :  ONT UNHANDLED restriction property: " + r);
                            }
                        } else if (intersectionOperand.isUnionClass()) {
                            UnionClass uc = intersectionOperand.asUnionClass();
                            RDFList members = uc.getOperands();
                            for (int i=0; i<members.size(); i++) {
                                RDFNode n = members.get(i);
                                assert(n instanceof Resource);
                                String eiClassURI = ((Resource) n).getURI();
                                if (eiClassURI == null) {
                                    logger.warn(entity + " :  ONT UNHANDLED, union member has a null URI");
                                    continue;
                                }
                                assert(!eiClassURI.equals(OWL.Thing.getURI()));
                                EIClass ec = getClass(eiClassURI);
                                listEquivalentToClasses.add(ec);
                            }
                        } else {
                            if (!(intersectionOperand instanceof Resource)) {
                                logger.warn(entity + " :  ONT UNHANDLED, intersection operand is not a Resource");
                                continue;
                            }
                            equivClassURI = ((Resource) intersectionOperand).getURI();
                            if (equivClassURI == null) {
                                logger.warn(entity + " :  ONT UNHANDLED, Resource intersection operand has null URI");
                                continue;
                            }
                            /*
                            if (isSubClass(equivClassURI, ontClass.getURI())) {
                                // TODO Handle ancestor equivalent classes
                                // Operand is a superclass of this class,
                                // this is probably a property restriction
                                logger.warn("Equivalence of: " + ontClass.getURI() + ":  is ancestor class: " + eiClassURI);
                            }
                            */
                            logger.warn(entity + " :  ONT UNHANDLED intersectionOperand: " + intersectionOperand);
                        }
                    }
                    // Check that we came out of this with at least one equivalent to class.
                    if(listEquivalentToClasses.size() > 0) {
                        listEquivalentClasses.add(new EIEquivalentClass(null, listEquivalentToClasses));
                        //logger.info(entity + " : Has equivalent classes: " + listEquivalentToClasses);
                    }
                    // Special case has_function
                    /*
                    if (hasFunctionRestriction) {
                        isInferred = true;
                        //logger.info("Inferred class: " + entity);
                    }
                    */
                } else {
                    logger.warn(entity + " :  ONT UNHANDLED equivalent class construct, not an intersection.");
                }
            }
        }
        boolean hasEquivalentClass = listEquivalentClasses != null ? listEquivalentClasses.size() > 0 : false;
        
        if (isInferred) {
            logger.debug(entity + " :  NO ASSERTED INSTANCES");
        }
        result = new EIClass(entity, isInferred, hasProperties, hasSubClass, hasSuperClass, hasEquivalentClass, isEagleIResource);
        result.setEquivalentClasses(listEquivalentClasses);

        mapURIStrToEIClass.put(ontClass.getURI(), result);

        return result;
    }
    
    /*
     * Return a property restriction map for the given class
     */
    private Map<String,EIValueRestriction> getRestrictions(String classURIStr) {
        Map<String,EIValueRestriction> mapValueRestrictions = mapURIStrToRestrictionMap.get(classURIStr);    
        if (mapValueRestrictions != null) {
            return mapValueRestrictions;
        }
        // Map property URI string to EIValueRestriction
        mapValueRestrictions = new HashMap<String,EIValueRestriction>();
        OntClass ontClass = getOntologyClass(classURIStr);
        computeAllRestrictions(ontClass, mapValueRestrictions);
        mapURIStrToRestrictionMap.put(classURIStr, mapValueRestrictions);
        return mapValueRestrictions;
    }
    
    /*
     * Puts the full set of restrictions for
     */
    private Map<String,EIValueRestriction> computeAllRestrictions(OntClass ontClass, Map<String,EIValueRestriction> mapValueRestrictions) {
        for (OntClass ontSuperClass : ontClass.listSuperClasses().toList()) {
            if (ontSuperClass.isRestriction()) {
                /*
                 * Restriction expressed in a subClassOf element
                 * 
                 *   <owl:Class rdf:about="http://purl.obolibrary.org/obo/ERO_0000009">
                 *      <rdfs:subClassOf>
                 *        <owl:Restriction>
                 *          <owl:someValuesFrom rdf:resource="http://purl.obolibrary.org/obo/OBI_0100026"/>
                 *          <owl:onProperty rdf:resource="http://www.obofoundry.org/ro/ro.owl#derives_from"/>
                 *        </owl:Restriction>
                 *      </rdfs:subClassOf>
                 *   </owl:Class>   
                 */
                Restriction r = ontSuperClass.asRestriction();
                parseRestriction(ontClass.getURI(), r, mapValueRestrictions);
            } else {
                // Recursively for restrictions on superclasses and merge those
                // into the restrictions on this class's properties.
                // (inherit restrictions)
                String parentClassURIStr = ontSuperClass.getURI();
                if (parentClassURIStr != null) {
                    Map<String,EIValueRestriction> parentRestrictionMap = 
                        getRestrictions(parentClassURIStr);
                    mapValueRestrictions.putAll(parentRestrictionMap);
                }
            }
        }
        /* 
         * Restrictions expressed using a superclass equivalence class
         * 
         * <owl:Class rdf:about="http://purl.obolibrary.org/obo/ERO_0000015">
         *    <owl:equivalentClass>
         *         <owl:Class>
         *           <owl:intersectionOf rdf:parseType="Collection">
         *             <owl:Class rdf:about="http://purl.obolibrary.org/obo/ERO_0000014"/>
         *             <owl:Restriction>
         *               <owl:someValuesFrom rdf:resource="http://purl.org/obo/owl/NCBITaxon#NCBITaxon_9606"/>
         *               <owl:onProperty>
         *                 <owl:ObjectProperty rdf:about="http://purl.obolibrary.org/obo/ERO_0000234"/>
         *               </owl:onProperty>
         *             </owl:Restriction>
         *           </owl:intersectionOf>
         *         </owl:Class>
         *    </owl:equivalentClass>
         * </owl:Class>
         */
        List<OntClass> listEquivOntClasses = ontClass.listEquivalentClasses().toList();
        for (OntClass equivOntClass : listEquivOntClasses) {
            if (equivOntClass.isIntersectionClass()) {
                IntersectionClass intersectionClass = equivOntClass.asIntersectionClass();
                String equivClassURI = null;
                Restriction r = null;
                for (OntClass intersectionOperand : intersectionClass.listOperands().toList()) {
                    if (intersectionOperand.isRestriction()) {
                        r = intersectionOperand.asRestriction();
                    } else {
                        if (!(intersectionOperand instanceof Resource)) {
                            logger.warn("Unexpected equivalent class construct in: " + ontClass.getURI() + ", intersection operand is not a Resource");
                            continue;
                        }
                        equivClassURI = ((Resource) intersectionOperand).getURI();
                        if (equivClassURI == null) {
                            logger.warn("Unexpected equivalent class construct in: " + ontClass.getURI() + ", Resource intersection operand has null URI");
                            continue;
                        }
                    }
                }
                if (equivClassURI != null && r != null) {
                    if (isSubClass(equivClassURI, ontClass.getURI())) {
                        parseRestriction(ontClass.getURI(), r, mapValueRestrictions);
                    }
                }
            }
        }
        
        return mapValueRestrictions;
    }
    
    private void parseRestriction(String classURIStr, Restriction r, Map<String,EIValueRestriction> mapValueRestrictions) {
        OntProperty ontProperty = r.getOnProperty();
        String propertyURI = ontProperty.getURI();
        if (propertyURI == null) {
            logger.warn(classURIStr + " : Restriction's onProperty property uri is null");
            return;
        }
        String valueRestrictionURI = null;;
        EIClass valueEIClass = null;;
        EIValueRestriction valueRestriction = null;
        if (r.isAllValuesFromRestriction()) {
            AllValuesFromRestriction allRestrict = r.asAllValuesFromRestriction();
            Resource valueResource = allRestrict.getAllValuesFrom();
            valueRestrictionURI = valueResource.getURI();
            if (! (valueResource instanceof OntClass)) {
                logger.warn(classURIStr + " : Unable to handle all values data range restriction on property " + propertyURI);
                return;             
            }
            valueEIClass = getClass(valueRestrictionURI);
            valueRestriction = new EIValueRestriction(EIValueRestriction.Type.ALL_VALUES, valueRestrictionURI, valueEIClass);
            //logger.info(entity + " has ALL VALUES restriction:  Prop: " + propertyURI + "  Value: " + valueRestrictionURI);
        } else if (r.isHasValueRestriction()) {
            HasValueRestriction hasRestrict = r.asHasValueRestriction();
            RDFNode node = hasRestrict.getHasValue();
            if (node instanceof Resource) {
                valueRestrictionURI = ((Resource) node).getURI();
            } else {
                logger.warn(classURIStr + " : Unsupported restriction value type. HAS VALUE restriction:  Prop: " + propertyURI + "  Value: " + node.toString());
                return;
            }
            valueRestriction = new EIValueRestriction(EIValueRestriction.Type.HAS_VALUE, valueRestrictionURI, null);
            //logger.info(entity + " has HAS VALUE restriction:  Prop: " + propertyURI + "  Value: " + valueRestrictionURI);
        } else if (r.isSomeValuesFromRestriction()) {
            SomeValuesFromRestriction someRestrict = r.asSomeValuesFromRestriction();
            Resource valueResource = someRestrict.getSomeValuesFrom();
            valueRestrictionURI = valueResource.getURI();
            if (! (valueResource instanceof OntClass)) {
                logger.warn(classURIStr + " : Unable to handle all values data range restriction on property " + propertyURI);
                return;             
            }
            valueEIClass = getClass(valueRestrictionURI);
            valueRestriction = new EIValueRestriction(EIValueRestriction.Type.SOME_VALUES, valueRestrictionURI, valueEIClass);
            //logger.info(entity + " has SOME VALUES restriction:  Prop: " + propertyURI + "  Value: " + valueRestrictionURI);
        }
        if (valueRestrictionURI == null && valueRestriction == null) {
            logger.warn(classURIStr + " : Unable to handle restriction on property " + propertyURI);
            return;
        }
        mapValueRestrictions.put(propertyURI, valueRestriction);
    }

    private EIProperty createProperty(final OntProperty ontProperty) {
        EIProperty result = null;
        final EIURI uri = EIURI.create(ontProperty.getURI());
        final String label = getPreferredLabel(uri);
        final EIEntity entity = EIEntity.create(uri, label);
        if (ontProperty.isObjectProperty()) {
            //logger.info("Obj : " + p.getURI() + " : " + p.getLabel(null));
            List<EIClass> listRangeClasses = new ArrayList<EIClass>();
            List<String> rangeConstraints = mapPropURIToRangeConstraints.get(ontProperty.getURI());
            if (rangeConstraints != null) {
                // This property has a fixed range list.
                for (String rangeURI : rangeConstraints) {
                    EIClass rangeClass = getClass(rangeURI); 
                    addObjectPropertyRangeClass(listRangeClasses, rangeClass);
                }
            } else {
                List<OntResource> listRange = (List<OntResource>) ontProperty.listRange().toList();
                for (OntResource rangeOntResource : listRange) {
                    OntClass rangeOntClass = (OntClass) rangeOntResource;
                    if (rangeOntClass.isUnionClass()) {
                        UnionClass uc = rangeOntClass.asUnionClass();
                        RDFList members = uc.getOperands();
                        for (int i=0; i<members.size(); i++) {
                            RDFNode rangeNode = members.get(i);
                            if (rangeNode instanceof Resource) {
                                String rangeURI = ((Resource) rangeNode).getURI();
                                if (!rangeURI.equals(OWL.Thing.getURI())) {
                                    EIClass rangeClass = getClass(rangeURI); 
                                    addObjectPropertyRangeClass(listRangeClasses, rangeClass);
                                }
                            }
                        }
                    } else if (!rangeOntClass.getURI().equals(OWL.Thing.getURI())) {
                        EIClass rangeClass = createClass(rangeOntClass); 
                        addObjectPropertyRangeClass(listRangeClasses, rangeClass);
                    }
                }
            }
            if (listRangeClasses.size() > 0) {
                result = new EIObjectProperty(entity, listRangeClasses);         
                OntProperty inverseOntProperty = ontProperty.getInverse();
                if (inverseOntProperty != null) {
                    ((EIObjectProperty) result).setInverseURI(inverseOntProperty.getURI());
                    //final EIURI inverseUri = EIURI.create(inverseOntProperty.getURI());
                    //final String inverseLabel = getPreferredLabel(inverseUri);
                    //System.out.println("inverse: " + inverseLabel);
                }
            } else {
                assert(false) : "Unable to determine range for object property " + entity.toString();
                result = new EIDatatypeProperty(entity, null);                    
            }
        } else if (ontProperty.isDatatypeProperty()) {
            //logger.info("Data : " + p.getURI() + " : " + p.getLabel(null));
            String typeLabel = null;
            OntResource rangeResource = ontProperty.getRange();
            if (rangeResource != null) {
                typeLabel = rangeResource.getLocalName();
            } else {
                //logger.warn("Datatype property has unknown range type: " + p.getURI());   
            }
            result = new EIDatatypeProperty(entity, typeLabel);
        } else {
            //logger.warn("createProperty called on property that is neither an object nor datatype property: " + p.getURI());   
        }
        
        // Compute and set the annotations at EIProperty construction
        Set<String> setAnnotationValueURI = getPropertyAnnotations(ontProperty);
        result.setAnnotations(setAnnotationValueURI);
        
        return result;
    }
    
    private void addObjectPropertyRangeClass(List<EIClass> listRangeClasses, EIClass rangeClassToAdd) {
        if (rangeClassToAdd.hasEquivalentClass()) {
            // This is transitional code that checks if a range class
            // has an equivalent class, and if it does, puts the equivalent to
            // list into the range list rather than the original range class.
            // When all clients are ready to handle occurrences of 
            // EIEquivalentClass in the range list, then this code can be removed.
            for (EIEquivalentClass equivalentClass : rangeClassToAdd.getEquivalentClasses()) {
                for (EIClass equivalentToClass : equivalentClass.getEquivalentTo()) {
                    listRangeClasses.add(equivalentToClass); 
                }
            }
        } else {
            listRangeClasses.add(rangeClassToAdd); 
        }
    }
    
    private boolean hasSuperClass(OntClass ontClass) {
        boolean hasSuperClass =
            !isResourceBaseClass(ontClass) && !isNonResourceBaseClass(ontClass);
        return hasSuperClass;
    }

    @Override
    public List<EIClass> getSuperClasses(EIURI classId) {
        List<EIClass> result;
        
        EIClass eiClass = getClass(classId);
        if (eiClass == null) {
            return Collections.emptyList();            
        }
        if (!eiClass.hasSuperClass()) {
            return Collections.emptyList();            
        }
        
        result = eiClass.getSuperClasses();
        if (result != null) {
            return result;
        }
        
        result = new ArrayList<EIClass>();
        EIClass childClass = getClass(classId);
        while (childClass.hasSuperClass()) {
            EIClass parentClass = getSuperClass(childClass);
            assert(parentClass != null) : "getSuperClasses: " + classId + "   childClass has null parentClass: " + childClass.getEntity();
            result.add(parentClass);
            childClass = parentClass;
        }
        
        eiClass.setSuperClasses(result);
        return result;
    }
    
    @Override
    public EIClass getSuperClass(EIClass eiClass) {
        assert(eiClass != null);
        if (!eiClass.hasSuperClass()) {
            //logger.debug("getSuperClass() called on class for which hasSuperClass() is false: "+eiClass.getEntity().toString());
            return null;
        }
        EIClass parentEIClass = eiClass.getSuperClass();
        if (parentEIClass != null) {
            return parentEIClass;
        }

        String uri = eiClass.getEntity().getURI().toString();
        if (isResourceBaseClass(uri)) {
            // Top-level classes have no super class
            return null;
        }
        if (isNonResourceBaseClass(uri)) {
            // Top-level classes have no super class
            return null;
        }
        
        OntClass childClass = getOntologyClass(uri);
        // Get from mapped class, if there is a mapping\
        /* ????
        if (resolveReference) {
            Resource resDefinedBy = parentClass.getIsDefinedBy();
            if (resDefinedBy != null) {
                String definedByURI = resDefinedBy.getURI();
                parentClass = getOntologyClass(definedByURI);
            }
        }
        */
        ExtendedIterator itrSuperClasses = childClass.listSuperClasses(true);
        while (itrSuperClasses.hasNext()) {
            OntClass parentClass = (OntClass) itrSuperClasses.next();
            String parentURI = parentClass.getURI();
            if (parentURI == null) {
                continue;
            }
            if (OWL.Thing.getURI().equals(parentURI)) {
                continue;
            }
            // Treat first non-null, non-THING parent class as the super
            parentEIClass = getClass(parentClass.getURI());   
            eiClass.setSuperClass(parentEIClass);
            return parentEIClass;
        }
        return null;
    }
    
    /*
     * Is the given ont class a resource or resource subtype.
     */
    private boolean isResource(OntClass ontClass) {
        if (isResourceBaseClass(ontClass)) {
            return true;
        }
        for (OntClass c : mapGroupURIToOntClasses.get(EIOntConstants.CG_RESOURCE_ROOT)) {
            if (c.hasSubClass(ontClass)) {
                return true;
            }
        }
        return false;
    }
    
    /*
     * Is the given ont class a non-resource or non-resource subtype.
     */
    private boolean isNonResource(OntClass ontClass) {
        if (isNonResourceBaseClass(ontClass)) {
            return true;
        }
        for (String strURI : mapGroupURIToClassURIs.get(EIOntConstants.CG_NON_RESOURCE_ROOT)) {
            OntClass c = getOntologyClass(strURI);
            if (c.hasSubClass(ontClass)) {
                return true;
            }
        }
        return false;
    }
    
    private boolean isResourceBaseClass(OntClass ontClass) {
        return isResourceBaseClass(ontClass.getURI());
    }

    private boolean isResourceBaseClass(String uri) {
        return mapGroupURIToClassURIs.get(EIOntConstants.CG_RESOURCE_ROOT).contains(uri);
    }

    private boolean isNonResourceBaseClass(OntClass ontClass) {
        return isNonResourceBaseClass(ontClass.getURI());
    }

    private boolean isNonResourceBaseClass(String uri) {
        return mapGroupURIToClassURIs.get(EIOntConstants.CG_NON_RESOURCE_ROOT).contains(uri);
    }

    private boolean hasSubClass(OntClass ontClass) {
        // Note: need to use listSubClasses() rather than hasSubClasses()
        // because that always returns
        // true when using a reasoner (since all OWL classes have owl:Nothing as
        // a subclass)
        ExtendedIterator<OntClass> itrSubClasses = ontClass.listSubClasses(true);
        while (itrSubClasses.hasNext()) {
            OntClass childClass = (OntClass) itrSubClasses.next();
            if (childClass == null || childClass.getURI() == null) {
                continue;
            }
            if (childClass.getURI().equals(OWL.Nothing.getURI())) {
                return false;
            } else {
                return true;
            }
        }
        return false;
    }
    
    public boolean isSubClass(EIURI ancestorURI, EIURI descendentURI) {
        return isSubClass(ancestorURI.toString(), descendentURI.toString());
    }

    public boolean isSubClass(String ancestorURI, String descendentURI) {
        OntClass ancestorOntClass = getOntologyClass(ancestorURI);
        OntClass descendentOntClass = getOntologyClass(descendentURI);
        if (ancestorOntClass == null || descendentOntClass == null) {
            return false;
        }
        return ancestorOntClass.hasSubClass(descendentOntClass);
    }

    @Override
    public List<EIClass> getSubClasses(EIURI classId) {
        List<EIClass> result;
        
        EIClass eiClass = getClass(classId);
        if (eiClass == null) {
            return Collections.emptyList();            
        }
        if (!eiClass.hasSubClass()) {
            logger.debug("getSubClasses() called on class for which hasSubClass() is false: "+eiClass.getEntity().toString());
            return new ArrayList<EIClass>();
        }
        
        result = eiClass.getSubClasses();
        if (result != null) {
            return result;
        }

        OntClass parentClass = getOntologyClass(classId.toString());

        result = getSubClasses(parentClass);
        eiClass.setSubClasses(result);
        return result;
    }

    OntClass getOntologyClass(String strURI) {
        return jenaOntModel.getOntClass(strURI);
    }

    private List<EIClass> getSubClasses(OntClass ontClass) {
        // TreeMap for alpha sort
        TreeMap<String, EIClass> tmSubClasses = new TreeMap<String, EIClass>();
        ExtendedIterator itrSubClasses = ontClass.listSubClasses(true);
        while (itrSubClasses.hasNext()) {
            OntClass childClass = (OntClass) itrSubClasses.next();
            if (childClass == null || childClass.getURI() == null) {
                continue;
            }
            if (childClass.getURI().equals(OWL.Nothing.getURI())) {
                continue;
            }
            EIClass eiChildClass = createClass(childClass);
            tmSubClasses.put(eiChildClass.getEntity().getLabel().toLowerCase(), eiChildClass);
        }
        List<EIClass> listSubClasses = new ArrayList<EIClass>(tmSubClasses.values());
        return listSubClasses;
    }

    /* TODO should figure out how to get this into sync
     *      with the getProperties implementation.
     *      Note that simply calling getProperties as it is
     *      currently implemented is bad because it currently
     *      creates an EIClass, so circular instantiation in the
     *      case where this method is called from createClass.
    private boolean hasProperty(OntClass ontClass) {
        ExtendedIterator itrProperties = ontClass.listDeclaredProperties();
        while (itrProperties.hasNext()) {
            try {
                Object o = itrProperties.next();
                if (o instanceof OntProperty) {
                    if (((OntProperty) o).isObjectProperty()) {
                        return true;
                    }
                }
            } catch (ConversionException ex) {
                //logger.warn(ex.getMessage());
                continue;
            }
        }
        return false;
    }
    */

    public List<EIEquivalentClass> getEquivalentClasses(EIURI classId) {
        EIClass eiClass = getClass(classId);
        assert(eiClass != null);
        // Should already be set.
        return eiClass.getEquivalentClasses();
    }
    
    public List<EIProperty> getProperties(EIURI classId) {
        return  getProperties(classId, null);
    }
    
    /**
     * Retrieves the properties with the inPropertyGroup annotation set to 
     * the given value.  Or all non-annotation properties if groupURI is null. 
     * @param classId ID of class.
     * @param groupURI URI of property group value to filter on.  May be null.
     * @return List of EIProperties that represent the properties.
     */
    public List<EIProperty> getProperties(final EIURI classId, final String groupURI) {
        List<EIProperty> result;
        EIClass eiClass = getClass(classId);
        assert(eiClass != null);
        if (!eiClass.hasProperty()) {
            logger.debug("getProperties() called on class for which hasProperty() is false: "+eiClass.getEntity().toString());
            return new ArrayList<EIProperty>();
        }
        
        result = eiClass.getProperties();
        if (result != null) {
            return result;
        }

        OntClass ontClass = getOntologyClass(classId.toString());
        result = getProperties(ontClass, groupURI);
        // Set property restrictions if there are any
        setValueRestriction(result, eiClass);
        // Do a sort
        final TreeSet<EIProperty> tsProperties = new TreeSet<EIProperty>(new EIPropertyComparator());
        tsProperties.addAll(result);
        result = new ArrayList<EIProperty>(tsProperties);
        eiClass.setProperties(result);
        
        return result;
    }
    
    /*
     * Configures value restrictions on properties in the list using the
     * restriction map in the given class.
     */
    private void setValueRestriction(List<EIProperty> propertyList, EIClass eiClass) {
        Map<String,EIValueRestriction> mapValueRestrictions = getRestrictions(eiClass.getEntity().getURI().toString());
        for (EIProperty eiProp : propertyList) {
            eiProp.setValueRestriction(mapValueRestrictions.get(eiProp.getEntity().getURI().toString()));
        }
    }

    private List<EIProperty> getProperties(final OntClass ontClass, final String groupURI) {
        final List<EIProperty> listProperties = new ArrayList<EIProperty>();
        final ExtendedIterator itrProperties = ontClass.listDeclaredProperties();
        while (itrProperties.hasNext()) {
            try {
                final Object o = itrProperties.next();
                if (o instanceof OntProperty) {
                    final OntProperty ontProperty = (OntProperty) o;
                    if (ontProperty.isAnnotationProperty()) {
                        continue;
                    }
                    if (ontProperty.hasProperty(inPropertyGroup, dataModelExclude)) {
                        continue;
                    }
                    // Check that if there are domain constraints for this property,
                    // and if so, that the class is among the allowed domain classes.
                    List<String> domainConstraints = mapPropURIToDomainConstraints.get(ontProperty.getURI());
                    if (domainConstraints != null) {
                        boolean withinConstraint = false;
                        for (String domainRootURI : domainConstraints) {
                            if (domainRootURI.equals(ontClass.getURI())) {
                                // class is itself a domain constraint
                                withinConstraint = true;
                                break;
                            } else {
                                OntClass domainRootOntClass = getOntologyClass(domainRootURI);
                                if (domainRootOntClass.hasSubClass(ontClass)) {
                                    // class is a subclass of a domain constraint.
                                    withinConstraint = true;
                                    break;
                                }
                            }
                        }
                        // If class is not within the specified constraint,
                        // ignore the property.
                        if (!withinConstraint) continue;
                    }
                    final EIProperty property = createProperty(ontProperty);
                    if (property != null) {
                        listProperties.add(property);
                    }
                }
            } catch (ConversionException ex) {
                //logger.warn(ex.getMessage());
                continue;
            }
        }

        return listProperties;
    }
    
    @Override
    public List<String> getClassDefinitions(List<EIURI> classURIs) {
        List<String> results = new ArrayList<String>(classURIs.size());
        for (EIURI classURI : classURIs) {
            String def = getClassDefinition(classURI);
            results.add(def);
        }
        return results;
    }
    
    @Override
    public String getClassDefinition(EIURI classURI) {
        OntClass ontClass = getOntologyClass(classURI.toString());
        if (ontClass == null) {
            return null;
        }
        String def = null;
        RDFNode defNode = ontClass.getPropertyValue(eiPreferredDefinition);
        if (defNode != null) {
            def = ((Literal) defNode).getString();
        }
        if (def == null) {
            defNode = ontClass.getPropertyValue(obiDefinition);
            if (defNode != null) {
                def = ((Literal) defNode).getString();
            }
        }
        //if (def == null) {
        //    defNode = ontClass.getPropertyValue(RDFS.comment);
        //    if (defNode != null) {
        //        def = "COMMENT: " + ((Literal) defNode).getString();
        //    }
        //}
        return def;        
    }
    
    @Override
    public List<Set<String>> getClassAnnotations(List<EIURI> classURIs) {
        List<Set<String>> results = new ArrayList<Set<String>>(classURIs.size());
        for (EIURI classURI : classURIs) {
            Set<String> annos = getClassAnnotations(classURI);
            results.add(annos);
        }
        return results;
    }
    
    public Set<String> getClassAnnotations(EIURI classURI) {
        EIClass eiClass = getClass(classURI);
        if (eiClass == null) {
            return null;
        }
        Set<String> setAnnotationValueURI = eiClass.getAnnotations();
        if (setAnnotationValueURI == null) {
            OntClass ontClass = getOntologyClass(eiClass.getEntity().getURI().toString());
            setAnnotationValueURI = getClassAnnotations(ontClass);
            eiClass.setAnnotations(setAnnotationValueURI);
        }
        return setAnnotationValueURI;
    }
  
    private Set<String> getClassAnnotations(OntClass ontClass) { 
        HashSet<String> setAnnotationValueURI = new HashSet<String>();
        StmtIterator classGroupStatements = ontClass.listProperties(inClassGroup);
        while (classGroupStatements.hasNext()) {
            Statement stmt = classGroupStatements.next();
            RDFNode o = stmt.getObject();
            setAnnotationValueURI.add(((Resource) o).getURI());
        }
        return setAnnotationValueURI;
    }
    
    @Override
    public String getPreferredLabel(EIURI uri) {
        assert uri != null;
        final Resource resource = this.jenaOntModel.getResource(uri.toString());
        if (resource == null) {
            return null;
        }
        final List<String> labels = new ArrayList<String>();
        // try each of the preferred label props, if one of them has a value, return it
        for (Property prop: prefLabelProperties) {
            getLiteralsForProperty(resource, prop, labels, null);
            if (labels.size() > 0) {
                return labels.get(0);
            }
        }
        return null;
    }    
    
    public List<Property> getPrefLabelProperties() {
        return prefLabelProperties;
    }
    
    @Override
    public List<String> getLabels(EIURI uri) {
        assert uri != null;
        final Resource resource = this.jenaOntModel.getResource(uri.toString());
        if (resource == null) {
            return Collections.emptyList();
        }
        
        // Synonym list consists of the preferred labels plus
        // any synonyms specified using the IAO Alt Term property
        final List<String> labels = new ArrayList<String>();
        final List<String> labelsLower = new ArrayList<String>();
        for (Property prop: prefLabelProperties) {
            getLiteralsForProperty(resource, prop, labels, labelsLower);
        }
        getLiteralsForProperty(resource, iaoAltTerm, labels, labelsLower);
        
        return labels;
    }
    
    public static void getLiteralsForProperty(final Resource r, final Property p, final List<String> values, final List<String> valuesLower) {
        final StmtIterator it = r.listProperties(p);
        while (it.hasNext()) {
            final Statement stmt = it.next();
            final RDFNode node = stmt.getObject();
            if (node.isLiteral()) {
                String value = ((Literal) node).getLexicalForm();
                if (value == null) {
                    continue;
                }
                // Ignore empty string values
                value = value.trim();
                if (value.isEmpty()) {
                    continue;
                }              
                // For any label except the preferred label (which will be first in 
                // the list), replace any underscores.
                if (values.size() > 0) {
                    value = value.replace('_', ' ');
                }
                
                if (valuesLower != null) {
                    // If a lower case values list is passed in,
                    // then caller cares about checking for duplicates
                    String valueLower = value.toLowerCase();
                    if (!valuesLower.contains(valueLower)) {
                        values.add(value);
                        valuesLower.add(valueLower);
                    }
                } else {
                    values.add(value);
                }
            }
        }        
    }
    
    @Override
    public List<String> getPropertyDefinitions(List<EIURI> propertyURIs) {
        List<String> results = new ArrayList<String>(propertyURIs.size());
        for (EIURI propertyURI : propertyURIs) {
            String def = getPropertyDefinition(propertyURI);
            results.add(def);
        }
        return results;
    }
    
    @Override
    public String getPropertyDefinition(EIURI propertyURI) {
        String def = null;
        OntProperty ontProperty = jenaOntModel.getOntProperty(propertyURI.toString());
        RDFNode defNode = ontProperty.getPropertyValue(eiPreferredDefinition);
        if (defNode != null) {
            def = ((Literal) defNode).getString();
        }
        //if (def == null) {
        //    defNode = ontClass.getPropertyValue(obiDefinition);
        //    if (defNode != null) {
        //        def = ((Literal) defNode).getString();
        //    }
        //}
        //if (def == null) {
        //    defNode = ontProperty.getPropertyValue(RDFS.comment);
        //    if (defNode != null) {
        //        def = "COMMENT: " + ((Literal) defNode).getString();
        //    }
        //}
        
        return def;        
    }

    /*
    public Map<EIURI,Map<String,String>> getPropertyAnnotations(EIURI classURI) {
        Map<EIURI,Map<String,String>> result = new HashMap<EIURI,Map<String,String>>();
        List<EIProperty> listProperties = getProperties(classURI);
        for (EIProperty prop : listProperties) {
            Map<String,String> mapAnnotationURIToValue = prop.getAnnotations();
            if (mapAnnotationURIToValue == null) {
                OntProperty ontProperty = jenaOntModel.getOntProperty(prop.getEntity().getURI().toString());
                mapAnnotationURIToValue = getPropertyAnnotations(ontProperty);
                prop.setAnnotations(mapAnnotationURIToValue);
            }
            result.put(prop.getEntity().getURI(), mapAnnotationURIToValue);
        }
        return result;
    }
    */
    
    private Set<String> getPropertyAnnotations(OntProperty ontProperty) {
        HashSet<String> setAnnotationValueURI = new HashSet<String>();
        StmtIterator propGroupStatements = ontProperty.listProperties(inPropertyGroup);
        while (propGroupStatements.hasNext()) {
            Statement stmt = propGroupStatements.next();
            RDFNode o = stmt.getObject();
            setAnnotationValueURI.add(((Resource) o).getURI());
        }
        return setAnnotationValueURI;      
    }
    
    /*
    @Override 
    public Set<EIClass> getResourceProviders() {
        for (EIClass resourceClasses : getClassesInGroup(EIOntConstants.CG_DATA_MODEL_CREATE)) {
            getProperties(classId, groupURI)
        }
    }
    */
    
    @Override
    public void traverseDataModel(List<Visitor> visitors) {
        logger.info("Start traversal of data model");
        Set<EIClass> setChecked = new HashSet<EIClass>();
        Deque<String> stack = new ArrayDeque<String>();
        try {
            for (EIClass top : getClassesInGroup(EIOntConstants.CG_RESOURCE_ROOT)) {
                traverseDataModel(top, setChecked, stack, visitors);
            }
        } catch (Throwable e) {
            logger.error("Unexpected error.  Stack: \r\n" + generateStackTrace(stack));
            throw new RuntimeException(e);
        }
        logger.info("Traversal of data model complete");
    }

    private void traverseDataModel(EIClass c, Set<EIClass> setChecked, Deque<String> stack, List<Visitor> visitors) {
        if (setChecked.contains(c)) {
            return;
        }
        stack.addLast("CLASS: " + c.toString());
        for (Visitor v : visitors) {
            v.visit(c, stack);
        }
        setChecked.add(c);
        
        if (c.hasProperty()) {
            List<EIProperty> properties = getProperties(c.getEntity().getURI());
            for (EIProperty prop:properties) {
                stack.addLast("PROP: " + prop.toString());
                for (Visitor v : visitors) {
                    v.visit(prop, stack);
                }
                if (prop instanceof EIObjectProperty) {
                    List<EIClass> objPropClassList = ((EIObjectProperty) prop).getRangeList();
                    for (EIClass objPropClass : objPropClassList) {
                        //logger.debug("class: " + c.getEntity().getLabel() + "  prop: " + prop.getEntity().getLabel() + "  obj_prop: " + objPropClass.getEntity().getLabel());
                        if (objPropClass != null) {
                            traverseDataModel(objPropClass, setChecked, stack, visitors);
                        }
                    }
                }
                stack.removeLast();
            }
        }
        
        if (c.hasSubClass()) {
            List<EIClass> subclasses = getSubClasses(c.getEntity().getURI());
            for (EIClass subclass:subclasses) {
                //logger.debug("class: " + c.getEntity().getLabel() + " subclass " + subclass.getEntity().getLabel());
                stack.addLast("SUBCLASS");
                traverseDataModel(subclass, setChecked, stack, visitors);
                stack.removeLast();
            }
        }
        stack.removeLast();
    }

    @Override
    public String generateStackTrace(Deque<String> stack) {
        StringBuilder buf = new StringBuilder();
        for (String item : stack) {
            buf.append(item);
            buf.append("\r\n");
        }
        return buf.toString();
    }
}
