/**
 * The eagle-i consortium
 * Harvard University
 * Nov 3, 2010
 */
package org.eaglei.model.jena;

import static org.eaglei.model.jena.SPARQLConstants.ANY_PREDICATE_VARIABLE;
import static org.eaglei.model.jena.SPARQLConstants.CREATION_DATE_VARIABLE;
import static org.eaglei.model.jena.SPARQLConstants.IS_STUB_VARIABLE;
import static org.eaglei.model.jena.SPARQLConstants.LABEL_VARIABLE;
import static org.eaglei.model.jena.SPARQLConstants.MODIFIED_DATE_VARIABLE;
import static org.eaglei.model.jena.SPARQLConstants.OWNER_NAME_VARIABLE;
import static org.eaglei.model.jena.SPARQLConstants.OWNER_VARIABLE;
import static org.eaglei.model.jena.SPARQLConstants.PROVIDER_NAME_VARIABLE;
import static org.eaglei.model.jena.SPARQLConstants.PROVIDER_VARIABLE;
import static org.eaglei.model.jena.SPARQLConstants.STATE_VARIABLE;
import static org.eaglei.model.jena.SPARQLConstants.SUBJECT_VARIABLE;
import static org.eaglei.model.jena.SPARQLConstants.TYPE_VARIABLE;

import java.util.ArrayList;
import java.util.Collections;
import java.util.HashMap;
import java.util.List;
import java.util.Map;
import java.util.Set;

import org.apache.commons.logging.Log;
import org.apache.commons.logging.LogFactory;
import org.eaglei.model.EIClass;
import org.eaglei.model.EIEntity;
import org.eaglei.model.EIInstance;
import org.eaglei.model.EIInstanceMinimal;
import org.eaglei.model.EIObjectProperty;
import org.eaglei.model.EIOntConstants;
import org.eaglei.model.EIOntModel;
import org.eaglei.model.EIProperty;
import org.eaglei.model.EIURI;
import org.springframework.context.ApplicationContext;
import org.springframework.context.support.ClassPathXmlApplicationContext;

import com.hp.hpl.jena.query.QuerySolution;
import com.hp.hpl.jena.query.ResultSet;

/**
 * @author Daniela Bourges-Waldegg
 * 
 */
public class EIInstanceMinimalFactory {

	private static EIOntModel ontModel;

	private static final Log logger = LogFactory.getLog( EIInstanceMinimalFactory.class );

	private static final boolean isDebugEnabled = logger.isDebugEnabled();

	private static final EIInstanceMinimalFactory INSTANCE = new EIInstanceMinimalFactory();

	public static EIInstanceMinimalFactory getInstance() {
		return INSTANCE;
	}

	private EIInstanceMinimalFactory() {
		// create and configure spring beans
		final ApplicationContext context = new ClassPathXmlApplicationContext( new String[] { "jena-model-config.xml" } );
		// retrieve configured instance
		ontModel = context.getBean( EIOntModel.class );
	}

	public EIInstanceMinimal createEmptyMinimal(final EIURI typeUri, final EIEntity instanceEntity) {
		if ( instanceEntity == null ) {
			return null;
		}
		final EIClass instanceClass = safeGetClass( typeUri );
		if ( instanceClass == null ) {
			return null;
		}
		final String label = instanceClass.getEntity().getLabel();
		instanceClass.getEntity().setLabel( label );
		final EIInstanceMinimal ei = EIInstanceMinimal.create( instanceClass.getEntity(), instanceEntity );
		return ei;
	}

	// TODO replace with SearchResult
	/**
	 * Create method for abridged EIInstances (label, type, owner, WFState, creation date)
	 * 
	 * @param resultSet
	 * @return
	 */

	// TODO the query could contain labels of owner and state as well
	public List<EIInstanceMinimal> create(final ResultSet resultSet) {
		if ( resultSet == null ) {
			return Collections.emptyList();
		}
		final Map<EIURI, EIInstanceMinimal> instances = new HashMap<EIURI, EIInstanceMinimal>();
		// Need to preserve ordering to avoid a sort operation; this list will contain live elements
		final List<EIInstanceMinimal> instanceList = new ArrayList<EIInstanceMinimal>();
		while ( resultSet.hasNext() ) {
			final QuerySolution solution = resultSet.next();
			// Only type and label are mandatory
			if ( solution.contains( SUBJECT_VARIABLE ) && solution.contains( TYPE_VARIABLE ) && solution.contains( LABEL_VARIABLE ) ) {
				EIInstanceMinimal ei = processMandatoryVariables( solution, instances, instanceList );
				if ( ei == null ) {
					continue;
				}
				// Continue with setting optional parts
				ei.setWFOwner( getEntityFromSolution( solution, OWNER_VARIABLE, OWNER_NAME_VARIABLE ) );
				ei.setWFState( getEntityFromSolution( solution, STATE_VARIABLE, "" ) );
				ei.setCreationDate( getStringFromSolution( solution, CREATION_DATE_VARIABLE ) );
				ei.setModifiedDate( getStringFromSolution( solution, MODIFIED_DATE_VARIABLE ) );
				ei.setLab( getEntityFromSolution( solution, PROVIDER_VARIABLE, PROVIDER_NAME_VARIABLE ) );
				ei.setIsStub( getBooleanFromSolution( solution, IS_STUB_VARIABLE ) );
			} else {
				if ( isDebugEnabled ) {
					logger.debug( "Query solution without type, label or subject: " + solution );
				}
				continue;
			}
		}
		return instanceList;
	}

	private EIInstanceMinimal processMandatoryVariables(QuerySolution solution, Map<EIURI, EIInstanceMinimal> instances, List<EIInstanceMinimal> instanceList) {
		final EIURI instanceUri = getUriFromSolution( solution, SUBJECT_VARIABLE );
		final EIURI instanceType = getUriFromSolution( solution, TYPE_VARIABLE );
		final String label = getStringFromSolution( solution, LABEL_VARIABLE );
		// type is not an EIClass ; we can skip solution
		if ( safeGetClass( instanceType ) == null ) {
			if ( isDebugEnabled ) {
				logger.debug( "Query solution without EIClass: " + solution );
			}
			return null;
		}
		EIInstanceMinimal ei;
		// need to determine if value in the result set needs to be appended to an already processed instance,
		// as properties with multiple values in the graph generate duplicate query solutions
		if ( !instances.containsKey( instanceUri ) ) {
			ei = createEmptyMinimal( instanceType, EIEntity.create( instanceUri, label ) );
			instances.put( instanceUri, ei );
			instanceList.add( ei );

		} else {
			ei = instances.get( instanceUri );
			if ( isDebugEnabled ) {
				logger.debug( "Processing query solution for an already processed instance: " + ei );
			}
			ei.addEIType( EIEntity.create( instanceType, "" ) );
			// FIXME process additional labs here
		}
		return ei;
	}

	public void setReferencingInstances(final EIInstance instance, final ResultSet resultSet) {
		if ( resultSet == null ) {
			return;
		}
		final List<EIProperty> objectProperties = ontModel.getProperties( instance.getInstanceType().getURI() );
		final Map<EIURI, EIInstanceMinimal> instances = new HashMap<EIURI, EIInstanceMinimal>();
		// Need to preserve ordering to avoid a sort operation; this list will contain live elements
		final List<EIInstanceMinimal> instanceList = new ArrayList<EIInstanceMinimal>();
		while ( resultSet.hasNext() ) {
			final QuerySolution solution = resultSet.next();
			// Only type and label are mandatory
			if ( solution.contains( SUBJECT_VARIABLE ) && solution.contains( TYPE_VARIABLE ) && solution.contains( LABEL_VARIABLE ) ) {
				EIInstanceMinimal ei = processMandatoryVariables( solution, instances, instanceList );
				if ( ei == null ) {
					continue;
				}
				//Check that the referencing instance has an ontology class
				if( safeGetClass( ei.getInstanceType().getURI()) == null ) {
					continue;
				}
				
				
				// Set the root superclass once we have the instance; reminder: last in list is root
				EIClass currentClass = ontModel.getClass( ei.getInstanceType().getURI() );
				Set<String> currentClassAnnotations = ontModel.getClassAnnotations( currentClass.getEntity().getURI() );
				if ( (currentClass != null) && (currentClassAnnotations != null) && currentClassAnnotations.contains( EIOntConstants.CG_DATA_MODEL_CREATE ) ) {
					ei.setDataModelRootSuperClass( currentClass.getEntity() );
				} else { 
					List<EIClass> superclasses = ontModel.getSuperClasses( ei.getInstanceType().getURI() );
					if (superclasses == null || superclasses.size() == 0) {
						ei.setDataModelRootSuperClass( ei.getInstanceType() );
					} else {
						ei.setDataModelRootSuperClass( superclasses.get( superclasses.size() - 1 ).getEntity() );
					}
				}

				List<EIURI> referencedByClassProperties = getOntPropertyUris(ontModel.getProperties( ei.getInstanceType().getURI() ));
				// Perform on-the-fly inferencing of inverse property values
				EIURI referencePredicate = getUriFromSolution( solution, ANY_PREDICATE_VARIABLE );
				if ( referencePredicate != EIURI.NULL_EIURI && referencedByClassProperties.contains(referencePredicate)) {
					if ( logger.isDebugEnabled() ) {
						logger.debug( "Referencing property: " + referencePredicate );
					}
					String referencePredicateStr = referencePredicate.toString();
					for (EIProperty property : objectProperties) {
						if ( property instanceof EIObjectProperty ) {
							String inverseProperty = ( (EIObjectProperty)property ).getInverseURI();
							if ( referencePredicateStr.equals( inverseProperty ) ) {
								instance.addMaterializedInverseProperty( property.getEntity(), ei );
								instanceList.remove( ei );
								if ( logger.isDebugEnabled() ) {
									logger.debug( "Inferred inverse property: " + property.getEntity() + "   value: " + ei );
								}
								break;
							}
						}
					}
				}
			} else {
				if ( isDebugEnabled ) {
					logger.debug( "Query solution without type, label or subject: " + solution );
				}
				continue;
			}
		}
		instance.setReferencedByList( instanceList );
	}

	/**
	 * @param properties
	 * @return
	 */
	private List<EIURI> getOntPropertyUris(List<EIProperty> properties) {
		List<EIURI> propUris = new ArrayList<EIURI>(properties.size());
		for(EIProperty property : properties) {
			propUris.add( property.getEntity().getURI() );
		}
		return propUris;
	}



	private static EIClass safeGetClass(final EIOntModel ontModel, final EIURI typeUri) {
		if ( ontModel.isModelClassURI( typeUri.toString() ) ) {
			return ontModel.getClass( typeUri );
		} else {
			return null;
		}
	}

	private static EIInstanceMinimal createEmptyMinimal(final EIOntModel ontModel, final EIURI typeUri, final EIEntity instanceEntity) {
		if ( instanceEntity == null ) {
			return null;
		}
		final EIClass instanceClass = safeGetClass( ontModel, typeUri );
		if ( instanceClass == null ) {
			return null;
		}
		final String label = instanceClass.getEntity().getLabel();
		instanceClass.getEntity().setLabel( label );
		final EIInstanceMinimal ei = EIInstanceMinimal.create( instanceClass.getEntity(), instanceEntity );
		return ei;
	}

	private EIEntity getEntityFromSolution(final QuerySolution solution, final String uriVariable, final String labelVariable) {
		if ( solution.contains( uriVariable ) ) {
			final EIURI uri = EIURI.create( solution.getResource( uriVariable ).getURI() );
			if ( STATE_VARIABLE.equals( uriVariable ) ) {
				return MetadataConstants.getStatusEntity( uri );
			}
			String label = "<none>";
			if ( solution.contains( labelVariable ) ) {
				label = solution.getLiteral( labelVariable ).getString();
			}
			return EIEntity.create( uri, label );
		} else {
			return EIEntity.NULL_ENTITY;
		}
	}

	private EIURI getUriFromSolution(final QuerySolution solution, final String variable) {
		if ( solution.contains( variable ) ) {
			return EIURI.create( solution.getResource( variable ).getURI() );
		} else {
			return EIURI.NULL_EIURI;
		}
	}

	private String getStringFromSolution(final QuerySolution solution, final String variable) {
		if ( solution.contains( variable ) ) {
			return solution.getLiteral( variable ).getString();
		} else {
			return "";
		}
	}

	private boolean getBooleanFromSolution(final QuerySolution solution, final String variable) {
		if ( solution.contains( variable ) ) {
			return solution.getLiteral( variable ).getString().equals( "True" );
			// || solution.getLiteral( variable ).getBoolean(); // TODO: use this once type is fixed
		} else {
			return false;
		}
	}

	private EIClass safeGetClass(final EIURI typeUri) {
		if ( ontModel.isModelClassURI( typeUri.toString() ) ) {
			return ontModel.getClass( typeUri );
		} else {
			return null;
		}
	}

}
