package org.eaglei.repository.rid;

/**
 ***************************************************************************
 * RIDGenerator (=RepostoryID Generator)
 * The RIDGenerator is used to create ID's for eagle I repository. The ID's are
 * unique only within one repository.
 * The RIDGenerator creates java.util.UUID instances. The created IDs ARE NOT
 * UUIDs, the are only repository ID's. java.util.UUID's where reused for our
 * purpose since they offered many features we needed.
 * The created ID's consist of 3 values, a system clock time, a clockID and a
 * sequence number. The clockID is incremented in cases where more ID's are
 * requested in the same clock cycle than are available or if a clock error
 * has occured.
 * The system time will be stored on the most significant bits of the UUID.
 * clockID and sequence on the least significant bits where the clockID is
 * shifted onto the most significant 32 bits and the sequnce number on the
 * least significant ones. Since we dont have an unsigned datatype available
 * the sequence and clockID start counting at Integer.MIN_VALUE rather than 0.
 *
 * RIDGenerator allows for allocating sequences of UUIDs at once, preallocated
 * in the form of a UUID Array or a RIDSequence that then allocates new
 * UUID in sequence on demand in it's reserved range through the Iterator
 * interface.
 *
 * @author Carsten Schulz
 * @version $Id: $
 */

import java.util.UUID;
import java.security.SecureRandom;
import java.security.NoSuchAlgorithmException;

import org.apache.log4j.LogManager;
public final class RIDGenerator {
        
    private static org.apache.log4j.Logger logger = LogManager.getLogger(RIDGenerator.class);
    
    private final static RIDGenerator singleton = new RIDGenerator();
    private long previousTime;
    private int clockID=0;
    private int sequence=Integer.MIN_VALUE;
    private TimeProvider tp = null;  // parameterized for testing!!
    
    private RIDGenerator(){
        if (singleton!=null){
                throw new RuntimeException("Encountered RIDGenerator singleton contract violation attempt!");
        }
        tp = new SystemTimeProvider();
        init();
    }
    
    /**
     * <p>Constructor for RIDGenerator.</p>
     *
     * @param testTP a {@link org.eaglei.repository.rid.TimeProvider} object.
     */
    protected RIDGenerator (TimeProvider testTP){
        /* For unit testing only!!!
         * This constructor is only to be used for testing the ID generation. A TimeProvider from the testing framework
         * can be injected so that the TestCase can control it for testing purposes!!
         */
        tp = testTP;
        init();
    }
    
    /**
     * <p>getInstance</p>
     *
     * @return a {@link org.eaglei.repository.rid.RIDGenerator} object.
     */
    public static RIDGenerator getInstance(){
        return singleton;
    }
    private void init() {
        // We don't want to persist anything in a database because there might not
        // always be a database...therefore we will create a randomish clockid
        // on every startup since that seems a better strategy than always starting
        // with the same id...
        byte[] bytes = new byte[4];
        SecureRandom secureRandom =null;
        try {
            // Initialize a secure random number generator
                secureRandom = SecureRandom.getInstance("SHA1PRNG");
        } catch (NoSuchAlgorithmException e) {
                // fall back to default provider...
                logger.warn("RIDGenerator: Falling back to default random generator...");
                secureRandom = new SecureRandom();
        }
        secureRandom.nextBytes(bytes);

        clockID = 0;
        clockID |= bytes[0] & 0xFF;
        clockID <<= 8;
        clockID |= bytes[1] & 0xFF;
        clockID <<= 8;
        clockID |= bytes[2] & 0xFF;
        clockID <<= 8;
        clockID |= bytes[3] & 0xFF;
        // logworthy data...
        logger.info("Initialized RID clock-id to:" + clockID);
    }
    
    private void resetSequence(){
        sequence=Integer.MIN_VALUE;
    }
    private void resetClockID(){
        clockID=Integer.MIN_VALUE;
    }
    private void setPreviousTime(long value){
        previousTime=value;
    }
    private long getPreviousTime(){
        return previousTime;
    }
    private int getSequence(){
        return sequence;
    }
    private void incrementSequence(){
        sequence++;
    }
    private int getClockID(){
        return clockID;
    }
    private void incrementClockID(){
        if (clockID<Integer.MAX_VALUE){
                clockID++;
        } else {
                resetClockID();
        }
        logger.info("Incremented clock-id, value now:" + clockID);
    }
    private long getTime(){
        return tp.timeNow();
    }
    
    private void increaseSequenceBy(int anAmount){
        sequence+=anAmount;
    }
    /**
     * Test whether anAmount of sequence numbers remains
     * in current clockcycle.
     * @return true if anAmount of sequences can be allocated.
     */
    private boolean hasSequencesLeft(int anAmount){
        if (Integer.MAX_VALUE-anAmount>=sequence){
                return true;
        }
        return false;
    }

    /**
     * Generates a new time field. Each time field is unique and larger than the
     * previously generated time field.
     *
     * @return a new time value
     *
     *    public static synchronized long newTime() {
     *
     *        long timeNano=0l;
     *
     *        long timeMilli=System.currentTimeMillis();
     *        if (timeMilli > previousTime) {
     *                previousTime = timeNano;
     *        }
     *        else {
     *                if(timeMilli == previousTime)
     *                timeNano = ++previousTime;
     *        }
     *        // UTC time.
     *        //Convert to nanoseconds and add 122192928000000000
     *        timeNano = (timeMilli * 10000) + 0x01B21DD213814000L;
     *
     *
     *        time |= (timeNano & 0xFFFF00000000L) >> 16;
     *
     *        // time hi and version
     *
     *        time |= 0x1000 | ((timeNano >> 48) & 0x0FFF); // version 1
     *
     *        return time;
     *
     *    }
     */
    public boolean isUUID(){
        // IDs generated by this class are nor UUIDs, they are unique only
        // within one repository.
        return false;
    }

    /**
     * <p>newID</p>
     *
     * @return a {@link java.util.UUID} object.
     */
    public synchronized UUID newID(){
        long time=getTime();
        if (!(time == previousTime)){
                setPreviousTime(time);
                resetSequence();
                if (time < previousTime){
                        incrementClockID();
                }
        }
        int s = getSequence();
        long least = composeLeast(getClockID(), s);
        UUID result=new UUID(time, least);
        
        if(s < Integer.MAX_VALUE){
                incrementSequence();
        } else {
                incrementClockID();
                resetSequence();
        }
        return result;
    }
    
    private long composeLeast(int clock, int sequence){
        return composeLeast((((long) clock)<<32), sequence);
    }
    private long composeLeast(long clock, int sequence){
        // clock already shifted....
        return (clock | (((long) sequence) & 0xffffffffl));
     }

    public boolean flag=false;
    /**
     * <p>newIDs</p>
     *
     * @param amount a int.
     * @return an array of {@link java.util.UUID} objects.
     */
    public synchronized UUID[] newIDs(int amount){
        prepareForBulkAllocation(amount);
        int start=getSequence();
        increaseSequenceBy(amount);
        UUID results[]=new UUID[amount];
        //clockID stays constant in sequence range!
        long clockid=(long) getClockID();
        clockid<<=32;
        long time=getPreviousTime();
        long least=0l;
        for(int i=0;i<amount;i++){
                least=composeLeast(clockid,(start+i));
                results[i]= new UUID(time, least);
        }
        return results;
    }
    
    
    private void prepareForBulkAllocation(int amount){
        if (amount <=0){
            throw new RuntimeException("Invalid sequence range!");
        }
        long time=getTime();
        if (time > previousTime){
            setPreviousTime(time);
            resetSequence();
            return;
        } else {
            if (time==getPreviousTime()){
                if(hasSequencesLeft(amount)){
                    return;
                }
            }
        }
        // either clock error or no sequences left
        incrementClockID();
        resetSequence();
    }

    
    /**
     * <p>getIDSequence</p>
     *
     * @param amount a int.
     * @return a {@link org.eaglei.repository.rid.RIDSequence} object.
     */
    public synchronized RIDSequence getIDSequence(int amount){
        prepareForBulkAllocation(amount);
        int start=getSequence();
        increaseSequenceBy(amount);
        RIDSequence result = new RIDSequence(getPreviousTime(), getClockID(), start, amount);
        return result;
    }
    

}
