package org.eaglei.repository.servlet;

import java.io.IOException;
import java.io.FilterInputStream;
import java.io.InputStream;
import java.util.ArrayList;
import java.util.Date;
import java.net.URL;
import java.net.MalformedURLException;
import java.util.jar.JarInputStream;
import java.util.jar.JarEntry;
import java.util.regex.Pattern;
import java.util.regex.Matcher;

import javax.servlet.http.HttpServletRequest;
import javax.servlet.http.HttpServletResponse;
import javax.servlet.ServletException;

import org.apache.log4j.Logger;
import org.apache.log4j.LogManager;

import com.hp.hpl.jena.ontology.OntModel;
import com.hp.hpl.jena.ontology.Ontology;
import com.hp.hpl.jena.rdf.model.StmtIterator;
import com.hp.hpl.jena.util.iterator.ExtendedIterator;

import org.openrdf.OpenRDFException;
import org.openrdf.query.impl.MapBindingSet;
import org.openrdf.query.BindingSet;
import org.openrdf.model.Resource;
import org.openrdf.model.URI;
import org.openrdf.model.Literal;
import org.openrdf.model.Statement;
import org.openrdf.model.Value;
import org.openrdf.model.ValueFactory;
import org.openrdf.model.vocabulary.OWL;
import org.openrdf.rio.RDFFormat;
import org.openrdf.repository.RepositoryConnection;
import org.openrdf.repository.RepositoryException;
import org.openrdf.repository.RepositoryResult;
import org.openrdf.rio.RDFParseException;
import org.springframework.web.context.WebApplicationContext;
import org.springframework.web.context.support.WebApplicationContextUtils;

import org.eaglei.repository.Access;
import org.eaglei.repository.DataRepository;
import org.eaglei.repository.NamedGraph;
import org.eaglei.repository.Provenance;
import org.eaglei.repository.status.BadRequestException;
import org.eaglei.repository.status.ForbiddenException;
import org.eaglei.repository.util.SPARQL;
import org.eaglei.repository.vocabulary.DATAMODEL;

/**
 * Report on and load the data model ontology from various sources.
 *
 * @author Larry Stone
 * Started June, 2010
 * @version $Id: $
 */
public class Model extends RepositoryServlet
{
    private static Logger log = LogManager.getLogger(Model.class);

    /** {@inheritDoc} */
    @Override
    public void init() {
        WebApplicationContext ctx =
            WebApplicationContextUtils.getWebApplicationContext(getServletContext());
        // The pure jena OntModel
        ontModel = ctx.getBean("jenaOntModel", OntModel.class);
        // The EI* ont model
        //eagleiOntModel = ctx.getBean(EIOntModel.class);
    }
    
    /** {@inheritDoc} */
    protected void doGet(HttpServletRequest request, HttpServletResponse response)
        throws ServletException, java.io.IOException
    {
        // sanity check
        if (request.getParameter("action") != null)
            throw new BadRequestException("The 'action' argument is not allowed in the GET method, use POST.");

        String format = request.getParameter("format");
        try {
            RepositoryConnection rc = WithRepositoryConnection.get(request);

            // get the current version of datamodel ontology
            String loadedVersion = getLoadedVersionInfo(rc);
            if (loadedVersion == null)
                loadedVersion = "(not loaded or not in ontology)";

            // get version from jena model
            String availableVersion = getAvailableVersionInfo();
            if (availableVersion == null)
                availableVersion = "(failed or not in ontology)";

            ArrayList<BindingSet> results = new ArrayList<BindingSet>(1);
            MapBindingSet bs = new MapBindingSet(2);
            ValueFactory vf = rc.getValueFactory();
            bs.addBinding("loaded", vf.createLiteral(loadedVersion));
            bs.addBinding("available", vf.createLiteral(availableVersion));
            results.add(bs);
            SPARQL.sendTupleQueryResults(request, response, format, results);
        } catch (OpenRDFException e) {
            log.error(e);
            throw new ServletException(e);
        }
    }

    /** {@inheritDoc} */
    protected void doPost(HttpServletRequest request, HttpServletResponse response)
        throws ServletException, java.io.IOException
    {
        boolean force = false;
        long count = 0;
        long modTime = -1;
        RepositoryConnection rc = WithRepositoryConnection.get(request);
        ValueFactory vf = rc.getValueFactory();
        String action = request.getParameter("action");
        String source = request.getParameter("source");

        URI context = DATAMODEL.NAMESPACE_URI;

        if (action == null)
            throw new BadRequestException("Missing required argument 'action'");
        else if (action.equals("load"))
            force = true;
        else if (!action.equals("update"))
            throw new BadRequestException("Argument action must be one of: 'load', 'update'");

        if (source == null)
            throw new BadRequestException("Missing required argument 'source'");

        // check access
        if (!(Access.hasPermission(request, context, Access.ADD) &&
              Access.hasPermission(request, context, Access.REMOVE))) {
            throw new ForbiddenException("User is not permitted to replace the data model ontology graph");
        }

        // final sanity check, source=jar does not allow "update" since we don't have easy way to get version out of jarred owls.
        // (it *can* be done, but not worth the time for this kludge..)
        if (source.equals("jar") && !force)
            throw new BadRequestException("You must force-load the data model from the jar source since version is not available, although it SHOULD be the same as for the model source.");

        try {
            boolean created = rc.size(context) == 0;
            String loadedVersion = getLoadedVersionInfo(rc);
            String availableVersion = getAvailableVersionInfo();
            String sourceLoaded = null;
             
            if (availableVersion == null)
                throw new ServletException("Failed to get version of available data model ontology, perhaps failed to load ontology.");
             
            if (force || loadedVersion == null || !loadedVersion.equals(availableVersion)) {
                if (force)
                    log.debug("Forced reload of data model ontology.");
                else
                    log.debug("Updating data model ontology: loaded="+loadedVersion+", available="+availableVersion);
                 
                rc.clear(context);

                // load from EIOntModel
                if (source.equals("model")) {
                    // Need to get top-level Base Model *and* crawl all
                    // submodels (including recursive imports) to get their
                    // base models too..
                    //OntModel om = getOntModel();
                    count += convertOntModel(rc, ontModel, context);
                    ExtendedIterator<OntModel> mi = ontModel.listSubModels(true);
                    try {
                        while (mi.hasNext()) {
                            count += convertOntModel(rc, mi.next(),context);
                        }
                    } finally {
                        mi.close();
                    }
                    sourceLoaded = DATAMODEL.NAMESPACE_URI.toString();
                    log.info("Copied "+count+" statements from Data Model Ontology.");

                // load from OWL files in maven project jar:
                } else if (source.equals("jar")) {
                    // XXX MAGIC - assumes that this is the key data model
                    String toplevel = DataRepository.getInstance().getConfigurationProperty(
                            "eaglei.repository.datamodel.file", "ero.owl");
                     
                    String jar = getDataModelJarSpec(toplevel);
                    modTime = loadDataModelFromOWL(rc, context, toplevel, jar);
                    sourceLoaded = jar;

                } else
                    throw new BadRequestException("Argument source must be one of: 'model', 'jar'");

                // set named graph admin MD - force creation if necessary
                NamedGraph ng = NamedGraph.find(request, context, true);
                ng.setType(request, NamedGraph.Type.ontology);
                ng.setLabel(request, "eagle-i Data Model Ontology");

                // set provenance like any other graph
                Date now = new Date();
                Provenance p = new Provenance(context);
                if (created)
                    p.setCreated(request, now);
                else
                    p.setModified(request, now);
                Date sourceModified = null;
                if (modTime != -1)
                    sourceModified = new Date(modTime);
                p.setSource(request, sourceLoaded, sourceModified);
                rc.commit();
                response.setStatus(HttpServletResponse.SC_CREATED);
            }
            else
                response.setStatus(HttpServletResponse.SC_OK);
        } catch (OpenRDFException e) {
            log.error(e);
            throw new ServletException(e);
        }
    }

    // get the *BaseModel* and convert that, OntModel includes inferred stms
    private int convertOntModel(RepositoryConnection rc, OntModel om, URI context)
        throws RepositoryException
    {
        ValueFactory vf = rc.getValueFactory();
        int count = 0;
        StmtIterator si = om.getBaseModel().listStatements();
        try {
            while (si.hasNext()) {
                com.hp.hpl.jena.rdf.model.Statement js = si.nextStatement();
                Resource subject = convertResource(js.getSubject(), vf);
                URI predicate = vf.createURI(js.getPredicate().getURI());
                Value object = null;
                com.hp.hpl.jena.rdf.model.RDFNode jv = js.getObject();
                if (jv.isResource())
                    object = convertResource((com.hp.hpl.jena.rdf.model.Resource)jv, vf);
                else if (jv.isLiteral()) {
                    com.hp.hpl.jena.rdf.model.Literal jvl = (com.hp.hpl.jena.rdf.model.Literal)jv;
                    String lang = jvl.getLanguage();
                    String dt = jvl.getDatatypeURI();
                    if (dt != null)
                        object = vf.createLiteral(jvl.getLexicalForm(), vf.createURI(dt));
                    else if (lang != null)
                        object = vf.createLiteral(jvl.getLexicalForm(), lang);
                    else
                        object = vf.createLiteral(jvl.getLexicalForm());
                } else
                    log.error("Skipping unknown Value object (class="+jv.getClass().getName()+"): "+jv);
                // XXX DEBUG TEMP
                //log.debug("Adding statement = ("+subject+", "+predicate+", "+object+")");
                rc.add(subject, predicate, object, context);
                ++count;
            }
        } finally {
            si.close();
        }
        return count;
    }

    private Resource convertResource(com.hp.hpl.jena.rdf.model.Resource jr, ValueFactory vf)
    {
        if (jr.isURIResource())
            return vf.createURI(jr.getURI());
        else {
            String id = jr.getId().getLabelString();
            //XXX log.debug("Creating Blank Node with id="+id);
            return vf.createBNode(id);
        }
    }

    private static OntModel ontModel = null;
    /*
    private synchronized OntModel getOntModel()
    {
        if (ontModel == null) {
            log.info("Loading OntModel from JenaEIOntModel..");
            ontModel = org.eaglei.model.jena.JenaEIOntModel.INSTANCE.getOntModel();
        }
        return ontModel;
    }
    */

    // get versionInfo from "available" data model onto, i.e. EIOntModel
    private String getAvailableVersionInfo()
    {
            //OntModel om = getOntModel();
            Ontology oo = ontModel.getOntology(DATAMODEL.NAMESPACE);
            if (oo == null)
                log.warn("Failed to find ontology in OntModel, uri="+DATAMODEL.NAMESPACE);
            else
                return oo.getVersionInfo();
            return null;
    }

    // get versionInfo from loaded data model onto, i.e. EIOntModel
    private String getLoadedVersionInfo(RepositoryConnection rc)
        throws RepositoryException
    {
        RepositoryResult<Statement> rr =
                rc.getStatements(DATAMODEL.NAMESPACE_URI, OWL.VERSIONINFO, null, false, DATAMODEL.NAMESPACE_URI);
        try {
            if (rr.hasNext()) {
                Value vers = rr.next().getObject();
                if (vers instanceof Literal)
                    return ((Literal)vers).getLabel();
            }
        } finally {
            rr.close();
        }
        return null;
    }


    private String getDataModelJarSpec(String toplevel)
        throws ServletException
    {
        URL ero = this.getClass().getClassLoader().getResource(toplevel);
        if (ero == null)
            throw new ServletException("Cannot load data model OWL, since the top-level file \""+toplevel+"\"  is not available as a resource.  Check your configuration.");

        // The rest of this kludge assumes we have a jar-scheme URL, which
        // has the form:   jar:<url>!/<entry>
        // e.g.  jar:file:/..../lib/eagle-i-model-owl-0.4.3.jar!/ero.owl
        // so we can extract the URL of the jar itself:
        String eros = ero.toString();
        if (eros.startsWith("jar:")) {
            Pattern ep = Pattern.compile("jar:([^!]+)!/"+Pattern.quote(toplevel));
            log.debug("Groveling jar URL with pattern=\""+ep.pattern()+"\"");
            Matcher em = ep.matcher(eros);
            if (em.matches()) {
                String jarURLspec = em.group(1);
                log.debug("Got jar URL = "+jarURLspec);
                return jarURLspec;
            } else {
                log.warn("Failed matching pattern=\""+ep.pattern()+"\" against owl resource URL="+eros);
                throw new ServletException("Failed to extract jar's URL from resource (see log for details): "+eros);
            }
        } else {
            log.warn("Cannot load data model OWL, URL for the toplevel file is NOT in Jar scheme: "+eros);
            throw new ServletException("Cannot load data model OWL, URL for the toplevel file is NOT in Jar scheme: "+eros);
        }
    }

    /**
     * Grovel the data model OWL files out of available jars.
     *
     */
    private long loadDataModelFromOWL(RepositoryConnection rc, URI context, String toplevel, String jarURLspec)
        throws ServletException, IOException
    {
        long result = -1;

        log.info("Loading data model from jar at: "+jarURLspec);
        JarInputStream jis = null;
        try {
            URL jarURL = new URL(jarURLspec);
            jis = new JarInputStream(jarURL.openStream());
            JarEntry je;
            while ((je = jis.getNextJarEntry()) != null) {
                if (je.isDirectory())
                    continue;
                String fn = je.getName();
                if (fn.endsWith(".owl")) {
                    log.info("..Loading OWL from jar entry: "+fn);
                    try {
                        rc.add(new UnclosableInputStream(jis), "", RDFFormat.RDFXML, context);
                    } catch (RDFParseException e) {
                        log.warn("Skipping file="+fn+", failed parsing: "+e);
                     // XXX FIXME: this failure should appear in overall servlet status as well
                    }
                    if (fn.equals(toplevel) || fn.equals("/"+toplevel)) {
                        result = je.getTime();
                        log.debug("Found toplevel entry, setting mod time = "+result);
                    }
                } else
                    log.info("..Skipping non-OWL jar file = "+fn);
            }
        } catch (MalformedURLException e) {
            log.error(e);
            throw new ServletException(e);
        } catch (OpenRDFException e) {
            log.error(e);
            throw new ServletException(e);
        } finally {
            if (jis != null)
                jis.close();
        }
        return result;
    }

    // InputStream wrapper that ignores close() so RDF reader can
    // read from jar entry without closing the whole jar input stream
    private class UnclosableInputStream extends FilterInputStream {
        private UnclosableInputStream(InputStream in)
        {
            super(in);
        }

        @Override
        public void close()
        {
        }
    }
}
