package org.eaglei.repository;

import org.eaglei.repository.vocabulary.REPO;
import org.eaglei.repository.util.SPARQL;
import org.eaglei.repository.status.NotFoundException;
import java.util.ArrayList;
import java.util.List;
import java.util.Map;
import java.util.HashMap;
import javax.servlet.ServletException;
import javax.servlet.http.HttpServletRequest;
import org.apache.log4j.Logger;
import org.apache.log4j.LogManager;

import org.openrdf.OpenRDFException;
import org.openrdf.model.URI;
import org.openrdf.model.Literal;
import org.openrdf.model.Value;
import org.openrdf.model.Statement;
import org.openrdf.repository.RepositoryConnection;
import org.openrdf.repository.RepositoryException;
import org.openrdf.model.vocabulary.RDF;
import org.openrdf.model.vocabulary.RDFS;
import org.openrdf.repository.RepositoryResult;
import org.openrdf.query.TupleQuery;
import org.openrdf.query.TupleQueryResultHandlerBase;
import org.openrdf.query.TupleQueryResultHandlerException;
import org.openrdf.query.BindingSet;
import org.openrdf.query.QueryLanguage;
import org.openrdf.query.MalformedQueryException;

import org.eaglei.repository.servlet.WithRepositoryConnection;

/**
 * Role object model, reflects the :Role object in RDF database.
 * This is a READ ONLY model.
 *
 * Named Graph usage: Role looks for statements in ALL graphs, although
 *  they should really only be found in the Repository Ontology graph,
 *  and possibly the NG_Internal graph.  It requires inference for RDFS
 *  type propagation to pick up all the Roles since they are all subclasses
 *  of :Role in the repo ontology.
 *
 * @author Larry Stone
 * Started April 26, 2010
 * @version $Id: $
 */
public class Role
{
    private static Logger log = LogManager.getLogger(Role.class);

    private URI uri;
    private String label;

    // Cache, so we don't keep allocating new Roles - key is URI
    // Since there is no admin UI to change role names, we can
    // make almost-believable excuses to avoid decaching.
    private static Map<URI,Role> cache = new HashMap<URI,Role>();

    // SPARQL Query to get all roles and labels..
    // XXX KLUDGE NOTE: Since Roles depend on rdfs:subClassOf type propagation,
    // we need the Sesame FC inferencing and thus have to look in null
    // context for now..  XXX FIXME
    private final static String roleQuery =
          "SELECT DISTINCT ?ruri ?rlabel WHERE { "+
          " ?ruri <"+RDFS.SUBCLASSOF+"> <"+REPO.ROLE+"> . \n"+
          "  FILTER ( ?ruri != <"+REPO.ROLE+"> )\n"+
          "  OPTIONAL { ?ruri <"+RDFS.LABEL+"> ?rlabel } }";

    private Role(URI uri, String label)
    {
        super();
        this.uri = uri;
        this.label = label;
    }

    /**
     * Get the Role for given URI, lookup label.
     *
     * @param request a {@link javax.servlet.http.HttpServletRequest} object.
     * @param uri URI of the role
     * @return a {@link org.eaglei.repository.Role} object representing role.
     * @throws javax.servlet.ServletException if any.
     */
    public static Role find(HttpServletRequest request, URI uri)
        throws ServletException
    {
        RepositoryConnection rc = WithRepositoryConnection.get(request);
        String label = null;
        RepositoryResult<Statement> rr = null;
        try {
            // sanity check, should have right type
            if (!rc.hasStatement(uri, RDF.TYPE, REPO.ROLE, true))
                throw new NotFoundException("There is no Role of URI="+uri);
            try {
                rr = rc.getStatements(uri, RDFS.LABEL, null, false);
                if (rr.hasNext()) {
                    Value lv = rr.next().getObject();
                    if (lv instanceof Literal)
                        label = ((Literal)lv).getLabel();
                }
            } finally {
                rr.close();
            }
        } catch (RepositoryException e) {
            log.error("Failed finding Role for URI="+uri,e);
            throw new ServletException("Failed finding Role for URI="+uri,e);
        }
        if (label == null) {
            log.warn("No label found for Role URI="+uri);
            label = uri.getLocalName();
        }
        Role result = find(uri, label);
        log.debug("Role.find("+uri.stringValue()+") => "+result);
        return result;
    }

    /**
     * Alternate version of "find" for when the URI and label are already
     * known, e.g. when processign results of user SPARQL query.
     * This saves the overhead of another lookup when it isn't necessary.
     *
     * @param uri a {@link org.openrdf.model.URI} object.
     * @param label a {@link java.lang.String} object.
     * @return a {@link org.eaglei.repository.Role} object.
     */
    public static Role find(URI uri, String label)
    {
        if (cache.containsKey(uri))
            // XXX maybe should query and check if label still matches?
            return cache.get(uri);
        else {
            log.debug("find: caching a new Role("+uri.stringValue()+", "+label+")");
            Role result = new Role(uri, label);
            cache.put(uri, result);
            return result;
        }
    }

    /**
     * Get all known Roles - includes pseudo-Roles Anonymous and Authenticated
     * that should NOT be directly assigned to Users.
     * Make the query each time instead of relying on cache since some sneaky
     * admin may *add* new roles by uploading statements to NG_Internal graph.
     *
     * @param request a {@link javax.servlet.http.HttpServletRequest} object.
     * @return all roles in a {@link java.lang.Iterable} object.
     * @throws javax.servlet.ServletException if any.
     */
    public static Iterable<Role> findAll(HttpServletRequest request)
        throws ServletException
    {
        RepositoryConnection rc = WithRepositoryConnection.get(request);
        List<Role> result = new ArrayList<Role>();

        try {
            log.debug("All role SPARQL query = "+roleQuery);
            TupleQuery q = rc.prepareTupleQuery(QueryLanguage.SPARQL, roleQuery);
            q.setDataset(SPARQL.InternalGraphs);

            // need inference!
            q.setIncludeInferred(true);
            q.evaluate(new allRoleHandler(result));
        } catch (MalformedQueryException e) {
            log.error("Rejecting malformed query:"+e);
            throw new ServletException(e);
        } catch (OpenRDFException e) {
            log.error(e);
            throw new ServletException(e);
        }
        return result;
    }

    /**
     * <p>getURI</p>
     *
     * @return URI subject of the role, a {@link org.openrdf.model.URI} object.
     */
    public URI getURI()
    {
        return uri;
    }
    /**
     * <p>Getter for the field <code>label</code>.</p>
     *
     * @return label of the role, a {@link java.lang.String} object.
     */
    public String getLabel()
    {
        return label;
    }

    /** {@inheritDoc} */
    public boolean equals(Object other)
    {
        return other instanceof Role && uri.equals(((Role)other).uri);
    }

    /**
     * <p>toString</p>
     *
     * @return a {@link java.lang.String} object.
     */
    public String toString()
    {
        return "<#Role: uri="+uri.toString()+
            ", label="+label+">";
    }

    // tuple query result handler to gather list of all roles.
    private static class allRoleHandler extends TupleQueryResultHandlerBase
    {
        private List<Role> result = null;

        public allRoleHandler(List<Role> u)
        {
            super();
            allRoleHandler.this.result = u;
        }
        public void handleSolution(BindingSet bs)
            throws TupleQueryResultHandlerException
        {
            Value newURI = bs.getValue("ruri");
            if (newURI == null || !(newURI instanceof URI))
                log.error("Should not get null or non-URI result in allRoleHandler: "+newURI);

            // start a new one
            else {
                Value rlabel = bs.getValue("rlabel");
                String l = (rlabel != null && rlabel instanceof Literal) ?
                           ((Literal)rlabel).getLabel() : "";
                result.add(find((URI)newURI, l));
            }
        }
    }

}
