package org.eaglei.repository.servlet;

import java.io.InputStreamReader;
import java.io.IOException;
import java.io.Reader;
import java.io.StringReader;
import java.io.File;
import java.util.Arrays;
import java.util.ArrayList;
import java.util.Date;
import java.util.Iterator;
import java.util.List;
import javax.servlet.http.HttpServletRequest;
import javax.servlet.http.HttpServletResponse;
import javax.servlet.ServletException;

import org.apache.log4j.Logger;
import org.apache.log4j.LogManager;

import org.apache.commons.fileupload.servlet.ServletFileUpload;
import org.apache.commons.fileupload.FileUploadException;
import org.apache.commons.fileupload.disk.DiskFileItemFactory;
import org.apache.commons.fileupload.disk.DiskFileItem;

import org.openrdf.query.Dataset;
import org.openrdf.query.impl.DatasetImpl;
import org.openrdf.model.URI;
import org.openrdf.model.Graph;
import org.openrdf.model.Literal;
import org.openrdf.model.Resource;
import org.openrdf.model.Statement;
import org.openrdf.model.Value;
import org.openrdf.model.ValueFactory;
import org.openrdf.model.vocabulary.RDF;
import org.openrdf.model.vocabulary.XMLSchema;
import org.openrdf.model.impl.BooleanLiteralImpl;
import org.openrdf.model.impl.GraphImpl;
import org.openrdf.model.impl.LiteralImpl;
import org.openrdf.query.impl.MapBindingSet;
import org.openrdf.query.BindingSet;
import org.openrdf.repository.RepositoryConnection;
import org.openrdf.repository.RepositoryException;
import org.openrdf.repository.RepositoryResult;
import org.openrdf.OpenRDFException;
import org.openrdf.rio.RDFFormat;
import org.openrdf.rio.Rio;
import org.openrdf.rio.RDFParser;
import org.openrdf.rio.helpers.RDFHandlerBase;
import org.openrdf.rio.RDFHandlerException;
import org.openrdf.rio.RDFParseException;

import org.eaglei.repository.Access;
import org.eaglei.repository.DataRepository;
import org.eaglei.repository.EditToken;
import org.eaglei.repository.FakeFlow;
import org.eaglei.repository.NamedGraph;
import org.eaglei.repository.View;
import org.eaglei.repository.Provenance;
import org.eaglei.repository.status.BadRequestException;
import org.eaglei.repository.status.ConflictException;
import org.eaglei.repository.status.ForbiddenException;
import org.eaglei.repository.status.NotFoundException;
import org.eaglei.repository.status.HttpStatusException;
import org.eaglei.repository.util.SPARQL;
import org.eaglei.repository.util.Utils;
import org.eaglei.repository.vocabulary.DCTERMS;
import org.eaglei.repository.vocabulary.REPO;

/**
 * Create or update a single resource instance.
 *
 * @author Larry Stone
 * @version $Id: $
 */
public class Update extends RepositoryServlet
{
    private static Logger log = LogManager.getLogger(Update.class);

    // allowable values for Action arg
    private enum Action { create, update, gettoken };

    /** {@inheritDoc} */
    protected void doPost(HttpServletRequest request, HttpServletResponse response)
        throws ServletException, java.io.IOException
    {
        // web request parameters:
        String rawaction = null;
        String deleteMIMEType = null;
        Reader deleteStream = null;
        String insertMIMEType = null;
        Reader insertStream = null;
        String format = null;
        String workspace = null;
        String rawuri = null;
        String rawtoken = null;

        // if POST with multipart, grovel through args
        if (ServletFileUpload.isMultipartContent(request)) {
            try {
                ServletFileUpload upload = new ServletFileUpload();
                upload.setFileItemFactory(new DiskFileItemFactory(100000,
                  (File)getServletConfig().getServletContext().getAttribute("javax.servlet.context.tempdir")));
                for (DiskFileItem item : (List<DiskFileItem>)upload.parseRequest(request)) {
                    String ifn = item.getFieldName();
                    if (ifn.equals("action"))
                        rawaction = item.getString();
                    else if (ifn.equals("delete")) {
                        deleteStream = new InputStreamReader(item.getInputStream());
                        deleteMIMEType = item.getContentType();
                    } else if (ifn.equals("insert")) {
                        insertStream = new InputStreamReader(item.getInputStream());
                        insertMIMEType = item.getContentType();
                    } else if (ifn.equals("format"))
                        format = item.getString();
                    else if (ifn.equals("workspace"))
                        workspace = item.getString();
                    else if (ifn.equals("uri"))
                        rawuri = item.getString();
                    else if (ifn.equals("token"))
                        rawtoken = item.getString();
                    else
                        log.warn("Unrecoginized request argument: "+ifn);
                }
            } catch  (FileUploadException e) {
                log.error(e);
                throw new BadRequestException("failed parsing multipart request");
            }

        // gather args from input params instead
        } else {
            rawaction = request.getParameter("action");

            String delete = request.getParameter("delete");
            if (delete != null)
                deleteStream = new StringReader(delete);
            String insert = request.getParameter("insert");
            if (insert != null)
                insertStream = new StringReader(insert);
            format = request.getParameter("format");
            workspace = request.getParameter("workspace");
            rawuri = request.getParameter("uri");
            rawtoken = request.getParameter("token");
        }

        // argument sanity checks:
        // - uri is required, must be valid.
        // - action keyword is required.
        // - workspace required (must be valid URI) IFF creating new resource.
        // - format - global format for insert and delete graphs
        // - if neither insert nor delete supplied, it's a bad request.
        // - token required for update
        // - deletes not allowed on create

        // parse action
        if (rawaction == null)
            throw new BadRequestException("Missing required argument 'action'");
        Action action = null;
        try {
            action = Action.valueOf(rawaction);
        } catch (IllegalArgumentException e) {
            throw new BadRequestException("Illegal value for 'action', must be one of: "+Arrays.deepToString(Action.values()));
        }

        // derive and sanity-check URI
        if (rawuri == null) {
            String pi = request.getPathInfo();
            if (pi == null || pi.length() == 0)
                throw new BadRequestException("Missing required instance URI to view.");
            rawuri = DataRepository.getInstance().getDefaultNamespace() + pi.substring(1);
        }
        if (!Utils.isValidURI(rawuri))
            throw new BadRequestException("Instance URI must be a legal absolute URI: "+rawuri);
        log.debug("Got request uri="+rawuri);

        // Sanity-check workspace UNLESS action == getToken
        // - workspace is required if action == create
        // - if workspace is specified, make sure it is a valid URI
        if (action != Action.gettoken) {
            if (workspace == null) {
                if (action == Action.create)
                    throw new BadRequestException("The 'workspace' argument is required when creating a new instance.");
            } else if (!Utils.isValidURI(workspace))
                throw new BadRequestException("The 'workspace' must be a legal absolute URI: "+workspace);
        }

        // more sanity and security checking: to be reasonably sure
        // the URI refers to a resource instance, make sure there is
        // a rdf:type statement about it in a graph of type Public or WOrkspace
        try {
            boolean hasCreator = false;
            RepositoryConnection rc = WithRepositoryConnection.get(request);
            ValueFactory vf = rc.getValueFactory();
            URI uri = null;
            try {
                uri = vf.createURI(rawuri);
            } catch (IllegalArgumentException e) {
                throw new BadRequestException("Resource URI is malformed: "+rawuri);
            }

            // get User view mostly for access control:
            DatasetImpl ds = new DatasetImpl();
            View.addGraphs(request, ds, View.USER);

            //  Use prettyPrint because ds.toString() breaks when a default graph is null
            //  which is possible for the 'null' view.  don't ask, it's ugly.
            log.debug("Dataset for SPARQL query = "+Utils.prettyPrint(ds));

            URI wsURI = (workspace == null) ? null : vf.createURI(workspace);

            // when creating, check for resource in ANY context:
            if (action == Action.create) {
                if (rc.hasStatement(uri, RDF.TYPE, null, false))
                    throw new BadRequestException("This resource already exists so 'create' is not allowed: "+uri);

            // Choose home graph and test that it's a workspace:
            // if workspace arg was specified, it must agree with home graph
            // ..otherwise, use home graph *as* workspace.
            // This
            } else {
                URI homeGraph = getHomeGraph(rc, uri, ds);
                if (homeGraph == null)
                    throw new NotFoundException("Resource not found in this repository: "+uri.toString());
                if (wsURI == null)
                    wsURI = homeGraph;
                else if (!wsURI.equals(homeGraph))
                    throw new BadRequestException("Resource not found in specified workspace, its home graph is: "+homeGraph.toString());
            }
            log.debug("Final workspace URI = "+wsURI.toString());

            // Shortcut when action == gettoken:
            //  - find or create an edit token
            //  - format the results: columns are:
            //    token, created, creator, new
            if (action == Action.gettoken) {
                boolean created = false;
                EditToken et = EditToken.find(request, uri);
                if (et == null) {
                    created = true;
                    et = EditToken.create(request, uri);
                    // need to commit the statements added.
                    WithRepositoryConnection.get(request).commit();
                }

                ArrayList<BindingSet> results = new ArrayList<BindingSet>(1);
                MapBindingSet bs = new MapBindingSet(5);
                bs.addBinding("token", et.getURI());
                bs.addBinding("created", new LiteralImpl(et.getCreated(), XMLSchema.DATETIME));
                bs.addBinding("creator", et.getCreator());
                bs.addBinding("creatorLabel", new LiteralImpl(et.getCreatorLabel()));
                bs.addBinding("new", new BooleanLiteralImpl(created));
                results.add(bs);
                SPARQL.sendTupleQueryResults(request, response, format, results);
                return;
            }

            // another workspace sanity check:
            // - workspace must be a named graph of type published or workspace
            NamedGraph wng = NamedGraph.find(request, (URI)wsURI);
            if (wng == null)
                throw new BadRequestException("Invalid workspace, this is not a named graph: "+wsURI);
            NamedGraph.Type wngt = wng.getType();
            if (!(wngt == NamedGraph.Type.published || wngt == NamedGraph.Type.workspace))
                throw new BadRequestException("Resource is not in an appropriate workspace, ws = "+wsURI);

            // sanity check: no delete arg when action is create:
            if (action == Action.create && deleteStream != null)
                throw new BadRequestException("Delete is not allowed when creating a new resource.");

            // Token check: when updating, must present the current token
            if (action == Action.update) {
                if (rawtoken == null)
                    throw new BadRequestException("Missing the argument 'token', which is required when updating.");
                URI token = null;
                try {
                    token = vf.createURI(rawtoken);
                } catch (IllegalArgumentException e) {
                    throw new BadRequestException("URI for token is malformed: "+rawtoken);
                }

                EditToken et = EditToken.find(request, uri);
                if (et == null) {
                    // need to log aborted update in case of user complaint.
                    log.info("Update was not accepted because token was already gone: "+uri);
                    throw new ConflictException("The resource instance you were updating has already been changed, so this update cannot be accepted since it would undo the previous update. If you still wish to make changes to this resource, you must open it for editing again: "+uri);
                } else if (!et.getURI().equals(token)) {
                    log.info("Update was not accepted because token did not match: "+uri+", current token="+et);
                    throw new ConflictException(
                        "The resource instance you were updating has already been changed, AND "+
                        "ANOTHER UPDATE IS IN PROGRESS.  This update cannot be accepted since it "+
                        "would undo the previous update.  If you still wish to make changes to "+
                        "this resource, you must open it for editing again, but be warned that an "+
                        "update session was started at "+et.getCreated()+" by the user "+et.getCreatorLabel());
                }

                // remove the current token as part of this update.
                et.clear(request);
            }

            // access check first: if creating, check workspace,
            // otherwise check instance
            if (action == Action.create) {
                if (insertStream != null && !Access.hasPermission(request, wsURI, Access.ADD))
                    throw new ForbiddenException("User is not permitted to create new resource instance in named graph: "+wsURI);
            } else {
                if (insertStream != null && !Access.hasPermission(request, uri, Access.ADD))
                    throw new ForbiddenException("User is not permitted to insert into resource instance: "+uri);
                if (deleteStream != null && !Access.hasPermission(request, uri, Access.REMOVE))
                    throw new ForbiddenException("User is not permitted to delete from resource instance: "+uri);
            }

            // parse removes and inserts
            Graph removes = null;
            Graph inserts = null;
            if  (deleteStream != null) {
                removeHandler rh = new removeHandler(rc, uri, wsURI);
                removes = parseGraph(deleteStream, "delete", (format == null) ? deleteMIMEType: format,
                           rh, vf);
            }
            if  (insertStream != null) {
                insertHandler ih = new insertHandler(rc, action, uri, wsURI);
                inserts = parseGraph(insertStream, "insert", (format == null) ? insertMIMEType: format,
                           ih, vf);
                hasCreator = (ih.hasDCTERMSCreator);
                ih.addProvenance();
            }

            // optimize out any removed statements which are later restored.
            if (removes != null && inserts != null) {
                Iterator<Statement> ri = removes.iterator();
                while (ri.hasNext()) {
                    Statement rs = ri.next();
                    if (inserts.contains(rs)) {
                        ri.remove();
                        log.debug("Optimized out statement remove+insert: "+rs);
                        if (!inserts.remove(rs))
                            log.debug("Failed to remove statement from inserts too: "+rs);
                    }
                }
            }

            // execute removes first since they may include wildcards:
            if (removes != null)
                rc.remove(removes, wsURI);
            if (inserts != null)
                rc.add(inserts, wsURI);
            log.info(action+": Deleted "+
                     String.valueOf(removes == null ? 0 : removes.size())+
                     " statements and added "+
                     String.valueOf(inserts == null ? 0 : inserts.size())+
                     " statements for "+uri+" in workspace "+wsURI.toString());

            // 1. create final sanity checks:  must include insert
            // 2. also start workflow
            if (action == Action.create)
            {
                if (inserts == null)
                    throw new BadRequestException("Request to create an instance must include the insert graph.");
                FakeFlow ff = FakeFlow.create(request, uri);
                log.debug("Started fake workflow on resource inst="+uri+", WF state="+ff.getState());

            // update final sanity check:
            // if ANY properties are left, must include at least one rdf:type
            } else if (action == Action.update) {
                if (rc.hasStatement(uri, null, null, false, wsURI)) {
                    if (!rc.hasStatement(uri, RDF.TYPE, null, false, wsURI))
                        throw new BadRequestException("This update request would leave the resource instance without any rdf:type statements, which is not allowed unless ALL statements are removed.");
                } else {
                    log.info("Deleted resource: "+uri);
                }
            }

            if (inserts != null || removes != null) {
                Date now = new Date();
                Provenance p = new Provenance(uri);
                if (action == Action.create) {
                    if (hasCreator) {
                        p.setMediated(request, now);
                    } else
                        p.setCreated(request, now);
                } else
                    p.setModified(request, now);
                rc.commit();
            } else
                throw new BadRequestException("Request must include at least one of the insert or delete graphs.");

        // this "should never happen", means the SAIL or repo does not support generic prepareQuery()
        } catch (UnsupportedOperationException e) {
            log.error(e);
            throw new ServletException(e);
        } catch (RepositoryException e) {
            log.error(e);
            throw new ServletException(e);
        } catch (OpenRDFException e) {
            log.error(e);
            throw new ServletException(e);
        }
    }

    // feed graph stream to handler
    private Graph parseGraph(Reader graphStream, String mode, String rawFormat, graphHandler gh, ValueFactory vf)
        throws ServletException, IOException
    {
        if (rawFormat == null)
            throw new BadRequestException("Content-type (MIME Type) of the insert argument was not specified.");
        RDFFormat rf = RDFFormat.forMIMEType(rawFormat);
        if (rf == null) {
            throw new HttpStatusException(HttpServletResponse.SC_UNSUPPORTED_MEDIA_TYPE, "MIME type of "+mode+" argument is not supported: \""+rawFormat+"\"");
        }
        RDFParser parser = Rio.createParser(rf, vf);
        parser.setRDFHandler(gh);
        try {
            parser.parse(graphStream, "");
        } catch (RDFParseException e) {
            throw new BadRequestException("Error parsing "+mode+" statements: "+e.toString());
        } catch (RDFHandlerException e) {
            throw new ServletException(e);
        }
        return gh.getGraph();
    }

    /**
     * Heuristic to check whether URI is subject of a legitimate
     * resource: does it have a rdf:type statement in a named
     * graph that is expected to contain only resources?
     * Unfortunately SPARQL can't act on context URIs so we
     * have to do it the hard way:
     */
    private URI getHomeGraph(RepositoryConnection rc, URI instance, Dataset ds)
        throws RepositoryException
    {
        URI result = null;
        RepositoryResult<Statement> rr = null;
        try {
            rr = rc.getStatements(instance, RDF.TYPE, null, false);
            while (rr.hasNext()) {
                Statement s = rr.next();
                Resource ctx = s.getContext();
                log.debug("Found statement: "+instance+" rdf:type "+s.getObject()+", in graph "+ctx);

                // shortcut: test for read access by checking if context
                //  URI is listed in Dataset, instead of slower call
                //  to  Access.hasPermission(request, (URI)ctx, REPO.NAMED_GRAPH, Access.READ)
                if (ctx != null && ctx instanceof URI &&
                      (ds.getDefaultGraphs().contains(ctx) ||
                       ds.getNamedGraphs().contains(ctx))) {
                    if (result == null)
                        result = (URI)ctx;
                    else if (!result.equals(ctx))
                        log.warn("Found rdf:type statements for instance=<"+instance+"> in two different graphs: "+ctx+", "+result);
                }
            }
        } finally {
            rr.close();
        }
        return result;
    }

    // inner superclass of graph handlers
    private abstract class graphHandler extends RDFHandlerBase
    {
        protected Graph batch = new GraphImpl();

        private Graph getGraph()
        {
            return batch;
        }
    }

    // check and insert statements from submitted graph
    private class insertHandler extends graphHandler
    {
        private RepositoryConnection rc;
        private Action action;
        private URI uri;
        private URI wsURI;
        private boolean hasRDFType = false;
        private boolean hasDCTERMSCreator = false;
        private List<Statement> prov = new ArrayList<Statement>();

        protected insertHandler(RepositoryConnection rc, Action action, URI uri, URI wsURI)
        {
            insertHandler.this.rc = rc;
            insertHandler.this.action = action;
            insertHandler.this.uri = uri;
            insertHandler.this.wsURI = wsURI;
        }

        /**
         * Sanity-check that subject is eagle-i resource URI;
         * accumulate statements on a list while also checking:
         * 1. whether predicate is rdf:type (at least one is req'd for create,
         *    or else later the resource will not appear to exist.)
         * 2. whether predicate is dcterms:creator (affects provenance)
         * Throws error upon sanity check failure.
         */
        public void handleStatement(Statement s)
            throws RDFHandlerException
        {
            if (uri.equals(s.getSubject())) {
                URI p = s.getPredicate();
                if (RDF.TYPE.equals(p))
                    hasRDFType = true;
                else if (DCTERMS.CREATOR.equals(p))
                    hasDCTERMSCreator = true;
                else if (FakeFlow.isWorkflowPredicate(p)) {
                    log.info("Skipping insert of workflow property: "+p);
                    return;
                }

                // direct provenance statements to the provenance graph
                if (Provenance.isProvenancePredicate(p))
                    prov.add(s);
                else
                    batch.add(s);

            // XXX should we relax this to allow blank node subj as well??
            } else
                throw new BadRequestException("All inserted statements must belong to the resource, this subject does not: "+s.getSubject().toString());
        }

        /**
         * Sanity check, if creating do we have a type?
         * Then add the accumulated list of statements to the repo.
         */
        public void endRDF()
            throws RDFHandlerException
        {
            if (action == Action.create && !hasRDFType)
                throw new BadRequestException("Inserted statements for 'create' must include at least one rdf:type statement.");
        }

        // add the provenance statements to prov. graph
        private void addProvenance()
            throws RepositoryException
        {
            if (!prov.isEmpty())
                rc.add(prov, Provenance.PROVENANCE_GRAPH);
        }
    }

    // check and remove statements from submitted graph;
    // also implement wildcards.
    private class removeHandler extends graphHandler
    {
        private RepositoryConnection rc;
        private URI uri;
        private URI wsURI;

        protected removeHandler(RepositoryConnection rc, URI uri, URI wsURI)
        {
            removeHandler.this.rc = rc;
            removeHandler.this.uri = uri;
            removeHandler.this.wsURI = wsURI;
        }

        /**
         * Sanity-check that subject is eagle-i resource URI;
         * then accumulate statements to remove on a list.
         * ALSO: expand wildcard URIs in predicate and/or object.
         * Throws error upon sanity check failure.
         */
        public void handleStatement(Statement s)
            throws RDFHandlerException
        {
            if (uri.equals(s.getSubject())) {
                URI p = s.getPredicate();
                Value o = s.getObject();

                // if either predicate or object is wildcard, expand list:
                if (REPO.MATCH_ANYTHING.equals(p) ||
                      REPO.MATCH_ANYTHING.equals(o)) {
                    try {
                        RepositoryResult<Statement> rr = null;
                        try {
                            rr = rc.getStatements(s.getSubject(),
                                          REPO.MATCH_ANYTHING.equals(p) ? null : p,
                                          REPO.MATCH_ANYTHING.equals(o) ? null : o,
                                          false, wsURI);
                            while (rr.hasNext()) {
                                Statement rs = rr.next();
                                if (batch.add(rs))
                                    log.debug("removeHandler (wildcard) really removing: "+rs);
                            }
                        } finally {
                            rr.close();
                        }
                    } catch (RepositoryException e) {
                        log.error(e);
                        throw new RDFHandlerException(e);
                    }
                }
                else if (FakeFlow.isWorkflowPredicate(p))
                    log.info("Skipping delete of workflow property: "+p);
                else
                    if (batch.add(s))
                        log.debug("removeHandler really added: "+s);
            } else
                throw new BadRequestException("All statements to be removed must belong to the resource, this subject does not: "+s.getSubject().toString());
        }
    }
}
