package org.eaglei.repository;

import java.io.IOException;
import java.io.OutputStreamWriter;
import java.util.ArrayList;
import java.util.Iterator;
import java.util.Set;
import java.util.List;
import java.security.Principal;

import javax.servlet.ServletException;
import javax.servlet.http.HttpServletRequest;
import javax.servlet.http.HttpServletResponse;
import javax.servlet.http.HttpSession;

import org.apache.log4j.Logger;
import org.apache.log4j.LogManager;

import org.openrdf.rio.RDFFormat;
import org.openrdf.rio.Rio;
import org.openrdf.rio.RDFHandler;
import org.openrdf.OpenRDFException;
import org.openrdf.model.impl.ContextStatementImpl;
import org.openrdf.model.URI;
import org.openrdf.model.impl.URIImpl;
import org.openrdf.model.Literal;
import org.openrdf.model.Resource;
import org.openrdf.model.Statement;
import org.openrdf.model.Value;
import org.openrdf.repository.RepositoryConnection;
import org.openrdf.model.vocabulary.RDF;
import org.openrdf.model.vocabulary.RDFS;
import org.openrdf.query.Binding;
import org.openrdf.query.BindingSet;
import org.openrdf.query.MalformedQueryException;
import org.openrdf.query.TupleQuery;
import org.openrdf.query.TupleQueryResult;
import org.openrdf.query.TupleQueryResultHandler;
import org.openrdf.query.TupleQueryResultHandlerBase;
import org.openrdf.query.TupleQueryResultHandlerException;
import org.openrdf.query.BooleanQuery;
import org.openrdf.query.QueryLanguage;
import org.openrdf.query.Dataset;
import org.openrdf.query.impl.DatasetImpl;

import org.eaglei.repository.vocabulary.REPO;
import org.eaglei.repository.util.SPARQL;
import org.eaglei.repository.status.InternalServerErrorException;
import org.eaglei.repository.status.ForbiddenException;
import org.eaglei.repository.status.BadRequestException;
import org.eaglei.repository.servlet.WithRepositoryConnection;
import org.eaglei.repository.servlet.ImportExport.DuplicateArg;

/**
 * Access control for the repository.
 *
 * The Access class is two things:
 *  1. Enumerated type describing the type of operation the user is allowed to do.
 *     These correspond to REPO constants (and maybe should move there?)
 *  2. A collection of static utility methods to:
 *     a. Answer access-control questions
 *     b. Manage the records of access grants, including import/export
 *
 * Access permission is computed as follows:
 *  1. Does current user have the Superuser role?  If so, always "yes".
 *  2. Is there a direct grant, e.g. <resource> :has___Access <user> ?
 *  3. Indirect role grant?   e.g. <resource> :has___Access <role>, and
 *     user asserts that role (i.e. <user> :hasRole <role> )
 *  NOTES:
 *   - Roles are NOT hierarchical, each role is independent.
 *   - ALL users have :Role_Authenticated and :Role_Anonymous asserted
 *     invisibly (materialized but managed automatically)
 *   - A session without a logged-in user is identified as :Role_Anonymous
 *
 * Started April, 2010
 *
 * @author Larry Stone
 * @version $Id: $
 */
public enum Access
{
    /** Types of access to be granted */
    READ   (REPO.HAS_READ_ACCESS),
    ADD    (REPO.HAS_ADD_ACCESS),
    REMOVE (REPO.HAS_REMOVE_ACCESS),
    ADMIN  (REPO.HAS_ADMIN_ACCESS);

    // the related property URI for this access type
    private URI uri = null;

    /**
     * Superuser role name in Java servlet container,
     *  i.e. name of role that gets set by container authentication system
     */
    public static final String SUPERUSER_ROLE_NAME = "superuser";

    // Session Attr to cache User object for authenticated principal
    private static final String S_USER = "org.eaglei.repository.Access.User";

    // Session Attr to save original client IP for later checks
    private static final String S_REMOTE_IP = "org.eaglei.repository.Access.REMOTE_IP";

    private static String importGrantQuery = null;

    /**
     * SPARQL query to find grants on any subject.  Need to bind ?instance
     * to the subject URI.  Output columns are:
     *  ?access - type of access being granted
     *  ?agent - grantee
     *  ?agentType - either :Role or :Person
     *  (other result columsn are optional labels for these 3 URIs)
     */
    private static final String getGrantsQueryPrefix =
        "SELECT DISTINCT * WHERE { \n"+
        "GRAPH <"+REPO.NG_INTERNAL+"> { ?instance ?access ?agent } . \n"+
        "?access <"+RDFS.SUBPROPERTYOF+"> <"+REPO.HAS_ANY_ACCESS+"> \n"+
        "OPTIONAL { ?access <"+RDFS.LABEL+"> ?accessLabel }\n"+
        "OPTIONAL { ?agent <"+RDFS.LABEL+"> ?agentLabel }\n"+
    // XXX FIXME TODO: probably don't need the UNION ...subclassOf part of
    //      this query, but try deleting it when we have some tests.
        "OPTIONAL { {{?agent <"+RDF.TYPE+"> ?agentType} UNION {?agent <"+RDFS.SUBCLASSOF+"> ?agentType}}\n"+
        "  FILTER (?agentType = <"+REPO.ROLE+"> || ?agentType = <"+REPO.PERSON+">) \n"+
        "  OPTIONAL { ?agentType <"+RDFS.LABEL+"> ?agentTypeLabel }}\n";

    private static final String getGrantsQuery =
        getGrantsQueryPrefix + "}";

    // This query matches grants to a specific user or its roles.
    // Also need to bind ?user when using this
    private static final String getMyGrantsQuery =
        getGrantsQueryPrefix +
        "{{?user <"+REPO.HAS_ROLE+"> ?agent} UNION {?instance ?access ?user}}}";
              
    // SPARQL query to get subjects out of import graph
    private static final String getImportGrantsQuery =
        "SELECT DISTINCT ?subject WHERE { ?subject ?access ?agent }";

    // query string used in hasPermission()
    // Expect to have bindings for: ?user, ?access, and ?resource
    private static final String hasPermissionQuery = makeAccessQuery("resource", "ASK", null);

    // dataset of internal graphs plus NG_Users to get user's rdf:type
    private static DatasetImpl internalAccessGraphs = SPARQL.copyDataset(SPARQL.InternalGraphs);
    static {
        internalAccessGraphs.addDefaultGraph(REPO.NG_USERS);
    }

    private static Logger log = LogManager.getLogger(Access.class);

    // constructor for enum with uri
    private Access(URI uri)
    {
        this.uri = uri;
    }

    /**
     * Predicate testing whether a URI is a valid access grant property.
     *
     * @param uri the uri to test
     * @return true if uri is the URI value of an access grant keyword.
     */
    public static boolean isAccessPredicate(URI uri)
    {
        for (Access a : values()) {
            if (a.uri.equals(uri))
                return true;
        }
        return false;
    }

    /**
     * Get the URI referenced by this access type.
     *
     * @return a {@link org.openrdf.model.URI} object.
     */
    public URI getURI()
    {
        return uri;
    }

    public String toString()
    {
        return name().toUpperCase();
    }

    /**
     * <p>hasPermission - predicate, general permission test.</p>
     * Does current user have the indicated
     * permission on this resource?  See the general formula and rules
     * for computing access in comments at the head of this class.
     *
     * @param request a {@link javax.servlet.http.HttpServletRequest} object.
     * @param subject the object being tested for access
     * @param pred the type of access
     * @return a boolean, true if access was granted.
     */
    public static boolean hasPermission(HttpServletRequest request, Resource subject, Access pred)
    {
        if (isSuperuser(request)) {
            if (log.isDebugEnabled())
                log.debug("Superuser elides check: hasPermission("+subject+", "+pred+") => true");
            return true;
        }

        try {
            URI pu = getPrincipalURI(request);
            RepositoryConnection rc = WithRepositoryConnection.get(request);

            // XXX FIXME could optimize later as "prepared" BooleanQuery,
            // XXX maybe store in app context; is is bound to the RepositoryConnection
            // XXX  Parameterized prepared query - should be able to
            // XXX re-use prepared query object so long as Repository doesn't change.

            // Build SPARQL ASK query to get permission;
            // This is like an SQL prepared statement, setBinding plugs
            // values into the variables.  Variables are:
            //  ?user - agent seeking access (user or role)
            //  ?access - uri of access type/predicate
            //  ?resource - uri of object/subject
            BooleanQuery q = rc.prepareBooleanQuery(QueryLanguage.SPARQL, hasPermissionQuery);
            q.setIncludeInferred(true);
            q.setDataset(internalAccessGraphs);
            q.clearBindings(); // needed if re-using query
            q.setBinding("user", pu);
            q.setBinding("access", pred.uri);
            q.setBinding("resource", subject);
            boolean result = q.evaluate();
            if (log.isDebugEnabled()) {
                log.debug("Access Query: user="+pu+", access="+pred+", query=\n"+hasPermissionQuery);
                log.debug("hasPermission("+subject+", "+pred+", "+pu+") => "+result);
            }
            return result;
        } catch (OpenRDFException e) {
            log.error(e);
            throw new InternalServerErrorException("Failed in access check: ",e);
        }
    }

    /**
     * Special case access predicate on User objects.
     * Does CURRENT AUTHENTICATED user have permission TO MODIFY the
     * User object associated with this username?
     * True if it matches the current logged-in user, or we are superuser.
     *
     * @param request a {@link javax.servlet.http.HttpServletRequest} object.
     * @param username principal (i.e. RDBMS username, value of :hasPrincipal), a {@link java.lang.String} object.
     * @return a boolean, true if permission is gratned.
     */
    public static boolean hasPermissionOnUser(HttpServletRequest request, String username)
    {
        Principal p = request.getUserPrincipal();
        return isSuperuser(request) ||
               (username != null && p != null && username.equals(p.getName()));
    }

    /**
     * Filters results of query by what the current user has indicated permission
     * on.. Resource (URI) expected to be in  variable named "?{name}"  so
     * this SPARQL pattern group fragment (in "{ }") can be combined with the
     * rest of the query.
     *
     * See hasPermission() for algorithm to figure permission.  The only
     * difference is that this does NOT work for superuser.  (It could
     * be added if there is a need.)
     *
     * Results are returned by calling tuple query handler.
     *
     *   ***** WARNING *****
     * DO NOT call this if you are superuser! It will not work.
     * Test for superuser before calling this filter and use alternate query.
     *
     * @param request a {@link javax.servlet.http.HttpServletRequest} object.
     * @param principal URI of user or role being checked for permission
     * @param name bare name of the variable in query containing URI to test for
     *        access. NOTE: 'name' MUST be a variable in 'results' list
     * @param results query results clause, i.e. SELECT <results> WHERE ...
     * @param patternGroup query fragment
     * @param pred type of access being tested, a {@link org.eaglei.repository.Access} object.
     * @param dataset the dataset on whcih to operate, MUST not be null
     * @param handler a {@link org.openrdf.query.TupleQueryResultHandler} object.
     */
    public static void filterByPermission(HttpServletRequest request,
                           URI aprincipal, String name, String results,
                           String patternGroup, Access pred, Dataset dataset,
                           BindingSet bindings, TupleQueryResultHandler handler)
    {
        try {
            RepositoryConnection rc = WithRepositoryConnection.get(request);
            URI principal = aprincipal == null ? getPrincipalURI(request) : aprincipal;

            // Parameterized prepared query - should be able to
            // re-use prepared query object so long as Repository doesn't change.
            // replacement because there's no equiv. of a Prepared Statement..
            // This is like an SQL prepared statement, setBinding plugs
            // values into the variables and seems to work..

            // XXX maybe optimize later by storing "prepared" BooleanQuery,
            //  maybe in app context. it depends on the sesame repo obj.
            String qs = makeAccessQuery(name, "SELECT "+results+" WHERE", patternGroup);
            if (log.isDebugEnabled())
                log.debug("SPARQL query in filterByPermission, name="+name+", query=\n  "+qs);
            TupleQuery q = rc.prepareTupleQuery(QueryLanguage.SPARQL, qs);
            q.setIncludeInferred(true);
            q.setDataset(dataset);
            q.clearBindings(); // needed if re-using query
            q.setBinding("user", principal);
            q.setBinding("access", pred.uri);
            if (bindings != null && bindings.size() > 0) {
                for (Iterator<Binding> bi = bindings.iterator(); bi.hasNext();) {
                    Binding b = bi.next();
                    q.setBinding(b.getName(), b.getValue());
                }
            }
            q.evaluate(handler);

        } catch (OpenRDFException e) {
            log.error(e);
            throw new InternalServerErrorException("Failed in access check: ",e);
        }
    }

    /**
     * Find (and create if necessary) the URI of the :Person object
     * for the current authenticated user, if any.
     * Returns the anonymous user when not authenticated.
     *
     * @param request a {@link javax.servlet.http.HttpServletRequest} object.
     * @return the :Person object of current authenticated user, as a {@link org.openrdf.model.URI} object.
     */
    public static URI getPrincipalURI(HttpServletRequest request)
    {
        User u = getPrincipalUser(request);
        if (u == null) {
            log.debug("getPrincipalURI: Returning :Anonymous for unauthenticated user.");
            return request.isUserInRole(SUPERUSER_ROLE_NAME) ?
                   REPO.ROLE_SUPERUSER : REPO.ROLE_ANONYMOUS;
        }
        return u.getURI();
    }

    /**
     * Invalidate the cached authenticated User if it matches the one
     * that was modified by the User API.
     *
     * @param request a {@link javax.servlet.http.HttpServletRequest} object.
     * @param u user to decache, a {@link org.eaglei.repository.User} object.
     */
    public static void decacheAuthentication(HttpServletRequest request, User u)
    {
        HttpSession session = getValidatedSession(request);
        User cu = (User)session.getAttribute(S_USER);
        if (cu != null && cu.equals(u)) {
            session.removeAttribute(S_USER);
            log.debug("Decached authenticated user from Session.");
            logout(request);
        }
    }

    /**
     * Get the current authenticated username from container or other auth'n.
     * @return the username or null if not auth'nd
     */
    public static String getAuthenticatedUsername(HttpServletRequest request)
    {
        Principal p = request.getUserPrincipal();
        return p == null ? null :  p.getName();
    }

    /**
     * Find a User object for the current
     * authenticated user, if one is available.  Note that there MIGHT NOT
     * be a User object if the authencated user has no RDF metadata; in
     * this case it returns null.  It also returns null when there is no
     * authenticated user so this is not a good test for auth'n.
     *
     * @param request a {@link javax.servlet.http.HttpServletRequest} object.
     * @return a User object or null if not auth'n or no there is no RDF metadata for auth'n user
     */
    public static User getPrincipalUser(HttpServletRequest request)
    {
        // 0. Check for cache in the Request - this overrides the Session
        //    cache for both speed and consistency (when Session is decached
        //    because we modified the current user)
        User result = (User)request.getAttribute(S_USER);
        if (result != null) {
            if (log.isDebugEnabled())
                log.debug("Returning Request-cached user = "+result);
            return result;
        }

        // 1. are we authenticated?
        Principal p = request.getUserPrincipal();
        if (p == null)
            return null;
        String pname = p.getName();

        // 2. XXX KLUDGE: if an "anonymous" user is configured, short-circuit
        //  here if it matches.
        String anonUser = DataRepository.getInstance().getConfigurationProperty("eaglei.repository.anonymous.user");
        if (anonUser != null && anonUser.equals(pname)) {
            log.info("Anonymous login because of configured anonymous user: "+pname);
            return null;
        }

        // 3. check for cached answer first:
        //    make sure session isn't invalidated by (a) Remote IP addr check,
        //    and (b) generation count staleness
        // XXX FIXME: consider setting session timeout interval.. can do it in web.xml
        HttpSession session = getValidatedSession(request);
        if (DataRepository.getInstance().isSessionStale(session)) {
            session.removeAttribute(S_USER);
            log.debug("Invalidating cached user URI in stale session, principal="+pname);
        } else {
            result = (User)session.getAttribute(S_USER);
            if (result != null) {
                if (log.isDebugEnabled())
                    log.debug("Returning cached user for principal="+pname+" from session context: "+result);
                request.setAttribute(S_USER, result);
                return result;
            }
        }

        try {
            // 4. Look for a :Person for this Principal
            User u = User.findByUsername(request, pname);
            if (u != null) {

                // check for Superuser granted by container auth'n which is
                // NOT in the RDF metadata, and override:
                // (Note we CANNOT correct the RDF since we cannot write to Sesame here).
                if (request.isUserInRole(SUPERUSER_ROLE_NAME) && !u.isSuperuser()) {
                    u.setIsSuperuser(true);
                }
                session.setAttribute(S_USER, u);
                request.setAttribute(S_USER, u);
            }
            return u;
        } catch (ServletException e) {
            throw new InternalServerErrorException(e.getMessage(), e);
        }
    }

    /**
     * <p>isSuperuser</p>
     * Predicate, true if current auth'd user has superuser role either in RDF
     * assertion or in the container's authz.
     *
     * @param request a {@link javax.servlet.http.HttpServletRequest} object.
     * @return a boolean, true if current authenticated user has Superuser (Admin) privilege.
     */
    public static boolean isSuperuser(HttpServletRequest request)
    {
        User u = getPrincipalUser(request);
        if (u == null)
            return request.isUserInRole(SUPERUSER_ROLE_NAME);
        else
            return u.isSuperuser();
    }

    /**
     * Utility to get a session and check that it came from the client
     * IP that originally created it, to prevent session hijacking
     * through cookie stealing (note..  this probably wouldn't help much
     * for attacker and victim behind the same proxy or NAT, but there's
     * only so much you can do.)
     *
     */
    private static HttpSession getValidatedSession(HttpServletRequest request)
    {
        HttpSession result = request.getSession(false);
        String remoteIP = request.getRemoteAddr();

        // need to create a new session, save a record of the creator's IP
        if (result == null) {
            result = request.getSession(true);
            result.setAttribute(S_REMOTE_IP, remoteIP);

        // confirm that accessor has same IP as creator.
        } else {
            String testIP = (String)result.getAttribute(S_REMOTE_IP);
            if (testIP == null) {
                log.debug("Initializing session's record of remote address, IP="+remoteIP);
                result.setAttribute(S_REMOTE_IP, remoteIP);
            } else if (!testIP.equals(remoteIP)) {
                // XXX FIXME: maybe call logout to wipe session now?
                log.error("POSSIBLE SESSION HIJACKING: session created by IP="+testIP+", but is being accessed by IP="+remoteIP);
                throw new BadRequestException("Authentication denied, this session may only be accessed from the address that created it. Please login again.");
            }
        }
        return result;
    }

    /**
     * Destroy current session and credentials (if possible)
     * most web browsers cache the HTTP Basic creds so user needs to
     * trash those explicitly right after running this.
     *
     * @param request a {@link javax.servlet.http.HttpServletRequest} object.
     */
    public static void logout(HttpServletRequest request)
    {
        HttpSession s = request.getSession(false);
        if (s == null)
            log.debug("Logout finds no session to destroy!");
        else {
            s.invalidate();
            log.debug("Logout is destroying session ID="+s.getId());
        }
    }

    // cons up a sparql triple pattern to test access.
    // XXX note the parentheses don't balance
    private static String makeAccessQuery(String resourceName, String prologue, String patternGroup)
    {
        StringBuilder result = new StringBuilder();
        result.append(prologue).append(" { ");
        if (patternGroup != null)
            result.append(patternGroup);

        // user is in a role that has access
        result.append("{ { ?user <").append(REPO.HAS_ROLE)
              .append("> ?r . ?").append(resourceName).append(" ?access ?r }\n");

        // direct grant to user (or Role if login == ANONYMOUS)
        result.append(" UNION { ?").append(resourceName).append(" ?access ?user } } }");
        return result.toString();
    }

    /**
     * Remove specified grant of access from an instance.
     * Returns true if grant was there, false if not.
     * WARNING: You will need to commit() these changes to the repo connection!
     *
     * @param request a {@link javax.servlet.http.HttpServletRequest} object.
     * @param instance subject from which to remove access grant a {@link org.openrdf.model.URI} object.
     * @param agent principal to whom the access was granted, a {@link org.openrdf.model.URI} object.
     * @param access type of access, a {@link org.openrdf.model.URI} object.
     * @return a boolean, true if there was a grant to be removed.
     */
    public static boolean removeGrant(HttpServletRequest request, URI instance, URI agent, URI access)
    {
        if (hasPermission(request, instance, Access.ADMIN)) {
            return removeGrantAsAdministrator(request, instance, agent, access);
        } else
              throw new ForbiddenException("You are not allowed to change access controls on "+instance);
    }

    /**
     * Remove specified grant of access from a URI, but WITHOUT cehcking for
     * ADMIN access.  This is meant for INTERNAL user where the program
     * logic mediates access, e.g. workflow.
     * Returns true if grant was there, false if not.
     * WARNING: You will need to commit() these changes to the repo connection!
     *
     * @param request a {@link javax.servlet.http.HttpServletRequest} object.
     * @param instance subject from which to remove access grant a {@link org.openrdf.model.URI} object.
     * @param agent principal to whom the access was granted, a {@link org.openrdf.model.URI} object.
     * @param access type of access, a {@link org.openrdf.model.URI} object.
     * @return a boolean, true if there was a grant to be removed.
     */
    public static boolean removeGrantAsAdministrator(HttpServletRequest request, URI instance, URI agent, URI access)
    {
        // sanity check: do not allow null, which is wildcard delete:
        if (instance == null || access == null || agent == null)
            throw new BadRequestException("removeGrant called with an illegal null URI.");
        if (isAccessPredicate(access)) {
            RepositoryConnection rc = WithRepositoryConnection.get(request);
            try {
                if (rc.hasStatement(instance, access, agent, false, REPO.NG_INTERNAL)) {
                    rc.remove(instance, access, agent, REPO.NG_INTERNAL);
                    return true;
                } else {
                    return false;
                }
            } catch (OpenRDFException e) {
                log.error(e);
                throw new InternalServerErrorException("Failed in remove ACL: ",e);
            }
        } else {
            throw new IllegalArgumentException("Access URI is not a valid access predicate: "+access.stringValue());
        }
    }

    /**
     * Add the specified grant to the instance.  Requires ADMIN access.
     * WARNING: You will need to commit() these changes to the repo connection!
     *
     * @param request a {@link javax.servlet.http.HttpServletRequest} object.
     * @param instance subject from which to add access grant a {@link org.openrdf.model.URI} object.
     * @param agent principal to whom the access was granted, a {@link org.openrdf.model.URI} object.
     * @param access type of access, a {@link org.openrdf.model.URI} object.
     */
    public static void addGrant(HttpServletRequest request, URI instance, URI agent, URI access)
    {
        if (hasPermission(request, instance, Access.ADMIN)) {
            addGrantAsAdministrator(request, instance, agent, access);
        } else
            throw new ForbiddenException("You are not allowed to change access controls on "+instance);
    }

    /**
     * Add the specified grant to the instance, but WITHOUT cehcking for
     * ADMIN access.  This is meant for INTERNAL user where the program
     * logic mediates access, e.g. workflow.
     * WARNING: You will need to commit() these changes to the repo connection!
     *
     * @param request a {@link javax.servlet.http.HttpServletRequest} object.
     * @param instance subject from which to add access grant a {@link org.openrdf.model.URI} object.
     * @param agent principal to whom the access was granted, a {@link org.openrdf.model.URI} object.
     * @param access type of access, a {@link org.openrdf.model.URI} object.
     */
    public static void addGrantAsAdministrator(HttpServletRequest request, URI instance, URI agent, URI access)
    {
        if (isAccessPredicate(access)) {
            RepositoryConnection rc = WithRepositoryConnection.get(request);
            try {
                if (!rc.hasStatement(instance, access, agent, false, REPO.NG_INTERNAL)) {
                    rc.add(instance, access, agent, REPO.NG_INTERNAL);
                }
            } catch (OpenRDFException e) {
                log.error(e);
                throw new InternalServerErrorException("Failed in add ACL: ",e);
            }
        } else {
            throw new IllegalArgumentException("Access URI is not a valid access predicate: "+access.stringValue());
        }
    }

    public static Iterable<AccessGrant> getGrants(HttpServletRequest request, URI uri)
    {
        return getGrantsInternal(request, WithRepositoryConnection.get(request), uri, false);
    }
    public static Iterable<AccessGrant> getGrants(RepositoryConnection rc, URI uri)
    {
        return getGrantsInternal(null, rc, uri, false);
    }

    public static Iterable<AccessGrant> getMyGrants(HttpServletRequest request, URI uri)
    {
        return getGrantsInternal(request, WithRepositoryConnection.get(request), uri, true);
    }

    /**
     * Get list of access grants on this instance
     *
     * @param request a {@link javax.servlet.http.HttpServletRequest} object.
     * @param rc the repository connection - MIGHT not be from request
     * @param uri subject on which to find grants, a {@link org.openrdf.model.URI} object.
     * @param mine when true, only return grants affecting current user
     * @return all grants in a {@link java.lang.Iterable} object, possibly empty.
     */
    private static Iterable<AccessGrant> getGrantsInternal(HttpServletRequest request,
                                          RepositoryConnection rc,
                                          URI uri, boolean mine)
    {
        try {
            String qs = mine ? getMyGrantsQuery : getGrantsQuery;
            if (log.isDebugEnabled())
                log.debug("SPARQL query to get "+(mine?"MY ("+getPrincipalURI(request).stringValue()+")":"")+"grants, instance="+uri.stringValue()+", query=\n"+qs);
            TupleQuery q = rc.prepareTupleQuery(QueryLanguage.SPARQL, qs);
            q.setBinding("instance", uri);
            if (mine) {
                if (request == null)
                    throw new IllegalArgumentException("Cannot get grants for current user when request is null.");
                q.setBinding("user", getPrincipalURI(request));
            }
            q.setDataset(internalAccessGraphs);
            grantHandler h = new grantHandler();
            q.evaluate(h);
            return h.result;
        } catch (OpenRDFException e) {
            log.error(e);
            throw new InternalServerErrorException("Failed in query: "+e,e);
        }
    }

    /**
     *  Translate internal grant objects into exportable statements on a
     *  given subject.
     *
     * @param uri - the subject
     * @param grants - grant objects
     * @return iterable list of Sesame Statement objects
     */
    public static Iterable<Statement> exportGrants(URI uri, Iterable<AccessGrant> grants)
    {
        List<Statement> result = new ArrayList<Statement>();
        for (AccessGrant g : grants) {
            result.add(new ContextStatementImpl(uri, g.access.uri, g.agent.uri, REPO.NG_INTERNAL));
        }
        return result;
    }

    /**
     * Get importable access grant statements for URI from import document.
     * NOTE that it looks up grants on oldURI, but puts newURI in the
     * new grant statements; this is in case subject URI got transformed
     * on the import.
     * Also note that source grants come from 'content' (could be a memory repo)
     * but we need still the regular repo to compute ACL query the first time.
     */
    public static Iterable<Statement> importGrants(HttpServletRequest request,
                                        RepositoryConnection content, URI oldURI, URI newURI)
    {
        try {
            // compute the grant query based on grant predicates in current onto:
            if (importGrantQuery == null) {
                StringBuilder igq = new StringBuilder("SELECT ?p ?o WHERE {?s ?p ?o FILTER(" /*)}*/);
                RepositoryConnection rc = WithRepositoryConnection.get(request);
                TupleQuery q = rc.prepareTupleQuery(QueryLanguage.SPARQL,
                    "SELECT * WHERE { ?accessPred <"+RDFS.SUBPROPERTYOF+"> <"+REPO.HAS_ANY_ACCESS+">}");
                q.setDataset(internalAccessGraphs);
                TupleQueryResult qr = null;
                try {
                    qr = q.evaluate();
                    boolean first = true;
                    while (qr.hasNext()) {
                        String ap = qr.next().getValue("accessPred").stringValue();
                        if (first)
                            first = false;
                        else
                            igq.append(" || ");
                        igq.append("?p = <").append(ap).append(">");
                    }
                    igq.append(/*{(*/ ")}");
                    importGrantQuery = igq.toString();
                    log.debug("Generating fixed Access Import query: \""+importGrantQuery+"\"");
                } finally {
                    if (qr != null)
                        qr.close();
                }
            }

            TupleQuery q = content.prepareTupleQuery(QueryLanguage.SPARQL, importGrantQuery);
            q.setDataset(internalAccessGraphs);
            q.setBinding("s", oldURI);
            TupleQueryResult qr = null;
            List<Statement> result = new ArrayList<Statement>();
            try {
                qr = q.evaluate();
                while (qr.hasNext()) {
                    BindingSet bs= qr.next();
                    result.add(new ContextStatementImpl(newURI, (URI)bs.getValue("p"),
                           bs.getValue("o"), REPO.NG_INTERNAL));
                }
            } finally {
                if (qr != null)
                    qr.close();
            }
            log.debug("importGrants("+oldURI.stringValue()+"): returning "+result.size()+" grant statements.");
            return result;
        } catch (OpenRDFException e) {
            log.error("Failed in one of the Import Grant queries: ",e);
            throw new InternalServerErrorException(e);
        }
    }

    private static final String allAccessesQuery =
      "SELECT DISTINCT * WHERE { "+
      "?uri <"+RDFS.SUBPROPERTYOF+"> <"+REPO.HAS_ANY_ACCESS+"> \n"+
      " OPTIONAL { ?uri <"+RDFS.LABEL+"> ?label }}";

    /**
     *  Get all access types described in the RDF - note this CAN be extended
     *  by each repository instance by adding statements but it is NOT
     *  likely since the practical implementation of access is hardcoded.
     *
     * @param request a {@link javax.servlet.http.HttpServletRequest} object.
     * @return all access types in a {@link java.lang.Iterable} object.
     */
    public static Iterable<AccessGrant.Term> getAllAccesses(HttpServletRequest request)
    {
        if (log.isDebugEnabled())
            log.debug("getAllAccesses query = "+allAccessesQuery);
        return getAllTermsInternal(request, allAccessesQuery);
    }

    // utility to fetch terms from RDF
    private static Iterable<AccessGrant.Term> getAllTermsInternal(HttpServletRequest request, String query)
    {
        RepositoryConnection rc = WithRepositoryConnection.get(request);
        try {
            TupleQuery q = rc.prepareTupleQuery(QueryLanguage.SPARQL, query);
            q.setDataset(SPARQL.InternalGraphs);
            termHandler h = new termHandler();
            q.evaluate(h);
            return h.result;
        } catch (OpenRDFException e) {
            log.error(e);
            throw new InternalServerErrorException("Failed in query: ",e);
        }

    }

    // Term query result handler: stash tuple query results in List<AccessGrant.Term>
    // Looks for bindings of "uri" and "label"
    private static class termHandler extends TupleQueryResultHandlerBase
    {
        private List<AccessGrant.Term> result = new ArrayList<AccessGrant.Term>();

        // columns: namedGraphURI, namedGraphLabel, typeURI, typeLabel, anon
        public void handleSolution(BindingSet bs)
            throws TupleQueryResultHandlerException
        {
            Value theURI = bs.getValue("uri");
            Value theLabel = bs.getValue("label");
            if (theURI == null || !(theURI instanceof URI))
                throw new TupleQueryResultHandlerException(
                    "The value for 'uri' was null or not a URI type in termHandler: "+theURI);
            else {
                result.add(new AccessGrant.Term((URI)theURI,
                    (theLabel != null && theLabel instanceof Literal) ? ((Literal)theLabel).getLabel() : null));
            }
        }
    }

    // Grant query result handler: stash tuple query results in List<Grant>
    // Looks for bindings of "uri" and "label"
    private static class grantHandler extends TupleQueryResultHandlerBase
    {
        private List<AccessGrant> result = new ArrayList<AccessGrant>();

        protected grantHandler()
        {
            super();
        }

        // columns: namedGraphURI, namedGraphLabel, typeURI, typeLabel, anon
        public void handleSolution(BindingSet bs)
            throws TupleQueryResultHandlerException
        {
            Value agent = bs.getValue("agent");
            Value agentLabel = bs.getValue("agentLabel");
            Value agentType = bs.getValue("agentType");
            Value agentTypeLabel = bs.getValue("agentTypeLabel");
            Value access = bs.getValue("access");
            Value accessLabel = bs.getValue("accessLabel");
            Value ggraph = bs.getValue("graph");
            if (agent == null || !(agent instanceof URI))
                throw new TupleQueryResultHandlerException(
                    "The value for 'agent' was null or not a URI type in grantHandler: "+agent);
            else if (access == null || !(access instanceof URI))
                throw new TupleQueryResultHandlerException(
                    "The value for 'access' was null or not a URI type in grantHandler: "+access);
            else {
                // kludge: if agentType not set, grab repo:Agent
                if (agentType == null)
                    agentType = REPO.AGENT;
                // is this grant from the repo ontology, hence immutable??
                boolean builtin = ggraph != null && REPO.NG_REPO_ONTOLOGY.equals((URI)ggraph);
                if (log.isDebugEnabled())
                    log.debug("getGrants: Adding Grant(agent="+agent+
                        ", agentLabel="+agentLabel+", agentType="+agentType+
                        ", agentTypeLabel="+agentTypeLabel+
                        ", access="+access+", accessLabel="+accessLabel+
                        ", builtin="+builtin);
                result.add(new AccessGrant((URI)agent,
                    (agentLabel != null && agentLabel instanceof Literal) ? ((Literal)agentLabel).getLabel() : null,
                    (URI)agentType,
                    (agentTypeLabel != null && agentTypeLabel instanceof Literal) ? ((Literal)agentTypeLabel).getLabel() : null,
                    (URI)access,
                    (accessLabel != null && accessLabel instanceof Literal) ? ((Literal)accessLabel).getLabel() : null,
                    builtin));
            }
        }
    }

    /**
     * Export description of access grants as serialized RDF quads.
     * Handler for the ImportExport servlet.
     */
    public static void doExportGrants(HttpServletRequest request, HttpServletResponse response,
            RDFFormat format, Set<String> includes, Set<String> excludes)
        throws ServletException, IOException
    {
        try {
            RDFHandler out = Rio.createWriter(format, new OutputStreamWriter(response.getOutputStream(), "UTF-8"));
            out.startRDF();

            if (includes.isEmpty())
                throw new BadRequestException("Export of grants requires an include list of URIs");
            if (!excludes.isEmpty())
                throw new BadRequestException("Export of grants does not support an exclude list");

            for (String iu : includes) {
                URI uri = new URIImpl(iu);
                for (Statement s : exportGrants(uri, getGrants(request, uri))) {
                    out.handleStatement(s);
                }
            }
            out.endRDF();
        } catch (OpenRDFException e) {
            throw new InternalServerErrorException(e);
        }
    }

    /**
     * Import description of access grants from serialized RDF quads.
     * Handler for the ImportExport servlet.
     */
    public static void doImportGrants(HttpServletRequest request, HttpServletResponse response,
            RepositoryConnection content,
            Set<String> includes, Set<String> excludes,
            DuplicateArg duplicate,
            boolean transform, boolean ignoreACL)

        throws ServletException, IOException
    {
        RepositoryConnection rc = WithRepositoryConnection.get(request);
        try {
            log.debug("SPARQL query on imported GRANT statements: \n  "+getImportGrantsQuery);
            TupleQuery q = content.prepareTupleQuery(QueryLanguage.SPARQL, getImportGrantsQuery);
            q.setDataset(SPARQL.InternalGraphs);
            TupleQueryResult qr = null;
            try {
                qr = q.evaluate();
                while (qr.hasNext()) {
                    BindingSet bs= qr.next();
                    URI uri = (URI)bs.getValue("subject");
                    String us = uri.stringValue();
                    if (excludes.contains(us)) {
                        log.debug("SKIP IMPORT GRANT because of exclude: uri="+us);
                        continue;
                    } else if (!(includes.isEmpty() || includes.contains(us))) {
                        log.debug("SKIP IMPORT GRANT because of include: uri="+us);
                        continue;
                    }
                    for (Statement s : importGrants(request, content, uri, uri)) {
                        log.debug("Adding access grant statement: "+s);
                        rc.add(s, REPO.NG_INTERNAL);
                    }
                }
            } finally {
                if (qr != null)
                    qr.close();
            }
        } catch (MalformedQueryException e) {
            log.error("Rejecting malformed query:"+e);
            throw new ServletException(e);
        } catch (OpenRDFException e) {
            log.error(e);
            throw new ServletException(e);
        }
    }
}
