package org.eaglei.repository;

import java.io.File;
import java.io.FileOutputStream;
import java.io.InputStreamReader;
import java.io.Reader;
import java.io.IOException;
import java.io.UnsupportedEncodingException;
import java.net.URL;
import java.net.URLConnection;
import java.net.InetAddress;
import java.nio.charset.IllegalCharsetNameException;
import java.nio.charset.UnsupportedCharsetException;
import java.util.Date;
import java.util.Enumeration;
import java.util.HashMap;
import java.util.Map;
import java.util.Iterator;
import java.util.Properties;
import java.util.regex.Pattern;
import java.util.regex.Matcher;

import javax.servlet.ServletException;
import javax.servlet.ServletContext;
import javax.servlet.http.HttpSession;

import org.apache.log4j.AppenderSkeleton;
import org.apache.log4j.Logger;
import org.apache.log4j.LogManager;
import org.apache.log4j.PatternLayout;
import org.apache.log4j.PropertyConfigurator;
import org.apache.log4j.RollingFileAppender;

import org.apache.commons.configuration.PropertiesConfiguration;

import org.reflections.Reflections;

import org.openrdf.repository.Repository;
import org.openrdf.repository.RepositoryConnection;
import org.openrdf.repository.RepositoryException;
import org.openrdf.repository.sail.SailRepository;
import org.openrdf.sail.nativerdf.NativeStore;
import org.openrdf.model.URI;
import org.openrdf.model.Literal;
import org.openrdf.model.Resource;
import org.openrdf.model.Statement;
import org.openrdf.model.ValueFactory;
import org.openrdf.model.vocabulary.OWL;
import org.openrdf.rio.RDFFormat;
import org.openrdf.rio.RDFParseException;
import org.openrdf.rio.Rio;
import org.openrdf.rio.RDFHandlerException;
import org.openrdf.rio.RDFHandler;
import org.openrdf.rio.helpers.RDFHandlerBase;
import org.openrdf.rio.RDFParser;
import org.openrdf.OpenRDFException;

import org.eaglei.repository.HasContentCache;
import org.eaglei.repository.inferencer.MinimalInferencer;
import org.eaglei.repository.vocabulary.REPO;
import org.eaglei.repository.vocabulary.DCTERMS;
import org.eaglei.repository.util.Utils;

/**
 * This is a singleton class that encompasses the application state.
 * It is created and destroyed by Java Servlet Container webapp
 * lifecycle events.
 *
 * Important configuration variables:
 *  - Sesame repository object (not the connection, but what makes connections)
 *  - Default namespace
 *  - Title (in configuration)
 *
 * @author Larry Stone
 * Started April 2010
 * @version $Id: $
 */
public class DataRepository
{
    private static Logger log = LogManager.getLogger(DataRepository.class);

    // root of repository log4j hierarchy
    private static final String REPO_ROOT = "org.eaglei.repository";

    // root of Sesame's log4j hierarchy
    private static final String OPENRDF_ROOT = "org.openrdf";

    // subversion conventions for branch and tag paths
    private static final String SVN_BRANCHES_PREFIX = "branches/";
    private static final String SVN_TAGS_PREFIX = "tags/";

    /** name of Derby database for user authentication */
    /** XXX this needs to move */
    public  static final String AUTH_DB_NAME = "eagle-i-users.derby";

    /** Key to ServletContext attribute for open Sesame repository object */
    private static final String SCTX_REPOSITORY = "org.eaglei.repository.SesameRepository";

    /** Key to generation counter in current app context */
    private static final String SCTX_GENERATION = "org.eaglei.repository.Generation";

    /** Configuration property keys */
    private static final String CONFIG_NAMESPACE = "eaglei.repository.namespace";
    private static final String CONFIG_LOGDIR = "eaglei.repository.log.dir";
    private static final String CONFIG_SESAMEDIR = "eaglei.repository.sesame.dir";
    private static final String CONFIG_SESAMEINDEXES = "eaglei.repository.sesame.indexes";

    /** title of repository for UI labels */
    public static final String CONFIG_TITLE = "eaglei.repository.title";

    /** stylesheet for transforming HTML dissemination output, if any */
    public static final String CONFIG_INSTANCE_XSLT = "eaglei.repository.instance.xslt";

    /** CSS stylesheet for HTML dissemination output, if any */
    public static final String CONFIG_INSTANCE_CSS = "eaglei.repository.instance.css";

    /** List of JavaScript URLs to be included in HTML dissemination output */
    public static final String CONFIG_INSTANCE_JS = "eaglei.repository.instance.js";

    // name of system property where home directory is specified (mandatory)
    private static final String REPO_HOME = "org.eaglei.repository.home";

    // name of system property where RDBMS backup directory is specified (optional)
    private static final String BACKUP_HOME = "org.eaglei.repository.backup";

    // map of named graph URI to resource filename from which it is
    // initialized if empty
    private static final Map<URI,String> graphInitFile = new HashMap<URI,String>();
    static {
        graphInitFile.put(REPO.NAMESPACE_URI, "repository-ont.n3");
        graphInitFile.put(REPO.NG_INTERNAL, "repository-internal.n3");
        graphInitFile.put(REPO.NG_QUERY, "query-macros.n3");
    }

    /** Graphs that can be reloaded as part of upgrade */
    public static final URI[] UPGRADEABLE_GRAPHS = {
        REPO.NAMESPACE_URI,
        REPO.NG_QUERY
    };

    /** Mark the startup time for display in admin page, etc. */
    public static final Date STARTUP = new Date();

    //---------------------- instance variables:

    // cache of the last servlet context
    private ServletContext servletContext = null;

    // cache of default namespace prefix - from congif or defaulted
    private String defaultNamespace = null;

    /** Configuration properties */
    private PropertiesConfiguration configuration = null;

    // cache of properties describing how this software was built.
    private Properties buildProperties = null;

    // timestamp of most recent last-modified setting, useful
    // for optimizing time-bounded metadata harvesting queries.
    private Date lastModified = new Date();


    // the singleton instance
    private static DataRepository instance = null;

    /**
     * Get the singleton instance.  Will return null unless the initialization has run.
     *
     * @return the singleton, a {@link org.eaglei.repository.DataRepository} object.
     */
    public static DataRepository getInstance()
    {
        return instance;
    }

    // handle error before logging is working
    private static void printError(String msg)
    {
        System.err.println("***ERROR*** (at "+(new Date().toString())+
            ") eagle-i DataRepository: "+msg);
    }

    // handle info message before logging is working
    private static void printInfo(String msg)
    {
        System.err.println("***INFO*** (at "+(new Date().toString())+
            ") eagle-i DataRepository: "+msg);
    }

    /**
     * Web app initialization hook:
     *  1. find and load the configuration properties file.
     *  2. configure log4j
     *  3. set up sesame repository based on configs
     *
     * @param sc the {@link javax.servlet.ServletContext} object.
     */
    public static void initialize(ServletContext sc)
    {
        if (instance == null) {
            try {
                instance = new DataRepository(sc);
                instance.finishInitialize();
            } catch (Exception e) {
                printError("Got exception in Webapp context initialization, DO NOT EXPECT ANYTHING TO WORK NOW:"+e.toString());
                log.fatal("Got exception in Webapp context initialization, DO NOT EXPECT ANYTHING TO WORK NOW:", e);
            }
        }
        else
            log.fatal("Initialize was called after DataRepository already initialized!");
    }

    /**
     * <p>Constructor for DataRepository.</p>
     *
     * @param sc a {@link javax.servlet.ServletContext} object.
     */
    public DataRepository(ServletContext sc)
    {
        super();
        servletContext = sc;
    }

    /**
     * <p>finishInitialize - setup at construct time</p>
     *
     * @throws java$io$IOException if any.
     * @throws org.openrdf.OpenRDFException if any.
     */
    public void finishInitialize()
        throws IOException, OpenRDFException
    {
        // 0. Set up home directory, creating it if necessary
        File home = getHomeDirectory();
        printInfo("Configuring with Eagle-I Repository Home dir = "+home.toString());

        if (!home.exists()) {
            if (!home.mkdirs()) {
                printError("Failed to create repository home directory, configured path = "+home);
                printError("THIS WILL CAUSE MANY OTHER ERRORS, DO NOT EXPECT THE REPOSITORY TO WORK!");
            }

        // If homedir path points to something that is not a directory
        // there isn't much we can do but complain about certain doom:
        } else if (!home.isDirectory()) {
            printError("Path configured as repository home is not a directory: "+home);
            printError("THIS WILL CAUSE MANY OTHER ERRORS, DO NOT EXPECT THE REPOSITORY TO WORK!");
        }

        // 1. Setup configuration properties file:
        // If file does not exist, copy in the default template.
        File pf = new File(home,"configuration.properties");
        if (!pf.exists()) {
            try {
                Utils.copyStream(this.getClass().getClassLoader().getResourceAsStream("default.configuration.properties"),
                          new FileOutputStream(pf));

            } catch (IOException e) {
                try {
                    pf.delete();
                } catch (Exception ee) { }
                printError("Failed to copy default configuration file to repository home dir: "+e);
            }
        }

        // 2. load the configuration properties.
        try {
            log.debug("Loading configuration from file="+pf);
            configuration = new PropertiesConfiguration();
            // do not break up any property value into multiple values, default=','!
            configuration.setListDelimiter((char)0);
            configuration.load(pf);
        } catch (Exception e) {
            printError("Failed to read configuration, file="+pf+": "+e.toString());
        }

        // 3. set up logging configuration: if it looks like this is
        // the bootstrap sequence or there is no log4j config,
        // set up default logging and then save it out
        // to the (presumably new) properties file for next time
        // and so the admin can edit it.

        // copy over just the log4j properties -- allow these to be
        // set in the main properties because:
        // (1) simpler to have only one config file
        // (2) apache config lets you interpolate variables, would
        //   be confusing to have that in main config but not log4j's..
        Properties lp = new Properties();
        for (Iterator pi = configuration.getKeys("log4j"); pi.hasNext();) {
            String k = (String)pi.next();
            lp.setProperty(k, configuration.getString(k));
        }
        setupDefaultLogging(home, lp);

        // 4. configure the Sesame directory and start up the repository
        String sesPath = configuration.getString(CONFIG_SESAMEDIR);
        File sesDir =  (sesPath == null) ? new File(home, "sesame") : new File(sesPath);
        if (!sesDir.exists())
            sesDir.mkdirs();

        // 5. get optional custom index list from configuration
        String indexes = configuration.getString(CONFIG_SESAMEINDEXES);

        NativeStore ns = new NativeStore(sesDir);
        Repository r = new SailRepository(new MinimalInferencer(ns));

        if (indexes != null)
            ns.setTripleIndexes(indexes);
        r.initialize();
        if (!r.isWritable())
            log.fatal("Sesame repo is not writable, this is going to cause trouble soon!");
        servletContext.setAttribute(SCTX_REPOSITORY, r);
        log.info("Sesame Repository open, directory="+sesDir.toString());

        servletContext.setAttribute(SCTX_GENERATION, Integer.valueOf(1));
        log.debug("Generation initialized to 1.");

        // 5. Load initial content into graphs, e.g.
        // the repo ontology,  initial local admin MD
        for (URI key : graphInitFile.keySet()) {
            loadGraphFromInit(r, key, true);
        }
    }


    /**
     * <p>destroy - shut down this application and release resources.</p>
     *
     * @param sc a {@link javax.servlet.ServletContext} object.
     * @throws org.openrdf.repository.RepositoryException if any.
     */
    public void destroy(ServletContext sc)
        throws RepositoryException
    {
        // Shutdown sesame repository - MUST be done to protect
        // the integrity of Native repository files.
        Repository r = (Repository)sc.getAttribute(SCTX_REPOSITORY);
        sc.removeAttribute(SCTX_REPOSITORY);
        if (r != null) {
            r.shutDown();
            log.info("Sesame Repository closed.");
        }
        instance = null;
    }

    // add property if user override not in place already
    private void setPropertyUnlessOverride(Properties lp, String key, String value)
    {
        if (!lp.containsKey(key))
            lp.setProperty(key, value);
    }

    /**
     * Initialize log4j:
     *  - All our Loggers are under org.eaglei.repository
     *  - Turn off additivity so events DO NOT go up to root, logger e.g in case
     *    tehre's a console appender, prevent cluttering up the container (Tomcat) log.
     *  - NOTE that programmatic setup of log4j does NOT get along with
     *    property-based configuration, e.g. prop file ONLY sees its own
     *    appenders.. so, we have to insert default properties here.
     *  - Default configuration in absence of overrides:
     *    - log at the INFO level, to "repository" appender, NOT additive.
     *    - the "repository" appender writes to "repository.log" in log dir.
     *  - Any log4j configs on the "org.eaglei.repository" logger will
     *    override these defaults.
     *
     * XXX NOTE: If buffered or non-immediate-flush is used in FileAppender,
     * you MUST call LoManager.shutdown(), see RepositoryContextListener
     */
    private void setupDefaultLogging(File home, Properties lp)
        throws IOException
    {
        // configure log directory and log4j
        String logPath = configuration.getString(CONFIG_LOGDIR);
        File logDir =  (logPath == null) ? new File(home, "logs") : new File(logPath);
        printInfo("Set up default log4j config: logfile directory: "+logDir.toString());
        if (!logDir.exists())
            logDir.mkdirs();
        Logger repoRoot = LogManager.getLogger(REPO_ROOT);

        // add default configuration unless properties already present:
        printInfo("Got "+String.valueOf(lp.size())+" initial log4j properties from repo configuration.");
        setPropertyUnlessOverride(lp, "log4j.logger."+REPO_ROOT, "INFO, repository");
        setPropertyUnlessOverride(lp, "log4j.additivity."+REPO_ROOT, "false");
        setPropertyUnlessOverride(lp, "log4j.appender.repository", RollingFileAppender.class.getName());
        setPropertyUnlessOverride(lp, "log4j.appender.repository.File", new File(logDir, "repository.log").toString());
        setPropertyUnlessOverride(lp, "log4j.appender.repository.ImmediateFlush", "false");
        setPropertyUnlessOverride(lp, "log4j.appender.repository.BufferedIO", "true");
        setPropertyUnlessOverride(lp, "log4j.appender.repository.Append", "true");
        setPropertyUnlessOverride(lp, "log4j.appender.repository.Encoding", "UTF-8");
        setPropertyUnlessOverride(lp, "log4j.appender.repository.layout", PatternLayout.class.getName());
        setPropertyUnlessOverride(lp, "log4j.appender.repository.layout.ConversionPattern", "%d{ISO8601} T=%t %p %c - %m%n");
        setPropertyUnlessOverride(lp, "log4j.logger."+OPENRDF_ROOT, "WARN, repository");

        printInfo("Configuring log4j with properties: "+lp.toString());
        PropertyConfigurator.configure(lp);

        // XXX NOTE: log4j's PropertyConfigurator does NOT call activateOptions
        // on all the Appenders that get created and/or modified by the
        // the properties.  We have to do it explicitly, but only for
        // the repository root logger.  If you configure any appenders on its
        // descendents, they won't work.

        // Need to explicitly activate any appenders the set up by the
        // config properties. log4j is such a crock.
        Enumeration rae = repoRoot.getAllAppenders();
        if (!rae.hasMoreElements())
            printError("The "+REPO_ROOT+" logger has no appenders!  This may result in no log records. Check your log4j configuration properties.");
        while (rae.hasMoreElements()) {
            AppenderSkeleton ra = (AppenderSkeleton)rae.nextElement();
            printInfo("Activating log4j appender \""+ra.getName()+"\" for logger "+REPO_ROOT);
            ra.activateOptions();
        }
        log.debug("logger "+repoRoot.getName()+" additivity="+repoRoot.getAdditivity());
    }

    /**
     * Gets the version that would be set by loading the RDF from given
     * init file; this is the value of owl:versionInfo on the given URI,
     * if any, in the serialized RDF fiel found on resourcePath.
     *
     * @param uri the uri of the graph, subject of owl:versionInfo
     * @return the exact content of the versionInfo literal or null if none found.
     */
    public static String getInitFileVersion(URI uri)
        throws IOException, RDFParseException, RDFHandlerException
    {
        String resourcePath = graphInitFile.get(uri);
        if (resourcePath != null) {
            RDFParser parser = Rio.createParser(RDFFormat.forFileName(resourcePath, RDFFormat.N3));
            VersionFinder vf = new VersionFinder(uri);
            parser.setRDFHandler(vf);
            parser.parse(new InputStreamReader(DataRepository.class.getClassLoader().getResourceAsStream(resourcePath), "UTF-8"), "");
            return vf.result;
        } else {
            log.warn("No resource path found for uri="+uri);
            return null;
        }
    }

    // handler for parser to extract the specific owl:versionInfo value
    // used by getInitFileVersion()
    private static class VersionFinder extends RDFHandlerBase
    {
        private String result = null;
        private URI name = null;

        private VersionFinder(URI uri)
        {
            super();
            name = uri;
        }

        public void handleStatement(Statement s)
            throws RDFHandlerException
        {
            Resource subject = s.getSubject();
            if (subject instanceof URI && name.equals(subject) &&
                  (OWL.VERSIONINFO.equals(s.getPredicate()) ||
                   REPO.VERSION_INFO.equals(s.getPredicate()))) {
                result = s.getObject().stringValue();
            }
        }
    }

    /**
     * Get the init file (relative resource path) for given graph URI.
     * @return relative path or null if there is none for this URI.
     */
    public static String getGraphInitFile(URI uri)
    {
        return graphInitFile.get(uri);
    }


    // Initialize a named graph if (presumably) this is the first
    // time the repo is starting up and RDF quad-store is empty.
    // If there are no statements in the named graph, load the given
    // resource file into it - it is a resource in this webapp.
    private void loadGraphFromInit(Repository r, URI graphURI, boolean ifEmpty)
        throws RepositoryException, IOException, RDFParseException
    {
        RepositoryConnection rc = null;
        try {
            rc = r.getConnection();
            loadGraphFromInit(rc, graphURI, ifEmpty);
        } finally {
            if (rc != null)
                rc.close();
        }
    }

    public void loadGraphFromInit(RepositoryConnection rc, URI graphURI, boolean ifEmpty)
        throws RepositoryException, IOException, RDFParseException
    {
        if (!ifEmpty || rc.size((Resource)graphURI) == 0) {
            String resourcePath = graphInitFile.get(graphURI);
            if (resourcePath == null) {
                log.error("No local resource path found (NOT loading anything) for URI="+graphURI);
                return;
            }
            URL rurl = this.getClass().getClassLoader().getResource(resourcePath);
            if (rurl == null)
                log.error("Cannot find webapp resource at path="+resourcePath);
            else {
                URLConnection ruc = rurl.openConnection();
                long rdate = ruc.getLastModified();
                log.debug("Loading graph="+graphURI.toString()+" from URL="+rurl.toString()+", mod-date="+rdate);
                Reader is = new InputStreamReader(ruc.getInputStream(), "UTF-8");
                if (is == null)
                    log.error("Cannot open webapp resource at url="+rurl);
                else {
                    rc.add(is, graphURI.toString(), RDFFormat.forFileName(resourcePath, RDFFormat.N3), graphURI);

                    // make up some provenance - datestamp should match date of resource in WAR file
                    // which is useful for checking the actual "provenance".
                    Provenance gp = new Provenance(graphURI);
                    gp.setProvenance(rc, DCTERMS.CREATED, Provenance.getDateTime(new Date()));
                    gp.setProvenance(rc, DCTERMS.CREATOR, REPO.ROLE_SUPERUSER);
                    gp.setSource(rc, rc.getValueFactory().createURI(rurl.toString()),
                                         Provenance.getDateTime(new Date(rdate)));
                    rc.commit();
                    log.info("Initialized the empty named graph, name="+graphURI+", from resource path="+resourcePath);
                }
            }
        }
    }

    /**
     * <p>getSesameRepository</p>
     *
     * @return the {@link org.openrdf.repository.Repository} object.
     * @throws javax.servlet.ServletException if any.
     */
    public Repository getSesameRepository()
        throws ServletException
    {
        Repository result = (Repository)servletContext.getAttribute(SCTX_REPOSITORY);
        if (result == null)
            throw new ServletException("No RDF database connection, probably because of a failure in initialization.");

        // confirm sesame repo setup in log
        //XXX log.debug("sesame repository type = "+result.getClass().getName());
        //XXX log.debug("sesame repository's top SAIL type = "+((org.openrdf.repository.sail.SailRepository)result).getSail().getClass().getName());
        return result;
    }
    
    /**
     * Get value of default namespace prefix for URIs created by the
     * repository.  This is a critically important value that ought to be
     * configured by the administrator, so that URIs are resolved correctly
     * by the server.  Since it is so essential, we have to provide a
     * usable default.
     *
     * Also, sanity-check the configured value.  It must be a legal absolute
     * URI.
     *
     * @return the default namespace prefix as a {@link java.lang.String} object.
     */
    public String getDefaultNamespace()
    {
        if (defaultNamespace == null) {
            String cfg = configuration.getString(CONFIG_NAMESPACE);
            if (cfg != null) {
                log.debug("Got raw configured default namespace="+cfg);
                try {
                    java.net.URI juri = new java.net.URI(cfg);
                    if (!juri.isAbsolute())
                        log.error("The configured namespace prefix must be an absolute URI: "+CONFIG_NAMESPACE+" = "+cfg);
                    else if (juri.getPath() == null)
                        log.error("The configured namespace prefix must include a path ending with a slash ('/'): "+CONFIG_NAMESPACE+" = "+cfg);
                    else if (!juri.getPath().endsWith("/"))
                        log.error("The configured namespace prefix must end with a slash ('/'): "+CONFIG_NAMESPACE+" = "+cfg);
                    else {
                        String scheme = juri.getScheme();
                        if (scheme.equalsIgnoreCase("http") || scheme.equalsIgnoreCase("https"))
                            defaultNamespace = cfg;
                        else
                            log.error("The configured namespace prefix must be an absolute URI: "+CONFIG_NAMESPACE+" = "+cfg);
                    }
                } catch (java.net.URISyntaxException e) {
                    log.error("The configured namespace prefix has illegal URI syntax: "+CONFIG_NAMESPACE+" = "+cfg+": "+e);
                }
            }

            // Fallback default: construct URI out of hostname.  This is
            // unlikely to be correct since JVM doesn't know the full,
            // actual, domain name for the webserver..
            if (defaultNamespace == null) {
                try {
                    defaultNamespace = "http://"+InetAddress.getLocalHost().getHostName()+"/i/";
                } catch (java.net.UnknownHostException e) {
                    log.error("Failed to get hostname for default URI prefix",e);
                    defaultNamespace = "http://failed-to-get-local-host-name/i/";
                }
                if (cfg == null)
                    log.warn("Default Namespace Prefix URI not configured.  You really should set a configuration value for "+CONFIG_NAMESPACE);
                log.warn("Using emergency default for namespace prefix URI (probably wrong) = "+defaultNamespace);
            }
        }
        return defaultNamespace;
    }

    /**
     * <p>getConfigurationProperty</p>
     *
     * @param key a {@link java.lang.String} object.
     * @return property value or null if not set.
     */
    public String getConfigurationProperty(String key)
    {
        return configuration.getString(key);
    }

    /**
     * return default if there is no value for the indicated property.
     *
     * @param key a {@link java.lang.String} object.
     * @param dflt default value to return if not set a {@link java.lang.String} object.
     * @return the property value or dflt if not set.
     */
    public String getConfigurationProperty(String key, String dflt)
    {
        String result = configuration.getString(key);
        return (result == null) ? dflt : result;
    }

    /**
     * Gets a configuration property with multiple values, returned in an array.
     * The values are expected to be separated by comma and optional whitespace.
     * Returns the default value (which may be null) when there is no such key
     *
     * @param key a {@link java.lang.String} object.
     * @return property value as an array of {@link java.lang.String} objects, or null if not set.
     */
    public String[] getConfigurationPropertyArray(String key, String[] dfault)
    {
        // XXX FIXME - can't use getStringArray since we disable the delimiter
        // char earlier so that converting log4j configs will work...
        // maybe fix that by loading config twice in different modes?
        //return configuration.containsKey(key) ? configuration.getStringArray(key) : dfault;

        // XXX workaround, split on comma
        return configuration.containsKey(key) ? configuration.getString(key).split("\\s*,\\s*") : dfault;
    }

    /**
     * Hook to signal to the application that all RDF data has been replaced,
     * e.g. after restoring a complete backup of all graphs.  This is
     * responsible for updating or invalidating all memory caches that depend
     * on the RDF contents.
     */
    public void notifyDataReplaced()
    {
        incrementGeneration();
        decacheAll();
    }

    /**
     * Bump the generation counter to signal that RDF database contents
     * have been replaced (i.e. a "new generation") and existing sessions
     * should be stale.   This is used to invalidate any cached URIs, e.g.
     * the user's Person-object URI mapped from authenticated principal name.
     */
    private void incrementGeneration()
    {
        int gen = 1 + ((Integer)servletContext.getAttribute(SCTX_GENERATION)).intValue();
        servletContext.setAttribute(SCTX_GENERATION, Integer.valueOf(gen));
        log.debug("Generation is incremented to "+gen);
    }

    /**
     * Compare current webapp generation with value (if any) cached
     * in session, also update session's generation to current.
     *
     * @param session a {@link javax.servlet.http.HttpSession} object.
     * @return a boolean, true if the session is "stale" (e.g. after a global restore of RDF db)
     */
    public boolean isSessionStale(HttpSession session)
    {
        Integer sessionGen = (Integer)session.getAttribute(SCTX_GENERATION);
        Integer ctxGen = (Integer)servletContext.getAttribute(SCTX_GENERATION);
        boolean result = sessionGen != null &&
                 sessionGen.intValue() < ctxGen.intValue();
        session.setAttribute(SCTX_GENERATION, ctxGen);
        return result;
    }

    /**
     * Read a resource file out of the webapp, path is relative to webapp root
     * and MUST NOT begin with '/' -- e.g. "repository/styles/foo.xsl".
     *
     * @param path relative path to resource file, a {@link java.lang.String} object.
     * @return a {@link java.io.Reader} object open on the resource, or null if not available.
     */
    public Reader getWebappResourceAsReader(String path)
        throws ServletException
    {
        if (servletContext == null) {
            log.error("getWebappResourceAsStream: currentContext not set!");
            return null;
        }
        try {
            return new InputStreamReader(servletContext.getResourceAsStream(path), "UTF-8");
        } catch  (UnsupportedEncodingException e) {
            throw new ServletException("Unsupported encoding content-type spec: "+e);
        } catch  (IllegalCharsetNameException e) {
            throw new ServletException("Illegal character set name in content-type spec: "+e);
        } catch  (UnsupportedCharsetException e) {
            throw new ServletException("Unsupported character set name in content-type spec: "+e);
        }
    }

    /**
     * Get the Maven project version
     *
     * @return version as an opaque a {@link java.lang.String} object.
     */
    public String getProjectVersion()
    {
        return getBuildProperties().getProperty("version");
    }

    /**
     * Get the source revision, e.g. the subversion revision number
     *
     * @return revision as an opaque a {@link java.lang.String} object.
     */
    public String getRevision()
    {
        return getBuildProperties().getProperty("revision");
    }

    /**
     * Get the source SCM branch, if any.
     *
     * @return branch or tag name, or null if on the trunk.
     */
    public String getBranch()
    {
        String branch = getBuildProperties().getProperty("branch");
        if (branch == null) {
            log.warn("Cannot find build property \"branch\".");
            return null;
        } else if (branch.startsWith(SVN_BRANCHES_PREFIX)) {
            int nextSlash = branch.indexOf('/', SVN_BRANCHES_PREFIX.length());
            return nextSlash < 0 ? branch.substring(9) : branch.substring(9, nextSlash);
        } else if (branch.startsWith(SVN_TAGS_PREFIX)) {
            int nextSlash = branch.indexOf('/', SVN_TAGS_PREFIX.length());
            return nextSlash < 0 ? branch.substring(5) : branch.substring(5, nextSlash);

        // maven buildnumber plugin sometimes leaves scmBranch unset on
        // trunk, so value is literally "${scmBranch}".  Ugh.
        } else if (!(branch.equals("trunk") || branch.startsWith("${"/*}*/))) {
            log.warn("Build property \"branch\" has unexpected value: \""+branch+"\"");
        }
        return null;
    }

    /**
     * Get the time when this software version was built
     *
     * @return time as an opaque a {@link java.lang.String} object.
     */
    public String getTimestamp()
    {
        return getBuildProperties().getProperty("timestamp");
    }

    // Read the "build.properties" resource into a cached Properties.
    // It is used to obtain the version and timestamp info
    private Properties getBuildProperties()
    {
        if (buildProperties == null) {
            buildProperties = new Properties();
            try {
                buildProperties.load(this.getClass().getClassLoader().getResourceAsStream("build.properties"));
            } catch (IOException e) {
                log.error("Failed loading build.properties: ",e);
            }
        }
        return buildProperties;
    }


    /*  Find the root of the repository configuration and data hierarchy.
     *  It will be the first one of these that exists:
     *     a. value of system property "org.eaglei.repository.home"
     *     b. otherwise, default is "$HOME/eaglei/repository/"
     *       (where $HOME == System.getProperty("user.home") in Java-land)
     *  ALSO, this sets the system property to the default if it
     *   was unset before, so e.g. Apache config interpolation will work.
     */
    /**
     * <p>getHomeDirectory - get configured repository home directory.</p>
     *
     * @return repository home directory as a {@link java.io.File} object.
     * @throws java$io$IOException if any.
     */
    public File getHomeDirectory()
        throws IOException
    {
        String home = System.getProperty(REPO_HOME);
        File result = null;
        boolean haveSysProp = (home != null);
        if (!haveSysProp)
            result = new File(System.getProperty("user.home"),
                        "eaglei"+File.separator+"repository");
        else
            result = new File(home);
        if (!result.exists() && !result.mkdirs()) {
            log.fatal("The configured home directory does not exist: "+result.toString());
            throw new IOException("The configured home directory does not exist: "+result.toString());
        }
        if (!result.isDirectory()) {
            log.fatal("The configured home directory is not a directory or is protected: "+result.toString());
            throw new IOException("The configured home directory is not a directory or is protected: "+result.toString());
        }
        log.debug("repository home directory = "+result.toString());
        if (!haveSysProp)
            System.setProperty(REPO_HOME, result.toString());
        return result;
    }

    /**
     * <p>getBackupDirectory- get configured repository backup directory</p>
     *
     * @return repository backup directory as a {@link java.io.File} object.
     * @throws java$io$IOException if any.
     */
    public File getBackupDirectory()
       throws IOException
    {
        String home = System.getProperty(BACKUP_HOME);
        File result = null;
        boolean haveProp = (home != null);
        if (!haveProp)
            result = new File(getHomeDirectory(), "backup");
        else
            result = new File(home);
        if (!result.exists() && !result.mkdirs()) {
            log.fatal("The configured home directory does not exist: "+result.toString());
            throw new IOException("The configured home directory does not exist: "+result.toString());
        }
        if (!result.isDirectory()) {
            log.fatal("The configured home directory is not a directory or is protected: "+result.toString());
            throw new IOException("The configured home directory is not a directory or is protected: "+result.toString());
        }
        log.debug("backup home directory = "+result.toString());
        if (!haveProp)
            System.setProperty(BACKUP_HOME, result.toString());
        return result;
    }

    /**
     * Update the most recent "last modified" timestamp if profferred
     * value is newer
     *
     * @param lm date of new most recent dcterms:modified
     */
    public synchronized void setLastModified(Date lm)
    {
        if (lastModified.before(lm))
            lastModified = lm;
    }

    /**
     * Gets the most recent "last modified" timestamp applied to
     * provenance metadata.
     *
     * @return date of most recent dcterms:modified, or repository startup time by default
     */
    public Date getLastModified()
    {
        return lastModified;
    }


    public String getSesameVersion()
        throws IOException
    {
        // easy way -- if hte package has it
        String result = null;
        Package rcp = RepositoryConnection.class.getPackage();
        if (rcp == null) {
            log.debug("Cannot get Sesame version from package, no package for class "+RepositoryConnection.class.getName());
        } else {
            result = rcp.getImplementationVersion();
            if (result == null) {
                log.debug("Cannot get Sesame version, package has no ImplementationVersion: "+rcp.getName());
            } else {
                return result;
            }
        }

        // Try the hard way -- find Jar that this was loaded from.
        // The rest of this kludge assumes we have a jar-scheme URL, which
        // has the form:   jar:<url>!/<entry>
        // e.g.  jar:file:/..../lib/eagle-i-model-owl-0.4.3.jar!/ero.owl
        // so we can extract the URL of the jar itself:
        String classRes = "/"+RepositoryConnection.class.getName().replace('.','/')+".class";
        java.net.URL rurl = RepositoryConnection.class.getResource(classRes);
        if (rurl == null) {
            log.warn("Failed to find Sesame class resource path="+classRes);
        } else {
            log.debug("Got sesame class resource URL ="+rurl);
            String surl = rurl.toString();
            if (surl.startsWith("jar:")) {
                Pattern ep = Pattern.compile("jar:([^!]+)!.*");
                log.debug("Groveling jar URL with pattern=\""+ep.pattern()+"\"");
                Matcher em = ep.matcher(surl);
                if (em.matches()) {
                    String jarURLspec = em.group(1);
                    log.debug("Got jar URL = "+jarURLspec);
                    result = Utils.getMavenVersionFromJar(jarURLspec);
                    if (result != null)
                        return result;
                } else {
                    log.warn("Failed matching jar URL regex on Sesame class URL; pattern=\""+ep.pattern()+"\" against owl resource URL="+surl);
                }
            } else {
                log.warn("Sesame Class Resource URL is NOT in Jar scheme: "+surl);
            }
        }
        return "(unknown)";
    }

    /**
     * Call the decache() method of all classes with the HasContentCache
     * annotation.  This gets called when the RDF database was replaced
     * so in-memory caches must be invalidated.
     */
    public static void decacheAll()
    {
        log.info("Decaching all content caches..");
        Reflections r = new Reflections("org.eaglei.repository");
            for (Class c : r.getTypesAnnotatedWith(HasContentCache.class)) {
                log.debug("  Decaching "+c.getName());
            try {
                c.getMethod("decache").invoke(null);
            } catch (NoSuchMethodException e) {
                log.error("Class is marked as content cacher but has no static decache() method - "+c.getName()+": "+e);
            } catch (IllegalAccessException e) {
                log.error("Failed calling decache() on content cacher class="+c.getName()+": "+e);
            } catch (java.lang.reflect.InvocationTargetException e) {
                log.error("Failed calling decache() on content cacher class="+c.getName()+": "+e);
            }
        }
    }
}
