package org.eaglei.repository;

import java.util.ArrayList;
import java.util.Collection;
import java.util.Collections;
import java.util.List;
import java.util.Map;
import java.util.Set;
import java.util.HashMap;
import java.io.IOException;
import java.io.OutputStreamWriter;

import javax.servlet.ServletException;
import javax.servlet.http.HttpServletRequest;
import javax.servlet.http.HttpServletResponse;

import org.apache.log4j.Logger;
import org.apache.log4j.LogManager;

import org.openrdf.OpenRDFException;
import org.openrdf.model.URI;
import org.openrdf.model.Literal;
import org.openrdf.model.Value;
import org.openrdf.model.impl.LiteralImpl;
import org.openrdf.model.impl.ContextStatementImpl;
import org.openrdf.repository.RepositoryConnection;
import org.openrdf.repository.RepositoryException;
import org.openrdf.model.vocabulary.RDF;
import org.openrdf.model.vocabulary.RDFS;
import org.openrdf.model.vocabulary.OWL;
import org.openrdf.model.ValueFactory;
import org.openrdf.rio.RDFFormat;
import org.openrdf.rio.Rio;
import org.openrdf.rio.RDFHandler;
import org.openrdf.query.TupleQuery;
import org.openrdf.query.TupleQueryResult;
import org.openrdf.query.TupleQueryResultHandlerBase;
import org.openrdf.query.TupleQueryResultHandlerException;
import org.openrdf.query.BindingSet;
import org.openrdf.query.QueryLanguage;
import org.openrdf.query.MalformedQueryException;

import org.eaglei.repository.vocabulary.REPO;
import org.eaglei.repository.util.SPARQL;
import org.eaglei.repository.util.HandlerBadRequest;
import org.eaglei.repository.status.NotFoundException;
import org.eaglei.repository.status.BadRequestException;
import org.eaglei.repository.status.ForbiddenException;
import org.eaglei.repository.status.InternalServerErrorException;
import org.eaglei.repository.servlet.WithRepositoryConnection;
import org.eaglei.repository.rid.RIDGenerator;
import org.eaglei.repository.servlet.ImportExport.DuplicateArg;

/**
 * Role object model, reflects the :Role object in RDF database.
 *
 * A role represents an entity that can be granted privileges in the access
 * control system.  Users who are <i>members</i> of a Role get all its
 * grants.  Roles are "flat" objects, i.e. there is NO hierarchy of Roles --
 * although it would be convenient, it is too difficult to implement given
 * the current inferencing support available in the RDF database.
 *
 * Named Graph usage: Role looks for statements in ALL graphs, although
 *  they should really only be found in the :NG_Internal graph.
 *
 * @author Larry Stone
 * Started April 26, 2010
 * @version $Id: $
 */
@HasContentCache
public class Role implements Comparable
{
    private static Logger log = LogManager.getLogger(Role.class);

    private URI uri;                  // subject: required
    private String label = null;      // value of rdfs:label
    private String comment = null;    // value of rdfs:comment
    private boolean implicit = false; // value of :isImplicit proeprty
    private boolean dirty = false;    // anything modified?

    // Caches, so we don't keep allocating new Roles - key is URI
    private static volatile Map<URI,Role> cacheInternal = null;

    // get role descriptions, works on both internal repo graph and exports.
    private static final String roleQuery =
          "SELECT DISTINCT ?uri ?label ?comment ?implicit WHERE {\n"+
          " ?uri a <"+REPO.ROLE+"> ; <"+RDFS.LABEL+"> ?label .\n"+
          "  OPTIONAL { ?uri <"+REPO.IS_IMPLICIT+"> ?implicit }\n"+
          "  OPTIONAL { ?uri <"+RDFS.COMMENT+"> ?comment }}";

    private Role(URI uri, String label, String comment, boolean implicit)
    {
        super();
        this.uri = uri;
        this.label = label;
        this.comment = comment;
        this.implicit = implicit;
    }

    /**
     * Get the Role for given URI, lookup label.
     *
     * @param request a {@link javax.servlet.http.HttpServletRequest} object.
     * @param uri URI of the role
     * @return a {@link org.eaglei.repository.Role} object representing role.
     * @throws javax.servlet.ServletException if any.
     */
    public static Role find(HttpServletRequest request, URI uri)
        throws ServletException
    {
        Map<URI,Role> cache = getCache(request);
        if (cache.containsKey(uri)) {
            if (log.isDebugEnabled())
                log.debug("find("+uri.stringValue()+") => "+cache.get(uri));
            return cache.get(uri);
        } else
            throw new NotFoundException("There is no Role of URI="+uri);
    }

    /**
     * Get all known Roles - includes pseudo-Roles Anonymous and Authenticated
     * that should NOT be directly assigned to Users.
     * Make the query each time instead of relying on cache since some sneaky
     * admin may *add* new roles by uploading statements to NG_Internal graph.
     *
     * @param request a {@link javax.servlet.http.HttpServletRequest} object.
     * @return all roles in a {@link java.lang.Iterable} object.
     * @throws javax.servlet.ServletException if any.
     */
    public static Collection<Role> findAll(HttpServletRequest request)
        throws ServletException
    {
        Map<URI,Role> cache = getCache(request);
        return cache.values();
    }

    /**
     * Get all Roles that can be granted in a :has___Access statement.
     * This works out to all roles *except* SuperUser.
     *
     * @param request a {@link javax.servlet.http.HttpServletRequest} object.
     * @return all roles in a {@link java.lang.Iterable} object.
     * @throws javax.servlet.ServletException if any.
     */
    public static Collection<Role> findAllGrantable(HttpServletRequest request)
        throws ServletException
    {
        Map<URI,Role> cache = getCache(request);
        List<Role> result = new ArrayList(cache.values());
        result.remove(find(request, REPO.ROLE_SUPERUSER));
        Collections.sort(result);
        return result;
    }

    /**
     * Get all Roles that ought to be assigned to users - EXCLUDES
     * "implicit" pseudo-Roles Anonymous and Authenticated.
     * Make the query each time instead of relying on cache since some sneaky
     * admin may *add* new roles by uploading statements to NG_Internal graph.
     *
     * @param request a {@link javax.servlet.http.HttpServletRequest} object.
     * @return all roles in a {@link java.lang.Iterable} object.
     * @throws javax.servlet.ServletException if any.
     */
    public static Collection<Role> findAllUserRoles(HttpServletRequest request)
        throws ServletException
    {
        Map<URI,Role> cache = getCache(request);
        List<Role> result = new ArrayList<Role>();
        for (Role r : cache.values()) {
            if (!r.implicit)
                result.add(r);
        }
        return result;
    }

    /**
     * Invalidate the role cache, called when RDF has changed.
     */
    public static void decache()
    {
        synchronized (Role.class) {
            cacheInternal = null;
        }
    }

    // reload role cache if needed.
    private static Map<URI,Role> getCache(HttpServletRequest request)
        throws ServletException
    {
        // this looks weird but is necessary to minimize sync block.
        synchronized (Role.class) {
            if (cacheInternal != null)
                return cacheInternal;
            cacheInternal = new HashMap<URI,Role>();
        }
        try {
            RepositoryConnection rc = WithRepositoryConnection.get(request);
            log.debug("All role SPARQL query = "+roleQuery);
            TupleQuery q = rc.prepareTupleQuery(QueryLanguage.SPARQL, roleQuery);
            q.setDataset(SPARQL.InternalGraphs);
         
            // need inference!
            q.setIncludeInferred(true);
            q.evaluate(new allRoleHandler(cacheInternal));
        } catch (MalformedQueryException e) {
            log.error("Rejecting malformed query:"+e);
            throw new ServletException(e);
        } catch (OpenRDFException e) {
            log.error(e);
            throw new ServletException(e);
        }
        return cacheInternal;
    }

    /**
     * <p>getURI</p>
     *
     * @return URI subject of the role, a {@link org.openrdf.model.URI} object.
     */
    public URI getURI()
    {
        return uri;
    }

    /**
     * <p>isSuperuser</p>
     * predicate that answers whether this role confers Administrator
     * privileges, an attempt at preserving modularity.
     *
     * @return true if this role determins superuser (admin) privileges
     */
    public boolean isSuperuser()
    {
        return uri.equals(REPO.ROLE_SUPERUSER);
    }

    /**
     * <p>Getter for the field <code>label</code>.</p>
     *
     * @return label of the role, a {@link java.lang.String} object.
     */
    public String getLabel()
    {
        return label;
    }

    /**
     * <p>Getter for the field <code>comment</code>.</p>
     *
     * @return comment of the role, a {@link java.lang.String} object, MIGHT be null.
     */
    public String getComment()
    {
        return comment;
    }

    /** {@inheritDoc} */
    public boolean equals(Object other)
    {
        return other instanceof Role && uri.equals(((Role)other).uri);
    }

    /** {@inheritDoc} */
    public int hashCode()
    {
        return uri.hashCode();
    }

    /**
     * <p>toString</p>
     *
     * @return a {@link java.lang.String} object.
     */
    public String toString()
    {
        return "<#Role: uri="+uri.toString()+
            ", label="+label+">";
    }

    public int compareTo(Object o)
    {
        return label.compareTo(((Role)o).label);
    }

    // predicate - is this one of the permanent builtin roles?
    public boolean isBuiltin()
    {
        return implicit || REPO.ROLE_SUPERUSER.equals(uri);
    }

    // predicate - value of implicit
    public boolean isImplicit()
    {
        return implicit;
    }

    /**
     * <p>Make a new Role</p> at a specific URI (assumes it is not
     * already the subject of any existing statements).  If uri == null,
     * a new URI is allocated.
     *
     * @param request a {@link javax.servlet.http.HttpServletRequest} object.
     * @param label short name for role, SHOULD be non-null
     * @param comment longer description of role, may be null
     * @throws javax.servlet.ServletException if any.
     */
    public static Role create(HttpServletRequest request, URI subject, String label, String comment)
        throws ServletException
    {
        if (!Access.isSuperuser(request))
            throw new ForbiddenException("You must be an Administrator to create a new Role.");

        // First, sanity check that the URI is not already *any* kind of object
        // Then, add role property statements.
        try {
            RepositoryConnection rc = WithRepositoryConnection.get(request);
            ValueFactory vf = rc.getValueFactory();
            if (label == null || label.length() == 0)
                throw new BadRequestException("Label is required when creating a new Role.");

            // forge a new subject URI
            if (subject == null)
                subject = vf.createURI(DataRepository.getInstance().getDefaultNamespace(),
                            RIDGenerator.getInstance().newID().toString());

            // rdf description of role:
            rc.add(subject, RDF.TYPE, REPO.ROLE, REPO.NG_INTERNAL);
            rc.add(subject, RDFS.LABEL, vf.createLiteral(label), REPO.NG_INTERNAL);
            if (comment != null && comment.length() > 0)
                rc.add(subject, RDFS.COMMENT, vf.createLiteral(comment), REPO.NG_INTERNAL);
            Role result = new Role(subject, label, comment, false);
            result.dirty = true;
            decache();
            log.debug("Created Role uri="+subject.stringValue()+", label="+label);
            return result;
        } catch (RepositoryException e) {
            throw new ServletException(e);
        }
    }

    public void delete(HttpServletRequest request)
        throws ServletException
    {
        if (!Access.isSuperuser(request))
            throw new ForbiddenException("You must be an Administrator to delete a Role.");
        if (isBuiltin())
            throw new ForbiddenException("Built-in Roles may not be deleted.");

        try {
            RepositoryConnection rc = WithRepositoryConnection.get(request);
            rc.remove(uri, null, null, REPO.NG_INTERNAL);
            dirty = true;
            decache();
        } catch (RepositoryException e) {
            throw new ServletException(e);
        }
    }

    /**
     * <p>Setter for the field <code>label</code>.</p>
     *
     * @param request a {@link javax.servlet.http.HttpServletRequest} object.
     * @param label a new label value
     * @throws javax.servlet.ServletException if any.
     */
    public void setLabel(HttpServletRequest request, String label)
        throws ServletException
    {
        if (label == null || label.length() == 0)
            throw new BadRequestException("Label must be a non-empty string.");
        setMetadataInternal(request, RDFS.LABEL, label);
    }

    public void setComment(HttpServletRequest request, String newVal)
        throws ServletException
    {
        setMetadataInternal(request, RDFS.COMMENT, newVal);
    }


    // update one of the metadata properties.
    private void setMetadataInternal(HttpServletRequest request, URI property, String newVal)
        throws ServletException
    {
        if (!Access.isSuperuser(request))
            throw new ForbiddenException("You must be an Administrator to modify Roles.");
        if (isBuiltin())
            throw new ForbiddenException("Builtin Roles may not be modified.");
        try {
            RepositoryConnection rc = WithRepositoryConnection.get(request);
            ValueFactory vf = rc.getValueFactory();
            rc.remove(uri, property, null, REPO.NG_INTERNAL);
            if (newVal != null) {
                rc.add(uri, property, vf.createLiteral(newVal), REPO.NG_INTERNAL);
                log.debug("Setting Role uri="+uri+", property="+property+" to value="+newVal);
            } else
                log.debug("Clearing Role uri="+uri+", property="+property);
            // XXX this may lie if we didn't really change anything, but who cares.
            dirty = true;
        } catch (RepositoryException e) {
            throw new ServletException(e);
        }
    }

    /**
     * <p>commit - commit any changes made to this object (and any others)</p>
     *
     * @param request a {@link javax.servlet.http.HttpServletRequest} object.
     * @throws javax.servlet.ServletException if any.
     */
    public void commit(HttpServletRequest request)
        throws ServletException
    {
        try {
            if (dirty) {
                WithRepositoryConnection.get(request).commit();
                decache();
            }
            dirty = false;
        } catch (OpenRDFException e) {
            throw new ServletException(e);
        }
    }

    // tuple query result handler to gather list of all roles.
    private static class allRoleHandler extends TupleQueryResultHandlerBase
    {
        private Map<URI,Role> result = null;

        public allRoleHandler(Map<URI,Role> result)
        {
            super();
            allRoleHandler.this.result = result;
        }

        public void handleSolution(BindingSet bs)
            throws TupleQueryResultHandlerException
        {
            Value newURI = bs.getValue("uri");
            if (newURI == null || !(newURI instanceof URI))
                log.error("Should not get null or non-URI result in allRoleHandler: "+newURI);
            else {
                URI u = (URI)newURI;
                    // If search is limited to "roles that can be assigned to users"
                    // skip anonymous and authenticated, they are virtual roles.
                Value rlabel = bs.getValue("label");
                String label = (rlabel != null && rlabel instanceof Literal) ?
                           ((Literal)rlabel).getLabel() : "";
                Value rcomment = bs.getValue("comment");
                String comment = null;
                if (rcomment != null && rcomment instanceof Literal)
                    comment = ((Literal)rcomment).getLabel();
                boolean implicit = false;
                Value rimplicit = bs.getValue("implicit");
                if (rimplicit instanceof Literal)
                    implicit = ((Literal)rimplicit).booleanValue();
                result.put(u, new Role(u, label, comment, implicit));
            }
        }
    }

    /**
     * Export roles to serialized quad format.
     * Write a stylized representation of essential data, since the
     * actual statements implementing a Role are bound to chagne soon.
     * The export contains these statements for each role:
     *
     * {All statements in :NG_Internal graph}
     *     <role-uri> a :Role .
     *     <role-uri> rdfs:label "label" .
     *     {optional} <role-uri> rdfs:comment  "comment" .
     *
     */
    public static void doExportRoles(HttpServletRequest request, HttpServletResponse response,
            RDFFormat format, Set<String> includes, Set<String> excludes)
        throws ServletException, IOException
    {
        try {
            RDFHandler out = Rio.createWriter(format, new OutputStreamWriter(response.getOutputStream(), "UTF-8"));
            out.startRDF();
            for (Role r : findAll(request)) {
                if (!r.isBuiltin()) {
                    URI ru = r.getURI();
                    String rus = ru.stringValue();
                    String rc = r.getComment();
                    String rl = r.getLabel();
                    if (excludes.contains(rl) || excludes.contains(rus)) {
                        log.debug("SKIP ROLE because of exclude: "+r);
                    } else if (includes.isEmpty() ||
                             includes.contains(rl) || includes.contains(rus)) {
                        // NOTE: synthesize a bogus rdf:type, just to reassure import
                        out.handleStatement(new ContextStatementImpl(
                            ru, RDF.TYPE, REPO.ROLE, REPO.NG_INTERNAL));
                        out.handleStatement(new ContextStatementImpl(
                            ru, RDFS.LABEL, new LiteralImpl(r.getLabel()), REPO.NG_INTERNAL));
                        if (rc != null)
                            out.handleStatement(new ContextStatementImpl(
                                ru, RDFS.COMMENT, new LiteralImpl(rc), REPO.NG_INTERNAL));
                    } else {
                        log.debug("SKIP ROLE because of include: "+r);
                    }
                }
            }
            out.endRDF();
        } catch (OpenRDFException e) {
            throw new InternalServerErrorException(e);
        }
    }

    /**
     * Import description of roles from serialized RDF quads.
     */
    public static void doImportRoles(HttpServletRequest request, HttpServletResponse response,
            RepositoryConnection content,
            Set<String> includes, Set<String> excludes,
            DuplicateArg duplicate,
            boolean transform)

        throws ServletException, IOException
    {
        try {
            TupleQuery q = content.prepareTupleQuery(QueryLanguage.SPARQL, roleQuery);
            if (log.isDebugEnabled())
                log.debug("SPARQL query to get IMPORTED ROLES (against internal memory repo) =\n\t"+roleQuery);
            q.setDataset(SPARQL.InternalGraphs);
            q.setIncludeInferred(false);
            importRoleHandler ih = new importRoleHandler(request, includes, excludes, duplicate, transform);
            q.evaluate(ih);

            // if there are statements but nothing matching roles, flag an
            // error because it may be an invalid export file.
            if (ih.count == 0 && content.size() > 0)
                throw new BadRequestException("Invalid input: None of the statements in the input data are valid Role representations.");
        } catch (HandlerBadRequest e) {
            throw new BadRequestException(e.getMessage());
        } catch (OpenRDFException e) {
            throw new InternalServerErrorException(e);
        }
    }

    private static class importRoleHandler extends TupleQueryResultHandlerBase
    {
        private static final String dupeQueryString =
            "SELECT DISTINCT ?uri WHERE {\n"+
            " ?uri <"+RDFS.SUBCLASSOF+"> <"+REPO.ROLE+">; \n"+
            "  <"+RDFS.LABEL+"> ?label. \n"+
            "FILTER (?uri = ?quri || ?label = ?qlabel)}";

        private HttpServletRequest request;
        private boolean transform;
        private DuplicateArg duplicate;
        private Set<String> includes;
        private Set<String> excludes;
        private TupleQuery dupeQuery = null;
        private int count = 0;

        public importRoleHandler(HttpServletRequest arequest,
                Set<String> aincludes, Set<String> aexcludes,
                DuplicateArg aduplicate,
                boolean atransform)
            throws OpenRDFException
        {
            super();
            request = arequest;
            transform = atransform;
            duplicate = aduplicate;
            includes = aincludes;
            excludes = aexcludes;

            // re-use the same query with different bindings
            RepositoryConnection rc = WithRepositoryConnection.get(request);
            dupeQuery = rc.prepareTupleQuery(QueryLanguage.SPARQL, dupeQueryString);
            dupeQuery.setDataset(SPARQL.InternalGraphs);
            dupeQuery.setIncludeInferred(false);
        }

        // process result of uri, label, comment
        public void handleSolution(BindingSet bs)
            throws TupleQueryResultHandlerException
        {
            ++count;
            Value vuri = bs.getValue("uri");
            Value label = bs.getValue("label");
            String us = vuri.stringValue();
            String ls = label.stringValue();
            Value comment = bs.getValue("comment");
            boolean implicit = false;
            Value rimplicit = bs.getValue("implicit");
            if (rimplicit instanceof Literal)
                implicit = ((Literal)rimplicit).booleanValue();
            if (log.isDebugEnabled())
                log.debug("importRoleHandler: Got result: uri="+vuri+", label="+label+", implicit="+implicit);

            // first, apply include/exclude filter
            if (excludes.contains(us) || excludes.contains(ls)) {
                log.debug("SKIP IMPORT ROLE because of exclude: uri="+us+", label="+ls);
                return;
            } else if (!includes.isEmpty() && !(includes.contains(us) || includes.contains(ls))) {
                log.debug("SKIP IMPORT ROLE because of include: uri="+us+", label="+ls);
                return;
            }

            // check for duplicate - horrible nested query but necessary
            // if we are to check label AND uri.. at least there are
            // only a few roles expected.
            TupleQueryResult qr = null;
            try {
                dupeQuery.clearBindings();
                dupeQuery.setBinding("quri", vuri);
                dupeQuery.setBinding("qlabel", label);
                qr = dupeQuery.evaluate();
                if (qr.hasNext()) {
                    Value du = qr.next().getValue("uri");
                    log.debug("Found a duplicate Role, uri="+du.stringValue());
                    if (duplicate == DuplicateArg.ignore) {
                        log.debug("SKIPPING duplicate role, input uri="+us);
                        return;
                    } else if (duplicate == DuplicateArg.abort) {
                        throw new HandlerBadRequest("Import contains a duplicate Role for existing Role uri="+du.stringValue());

                    // to replace -- delete old role first.
                    } else if (duplicate == DuplicateArg.replace) {
                        Role r = find(request, (URI)du);
                        r.delete(request);
                    }
                }
                Role nr = create(request, transform ? null : (URI)vuri,
                            ls, comment == null ? null : comment.stringValue());
                nr.implicit = implicit;
            } catch (OpenRDFException e) {
                throw new InternalServerErrorException(e);
            } catch (ServletException e) {
                throw new TupleQueryResultHandlerException(e);
            } finally {
                try {
                    if (qr != null)
                        qr.close();
                } catch (OpenRDFException e) {
                    log.warn("Ignoring exception while closing result: "+e);
                }
            }

        }
    }
}
