package org.eaglei.repository.admin;

import java.sql.Connection;
import java.sql.PreparedStatement;
import java.sql.ResultSet;
import java.sql.SQLException;
import javax.sql.DataSource;
import javax.naming.Context;
import javax.naming.InitialContext;
import javax.naming.NamingException;
import java.util.HashMap;
import java.util.Map;
import java.util.HashSet;
import java.util.Set;

import org.apache.log4j.Logger;
import org.apache.log4j.LogManager;

import org.eaglei.repository.Access;

/**
 * Simple lightweight class to manage RDBMS authorization system users
 * for the purpose of importing and exporting auth. users.
 *
 * @author Larry Stone
 * Started June 24 2010
 */
public class AuthUser
{
    private static Logger log = LogManager.getLogger(AuthUser.class);

    private String username;
    private String password;
    private boolean isSuperuser;
    private String oldPassword;
    private boolean oldIsSuperuser;

    private AuthUser(String u, String p, boolean s)
    {
        super();
        username = u;
        password = p;
        oldPassword = password;
        isSuperuser = s;
        oldIsSuperuser = isSuperuser;
    }

    /**
     * Get the auth-db description of the indicated user, if present.
     * Returns a new AuthUser object or null if not found.
     */
    public static AuthUser find(String targetUsername)
        throws NamingException, SQLException
    {
        Connection c = null;
        PreparedStatement s = null;
        try {
            c = getConnection();
            s = c.prepareStatement(
                "SELECT Users.Username,Users.Password,Roles.Rolename "+
                " FROM Users LEFT OUTER JOIN Roles "+
                " ON Users.Username = Roles.Username "+
                "    AND Roles.Rolename = '"+Access.SUPERUSER_ROLE_NAME+"'"+
                " WHERE Users.Username = ?");
            s.setString(1, targetUsername);
            ResultSet r = s.executeQuery();
            int count = 0;
            AuthUser result = null;
            while (r.next()) {
                String dusername = r.getString(1);
                String dpassword = r.getString(2);
                String role = r.getString(3);
                boolean isAdmin = role != null && Access.SUPERUSER_ROLE_NAME.equalsIgnoreCase(role);
                if (log.isDebugEnabled())
                    log.debug("Got row, user="+dusername+", passwd="+dpassword+", role="+role);
                ++count;
                if (result == null)
                    result = new AuthUser(dusername, dpassword, isAdmin);
                else
                    log.error("Got extra results from single user query: user="+dusername+", pw="+dpassword+", role="+role);
            }
            if (log.isDebugEnabled())
                log.debug("Processed "+String.valueOf(count)+" rows of results.");
            return result;
        } finally {
            if (s != null)
                s.close();
            if (c != null)
                c.close();
        }
    }

    /**
     * Get a Map of each username key to AuthUser object.
     */
    public static Map<String,AuthUser> findAllAsMap()
        throws NamingException, SQLException
    {
        Map<String,AuthUser> result = new HashMap<String,AuthUser>();
        Connection c = null;
        PreparedStatement s = null;
        try {
            c = getConnection();
            s = c.prepareStatement(
                "SELECT Users.Username,Users.Password,Roles.Rolename "+
                " FROM Users LEFT OUTER JOIN Roles "+
                " ON Users.Username = Roles.Username"+
                " AND Roles.Rolename = '"+Access.SUPERUSER_ROLE_NAME+"'");
            ResultSet r = s.executeQuery();
            while (r.next()) {
                String dusername = r.getString(1);
                String role = r.getString(3);
                boolean isAdmin = role != null && Access.SUPERUSER_ROLE_NAME.equalsIgnoreCase(role);
                if (result.containsKey(dusername)) {
                    log.error("Got duplicate SQL result for user="+dusername+", role="+role);
                } else {
                    String dpassword = r.getString(2);
                    result.put(dusername, new AuthUser(dusername, dpassword, isAdmin));
                    if (log.isDebugEnabled())
                        log.debug("Got new user, user="+dusername+", passwd="+dpassword+", role="+role);
                }
            }
            return result;
        } finally {
            if (s != null)
                s.close();
            if (c != null)
                c.close();
        }
    }

    /**
     * Get all username keys in the DB.
     */
    public static Set<String> getAllUsernames()
        throws NamingException, SQLException
    {
        Connection c = null;
        PreparedStatement s = null;
        Set<String> result = new HashSet<String>();
        try {
            c = getConnection();
            s = c.prepareStatement("SELECT DISTINCT Username FROM Users");
            ResultSet r = s.executeQuery();
            while (r.next()) {
                result.add(r.getString(1));
            }
            log.debug("getAllUsernames found  count="+result.size());
            return result;
        } finally {
            if (s != null)
                s.close();
            if (c != null)
                c.close();
        }
    }

    /**
     * Hook to create a connection to manage one transaction.
     */
    public static Connection startTransaction()
        throws NamingException, SQLException
    {
        Connection c = getConnection();
        c.setAutoCommit(false);
        return c;
    }

    /**
     * Hook to commit and close the current transaction.
     */
    public static void commitTransaction(Connection c)
        throws NamingException, SQLException
    {
        log.debug("Committing current transaction..");
        c.commit();
        c.close();
    }

    /**
     * Hook to rollback and close the current transaction.
     */
    public static void abortTransaction(Connection c)
        throws NamingException, SQLException
    {
        c.rollback();
        c.close();
    }

    /**
     * Sanity-check content of username and password strings:
     * Username MUST not include ':', and (according to HTTP/1.1 MAY
     * include graphic chars in the ISO-8859-1 codeset, linear whitespace,
     * and other chars WITH special MIME RFC-2047 encoding rules.  Since
     * this is a PITA and is not likely to be implemented right by HTTP
     * clients anyway, we will further restrict usernames and passwords to:
     *  - letter or digit characters in Unicode C0 an C1 (basic latin, latin-1)
     *  - punctuation: ~, @, #, $, %, _, -, . (dot)
     *
     * Throw a runtime error upon failure.
     */
    private static void nameCheck(String s, String field)
    {
        for (int i = 0; i < s.length(); ++i) {
            char c = s.charAt(i);
            if (!((Character.isLetterOrDigit(c) && c < 0x100) ||
                  c == '~' || c == '@' || c == '#' || c == '$' ||
                  c == '%' || c == '_' || c == '-' || c == '.'))
                throw new IllegalArgumentException(field+" contains an illegal character: '"+c+"'");
        }
    }

    /**
     * Create a new auth user in the DB with indicated username, password,
     * and superuser status.  Always add a role of "authenticated", since
     * the web container security has to see *some* role to recognize the user.
     */
    public static void create(Connection c, String ausername, String apassword, boolean aisSuperuser)
        throws NamingException, SQLException
    {
        nameCheck(ausername, "Username");
        nameCheck(apassword, "Password");
        if (ausername.length() == 0)
            throw new IllegalArgumentException("Username may not be empty.");

        PreparedStatement iu = null;
        PreparedStatement ia = null;
        PreparedStatement ir = null;
        try {
            iu = c.prepareStatement("INSERT INTO Users (Username, Password) VALUES (?, ?)");
            iu.setString(1, ausername);
            iu.setString(2, apassword);
            if (iu.executeUpdate() != 1)
                throw new SQLException("INSERT INTO Users failed, row count incorrect.");
            ia = c.prepareStatement("INSERT INTO Roles (RoleName, Username) VALUES ('authenticated', ?)");
            ia.setString(1, ausername);
            if (ia.executeUpdate() != 1)
                throw new SQLException("INSERT INTO Roles failed, row count incorrect.");
            if (aisSuperuser) {
                ir = c.prepareStatement("INSERT INTO Roles (RoleName, Username) VALUES (?, ?)");
                ir.setString(1, Access.SUPERUSER_ROLE_NAME);
                ir.setString(2, ausername);
                if (ir.executeUpdate() != 1)
                    throw new SQLException("INSERT INTO Roles failed, row count incorrect.");
            }
            log.debug("Created Auth DB entries for user="+ausername);
        } finally {
            if (iu != null)
                iu.close();
            if (ia != null)
                ia.close();
            if (ir != null)
                ir.close();
        }
    }

    /**
     * Update auth DB entries if any values have changed.
     * This version creates its own transaction.
     */
    public void update()
        throws NamingException, SQLException
    {
        if (password.equals(oldPassword) && isSuperuser == oldIsSuperuser)
            return;

        Connection c = null;
        try {
            c = getConnection();
            c.setAutoCommit(false);
            update(c);
            c.commit();
        } finally {
            if (c != null)
                c.close();
        }
    }

    /**
     * Update auth DB entries if any values have changed.
     * Pass in a DB connection so it can share an external transaction
     * with other operations - create and update.
     */
    public void update(Connection c)
        throws NamingException, SQLException
    {
        if (password.equals(oldPassword) && isSuperuser == oldIsSuperuser)
            return;

        PreparedStatement iu = null;
        PreparedStatement ir = null;
        try {
            if (!password.equals(oldPassword)) {
                iu = c.prepareStatement("UPDATE Users SET Password = ? WHERE Username = ?");
                iu.setString(1, password);
                iu.setString(2, username);
                if (iu.executeUpdate() != 1)
                    throw new SQLException("UPDATE Users failed, row count incorrect.");
            }
            if (isSuperuser != oldIsSuperuser) {
                ir = c.prepareStatement(isSuperuser ?
                  "INSERT INTO Roles (Username, RoleName) VALUES (?, ?)" :
                  "DELETE FROM Roles WHERE (Username = ?) AND (rolename = ?)");
                ir.setString(1, username);
                ir.setString(2, Access.SUPERUSER_ROLE_NAME);
                if (ir.executeUpdate() != 1)
                    throw new SQLException("Update: Modification of Roles failed, row count incorrect.");
            }
            log.debug("Updated Auth DB entries for user="+username);
        } finally {
            if (iu != null)
                iu.close();
            if (ir != null)
                ir.close();
        }
    }

    /**
     * Remove user from the auth DB.
     */
    public void delete(Connection c)
        throws NamingException, SQLException
    {
        PreparedStatement du = null;
        PreparedStatement dr = null;
        try {
            dr = c.prepareStatement("DELETE FROM Roles WHERE Username = ?");
            du = c.prepareStatement("DELETE FROM Users WHERE Username = ?");
            du.setString(1, username);
            dr.setString(1, username);
            dr.executeUpdate();
            if (du.executeUpdate() != 1)
                throw new SQLException("Update: Modification of Users failed, row count incorrect.");
        } finally {
            if (du != null)
                du.close();
            if (dr != null)
                dr.close();
        }
    }

    /** Getters */
    public String getUsername()
    {
        return username;
    }
    public String getPassword()
    {
        return password;
    }
    public boolean isSuperuser()
    {
        return isSuperuser;
    }

    public void setPassword(String pw)
    {
        nameCheck(pw, "Password");
        password = pw;
    }

    // separate method to test password, in case it's encrypted or something.
    // returns true when authentication succeeds.
    public boolean authenticate(String pw)
    {
        return password != null && password.equals(pw);
    }

    public void setIsSuperuser(boolean su)
    {
        isSuperuser = su;
    }

    // get a jdbc Connection to the authentication RDBMS
    private static Connection getConnection()
        throws NamingException, SQLException
    {
        Context initContext = new InitialContext();
        Context envContext  = (Context)initContext.lookup("java:/comp/env");
        DataSource ds = (DataSource)envContext.lookup("jdbc/eaglei");
        return ds.getConnection();
    }
}
