package org.eaglei.repository.format;

import org.apache.commons.lang.StringEscapeUtils;

import java.io.IOException;
import java.io.Writer;
import java.io.BufferedWriter;
import java.nio.charset.Charset;
import java.util.Arrays;

import org.openrdf.rio.RDFFormat;
import org.openrdf.rio.RDFWriter;
import org.openrdf.rio.helpers.RDFHandlerBase;
import org.openrdf.rio.RDFHandlerException;
import org.openrdf.model.Statement;
import org.openrdf.model.Value;
import org.openrdf.model.URI;
import org.openrdf.model.Literal;
import org.openrdf.model.Resource;
import org.openrdf.model.BNode;

/**
 * RDF HTML writer
 *
 * About the Format:
 *  Writes a simple HTML table.  It is intended to let the human operator
 *  quickly evaluate the results of a graph query or "getStatements" from
 *  a clean tabular display.  It is NOT intended to be styled and presented.
 *
 *  This implementation is crude and ugly, but at least it has a small
 *  memory footprint and does not depend on building a DOM model like e.g.
 *  the more sophisticated JDOM output drivers.
 *
 *  TODO: add namespace abbreviations of URIs for more compact and readable URIs
 *
 * @author Larry Stone
 * Started May 24, 2010
 * @version $Id: $
 */
public class RDFHTMLWriter extends RDFHandlerBase implements RDFWriter
{
    /** Constant <code>MIME="text/html"</code> */
    public static final String MIME = "text/html";

    private BufferedWriter out = null;

    private static final String mimeTypes[] = { "text/html", "application/xhtml+xml" };
    private static final String fileExts[] = { ".html" };

    /** Constant <code>FORMAT</code> */
    public final static RDFFormat FORMAT =
        new RDFFormat("RDFHTML", Arrays.asList(mimeTypes), Charset.defaultCharset(),
                      Arrays.asList(fileExts), false, true);

    /**
     * <p>getRDFFormat</p>
     *
     * @return a {@link org.openrdf.rio.RDFFormat} object.
     */
    public RDFFormat getRDFFormat()
    {
        return FORMAT;
    }

    /**
     * <p>Constructor for RDFHTMLWriter.</p>
     *
     * @param out a {@link java.io.Writer} object.
     */
    public RDFHTMLWriter(Writer out)
    {
        super();
        this.out = out instanceof BufferedWriter ? ((BufferedWriter)out) :
                                                   new BufferedWriter(out);
    }

    /** {@inheritDoc} */
    public void handleStatement(Statement s)
        throws RDFHandlerException
    {
        try {
            out.write("<tr>");
            out.newLine();

            // subject
            handleResource(s.getSubject());

            // predicate
            handleResource(s.getPredicate());

            // value
            out.write("<td><tt>");
            SPARQLHTMLWriter.printValue(out, s.getObject());
            out.write("</tt></td>");
            out.write("</tr>");
            out.newLine();
        } catch (IOException e) {
            throw new RDFHandlerException(e);
        }
    }

    private void handleResource(Resource ss)
        throws IOException
    {
        out.write("<td><tt>");
        if (ss instanceof URI) {
            out.write(SPARQLHTMLWriter.renderLinkedURI((URI)ss));

        // XXX kludge: all blank nodes look the same; may cause false
        // positives in comparison, but at least no false negatives..
        } else {
            out.write(StringEscapeUtils.escapeHtml("_:"+((BNode)ss).getID()));
        }
        out.write("</tt></td>");
        out.newLine();
    }

    /**
     * <p>endRDF</p>
     *
     * @throws org.openrdf.rio.RDFHandlerException if any.
     */
    public void endRDF()
        throws RDFHandlerException
    {
        try {
            out.write(SPARQLHTMLWriter.Postlogue);
            out.newLine();
            out.close();
        } catch (IOException e) {
            // log.error("Failed closign output stream: ",e);
            throw new RDFHandlerException(e);
        }
    }

    /**
     * <p>startRDF</p>
     *
     * @throws org.openrdf.rio.RDFHandlerException if any.
     */
    public void startRDF()
        throws RDFHandlerException
    {
        try {
            out.write(SPARQLHTMLWriter.Prologue);
            out.newLine();
            out.write("<tr>"+
                        "<th>Subject</th>"+
                        "<th>Predicate</th>"+
                        "<th>Object</th>"+
                        "</tr>");
            out.newLine();
        } catch (IOException e) {
            // log.error("Failed closign output stream: ",e);
            throw new RDFHandlerException(e);
        }
    }
}
