package org.eaglei.repository.servlet;

import java.io.InputStream;
import java.io.InputStreamReader;
import java.io.StringReader;
import java.io.Reader;
import java.io.File;
import java.io.IOException;
import java.nio.charset.IllegalCharsetNameException;
import java.nio.charset.UnsupportedCharsetException;
import java.nio.charset.Charset;
import java.util.Arrays;
import java.util.List;
import java.util.Set;
import java.util.HashSet;

import javax.servlet.ServletConfig;
import javax.servlet.ServletException;
import javax.servlet.http.HttpServletRequest;
import javax.servlet.http.HttpServletResponse;

import org.apache.commons.fileupload.servlet.ServletFileUpload;
import org.apache.commons.fileupload.FileUploadException;
import org.apache.commons.fileupload.disk.DiskFileItemFactory;
import org.apache.commons.fileupload.disk.DiskFileItem;

import org.apache.log4j.Logger;
import org.apache.log4j.LogManager;

import org.openrdf.OpenRDFException;
import org.openrdf.model.URI;
import org.openrdf.model.impl.URIImpl;
import org.openrdf.rio.RDFFormat;
import org.openrdf.repository.Repository;
import org.openrdf.repository.RepositoryConnection;
import org.openrdf.repository.sail.SailRepository;
import org.openrdf.sail.memory.MemoryStore;
import org.openrdf.repository.RepositoryException;

import org.eaglei.repository.Access;
import org.eaglei.repository.Formats;
import org.eaglei.repository.User;
import org.eaglei.repository.Role;
import org.eaglei.repository.workflow.WorkflowTransition;
import org.eaglei.repository.View;
import org.eaglei.repository.status.BadRequestException;
import org.eaglei.repository.status.ForbiddenException;
import org.eaglei.repository.status.HttpStatusException;
import org.eaglei.repository.status.InternalServerErrorException;
import org.eaglei.repository.util.Utils;

/**
 * Structured export and import of resource instances OR user instances.
 * The same servlet implementation is called by both /export and /import,
 * they should set the init parameter "import" or "export" appropriately.
 * e.g.
 * <pre>
 * <servlet>
 *   <servlet-name>Export</servlet-name>
 *   <servlet-class>org.eaglei.repository.servlet.ImportExport</servlet-class>
 *   <init-param>
 *     <param-name>export</param-name>
 *     <param-value>true</param-value>
 *   </init-param>
 * </servlet>
 * </pre>
 *
 * @author Larry Stone
 * @version $Id: $
 */
public class ImportExport extends RepositoryServlet
{
    private static Logger log = LogManager.getLogger(ImportExport.class);

    // values of 'duplicate' arg - must be public for parseKeyword.
    public enum DuplicateArg { abort, ignore, replace };

    // values of 'graph' arg
    public enum NewGraphArg { abort, create };

    // values of 'type' arg
    public enum TypeArg { resource, user, role, transition, grant };

    // identify which kind of servlet this is since both can POST
    private boolean isImport = false;
    private boolean isExport = false;

    /**
     * {@inheritDoc}
     *
     * Configure this servlet as either import or export mode.
     */
    public void init(ServletConfig sc)
        throws ServletException
    {
        super.init(sc);

        // if this is not the export servlet, don't implement GET:
        isExport = sc.getInitParameter("export") != null;
        isImport = sc.getInitParameter("import") != null;

        if (!(isExport || isImport))
            log.error("Servlet was initialized without either import or export mode set, THIS IS BAD.");
    }

    /**
     * {@inheritDoc}
     *
     * GET the contents of a graph - for EXPORT only
     * Query Args:
     *  - format = MIME type (overrides content-type)
     *  - view = view to query (mutually excl. w/workspace)
     *  - workspace = workspace to query (mutually excl. w/workspace)
     *     (NOTE: default to view = USER for type=resource)
     *  - type=(resource|user) - what to import
     *  - include="URI ..." or "username ..." if type=user
     *  - exclude="URI ..." or "username ..." if type=user
     * Result: HTTP status 200 for success, 400 or 4xx or 5xx otherwise.
     */
    protected void doGet(HttpServletRequest request, HttpServletResponse response)
        throws ServletException, IOException
    {
        if (!isExport)
            throw new HttpStatusException(HttpServletResponse.SC_NOT_IMPLEMENTED, "GET is not implemented by this service");

        String rawFormat = null;
        String rawView = null;
        String rawWorkspace = null;
        String rawType = null;
        String include = null;
        String exclude = null;

        // if we got here through POST with multipart, grovel through args
        if (ServletFileUpload.isMultipartContent(request)) {
            try {
                ServletFileUpload upload = new ServletFileUpload();
                File tmp = (File)getServletConfig().getServletContext().getAttribute("javax.servlet.context.tempdir");
                if (tmp == null)
                    throw new InternalServerErrorException("Cannot find servlet context attr = \"javax.servlet.context.tempdir\"");
                upload.setFileItemFactory(new DiskFileItemFactory(100000, tmp));
                for (DiskFileItem item : (List<DiskFileItem>)upload.parseRequest(request)) {
                    String ifn = item.getFieldName();
                    if (ifn.equals("format"))
                        rawFormat = item.getString();
                    else if (ifn.equals("view"))
                        rawView = item.getString();
                    else if (ifn.equals("workspace"))
                        rawWorkspace = item.getString();
                    else if (ifn.equals("type"))
                        rawType = item.getString();
                    else if (ifn.equals("include"))
                        include = item.getString();
                    else if (ifn.equals("exclude"))
                        exclude = item.getString();
                    else
                        log.warn("Unrecoginized request argument: "+ifn);
                }
            } catch  (FileUploadException e) {
                log.error(e);
                throw new BadRequestException("failed parsing multipart request");
            }

        // gather args from input params instead
        } else {
            request.setCharacterEncoding("UTF-8");
            rawFormat = request.getParameter("format");
            rawView = request.getParameter("view");
            rawWorkspace = request.getParameter("workspace");
            rawType = request.getParameter("type");
            include = request.getParameter("include");
            exclude = request.getParameter("exclude");
        }

        // defaulting and sanity-checking
        TypeArg type = (TypeArg)Utils.parseKeywordArg(TypeArg.class, rawType, "type", true, null);
        View view = (View)Utils.parseKeywordArg(View.class, rawView, "view", false, null);
        URI workspace = null;
        if (rawWorkspace != null) {
            if (Utils.isValidURI(rawWorkspace))
                workspace = new URIImpl(rawWorkspace);
            else
                throw new BadRequestException("Workspace is not a valid URI: "+rawWorkspace);
        }

        // sanity check, then set default view if needed
        if (workspace != null && view != null)
            throw new BadRequestException("Only one of the 'workspace' or 'view' args may be specified.");
        else if (workspace == null && view == null)
            view = (type == TypeArg.resource ? View.USER_RESOURCES : View.USER);

        Set<String> includes = parseXCludeList(include);
        Set<String> excludes = parseXCludeList(exclude);
        if (log.isDebugEnabled()) {
            log.debug("INCLUDES = "+Arrays.deepToString(includes.toArray()));
            log.debug("EXCLUDES = "+Arrays.deepToString(excludes.toArray()));
        }

        // sanity check format, also must be a quad format
        String mimeType = Formats.negotiateRDFContent(request, rawFormat);
        RDFFormat format = Formats.RDFOutputFormatForMIMEType(mimeType);
        if (format == null) {
            throw new HttpStatusException(HttpServletResponse.SC_UNSUPPORTED_MEDIA_TYPE, "MIME type of serialized RDF is not supported: \""+mimeType+"\"");
        }
        if (!format.supportsContexts()) {
            throw new HttpStatusException(HttpServletResponse.SC_UNSUPPORTED_MEDIA_TYPE, "Format does not support quad (graph) encoding: "+format);
        }
        log.debug("Output serialization format = "+format);

        if (type == TypeArg.user) {
            if (!Access.isSuperuser(request))
                throw new ForbiddenException("Export of users requires administrator privileges.");
            response.setContentType(Utils.makeContentType(mimeType, "UTF-8"));
            User.doExportUsers( request, response, format, includes, excludes);

        } else if (type == TypeArg.role) {
            if (!Access.isSuperuser(request))
                throw new ForbiddenException("Export of roles requires administrator privileges.");
            response.setContentType(Utils.makeContentType(mimeType, "UTF-8"));
            Role.doExportRoles(request, response, format, includes, excludes);

        } else if (type == TypeArg.transition) {
            if (!Access.isSuperuser(request))
                throw new ForbiddenException("Export of roles requires administrator privileges.");
            response.setContentType(Utils.makeContentType(mimeType, "UTF-8"));
            WorkflowTransition.doExportTransitions(request, response, format, includes, excludes);

        } else if (type == TypeArg.grant) {
            if (!Access.isSuperuser(request))
                throw new ForbiddenException("Export of roles requires administrator privileges.");
            response.setContentType(Utils.makeContentType(mimeType, "UTF-8"));
            Access.doExportGrants(request, response, format, includes, excludes);

        } else {
            throw new HttpStatusException(HttpServletResponse.SC_NOT_IMPLEMENTED, "export of "+type+" not implemented.");
        }
    }

    /**
     * {@inheritDoc}
     *
     * Import a collection of Users *or* Instances from an export serialization
     * generated by this service.
     * POST is for IMPORT only.
     *
     * Args:
     *  - format = MIME type (overrides content-type)
     *  - duplicate=(abort|ignore|replace)
     *  - transform=yes|no - no default
     *  - graph=(abort|create) -- what to do when import would create new graph
     *  - type=(resource|user) - what to import
     *  - ignoreACL=yes|no - default=no, skip access grants on import (transition only)
     * Result: HTTP status 200 for success, 400 or 4xx or 5xx otherwise.
     */
    protected void doPost(HttpServletRequest request, HttpServletResponse response)
        throws ServletException, IOException
    {
        // GET implements the export function, even for a POST.
        if (isExport) {
            doGet(request, response);
            return;
        }
        if (!isImport)
            throw new HttpStatusException(HttpServletResponse.SC_NOT_IMPLEMENTED, "Servlet must be configured for import or export.");

        String rawFormat = null;
        String rawDuplicate = null;
        String rawTransform = null;
        String rawIgnoreACL = null;
        String rawNewGraph = null;
        String rawType = null;
        String include = null;
        String exclude = null;
        String contentType = null;
        InputStream contentStream = null;
        Reader content = null;

        // if we got here through POST with multipart, grovel through args
        if (ServletFileUpload.isMultipartContent(request)) {
            try {
                ServletFileUpload upload = new ServletFileUpload();
                File tmp = (File)getServletConfig().getServletContext().getAttribute("javax.servlet.context.tempdir");
                if (tmp == null)
                    throw new InternalServerErrorException("Cannot find servlet context attr = \"javax.servlet.context.tempdir\"");
                upload.setFileItemFactory(new DiskFileItemFactory(100000, tmp));
                for (DiskFileItem item : (List<DiskFileItem>)upload.parseRequest(request)) {
                    String ifn = item.getFieldName();
                    if (ifn.equals("format"))
                        rawFormat = item.getString();
                    else if (ifn.equals("duplicate"))
                        rawDuplicate = item.getString();
                    else if (ifn.equals("transform"))
                        rawTransform = item.getString();
                    else if (ifn.equals("ignoreACL"))
                        rawIgnoreACL = item.getString();
                    else if (ifn.equals("newgraph"))
                        rawNewGraph = item.getString();
                    else if (ifn.equals("type"))
                        rawType = item.getString();
                    else if (ifn.equals("include"))
                        include = item.getString();
                    else if (ifn.equals("exclude"))
                        exclude = item.getString();
                    else if (ifn.equals("content")) {
                        contentStream = item.getInputStream();
                        contentType = item.getContentType();
                        log.debug("Got content stream, MIME type = "+contentType);
                    } else
                        log.warn("Unrecoginized request argument: "+ifn);
                }
            } catch  (FileUploadException e) {
                log.error(e);
                throw new BadRequestException("failed parsing multipart request");
            }

        // gather args from input params instead
        } else {
            request.setCharacterEncoding("UTF-8");
            rawFormat = request.getParameter("format");
            rawDuplicate = request.getParameter("duplicate");
            rawTransform = request.getParameter("transform");
            rawIgnoreACL = request.getParameter("ignoreACL");
            rawNewGraph = request.getParameter("newgraph");
            rawType = request.getParameter("type");
            include = request.getParameter("include");
            exclude = request.getParameter("exclude");
            String cs = request.getParameter("content");
            if (cs != null)
                content = new StringReader(cs);
        }

        // defaulting and sanity-checking
        TypeArg type = (TypeArg)Utils.parseKeywordArg(TypeArg.class, rawType, "type", true, null);
        DuplicateArg duplicate = (DuplicateArg)Utils.parseKeywordArg(DuplicateArg.class, rawDuplicate, "duplicate", false, DuplicateArg.abort);
        NewGraphArg newGraph = (NewGraphArg)Utils.parseKeywordArg(NewGraphArg.class, rawNewGraph, "newgraph", false, NewGraphArg.abort);
        boolean transform = Utils.parseBooleanParameter(rawTransform, "transform", true, false);
        boolean ignoreACL = Utils.parseBooleanParameter(rawIgnoreACL, "ignoreACL", false, false);
        Set<String> includes = parseXCludeList(include);
        Set<String> excludes = parseXCludeList(exclude);

        // sanity check format
        if (rawFormat == null)
            rawFormat = contentType;
        if (rawFormat == null) {
            throw new BadRequestException("Missing required argument: format (or content-type on input)");
        }

        if (contentStream != null) {
            try {
                String csn = Utils.contentTypeGetCharset(rawFormat, "UTF-8");
                log.debug("Reading serialized RDF from stream with charset="+csn);
                content = new InputStreamReader(contentStream, Charset.forName(csn));
            } catch  (IllegalCharsetNameException e) {
                throw new BadRequestException("Illegal character set name in content-type spec: "+e);
            } catch  (UnsupportedCharsetException e) {
                throw new BadRequestException("Unsupported character set name in content-type spec: "+e);
            }
        } else if (content == null)
            throw new BadRequestException("Missing the required content argument.");

        String mime = Utils.contentTypeGetMIMEType(rawFormat);
        RDFFormat format = Formats.RDFOutputFormatForMIMEType(mime);
        if (format == null) {
            throw new HttpStatusException(HttpServletResponse.SC_UNSUPPORTED_MEDIA_TYPE, "MIME type of serialized RDF is not supported: \""+mime+"\"");
        }
        if (!format.supportsContexts()) {
            throw new HttpStatusException(HttpServletResponse.SC_UNSUPPORTED_MEDIA_TYPE, "Format does not support quad (graph) encoding: "+format);
        }
        // read document into memory repo so we can query it without polluting main repo
        log.debug("Input serialization format = "+format);
        Repository mr = null;
        RepositoryConnection mrc = null;
        try {
            mr = new SailRepository(new MemoryStore());
            mr.initialize();
            mrc = mr. getConnection();
            mrc.add(content, "", format);

            if (type == TypeArg.user) {
                if (!Access.isSuperuser(request))
                    throw new ForbiddenException("This request requires administrator privileges.");
                User.doImportUsers(request, response, mrc, includes,
                        excludes, duplicate, transform);
             
            } else if (type == TypeArg.role) {
                if (!Access.isSuperuser(request))
                    throw new ForbiddenException("This request requires administrator privileges.");
                Role.doImportRoles(request, response, mrc, includes,
                        excludes, duplicate, transform);

            } else if (type == TypeArg.transition) {
                if (!Access.isSuperuser(request))
                    throw new ForbiddenException("This request requires administrator privileges.");
                WorkflowTransition.doImportTransitions(request, response, mrc, includes,
                        excludes, duplicate, transform, ignoreACL);

            } else if (type == TypeArg.grant) {
                if (!Access.isSuperuser(request))
                    throw new ForbiddenException("This request requires administrator privileges.");
                Access.doImportGrants(request, response, mrc, includes,
                        excludes, duplicate, transform, ignoreACL);
             
            } else {
              throw new HttpStatusException(HttpServletResponse.SC_NOT_IMPLEMENTED, "import of "+type+" not implemented.");
            }

            WithRepositoryConnection.get(request).commit();
        } catch (OpenRDFException e) {
            log.error("Failed loading import content or calling import: ",e);
            throw new ServletException(e);
        } finally {
            try {
                if (mrc != null && mrc.isOpen()) {
                    mrc.close();
                }
                if (mr != null)
                    mr.shutDown();
            } catch (RepositoryException e) {
                log.error("Failed while closing temporary repo of import content: ",e);
                throw new ServletException(e);
            }
        }
    }


    // parse excludes or includes list - space separated URI or string values
    private Set<String> parseXCludeList(String s)
    {
        Set<String> result = new HashSet<String>();
        if (s != null) {
            for (String e : s.split("\\s*,\\s*"))
                result.add(e);
        }
        return result;
   }
}
