package org.eaglei.repository.servlet;

import java.io.File;
import java.util.ArrayList;
import java.util.List;

import javax.servlet.http.HttpServletRequest;
import javax.servlet.http.HttpServletResponse;
import javax.servlet.ServletException;

import org.apache.log4j.Logger;
import org.apache.log4j.LogManager;

import org.apache.commons.fileupload.servlet.ServletFileUpload;
import org.apache.commons.fileupload.FileUploadException;
import org.apache.commons.fileupload.disk.DiskFileItemFactory;
import org.apache.commons.fileupload.disk.DiskFileItem;

import org.openrdf.query.impl.MapBindingSet;
import org.openrdf.query.BindingSet;
import org.openrdf.model.URI;
import org.openrdf.model.Literal;
import org.openrdf.model.impl.LiteralImpl;
import org.openrdf.model.impl.BooleanLiteralImpl;

import org.eaglei.repository.Access;
import org.eaglei.repository.Role;
import org.eaglei.repository.User;
import org.eaglei.repository.vocabulary.FOAF;
import org.eaglei.repository.vocabulary.REPO;
import org.eaglei.repository.util.SPARQL;
import org.eaglei.repository.util.Utils;
import org.eaglei.repository.status.BadRequestException;
import org.eaglei.repository.status.ConflictException;

/**
 * Get status of current authenticated user as tabular result.
 * Returns an empty result when there is no authenticated user.
 *
 * @author Larry Stone
 * @version $Id: $
 */
public class WhoAmI extends RepositoryServlet
{
    private static Logger log = LogManager.getLogger(WhoAmI.class);

    /** {@inheritDoc} */
    protected void doGet(HttpServletRequest request, HttpServletResponse response)
        throws ServletException, java.io.IOException
    {
        request.setCharacterEncoding("UTF-8");

        // sanity check - create arg only allowed on POST
        if (request.getParameter("create") != null)
            throw new BadRequestException("'create' arg is only allowed with POST method");

        whoami(request, response, request.getParameter("format"));
    }

    /** {@inheritDoc} */
    protected void doPost(HttpServletRequest request, HttpServletResponse response)
        throws ServletException, java.io.IOException
    {
        // web request parameters:
        String rawcreate = null;
        String firstName = null;
        String lastName = null;
        String mbox = null;
        String format = null;

        // if POST with multipart, grovel through args
        if (ServletFileUpload.isMultipartContent(request)) {
            try {
                ServletFileUpload upload = new ServletFileUpload();
                upload.setFileItemFactory(new DiskFileItemFactory(100000,
                  (File)getServletConfig().getServletContext().getAttribute("javax.servlet.context.tempdir")));
                for (DiskFileItem item : (List<DiskFileItem>)upload.parseRequest(request)) {
                    String ifn = item.getFieldName();
                    if (ifn.equals("create"))
                        rawcreate = item.getString();
                    else if (ifn.equals("firstname"))
                        firstName = item.getString();
                    else if (ifn.equals("lastname"))
                        lastName = item.getString();
                    else if (ifn.equals("mbox"))
                        mbox = item.getString();
                    else if (ifn.equals("format"))
                        format = item.getString();
                    else
                        log.warn("Unrecoginized request argument: "+ifn);
                }
            } catch  (FileUploadException e) {
                log.error(e);
                throw new BadRequestException("failed parsing multipart request");
            }

        // gather args from input params instead
        } else {
            request.setCharacterEncoding("UTF-8");
            rawcreate = request.getParameter("create");
            firstName = request.getParameter("firstname");
            lastName = request.getParameter("lastname");
            mbox = request.getParameter("mbox");
            format = request.getParameter("format");
        }
        boolean create = Utils.parseBooleanParameter(rawcreate, "create", false, false);

        // make a new User if current authenticated user is "undocumented"
        if (create)
        {
            String login = Access.getAuthenticatedUsername(request);
            User u = Access.getPrincipalUser(request);
            if (login == null) {
                throw new BadRequestException("No authentication found.");

            // no metadata, so create new User
            } else if (u == null) {
                u = User.createAsAdministrator(request, login);
                if (Access.isSuperuser(request))
                    u.addRoleAsAdministrator(request, Role.find(request, REPO.ROLE_SUPERUSER));
                if (firstName != null)
                    u.setFirstName(request, firstName);
                if (lastName != null)
                    u.setLastName(request, lastName);
                if (mbox != null)
                    u.setMbox(request, mbox);
                u.update(request);
                response.setStatus(HttpServletResponse.SC_CREATED);

            // when there is already user metadata, flag a conflict
            } else {
                throw new ConflictException("There is already User metadata for login name = "+login);
            }
        } else {
            whoami(request, response, format);
        }
    }

    // return "whoami" results in response document
    private void whoami(HttpServletRequest request, HttpServletResponse response, String format)
        throws ServletException, java.io.IOException
    {
        ArrayList<BindingSet> result = new ArrayList<BindingSet>();
        MapBindingSet nbs = new MapBindingSet(6);
        User u = Access.getPrincipalUser(request);
        log.debug("authenticated User = "+u);
        if (u == null) {
            nbs.addBinding("uri", Access.getPrincipalURI(request));
            nbs.addBinding("username", null);
            nbs.addBinding("firstname", null);
            nbs.addBinding("lastname", null);
            nbs.addBinding("mbox", null);
        } else {
            nbs.addBinding("uri", u.getURI());
            nbs.addBinding("username", new LiteralImpl(u.getUsername()));
            nbs.addBinding("firstname", u.getFirstName() == null ? null : new LiteralImpl(u.getFirstName()));
            nbs.addBinding("lastname", u.getLastName() == null ? null : new LiteralImpl(u.getLastName()));
            nbs.addBinding("mbox", u.getMbox() == null ? null : new LiteralImpl(u.getMbox()));
        }
        nbs.addBinding("isSuperuser", Access.isSuperuser(request) ?
                       BooleanLiteralImpl.TRUE : BooleanLiteralImpl.FALSE);
        result.add(nbs);
        SPARQL.sendTupleQueryResults(request, response, format, result);
    }
}
