package org.eaglei.repository.servlet;

import org.eaglei.repository.*;
import javax.servlet.Filter;
import javax.servlet.FilterConfig;
import javax.servlet.FilterChain;
import javax.servlet.ServletRequest;
import javax.servlet.ServletResponse;
import javax.servlet.ServletException;

import java.io.IOException;

import org.openrdf.repository.RepositoryConnection;
import org.openrdf.repository.RepositoryException;

import org.apache.log4j.Logger;
import org.apache.log4j.LogManager;

/**
 * Give wrapped servlet a Sesame repository connection with
 * automatic cleanup. Opens a new connection for each request
 * and ensures it is closed at the end.
 *
 * WARNING: It is the servlet implementation's responsibility to
 * commit() all changes made on the RepositoryConnection.  This
 * filter ONLY closes the connection (with a rollback if necessary), it
 * does NOT commit anything, since it might regain control after an
 * exception, in which case you wouldn't want any partial changes committed.
 *
 *
 *   XXX FIXME MAYBE: Is it worth using a pool of RepositoryConnections
 *    for efficiency?  Maybe they are just too cheap to set up. Need to
 *    check if it is safe to reuse a connection after commit/rollback too.
 *    see http://commons.apache.org/pool/
 *
 * @author Larry Stone
 * @version $Id: $
 */
public class WithRepositoryConnection implements Filter
{
    private static Logger log = LogManager.getLogger(WithRepositoryConnection.class);

    private static final String ATTR_CONNECTION = "org.eaglei.repository.RepositoryConnection";

    /** {@inheritDoc} */
    public void init(FilterConfig fc)
        throws ServletException
    {}
    /**
     * <p>destroy</p>
     */
    public void destroy()
    {}

    /**
     * Gets RepositoryConnection from where the filter stashed it.
     *
     * @param request a {@link javax.servlet.ServletRequest} object.
     * @return a {@link org.openrdf.repository.RepositoryConnection} object.
     */
    public static RepositoryConnection get(ServletRequest request)
    {
        return (RepositoryConnection)request.getAttribute(ATTR_CONNECTION);
    }

    /** {@inheritDoc} */
    public void doFilter(ServletRequest req, ServletResponse resp, FilterChain fc)
        throws IOException, ServletException
    {
        RepositoryConnection rc = null;
        try {
            try {
                rc = DataRepository.getInstance().getSesameRepository().getConnection();
                rc.setAutoCommit(false);
                req.setAttribute(ATTR_CONNECTION, rc);
                log.debug("Providing RepositoryConnection = "+rc);
                fc.doFilter(req, resp);

            } finally {

                /***
                 * IMPORTANT NOTE:
                 * It is the servlet implementation's responsibility to
                 * commit() all changes.  If we get here, it coudl be
                 * from a failure or exception so we should NOT commit.
                 * The rollback below prevents Sesame from complaining when
                 * you close() a connection that has pending changes.
                 */

                if (rc != null) {
                    if (rc.isOpen()) {
                        log.debug("Closing RepositoryConnection = "+rc);
                        rc.rollback();
                        rc.close();
                    } else {
                        log.warn("RepositoryConnection was already closed! rc="+rc);
                    }
                }
                req.setAttribute(ATTR_CONNECTION, null);
            }
        } catch (RepositoryException e) {
            throw new ServletException(e);
        }
    }
}
