package org.eaglei.repository.servlet;

import java.io.File;
import java.util.ArrayList;
import java.util.Arrays;
import java.util.Collection;
import java.util.List;
import javax.servlet.http.HttpServletRequest;
import javax.servlet.http.HttpServletResponse;
import javax.servlet.ServletException;

import org.apache.log4j.Logger;
import org.apache.log4j.LogManager;
import org.apache.commons.fileupload.servlet.ServletFileUpload;
import org.apache.commons.fileupload.FileUploadException;
import org.apache.commons.fileupload.disk.DiskFileItemFactory;
import org.apache.commons.fileupload.disk.DiskFileItem;

import org.openrdf.query.impl.MapBindingSet;
import org.openrdf.query.BindingSet;
import org.openrdf.model.URI;
import org.openrdf.model.Literal;
import org.openrdf.model.impl.URIImpl;
import org.openrdf.model.impl.LiteralImpl;
import org.openrdf.model.impl.BooleanLiteralImpl;

import org.eaglei.repository.Access;
import org.eaglei.repository.AccessGrant;
import org.eaglei.repository.workflow.WorkflowTransition;
import org.eaglei.repository.vocabulary.REPO;
import org.eaglei.repository.util.SPARQL;
import org.eaglei.repository.util.Utils;
import org.eaglei.repository.status.BadRequestException;
import org.eaglei.repository.status.InternalServerErrorException;

/**
 * Returns a tabular listing of available workflow transitions.
 * The 'format' arg selects one of the tuple query result formats.
 * Optional 'workspace' arg selects only transitions that apply to a given
 * workspace (which includes wildcards).
 * @see WorkflowTransition
 * @author Larry Stone
 * @version $Id: $
 * Started October 20, 2010
 */
public class WorkflowTransitions extends RepositoryServlet
{
    private static Logger log = LogManager.getLogger(WorkflowTransitions.class);

    /** {@inheritDoc} */
    protected void doPost(HttpServletRequest request, HttpServletResponse response)
        throws ServletException, java.io.IOException
    {
        doGet(request, response);
    }

    /** {@inheritDoc} */
    protected void doGet(HttpServletRequest request, HttpServletResponse response)
        throws ServletException, java.io.IOException
    {
        request.setCharacterEncoding("UTF-8");
        String format = null;
        String rawWorkspace = null;

        // if we got here through POST with multipart, grovel through args
        if (ServletFileUpload.isMultipartContent(request)) {
            try {
                ServletFileUpload upload = new ServletFileUpload();
                File tmp = (File)getServletConfig().getServletContext().getAttribute("javax.servlet.context.tempdir");
                if (tmp == null)
                    throw new InternalServerErrorException("Cannot find servlet context attr = \"javax.servlet.context.tempdir\"");
                upload.setFileItemFactory(new DiskFileItemFactory(100000, tmp));
                for (DiskFileItem item : (List<DiskFileItem>)upload.parseRequest(request)) {
                    String ifn = item.getFieldName();
                    if (ifn.equals("workspace"))
                        rawWorkspace = item.getString();
                    else if (ifn.equals("format"))
                        format = item.getString();
                    else
                        log.warn("Unrecoginized request argument: "+ifn);
                }
            } catch  (FileUploadException e) {
                log.error(e);
                throw new BadRequestException("failed parsing multipart request");
            }

        // gather args from input params instead
        } else {
            format = request.getParameter("format");
            rawWorkspace = request.getParameter("workspace");
        }

        URI workspace = null;
        if (rawWorkspace != null) {
            if (Utils.isValidURI(rawWorkspace))
                workspace = new URIImpl(rawWorkspace);
            else
                throw new BadRequestException("Workspace is not a valid URI: "+rawWorkspace);
        }
        log.debug("workspace parameter = "+workspace);

        ArrayList<BindingSet> result = new ArrayList<BindingSet>();
        for (WorkflowTransition t : WorkflowTransition.findAll(request)) {
            URI tw = t.getWorkspace();
            if (workspace == null || tw == null || tw.equals(workspace)) {
                URI subject = t.getURI();
                boolean allowed = Access.hasPermission(request, subject, Access.READ);
                MapBindingSet nbs = new MapBindingSet(10);
                nbs.addBinding("subject", subject);
                nbs.addBinding("label", new LiteralImpl(t.getLabel()));
                nbs.addBinding("initial", t.getInitial());
                nbs.addBinding("initialLabel", new LiteralImpl(t.getInitialLabel()));
                nbs.addBinding("final", t.getFinal());
                nbs.addBinding("finalLabel", new LiteralImpl(t.getFinalLabel()));
                String comment = t.getComment();
                nbs.addBinding("description", comment == null ? null : new LiteralImpl(comment));
                nbs.addBinding("workspace", tw);
                String wl = t.getWorkspaceLabel();
                nbs.addBinding("workspaceLabel", wl == null ? null : new LiteralImpl(wl));
                nbs.addBinding("allowed", BooleanLiteralImpl.valueOf(allowed));
                result.add(nbs);
            } else {
                log.debug("Skipping because of WORKSPACE restriction: "+t);
            }
        }
        SPARQL.sendTupleQueryResults(request, response, format, result);
    }
}
