package org.eaglei.repository.util;

import org.eaglei.repository.vocabulary.REPO;
import org.eaglei.repository.*;
import java.io.IOException;
import java.util.Arrays;
import java.util.ArrayList;
import java.util.Iterator;
import java.util.Set;

import org.apache.log4j.Logger;
import org.apache.log4j.LogManager;

import javax.servlet.http.HttpServletRequest;
import javax.servlet.http.HttpServletResponse;
import javax.servlet.ServletException;

import org.openrdf.OpenRDFException;
import org.openrdf.repository.RepositoryConnection;
import org.openrdf.model.URI;
import org.openrdf.model.vocabulary.RDF;
import org.openrdf.model.vocabulary.RDFS;
import org.openrdf.model.vocabulary.OWL;
import org.openrdf.query.resultio.TupleQueryResultFormat;
import org.openrdf.query.resultio.QueryResultIO;
import org.openrdf.query.TupleQueryResultHandler;
import org.openrdf.query.BindingSet;
import org.openrdf.query.Dataset;
import org.openrdf.query.impl.DatasetImpl;
import org.openrdf.query.TupleQuery;
import org.openrdf.query.QueryLanguage;
import org.openrdf.query.MalformedQueryException;

import org.eaglei.repository.servlet.WithRepositoryConnection;
import org.eaglei.repository.status.InternalServerErrorException;

/**
 * Constants and utilities helpful in building SPARQL queries.
 * Also houses the tuple-result output formatter.
 *
 * @author Larry Stone
 * @version $Id: $
 */
public class SPARQL
{
    private static Logger log = LogManager.getLogger(SPARQL.class);

    private SPARQL()
    {
    }

    // dataset for ACL queries - just internal graph and repo ontology
    /** Constant <code>InternalGraphs</code> */
    public static final DatasetImpl InternalGraphs = new DatasetImpl();
    static {
        addGraph(InternalGraphs,REPO.NAMESPACE_URI);
        addGraph(InternalGraphs,REPO.NG_INTERNAL);
    }

    // cache of dataset for queries involving metadata including User entries
    // should include internal, repo-onto, and ALL NG's of type Metadata..
    // must be initialized dynamically.
    private static DatasetImpl InternalAndMetadataGraphs = null;

    /**
     * <p>Get a Dataset including all Internal and Metadata named graphs</p>
     *
     * @param request a {@link javax.servlet.http.HttpServletRequest} object.
     * @return a {@link org.openrdf.query.Dataset} object.
     */
    public static Dataset getInternalAndMetadataGraphs(HttpServletRequest request)
    {
        if (InternalAndMetadataGraphs == null) {
            InternalAndMetadataGraphs = copyDataset(InternalGraphs);
            View.addGraphs(request, InternalAndMetadataGraphs, View.METADATA);
        }
        return InternalAndMetadataGraphs;
    }

    /**
     * <p>Create copy of given Dataset with same default- and named-graph contents</p>
     *
     * @param src the source {@link org.openrdf.query.Dataset} object.
     * @return a new {@link org.openrdf.query.impl.DatasetImpl} object.
     */
    public static DatasetImpl copyDataset(Dataset src)
    {
        DatasetImpl result = new DatasetImpl();
        for (URI g : src.getDefaultGraphs())
            result.addDefaultGraph(g);
        for (URI g : src.getNamedGraphs())
            result.addNamedGraph(g);
        return result;
    }

    /**
     * Add a single named graph explicitly to the dataset.  Used to
     * create custom dataset, or extend a View.  Add to both default
     * graph and named graphs to facilitate use of "GRAPH ?g" expressions.
     *
     * @param ds desintation dataset, a {@link org.openrdf.query.impl.DatasetImpl} object.
     * @param graphName the graph to add, a {@link org.openrdf.model.URI} object.
     */
    public static void addGraph(DatasetImpl ds, URI graphName)
    {
        ds.addDefaultGraph(graphName);
        ds.addNamedGraph(graphName);
    }

    /**
     * Generic code to implement a Servlet body whose main purpose is to
     * execute a tabular (Select) SPARQL query and return the results.
     * This encapsulates all the machinery to run the query, handle errors,
     * and encode the results in an acceptable format.
     *
     * @param request a {@link javax.servlet.http.HttpServletRequest} object.
     * @param response a {@link javax.servlet.http.HttpServletResponse} object.
     * @param query SPARQL query as a {@link java.lang.String} object.
     * @param format output format as MIME type, a {@link java.lang.String} object.
     * @param ds dataset for query, a {@link org.openrdf.query.Dataset} object.
     * @throws javax.servlet.ServletException if any.
     * @throws java$io$IOException if any.
     */
    public static void tupleQueryRequest(HttpServletRequest request, HttpServletResponse response,
                                    String query, String format, Dataset ds)
        throws ServletException, java.io.IOException
    {
        RepositoryConnection rc = WithRepositoryConnection.get(request);

        //  Use prettyPrint because ds.toString() breaks when a default graph is null
        //  which is possible for the 'null' view.  don't ask, it's ugly.
        if (log.isDebugEnabled()) {
            log.debug("Trying SPARQL query = "+query);
            log.debug("Dataset for SPARQL query = "+Utils.prettyPrint(ds));
        }
        TupleQuery q = null;
        try {
            q = rc.prepareTupleQuery(QueryLanguage.SPARQL, query);
        } catch (MalformedQueryException e) {
            log.info("Rejecting malformed query.");
            throw new InternalServerErrorException("Malformed query was generated internally: "+e.toString());
        } catch (OpenRDFException e) {
            log.error(e);
            throw new ServletException(e);
        }

        q.setIncludeInferred(true);
        q.setDataset(ds);

        String mimeType = Formats.negotiateTupleContent(request, format);
        TupleQueryResultFormat tqf = QueryResultIO.getWriterFormatForMIMEType(mimeType);
        if (tqf == null) {
            log.error("Failed to get tuple query format, for mime="+mimeType);
            throw new ServletException("Failed to get tuple query format that SHOULD have been available, for mime="+mimeType);
        } else {
            response.setContentType(Utils.makeContentType(mimeType, "UTF-8"));
            try {
                q.evaluate(QueryResultIO.createWriter(tqf, response.getOutputStream()));
            } catch (OpenRDFException e) {
                log.error(e);
                throw new ServletException(e);
            }
        }
    }

    /**
     * Sends HTTP result document for a list of artificially-generated
     * tuple-query results.  This lets anything that generates tabular
     * data leverage the Sesame query result infrastructure to format
     * results in a wide variety of formats, and likewise the recipient
     * can use their favorite result parser.
     *
     * @param request a {@link javax.servlet.http.HttpServletRequest} object.
     * @param response a {@link javax.servlet.http.HttpServletResponse} object.
     * @param format output format as MIME type, a {@link java.lang.String} object.
     * @param bindings tuple result bindings as a {@link java.lang.Iterable} object.
     * @throws javax.servlet.ServletException if any.
     * @throws java$io$IOException if any.
     */
    public static void sendTupleQueryResults(HttpServletRequest request, HttpServletResponse response,
                                      String format, Iterable<BindingSet> bindings)
        throws ServletException, IOException
    {
        // synthesize a SPARQL tuple query response by creating a result
        // writer and driving it directly with a made up binding listing:
        String mimeType = Formats.negotiateTupleContent(request, format);
        TupleQueryResultFormat tqf = QueryResultIO.getWriterFormatForMIMEType(mimeType);
        if (tqf == null) {
            log.error("Failed to get tuple query format, for mime="+mimeType);
            throw new ServletException("Failed to get tuple query format that SHOULD have been available, for mime="+mimeType);
        } else {
            response.setContentType(Utils.makeContentType(mimeType, "UTF-8"));
            try {
                TupleQueryResultHandler h = QueryResultIO.createWriter(tqf, response.getOutputStream());
                Iterator<BindingSet> bi = bindings.iterator();
                if (bi.hasNext()) {
                    BindingSet first = bi.next();
                    Set<String> bns = first.getBindingNames();
                    h.startQueryResult(Arrays.asList(bns.toArray(new String[bns.size()])));
                    h.handleSolution(first);
                    while (bi.hasNext()) {
                        h.handleSolution(bi.next());
                    }
                    h.endQueryResult();

                // best we can do when there are no bindings..
                } else {
                    h.startQueryResult(new ArrayList(0));
                    h.endQueryResult();
                }
            } catch (OpenRDFException e) {
                log.error(e);
                throw new ServletException(e);
            }
        }
    }
}
