package org.eaglei.repository.workflow;

import java.util.List;
import javax.servlet.http.HttpServletRequest;

import org.apache.log4j.Logger;
import org.apache.log4j.LogManager;

import org.openrdf.OpenRDFException;
import org.openrdf.model.URI;
import org.openrdf.model.URI;
import org.openrdf.model.Literal;
import org.openrdf.model.impl.LiteralImpl;
import org.openrdf.model.impl.NumericLiteralImpl;
import org.openrdf.model.impl.URIImpl;
import org.openrdf.model.Resource;
import org.openrdf.model.Statement;
import org.openrdf.model.Value;
import org.openrdf.repository.RepositoryConnection;
import org.openrdf.repository.RepositoryException;
import org.openrdf.model.vocabulary.RDF;
import org.openrdf.model.vocabulary.RDFS;
import org.openrdf.model.vocabulary.OWL;
import org.openrdf.model.vocabulary.XMLSchema;
import org.openrdf.model.ValueFactory;
import org.openrdf.repository.RepositoryResult;

import org.eaglei.repository.servlet.WithRepositoryConnection;
import org.eaglei.repository.status.NotFoundException;
import org.eaglei.repository.status.BadRequestException;
import org.eaglei.repository.status.ConflictException;
import org.eaglei.repository.status.ForbiddenException;
import org.eaglei.repository.status.InternalServerErrorException;

public class ActionMoveGraph implements WorkflowAction
{
    private static Logger log = LogManager.getLogger(ActionMoveGraph.class);

    // move resource's properties to new named graph
    // (i.e. to implement different access control and query response)
    public void onTransition(HttpServletRequest request, URI resource, Value param)
        throws Exception
    {
        if (param == null || !(param instanceof URI))
            throw new BadRequestException("ActionMoveGraph requires a valid URI parameter, naming the destination graph.");
        URI newGraph = (URI)param;
        URI homeGraph = null;
        RepositoryConnection rc = WithRepositoryConnection.get(request);

        // 1. get resource's home graph, ideally where its *asserted*
        //    rdf:type statements live - avoid inferred types.
        RepositoryResult<Statement> rr = null;
        try {
            rr = rc.getStatements(resource, RDF.TYPE, null, false);
            while (rr.hasNext()) {
                Statement s = rr.next();
                Resource ctx = s.getContext();
                log.debug("Found statement: "+resource+" rdf:type "+s.getObject()+", in graph "+ctx);
                if (ctx instanceof URI) {
                    homeGraph = (URI)ctx;
                    break;
                }
            }
        } finally {
            rr.close();
        }
        if (homeGraph == null) {
            log.error("Failed to find home graph for resource instance="+resource);
            throw new InternalServerErrorException("Failed to find home graph for resource instance="+resource);
        } else if (homeGraph.equals(newGraph)) {
            log.warn("Tried to move resource inst="+resource+" to the same named graph: "+homeGraph);
            return;
        }

        // 2. Move all statements to new graph.
        //   Need to create a List because removing them might
        //   mess up RepoResult's lazy evaluation.
        List<Statement> stms = null;
        try {
            rr = rc.getStatements(resource, null, null, false, homeGraph);
            stms = rr.asList();
        } finally {
            rr.close();
        }
        for (Statement s : stms) {
            rc.remove(s, homeGraph);
            rc.add(s, newGraph);
        }
    }
}
