package org.eaglei.repository.workflow;

import java.util.ArrayList;
import java.util.Collection;
import java.util.List;
import java.util.Map;
import java.util.HashMap;

import javax.servlet.ServletException;
import javax.servlet.http.HttpServletRequest;

import org.apache.log4j.Logger;
import org.apache.log4j.LogManager;

import org.openrdf.OpenRDFException;
import org.openrdf.model.URI;
import org.openrdf.model.Literal;
import org.openrdf.model.Value;
import org.openrdf.repository.RepositoryConnection;
import org.openrdf.model.vocabulary.RDFS;
import org.openrdf.query.TupleQuery;
import org.openrdf.query.TupleQueryResultHandlerBase;
import org.openrdf.query.TupleQueryResultHandlerException;
import org.openrdf.query.BindingSet;
import org.openrdf.query.QueryLanguage;
import org.openrdf.query.MalformedQueryException;

import org.eaglei.repository.HasContentCache;
import org.eaglei.repository.vocabulary.REPO;
import org.eaglei.repository.util.SPARQL;
import org.eaglei.repository.status.NotFoundException;
import org.eaglei.repository.servlet.WithRepositoryConnection;

/**
 * Workflow WorkflowState object model, reflects the :WorkflowState object in RDF database.
 * This is a READ ONLY model.  The objects are IMMUTABLE.   Of course, this
 * might change if we add an admin UI to add workflow states, but right
 * now that seems unlikely.
 *
 * @author Larry Stone
 * Started April 26, 2010
 * @version $Id: $
 */
@HasContentCache
public class WorkflowState implements Comparable
{
    private static Logger log = LogManager.getLogger(WorkflowState.class);

    private URI uri;                  // subject: required
    private String label = null;      // value of rdfs:label
    private String comment = null;    // value of rdfs:comment

    // Caches, so we don't keep allocating new States - key is URI
    private static volatile Map<URI,WorkflowState> cacheInternal = null;

    // cache of sorted list to return from findAll()
    private static volatile List<WorkflowState> cacheListInternal = null;

    // SPARQL Query to get all States and labels..
    private final static String stateQuery =
          "SELECT DISTINCT * WHERE { "+
          " ?uri a <"+REPO.WORKFLOW_STATE+"> ; \n"+
          "  <"+RDFS.LABEL+"> ?label ; "+
          "  <"+REPO.ORDER+"> ?order \n"+
          " OPTIONAL { ?uri <"+RDFS.COMMENT+"> ?comment }"+
          "} ORDER BY ?order";

    private WorkflowState(URI uri, String label, String comment)
    {
        super();
        this.uri = uri;
        this.label = label;
        this.comment = comment;
    }

    /**
     * Get the WorkflowState for given URI, lookup label.
     *
     * @param request a {@link javax.servlet.http.HttpServletRequest} object.
     * @param uri URI of the state
     * @return a {@link org.eaglei.repository.WorkflowState} object
     * @throws javax.servlet.ServletException if any.
     */
    public static WorkflowState find(HttpServletRequest request, URI uri)
        throws ServletException
    {
        Map<URI,WorkflowState> cache = getCache(request);
        if (cache.containsKey(uri)) {
            if (log.isDebugEnabled())
                log.debug("State.find("+uri.stringValue()+") => "+cache.get(uri));
            return cache.get(uri);
        } else
            throw new NotFoundException("There is no State of URI="+uri);
    }

    /**
     * Get all known States -- we expect them to be stable, hence read-only.
     *
     * @param request a {@link javax.servlet.http.HttpServletRequest} object.
     * @return all states in a {@link java.lang.Iterable} object.
     * @throws javax.servlet.ServletException if any.
     */
    public static Collection<WorkflowState> findAll(HttpServletRequest request)
        throws ServletException
    {
        return getCacheList(request);
    }

    /**
     * Invalidate any in-memory cache of RDF data.
     */
    public static void decache()
    {
        synchronized (WorkflowState.class) {
            cacheInternal = null;
        }
    }

    // return cached map of states
    private static Map<URI,WorkflowState> getCache(HttpServletRequest request)
        throws ServletException
    {
        // KLUDGE: this bit of awkwardness is to prevent entering a
        // Sesame transaction within a sync block.
        synchronized (WorkflowState.class) {
            if (cacheInternal != null)
                return cacheInternal;
            cacheInternal = new HashMap<URI,WorkflowState>();
            cacheListInternal = new ArrayList<WorkflowState>();
        }
        try {
            RepositoryConnection rc = WithRepositoryConnection.get(request);
            log.debug("All State SPARQL query = "+stateQuery);
            TupleQuery q = rc.prepareTupleQuery(QueryLanguage.SPARQL, stateQuery);
            q.setDataset(SPARQL.InternalGraphs);

            // don't need inference!
            q.setIncludeInferred(false);
            q.evaluate(new allStateHandler(cacheInternal, cacheListInternal));
        } catch (MalformedQueryException e) {
            log.error("Rejecting malformed query:"+e);
            throw new ServletException(e);
        } catch (OpenRDFException e) {
            log.error(e);
            throw new ServletException(e);
        }
        return cacheInternal;
    }

    // return cached, sorted list of states
    private static List<WorkflowState> getCacheList(HttpServletRequest request)
        throws ServletException
    {
        getCache(request);
        return cacheListInternal;
    }

    /**
     * <p>getURI</p>
     *
     * @return URI subject of the WorkflowState, a {@link org.openrdf.model.URI} object.
     */
    public URI getURI()
    {
        return uri;
    }

    /**
     * <p>Getter for the field <code>label</code>.</p>
     *
     * @return label of the WorkflowState, a {@link java.lang.String} object.
     */
    public String getLabel()
    {
        return label;
    }

    /**
     * <p>Getter for the field <code>comment</code>.</p>
     *
     * @return comment of the WorkflowState, a {@link java.lang.String} object, MIGHT be null.
     */
    public String getComment()
    {
        return comment;
    }

    /** {@inheritDoc} */
    public boolean equals(Object other)
    {
        return other instanceof WorkflowState && uri.equals(((WorkflowState)other).uri);
    }

    /** {@inheritDoc} */
    public int hashCode()
    {
        return uri.hashCode();
    }

    /**
     * <p>toString</p>
     *
     * @return a {@link java.lang.String} object.
     */
    public String toString()
    {
        return "<#State: uri="+uri.toString()+
            ", label="+label+">";
    }

    public int compareTo(Object o)
    {
        return label.compareTo(((WorkflowState)o).label);
    }

    // tuple query result handler to gather list of all States
    private static class allStateHandler extends TupleQueryResultHandlerBase
    {
        private Map<URI,WorkflowState> result = null;
        private List<WorkflowState> resultList = null;

        public allStateHandler(Map<URI,WorkflowState> result, List<WorkflowState> resultList)
        {
            super();
            allStateHandler.this.result = result;
            allStateHandler.this.resultList = resultList;
        }

        public void handleSolution(BindingSet bs)
            throws TupleQueryResultHandlerException
        {
            Value newURI = bs.getValue("uri");
            if (newURI == null || !(newURI instanceof URI))
                log.error("Should not get null or non-URI result in allStateHandler: "+newURI);
            else {
                URI u = (URI)newURI;
                Value rlabel = bs.getValue("label");
                String label = (rlabel != null && rlabel instanceof Literal) ?
                           ((Literal)rlabel).getLabel() : "";
                Value rcomment = bs.getValue("comment");
                String comment = null;
                if (rcomment != null && rcomment instanceof Literal)
                    comment = ((Literal)rcomment).getLabel();
                WorkflowState s = new WorkflowState(u, label, comment);
                result.put(u, s);
                resultList.add(s);
            }
        }
    }
}
