package org.eaglei.repository;

import org.eaglei.repository.vocabulary.REPO;
import org.eaglei.repository.util.SPARQL;
import java.util.ArrayList;
import java.util.List;
import java.util.Collection;
import java.util.HashMap;
import java.util.Map;
import javax.servlet.ServletException;
import javax.servlet.http.HttpServletRequest;
import org.apache.log4j.Logger;
import org.apache.log4j.LogManager;

import org.openrdf.OpenRDFException;
import org.openrdf.model.URI;
import org.openrdf.model.Literal;
import org.openrdf.model.impl.LiteralImpl;
import org.openrdf.model.Resource;
import org.openrdf.model.Value;
import org.openrdf.repository.RepositoryConnection;
import org.openrdf.repository.RepositoryResult;
import org.openrdf.model.vocabulary.RDF;
import org.openrdf.model.vocabulary.RDFS;
import org.openrdf.model.vocabulary.XMLSchema;
import org.openrdf.query.TupleQuery;
import org.openrdf.query.TupleQueryResultHandlerBase;
import org.openrdf.query.TupleQueryResultHandlerException;
import org.openrdf.query.BindingSet;
import org.openrdf.query.QueryLanguage;
import org.openrdf.query.MalformedQueryException;

import org.eaglei.repository.servlet.WithRepositoryConnection;

/**
 * Named Graph object model, reflects the named graph's properties
 *
 * @author Larry Stone
 * Started June 7, 2010
 * @version $Id: $
 */
public class NamedGraph implements Comparable<NamedGraph>
{
    private static Logger log = LogManager.getLogger(NamedGraph.class);

    /**
     * The allowable named graph types.  Although the type itself
     * is part of the repository's internal ontology and thus appears
     * that it might be easily changed, in reality, the semantics of
     * named graph types are hardcoded that if they are changed or
     * extended, updating this enum is the least of the work required.
     * The enum's symbolic name is how the named graph type appears in
     * the API, at least, where a keyword is used instead of the URI.
     */
    public enum Type
    {
        ontology        (REPO.NGTYPE_ONTOLOGY),
        metadata        (REPO.NGTYPE_METADATA),
        workspace       (REPO.NGTYPE_WORKSPACE),
        published       (REPO.NGTYPE_PUBLISHED),
        internal        (REPO.NGTYPE_INTERNAL);

        private URI uri = null;

        private Type (URI u) {
            uri = u;
        }

        /** Accessor */
        public URI getURI()
        {
            return uri;
        }

        /** Interpret string as one of the enumerated type names */
        public static Type parse(String v)
        {
            try {
                return valueOf(v);
            } catch (IllegalArgumentException e) {
                return null;
            }
        }

        /** Interpret URI as one of the enumerated type names */
        public static Type parse(URI v)
        {
            for (Type t : values()) {
                if (t.uri.equals(v))
                    return t;
                }
            return null;
        }

        /** Get pretty title to include in e.g. a menu */
        public String getTitle()
        {
            String title = toString();
            return String.format("%C%s", title.charAt(0), title.substring(1));
        }
    }

    // name of request attribute where cached table of NGs is kept
    private static final String R_NG_MAP = "org.eaglei.repository.NamedGraph.Map";

    // flag: true when cache map contains all graphs
    private static final String R_NG_ALL = "org.eaglei.repository.NamedGraph.All";

    // instance variables
    private URI name;
    private String label;
    private Type type;
    private String typeLabel;
    private boolean anonAccess;
    private boolean hasMetadata = false;
    private boolean dirty = false;    // anything modified?

    // full constructor for NG instance with metadata
    private NamedGraph(URI name, String label, URI type, String typeLabel, boolean anonAccess)
    {
        super();
        this.name = name;
        this.label = label;
        this.type = Type.parse(type);
        this.typeLabel = typeLabel;
        this.anonAccess = anonAccess;
        this.hasMetadata = true;
    }

    // constructor for NG without metadata, e.g. from Sesame context list
    private NamedGraph(URI name)
    {
        super();
        this.name = name;
        this.hasMetadata = false;
    }

    // query to get all graphs
    // XXX NOTE that the HAS_READ_ACCESS clause is a kludgy optimization
    //   that violates the access control abstraction barrier.
    // If you bind "namedGraphURI" this becomes a query for ONE named graph.
    private static final String ngQuery =
          "SELECT DISTINCT * WHERE { ?namedGraphURI a <"+REPO.NAMED_GRAPH+"> . \n"+
          " OPTIONAL { ?namedGraphURI <"+RDFS.LABEL+"> ?namedGraphLabel } \n"+
          " OPTIONAL { ?namedGraphURI <"+REPO.NG_TYPE+"> ?typeURI . "+
                       " ?typeURI <"+RDFS.LABEL+"> ?typeLabel } \n"+
          " OPTIONAL { ?namedGraphURI <"+REPO.HAS_READ_ACCESS+"> ?anon \n"+
          "            FILTER( ?anon = <"+REPO.ROLE_ANONYMOUS+"> )}}";

    /**
     * Get an iterable collection of all known named graphs AND Contexts.
     * If there is no Named Graph metadata for a Sesame context, its
     * isManaged() method will return false.
     * Note that named graphs are ONLY listed ONCE even if there is duplicate
     * metadata.
     *
     * @param request a {@link javax.servlet.http.HttpServletRequest} object.
     * @return all named graphs in a {@link java.util.Collection} object.
     * @throws javax.servlet.ServletException if any.
     */
    public static List<NamedGraph> findAll(HttpServletRequest request)
        throws ServletException
    {
        return new ArrayList<NamedGraph>(findInternal(request, null).values());
    }

    /**
     * Returns a named graph object, or null if this is not a valid
     * named graph OR context.
     *
     * @param request a {@link javax.servlet.http.HttpServletRequest} object.
     * @param name name of named-graph or context, a {@link org.openrdf.model.URI} object.
     * @return a {@link org.eaglei.repository.NamedGraph} object representing named graph, or null if no graph or context exists.
     * @throws javax.servlet.ServletException if any.
     */
    public static NamedGraph find(HttpServletRequest request, URI name)
        throws ServletException
    {
        return find(request, name, false);
    }
    /**
     * Returns a named graph object, or null if this is not a valid
     * named graph OR context and createp was false.
     *
     * @param request a {@link javax.servlet.http.HttpServletRequest} object.
     * @param name name of named-graph or context, a {@link org.openrdf.model.URI} object.
     * @param createp a boolean, true if graph should be created
     * @return a {@link org.eaglei.repository.NamedGraph} object representing named graph, or null if no graph or context exists UNLESS createp is true.
     * @throws javax.servlet.ServletException if any.
     */
    public static NamedGraph find(HttpServletRequest request, URI name, boolean createp)
        throws ServletException
    {
        Map<URI,NamedGraph> ngs = findInternal(request, name);
        NamedGraph result = ngs.get(name);
        if (result == null) {
            if (createp) {
                // XXX probably ought to attach it to the map..
                result = new NamedGraph(name);
            } else
                log.warn("Failed to find named graph or context for name="+name);
        }
        return result;
    }

    /**
     * Load the requested named graph(s) into the map -- ALL if name == null.
     * Returns the map after caching it in the request attribute.
     * Assumes
     */
    private static Map<URI,NamedGraph> findInternal(HttpServletRequest request, URI name)
        throws ServletException
    {
        // is requested result already in the cache?
        Map<URI,NamedGraph> result = (Map<URI,NamedGraph>)request.getAttribute(R_NG_MAP);
        if (result != null &&
            ((name == null && request.getAttribute(R_NG_ALL) != null) ||
             (name != null && result.containsKey(name))))
            return result;

        if (result == null)
            result = new HashMap<URI,NamedGraph>();
        RepositoryConnection rc = WithRepositoryConnection.get(request);
        try {
            if (log.isDebugEnabled())
                log.debug("NamedGraph SPARQL query = "+ngQuery);
            TupleQuery q = rc.prepareTupleQuery(QueryLanguage.SPARQL, ngQuery);
            if (name != null)
                q.setBinding("namedGraphURI", name);
            q.setDataset(SPARQL.InternalGraphs);
            q.setIncludeInferred(true);
            q.evaluate(new graphHandler(result, name));
        } catch (MalformedQueryException e) {
            log.error("Rejecting malformed query:"+e);
            throw new ServletException(e);
        } catch (OpenRDFException e) {
            log.error(e);
            throw new ServletException(e);
        }
         
        // Second pass, check for Sesame contexts that don't have NG metadata.
        // If one of these exists, it implies something was writing into
        // an "undeclared" named graph.
        if (name == null || !result.containsKey(name)) {
            try {
                RepositoryResult<Resource> rr = null;
                try {
                    rr = rc.getContextIDs();
                    while (rr.hasNext()) {
                        Resource ctx = rr.next();
                        if (ctx instanceof URI &&
                              (name == null || name.equals((URI)ctx)) &&
                              !result.containsKey((URI)ctx)) {
                            if (log.isDebugEnabled())
                                log.debug("Found 'undocumented' named graph (context), name="+ctx);
                            result.put((URI)ctx, new NamedGraph((URI)ctx));
                        }
                    }
                } finally {
                    rr.close();
                }
            } catch (OpenRDFException e) {
                log.error(e);
                throw new ServletException(e);
            }
        }

        // save results in cache
        request.setAttribute(R_NG_MAP, result);
        if (name == null)
            request.setAttribute(R_NG_ALL, Boolean.TRUE);
        return result;
    }

    /**
     * <p>Getter for the field <code>name</code>.</p>
     *
     * @return a {@link org.openrdf.model.URI} object.
     */
    public URI getName()
    {
        return name;
    }
    /**
     * <p>Getter for the field <code>label</code>.</p>
     *
     * @return a {@link java.lang.String} object.
     */
    public String getLabel()
    {
        if (label != null)
            return label;
        else
            return name.getLocalName();
    }
    /**
     * <p>Getter for the field <code>type</code>.</p>
     *
     * @return a {@link org.openrdf.model.URI} object.
     */
    public Type getType()
    {
        return type;
    }
    /**
     * <p>Getter for the field <code>type</code>, but returns URI form.</p>
     *
     * @return a {@link org.openrdf.model.URI} object.
     */
    public URI getTypeURI()
    {
        return type == null ? null : type.uri;
    }
    /**
     * <p>Getter for the field <code>typeLabel</code>.</p>
     *
     * @return a {@link java.lang.String} object.
     */
    public String getTypeLabel()
    {
        return typeLabel;
    }

    /**
     * Can be read by anonymous user (i.e. general public) if true.
     *
     * @return a boolean, true if graph is publically readable.
     */
    public boolean isAnonymousReadable()
    {
        return anonAccess;
    }

    /**
     * When true, graph has metadata and is managed by repository.
     * False for unmanaged sesame contexts.
     *
     * @return a boolean, true when graph is known NamedGraph.
     */
    public boolean isManaged()
    {
        return hasMetadata;
    }

    /**
     * <p>Setter for the field <code>label</code>.</p>
     *
     * @param request a {@link javax.servlet.http.HttpServletRequest} object.
     * @param nl a {@link java.lang.String} object.
     * @throws javax.servlet.ServletException if any.
     */
    public void setLabel(HttpServletRequest request, String nl)
        throws ServletException
    {
        setMetadataInternal(request, RDFS.LABEL, new LiteralImpl(nl, XMLSchema.STRING));
        label = nl;
    }

    /**
     * <p>Setter for the field <code>type</code>.</p>
     *
     * @param request a {@link javax.servlet.http.HttpServletRequest} object.
     * @param nt a {@link org.eaglei.repository.NamedGraph.Type} object.
     * @throws javax.servlet.ServletException if any.
     */
    public void setType(HttpServletRequest request, URI nt)
        throws ServletException
    {
        Type ntt = Type.parse(nt);
        if (ntt != null)
            setType(request, ntt);
    }

    /**
     * <p>Setter for the field <code>type</code>.</p>
     *
     * @param request a {@link javax.servlet.http.HttpServletRequest} object.
     * @param nt a {@link org.openrdf.model.URI} object.
     * @throws javax.servlet.ServletException if any.
     */
    public void setType(HttpServletRequest request, Type nt)
        throws ServletException
    {
        setMetadataInternal(request, REPO.NG_TYPE, nt.uri);
        type = nt;
        // XXX this is poor but UI doesn't need it so why bother?
        typeLabel = nt.uri.getLocalName();
    }

    /**
     * <p>commit - commit any changes made to this object (and any others)</p>
     *
     * @param request a {@link javax.servlet.http.HttpServletRequest} object.
     * @throws javax.servlet.ServletException if any.
     */
    public void commit(HttpServletRequest request)
        throws ServletException
    {
        try {
            if (dirty)
                WithRepositoryConnection.get(request).commit();
        } catch (OpenRDFException e) {
            throw new ServletException(e);
        }
    }

    // set metadata and also ensure there is a rdf:type statement
    private void setMetadataInternal(HttpServletRequest request, URI property, Value newVal)
        throws ServletException
    {
        try {
            RepositoryConnection rc = WithRepositoryConnection.get(request);
            boolean hasType = rc.hasStatement(name, RDF.TYPE, REPO.NAMED_GRAPH, false, REPO.NG_INTERNAL);
            if (hasType && rc.hasStatement(name, property, newVal, false, REPO.NG_INTERNAL)) {
                log.debug("Nothing to set, graph="+name+" already has "+property+" = "+newVal);
            } else {
                if (!hasType) {
                    rc.add(name, RDF.TYPE, REPO.NAMED_GRAPH, REPO.NG_INTERNAL);
                    hasMetadata = true;
                }
                rc.remove(name, property, null, REPO.NG_INTERNAL);
                rc.add(name, property, newVal, REPO.NG_INTERNAL);
                dirty = true;
                log.debug("Setting graph="+name+",  "+property+" = "+newVal);
            }
        } catch (OpenRDFException e) {
            throw new ServletException(e);
        }
    }


    // Consider two NGs equal if the names are equal, never mind the MD.
    /** {@inheritDoc} */
    public boolean equals(Object b)
    {
        return b instanceof NamedGraph && ((NamedGraph)b).name.equals(name);
    }

    /** {@inheritDoc} */
    public int hashCode()
    {
        return name.hashCode();
    }

    /** {@inheritDoc} */
    public int compareTo(NamedGraph o)
    {
        return name.toString().compareTo(o.name.toString());
    }

    /**
     * query result handler to populate result list with NamedGraph objects
     */
    private static class graphHandler extends TupleQueryResultHandlerBase
    {
        // key is graph URI for easy comparison and lookup of duplicates
        private Map<URI,NamedGraph> result = null;
        private URI name = null;
        private boolean checkDuplicates = false;

        public graphHandler(Map<URI,NamedGraph> result, URI name)
        {
            super();
            graphHandler.this.result = result;
            graphHandler.this.name = name;

            // if we're starting out with some graphs in map, ignore
            // "duplicates" since there would be false positives.
            checkDuplicates =  result.isEmpty();
        }

        // columns: namedGraphURI, namedGraphLabel, typeURI, typeLabel, anon
        public void handleSolution(BindingSet bs)
            throws TupleQueryResultHandlerException
        {
            Value ngURI = bs.getValue("namedGraphURI");
            Value ngLabel = bs.getValue("namedGraphLabel");
            Value typeURI = bs.getValue("typeURI");
            Value typeLabel = bs.getValue("typeLabel");
            Value anon = bs.getValue("anon");
            // plug in name if we forced a binding in teh query..
            // XXX is this needed???
            /***
            if (ngURI == null && name != null) {
                log.warn("Got null ngURI for name="+name);
                ngURI = name;
            }
                ***/
            if (ngURI == null || !(ngURI instanceof URI))
                throw new TupleQueryResultHandlerException(
                    "Should not get null or non-URI result in graphHandler: "+ngURI);

            // check for duplicate metadata on same graph URI, should NOT happen
            else if (checkDuplicates && result.containsKey(ngURI))
                log.warn("There are MULTIPLE named graph metadata entries for graph="+ngURI+", skipping extra: label="+ngLabel+", type="+typeURI);

            // looks like a good entry
            else {
                String ngl = (ngLabel != null && ngLabel instanceof Literal) ?
                           ((Literal)ngLabel).getLabel() : ngURI.toString();
                String tl = null;
                boolean anonAccess = (anon != null);
                if (typeLabel != null && typeLabel instanceof Literal)
                    tl = ((Literal)typeLabel).getLabel();
                else if (typeURI != null)
                    tl = ((URI)typeURI).getLocalName();
                result.put((URI)ngURI, new NamedGraph((URI)ngURI, ngl, (URI)typeURI, tl, anonAccess));
            }
        }
    }
}
