package org.eaglei.repository;

import java.io.IOException;
import java.io.OutputStreamWriter;
import java.util.Arrays;
import java.util.ArrayList;
import java.util.List;
import java.util.Map;
import java.util.HashMap;
import java.util.Set;
import java.util.HashSet;
import javax.servlet.ServletException;
import javax.servlet.http.HttpServletRequest;
import javax.servlet.http.HttpServletResponse;
import java.sql.Connection;
import java.sql.SQLException;
import javax.naming.NamingException;

import org.apache.log4j.Logger;
import org.apache.log4j.LogManager;

import org.openrdf.OpenRDFException;
import org.openrdf.model.URI;
import org.openrdf.model.impl.URIImpl;
import org.openrdf.model.impl.ContextStatementImpl;
import org.openrdf.model.BNode;
import org.openrdf.model.Literal;
import org.openrdf.model.Resource;
import org.openrdf.model.Statement;
import org.openrdf.model.Value;
import org.openrdf.model.impl.LiteralImpl;
import org.openrdf.model.ValueFactory;
import org.openrdf.model.vocabulary.RDF;
import org.openrdf.model.vocabulary.RDFS;
import org.openrdf.model.vocabulary.XMLSchema;
import org.openrdf.repository.RepositoryException;
import org.openrdf.repository.RepositoryConnection;
import org.openrdf.repository.RepositoryResult;
import org.openrdf.query.TupleQuery;
import org.openrdf.query.TupleQueryResultHandlerBase;
import org.openrdf.query.TupleQueryResultHandlerException;
import org.openrdf.query.BindingSet;
import org.openrdf.query.QueryLanguage;
import org.openrdf.query.Dataset;
import org.openrdf.query.impl.DatasetImpl;
import org.openrdf.query.MalformedQueryException;
import org.openrdf.rio.RDFFormat;
import org.openrdf.rio.Rio;
import org.openrdf.rio.RDFHandler;
import org.openrdf.rio.helpers.RDFHandlerWrapper;

import org.eaglei.repository.auth.Authentication;
import org.eaglei.repository.auth.AuthUser;
import org.eaglei.repository.servlet.WithRepositoryConnection;
import org.eaglei.repository.rid.RIDGenerator;
import org.eaglei.repository.vocabulary.REPO;
import org.eaglei.repository.vocabulary.FOAF;
import org.eaglei.repository.util.Utils;
import org.eaglei.repository.util.SPARQL;
import org.eaglei.repository.status.ForbiddenException;
import org.eaglei.repository.status.BadRequestException;
import org.eaglei.repository.status.InternalServerErrorException;
import org.eaglei.repository.servlet.ImportExport.DuplicateArg;
import org.eaglei.repository.util.HandlerBadRequest;

/**
 * User object model, reflects the user's properties in the both RDBMS
 * and RDF database.  Also manages the RDF descriptions of users.
 *
 * Named Graph Usage:
 *  1. Almost all statements about a repo User are in the graph named "repo:NG_Users".
 *  2. Ony the :hasPrincipalName statement is on the NG_Internal graph
 *     sicne it should not be exposed; there is no reason to let login
 *     names leak and it's good security practice to keep them hidden,
 *     since that just makes dictionary attacks so much easier.
 *
 * @see AuthUser, ImportExport
 *
 * @author Larry Stone
 * Started April 26, 2010
 * @version $Id: $
 */
public class User
{
    private static Logger log = LogManager.getLogger(User.class);

    /** Named graph where we create User objects.  Note taht the
     *  hasPrincipalName is still on the Internal graph, it's confidential.
     */
    private static final URI USER_GRAPH = REPO.NG_USERS;

    /** dataset that includes relevant graphs - internal and user */
    private static DatasetImpl userDataset = SPARQL.copyDataset(SPARQL.InternalGraphs);
    static {
        SPARQL.addGraph(userDataset, USER_GRAPH);
        if (log.isDebugEnabled())
            log.debug("User Dataset = "+Utils.prettyPrint(userDataset));
    }

    private final URI uri;  /*  immutable, the subject of this instance */
    private final String username; /* immutable, "user" (i.e. Principal) of this instance */
    private String firstName = null;
    private String lastName = null;
    private String mbox = null;
    private Set<URI> roles = new HashSet<URI>();

    // flag that says whether we have to update AuthUser's isSuperuser when committing
    private Boolean newIsSuperuser = null;
    private boolean dirty = false;    // anything modified?

    // special property only used in export and import of users to hold
    // RDBMS password entry
    private static final URI EXPORT_AUTH_PASSWORD = new URIImpl(REPO.NAMESPACE + "exportAuthPassword");

    // special property only used in export and import of users to
    // indicate which type of user authentication implements this user
    private static final URI EXPORT_AUTH_TYPE = new URIImpl(REPO.NAMESPACE + "exportAuthType");

    // special indtance only used in export and import of users to
    // indicate the builtin authentication (RDBMS) type
    private static final URI EXPORT_AUTH_TYPE_BUILTIN = new URIImpl(REPO.NAMESPACE + "exportAuthType_Builtin");

    // get contents of AuthUser entries to rebuild the RDBMS.
    // NOTE this is run against an in-memory graph of imported statements.
    private static final String importUserGetAuthUsers =
        "SELECT * WHERE { \n"+
        "GRAPH <"+REPO.NG_INTERNAL+"> { "+
        " ?authUser <"+EXPORT_AUTH_TYPE+"> <"+EXPORT_AUTH_TYPE_BUILTIN+">; "+
        " <"+REPO.HAS_PRINCIPAL_NAME+"> ?username; "+
        " <"+EXPORT_AUTH_PASSWORD+"> ?password "+
        " OPTIONAL { ?authUser <"+REPO.HAS_ROLE+"> ?su "+
        "   FILTER(?su = <"+REPO.ROLE_SUPERUSER+">)}}}";

    // get significant RDF statements from user import
    private static final String importUserGetStatements =
        "SELECT ?g ?s ?p ?v ?username WHERE { \n"+
        "GRAPH <"+REPO.NG_USERS+"> { ?s a <"+REPO.PERSON+"> }\n"+
        "OPTIONAL { GRAPH <"+REPO.NG_INTERNAL+"> { "+
        "    ?s <"+REPO.HAS_PRINCIPAL_NAME+"> ?username}}\n"+
        "GRAPH ?g {?s ?p ?v \n"+
        "  FILTER((?g = <"+REPO.NG_INTERNAL+"> && isURI(?s) &&\n"+
        "         (?p = <"+REPO.HAS_ROLE+"> || ?p = <"+REPO.HAS_PRINCIPAL_NAME+">)) ||\n"+
        "        (?g = <"+REPO.NG_USERS+">))}}";


    // "implicit" role URIs that should be added to EVERY user as statements.
    private static final URI implicitRoles[] = {
        REPO.ROLE_AUTHENTICATED,
        REPO.ROLE_ANONYMOUS
    };

    /**
     * @return dataset of graphs needed to query for all repo users' data
     */
    public static Dataset getUserDataset()
    {
        return userDataset;
    }

    // Constructor: initialize default implicit roles in case they aren't in the RDF.
    private User(URI uri, String username)
    {
        super();
        this.uri = uri;
        this.username = username;
        for (URI ruri : implicitRoles) {
            roles.add(ruri);
        }
    }

    // Find users query - get all or a specific one by binding 'uri'
    // note that we NEED the "order by" because the handler expects
    // all the hasRole bindings for a given URI to be clumped together.
    private static final String userForURIQuery =
          "SELECT * WHERE { ?uri a <"+REPO.PERSON+"> ; \n"+
          "<"+REPO.HAS_PRINCIPAL_NAME+"> ?hasPrincipalName .\n"+
          "  OPTIONAL { ?uri <"+FOAF.FIRST_NAME+"> ?firstName }\n"+
          "  OPTIONAL { ?uri <"+FOAF.SURNAME+"> ?surname }\n"+
          "  OPTIONAL { ?uri <"+FOAF.MBOX+"> ?mbox }\n"+
          "  OPTIONAL { ?uri <"+REPO.HAS_ROLE+"> ?hasRole . ?hasRole <"+RDFS.LABEL+"> ?roleLabel }\n"+
          " } ORDER BY ?hasPrincipalName";

    /**
     * find a record by username, i.e. login principal name
     * Returns null if NOT found..
     * NOTE that user instance gets auto-created for a "new" principal
     * upon login, so this CAN be called (by that code) when checking first
     * before creating a new Person instance.
     *
     * @param request a {@link javax.servlet.http.HttpServletRequest} object.
     * @param pname principal name (from user account RDBMS).
     * @return the User object, or null if none found.
     * @throws javax.servlet.ServletException if any.
     */
    public static User findByUsername(HttpServletRequest request, String pname)
        throws ServletException
    {
        return findOneUserInternal(request, pname, null);
    }

    /**
     * Find a user record by URI;  Returns null if NOT found..
     *
     * @param request a {@link javax.servlet.http.HttpServletRequest} object.
     * @param uri the URI of a person object
     * @return the User object, or null if none found.
     * @throws javax.servlet.ServletException if any.
     */
    public static User find(HttpServletRequest request, URI uri)
        throws ServletException
    {
        return findOneUserInternal(request, null, uri);
    }

    private static User findOneUserInternal(HttpServletRequest request, String pname, URI uri)
        throws ServletException
    {
        RepositoryConnection rc = WithRepositoryConnection.get(request);
        try {
            log.debug("Single user SPARQL query = "+userForURIQuery);
            TupleQuery q = rc.prepareTupleQuery(QueryLanguage.SPARQL, userForURIQuery);
            q.setDataset(userDataset);
            q.clearBindings(); // needed if re-using query
            if (pname != null)
                q.setBinding("hasPrincipalName", new LiteralImpl(pname));
            else if (uri != null)
                q.setBinding("uri", uri);
            else
                throw new ServletException("sanity check: findOneUserInternal called without either username OR uri");

            // do NOT infer any roles.
            q.setIncludeInferred(false);
            oneUserHandler h = new oneUserHandler(request);
            q.evaluate(h);
            log.debug("User.findByUsername("+pname+") => "+h.result);

            // check whether Superuser role is expressed in AuthUser:
            if (h.result != null) {
                AuthUser au = AuthUser.find(pname);
                if (au != null && au.isSuperuser())
                    h.result.roles.add(REPO.ROLE_SUPERUSER);
            }
            return h.result;
        } catch (MalformedQueryException e) {
            log.error("Rejecting malformed query:"+e);
            throw new ServletException(e);
        } catch (OpenRDFException e) {
            log.error(e);
            throw new ServletException(e);
        } catch (SQLException e) {
            log.error(e);
            throw new ServletException(e);
        } catch (NamingException e) {
            log.error(e);
            throw new ServletException(e);
        }
    }

    /**
     * <p>Get all known Users from RDF DB - may not match auth'n DB.
     * <br>XXX TODO should add start and count for pagination
     * </p>
     *
     * @param request a {@link javax.servlet.http.HttpServletRequest} object.
     * @return resulting Users in a {@link java.lang.Iterable} object.
     * @throws javax.servlet.ServletException if any.
     */
    public static Iterable<User> findAll(HttpServletRequest request)
        throws ServletException
    {
        RepositoryConnection rc = WithRepositoryConnection.get(request);
        List<User> result = new ArrayList<User>();

        try {
            log.debug("All user SPARQL query = "+userForURIQuery);
            TupleQuery q = rc.prepareTupleQuery(QueryLanguage.SPARQL, userForURIQuery);
            q.setDataset(userDataset);
            // do NOT infer any roles.
            q.setIncludeInferred(false);
            q.evaluate(new allUserHandler(request, result));

            // check whether Superuser role is expressed in AuthUser:
            Map<String,AuthUser> authUsers = AuthUser.findAllAsMap();
            for (User ru : result) {
                AuthUser au = authUsers.get(ru.username);
                if (au != null && au.isSuperuser())
                    ru.roles.add(REPO.ROLE_SUPERUSER);
            }
        } catch (MalformedQueryException e) {
            log.error("Rejecting malformed query:"+e);
            throw new ServletException(e);
        } catch (OpenRDFException e) {
            log.error(e);
            throw new ServletException(e);
        } catch (SQLException e) {
            log.error(e);
            throw new ServletException(e);
        } catch (NamingException e) {
            log.error(e);
            throw new ServletException(e);
        }
        return result;
    }

    /**
     * <p>Getter for the field <code>uri</code>.</p>
     *
     * @return a {@link org.openrdf.model.URI} object.
     */
    public URI getURI()
    {
        return uri;
    }
    /**
     * <p>Getter for the field <code>username</code>.</p>
     *
     * @return a {@link java.lang.String} object.
     */
    public String getUsername()
    {
        return username;
    }
    /**
     * <p>Getter for the field <code>firstName</code>.</p>
     *
     * @return a {@link java.lang.String} object or null if not set.
     */
    public String getFirstName()
    {
        return firstName;
    }
    /**
     * <p>Getter for the field <code>lastName</code>.</p>
     *
     * @return a {@link java.lang.String} object  or null if not set.
     */
    public String getLastName()
    {
        return lastName;
    }
    /**
     * <p>Getter for the field <code>mbox</code>.</p>
     *
     * @return a {@link java.lang.String} object  or null if not set.
     */
    public String getMbox()
    {
        return mbox;
    }

    /**
     * Computes the most informative "label" for a user, to present in UI.
     * Format is, ideally,   "username (firstname lastname)"
     * But it reverts to bare username or even URI if none is available.
     *
     * @return label as a {@link java.lang.String} object.
     */
    public String getTitle()
    {
        if (username != null) {
            String pn = getPersonalName();
            return pn.length() > 0 ? username+" ("+pn+")" : username;
        } else
            return uri.getLocalName();
    }

    /** returns "firstname lastname" if available */
    /**
     * <p>Get personal name, if available, as "firstname lastname". </p>
     *
     * @return name in String or else empty string if there is none; never null.
     */
    public String getPersonalName()
    {
        return (firstName == null) ?
                   ((lastName == null) ? "" : lastName) :
                   ((lastName == null) ? firstName : firstName+" "+lastName);
    }

    /**
     * <p>Get <code>Roles</code>.</p>
     *
     * @return all known Roles as an array of {@link org.eaglei.repository.Role} objects.
     */
    private Role[] getRoles(HttpServletRequest request)
        throws ServletException
    {
        Role result[] = new Role[roles.size()];
        int i = 0;
        for (URI ru : roles) {
            result[i] = Role.find(request, ru);
            ++i;
        }
        return result;
    }

    /**
     * has role predicate - the P suffix is a Lisp thing
     *
     * @param r role as a {@link org.eaglei.repository.Role} object.
     * @return a boolean, true if this User has indicated role.
     */
    public boolean hasRoleP(Role r)
    {
        return roles.contains(r.getURI());
    }

    /**
     * <p>hasRoleP</p>
     *
     * @param ru role as a {@link org.openrdf.model.URI} object.
     * @return a boolean, true if this User has indicated role.
     */
    public boolean hasRoleP(URI ru)
    {
        return roles.contains(ru);
    }

    /**
     * <p>isSuperuser</p> - convenience method to test for superuser (administrator) role
     *
     * @return a boolean, true if this User has Superuser (Admin) role.
     */
    public boolean isSuperuser()
    {
        return hasRoleP(REPO.ROLE_SUPERUSER);
    }

    /**
     * <p>setIsSuperuser</p> - change local value of Superuser bit WITHOUT
     *  updating the RDF description, ONLY for cached local User object.
     *
     * @param su a boolean, true if this User has Superuser (Admin) role.
     */
    public void setIsSuperuser(boolean su)
    {
        if (su)
            roles.add(REPO.ROLE_SUPERUSER);
        else
            roles.remove(REPO.ROLE_SUPERUSER);
    }


    /**
     * <p>
     * Ensure this user has the implicit Anonymous and Authenticated roles.
     *  </p>
     *
     * @param request a {@link javax.servlet.http.HttpServletRequest} object.
     * @throws javax.servlet.ServletException if any.
     */
    public void checkImplicitRoles(HttpServletRequest request)
        throws ServletException
    {
        for (URI ruri : implicitRoles) {
            if (!hasRoleP(ruri))
                addRoleAsAdministrator(request, Role.find(request, ruri));
        }
    }

    /**
     * <p>
     * Create new user instance for given username.
     * Returns User object with URI and username set; nothing else.
     * Fails if there is an existing user with the same principal.
     * Requires Administrator role.
     *  </p>
     *
     * @param request a {@link javax.servlet.http.HttpServletRequest} object.
     * @param username principal name, a {@link java.lang.String} object.
     * @return the new {@link org.eaglei.repository.User} object.
     * @throws javax.servlet.ServletException if any.
     */
    public static User create(HttpServletRequest request, String username)
        throws ServletException
    {
        return createInternal(request, username, false);
    }

    /**
     * <p>
     * Create new user instance for given username.
     * Returns User object with URI and username set; nothing else.
     * Fails if there is an existing user with the same principal.
     * Skip access test -- THIS IS VERY DANGEROUS, only for
     *  bootstrapping the authentication.
     *  </p>
     *
     * @param request a {@link javax.servlet.http.HttpServletRequest} object.
     * @param username principal name, a {@link java.lang.String} object.
     * @return the new {@link org.eaglei.repository.User} object.
     * @throws javax.servlet.ServletException if any.
     */
    public static User createAsAdministrator(HttpServletRequest request, String username)
        throws ServletException
    {
        return createInternal(request, username, true);
    }

    /**
     * <p>
     * Create new user instance for given username.
     * Returns User object with URI and username set; nothing else.
     * Fails if there is an existing user with the same principal.
     * Skip access test if skipAccessCheck bit is set -- dangerous, only for
     *  bootstrapping the authentication.
     *  </p>
     *
     * @param request a {@link javax.servlet.http.HttpServletRequest} object.
     * @param username principal name, a {@link java.lang.String} object.
     * @param skipAccessCheck a boolean.
     * @return the new {@link org.eaglei.repository.User} object.
     * @throws javax.servlet.ServletException if any.
     */
    private static User createInternal(HttpServletRequest request, String username, boolean skipAccessCheck)
        throws ServletException
    {
        if (!skipAccessCheck && !Access.hasPermissionOnUser(request, username))
            throw new ForbiddenException("Not allowed to create user: "+username);

        RepositoryConnection rc = WithRepositoryConnection.get(request);
        ValueFactory vf = rc.getValueFactory();
        User result = null;

        try {
            // sanity check, ensure there is no user with this principal already
            Literal lpname = vf.createLiteral(username);
            if (rc.hasStatement(null, REPO.HAS_PRINCIPAL_NAME, lpname, false, REPO.NG_INTERNAL))
                throw new BadRequestException("Cannot create user: there is already a repository user with the login principal name (username) \""+username+"\"");

            result = new User(vf.createURI(DataRepository.getInstance().getDefaultNamespace(),
                                      RIDGenerator.getInstance().newID().toString()),
                              username);
            rc.add(result.uri, RDF.TYPE, REPO.PERSON, USER_GRAPH);
            rc.add(result.uri, RDFS.LABEL, lpname, USER_GRAPH);
            rc.add(result.uri, REPO.HAS_PRINCIPAL_NAME, lpname, REPO.NG_INTERNAL);
            // KLUDGE: Assert the "implicit" roles..
            for (URI ruri : implicitRoles) {
                rc.add(result.uri, REPO.HAS_ROLE, ruri, REPO.NG_INTERNAL);
            }

            log.debug("create: created new User instance, username="+username+", uri="+result.uri);
            result.dirty = true;
            return result;
        } catch (RepositoryException e) {
            log.error("Failed creating user URI: ",e);
            throw new ServletException("Failed creating user URI: ",e);
        }
    }

    // Setters

    /**
     * Change value of first name. Setting it to null clears it.
     *
     * @param request a {@link javax.servlet.http.HttpServletRequest} object.
     * @param name first name a {@link java.lang.String} object.
     * @throws javax.servlet.ServletException if any.
     */
    public void setFirstName(HttpServletRequest request, String name)
        throws ServletException
    {
        firstName = name;
        setMetadataInternal(request, FOAF.FIRST_NAME, name);
    }

    /**
     * <p>Setter for the field <code>lastName</code>.</p>
     *
     * @param request a {@link javax.servlet.http.HttpServletRequest} object.
     * @param name a {@link java.lang.String} object.
     * @throws javax.servlet.ServletException if any.
     */
    public void setLastName(HttpServletRequest request, String name)
        throws ServletException
    {
        lastName = name;
        setMetadataInternal(request, FOAF.SURNAME, name);
    }
    /**
     * <p>Setter for the field <code>mbox</code>.</p>
     *
     * @param request a {@link javax.servlet.http.HttpServletRequest} object.
     * @param mbox a {@link java.lang.String} object.
     * @throws javax.servlet.ServletException if any.
     */
    public void setMbox(HttpServletRequest request, String mbox)
        throws ServletException
    {
        this.mbox = mbox;
        setMetadataInternal(request, FOAF.MBOX, mbox);
    }

    // update one of the metadata properties.
    private void setMetadataInternal(HttpServletRequest request, URI property, String newVal)
        throws ServletException
    {
        if (!Access.hasPermissionOnUser(request, username))
            throw new ForbiddenException("Not allowed to modify user: "+username);
        try {
            RepositoryConnection rc = WithRepositoryConnection.get(request);
            boolean hasProp = rc.hasStatement(uri, property, null, false, USER_GRAPH);
            boolean hasLabel = rc.hasStatement(uri, RDFS.LABEL, null, false, USER_GRAPH);
            ValueFactory vf = rc.getValueFactory();
            if (hasProp)
                rc.remove(uri, property, null, USER_GRAPH);
            if (newVal != null)
                rc.add(uri, property, vf.createLiteral(newVal), USER_GRAPH);
            // recompute label
            String label = getTitle();
            log.debug("Setting User label, uri="+uri+", label="+label);
            if (hasLabel)
                rc.remove(uri, RDFS.LABEL, null, USER_GRAPH);
            rc.add(uri, RDFS.LABEL, vf.createLiteral(label), USER_GRAPH);

            // XXX this may lie if we didn't change anything, but who cares.
            dirty = true;
        } catch (RepositoryException e) {
            throw new ServletException(e);
        }
    }

    /**
     * <p>addRole - add a role</p>
     *
     * @param request a {@link javax.servlet.http.HttpServletRequest} object.
     * @param ru Role to add, as a {@link org.openrdf.model.URI} object.
     * @throws javax.servlet.ServletException if any.
     */
    public void addRole(HttpServletRequest request, URI ru)
        throws ServletException
    {
        addRole(request, Role.find(request, ru));
    }

    /**
     * <p>addRole - add a role</p>
     *
     * @param request a {@link javax.servlet.http.HttpServletRequest} object.
     * @param r role to add as a {@link org.eaglei.repository.Role} object.
     * @throws javax.servlet.ServletException if any.
     */
    public void addRole(HttpServletRequest request, Role r)
        throws ServletException
    {
        addRoleInternal(request, r, false);
    }
    /**
     * <p>addRoleAsAdministrator - add a role WITHOUT access control checks</p>
     *
     * @param request a {@link javax.servlet.http.HttpServletRequest} object.
     * @param r role to add as a {@link org.eaglei.repository.Role} object.
     * @throws javax.servlet.ServletException if any.
     */
    public void addRoleAsAdministrator(HttpServletRequest request, Role r)
        throws ServletException
    {
        addRoleInternal(request, r, true);
    }

    private void addRoleInternal(HttpServletRequest request, Role r, boolean skipAccessCheck )
        throws ServletException
    {
        if (!skipAccessCheck && !Authentication.isSuperuser(request))
            throw new ForbiddenException("Only the administrator is allowed to modify user roles.");

        if (hasRoleP(r))
            return;
        RepositoryConnection rc = WithRepositoryConnection.get(request);
        try {
            if (roles.add(r.getURI())) {
                rc.add(uri, REPO.HAS_ROLE, r.getURI(), REPO.NG_INTERNAL);
                if (r.isSuperuser())
                    newIsSuperuser = Boolean.TRUE;
                dirty = true;
                log.debug("Added role: "+r);
            }
        } catch (RepositoryException e) {
            log.error("Failed adding role ",e);
            throw new ServletException("Failed adding role ",e);
        }
    }

    /**
     * <p>removeRole</p>
     *
     * @param request a {@link javax.servlet.http.HttpServletRequest} object.
     * @param ru role to remove as a {@link org.openrdf.model.URI} object.
     * @throws javax.servlet.ServletException if any.
     */
    public void removeRole(HttpServletRequest request, URI ru)
        throws ServletException
    {
        removeRole(request, Role.find(request, ru));
    }

    /**
     * <p>removeRole</p>
     *
     * @param request a {@link javax.servlet.http.HttpServletRequest} object.
     * @param r role to remove as a {@link org.eaglei.repository.Role} object.
     * @throws javax.servlet.ServletException if any.
     */
    public void removeRole(HttpServletRequest request, Role r)
        throws ServletException
    {
        if (!Authentication.isSuperuser(request))
            throw new ForbiddenException("Only the administrator is allowed to modify user roles.");

        if (!hasRoleP(r))
            return;
        RepositoryConnection rc = WithRepositoryConnection.get(request);
        try {
            rc.remove(uri, REPO.HAS_ROLE, r.getURI());
            roles.remove(r.getURI());
            if (r.isSuperuser())
                newIsSuperuser = Boolean.FALSE;
            dirty = true;
            log.debug("Removed role: "+r);
        } catch (RepositoryException e) {
            log.error("Failed adding role ",e);
            throw new ServletException("Failed removing role ",e);
        }
    }


    /**
     * <p>setRoles</p>  make the user's set of roles equal to contents of this array
     *
     * @param request a {@link javax.servlet.http.HttpServletRequest} object.
     * @param role array of Role objects which should be the new role set
     * @return true if any roles were actually changed.
     * @throws javax.servlet.ServletException if any.
     */
    public boolean setRoles(HttpServletRequest request, Role newRoles[])
        throws ServletException
    {
        List<Role> addRoles = new ArrayList<Role>();
        Set<Role> deleteRoles = new HashSet<Role>();
        for (Role r : getRoles(request)) {
            if (!r.isImplicit())
                deleteRoles.add(r);
        }

        // remove matches from existing set so any remaining ones must be deleted
        for (Role r : newRoles) {
            if (deleteRoles.contains(r))
                deleteRoles.remove(r);
            else
                addRoles.add(r);
        }
        if (deleteRoles.isEmpty() && addRoles.isEmpty())
            return false;
        for (Role r : deleteRoles)
            removeRole(request, r);
        for (Role r : addRoles)
            addRole(request, r);
        return true;
    }

    /**
     * <p>update -
     * Commits any changes to RDF database *and* RDBMS.
     * Also recomputes rdfs:label, passes superuser bit into RDBMS.
     * Does *not* need to check access again here because the only
     * operatiosn taht set "dirty" are already access-protected..
     * </p>
     *
     * @param request a {@link javax.servlet.http.HttpServletRequest} object.
     * @throws javax.servlet.ServletException if any.
     */
    public void update(HttpServletRequest request)
        throws ServletException
    {
        RepositoryConnection rc = WithRepositoryConnection.get(request);
        try {

            // label is either full name (if avail.) or principal
            if (dirty) {
                Authentication.decacheAuthentication(request, this);
                rc.commit();
                if (newIsSuperuser != null) {
                    AuthUser au = AuthUser.find(username);
                    if (au != null) {
                        au.setIsSuperuser(newIsSuperuser.booleanValue());
                        log.debug("update(): Setting AuthUser isSuperuser = "+newIsSuperuser);
                        au.update();
                    }
                }
            }
            dirty = false;
        } catch (SQLException e) {
            log.error("Failed updating user, URI="+uri,e);
            throw new ServletException("Failed updating user, URI="+uri,e);
        } catch (NamingException e) {
            log.error("Failed updating user, URI="+uri,e);
            throw new ServletException("Failed updating user, URI="+uri,e);
        } catch (RepositoryException e) {
            log.error("Failed updating user, URI="+uri,e);
            throw new ServletException("Failed updating user, URI="+uri,e);
        }
    }

    /**
     * <p>finalize - flag error if GC'ing a dirty instance, its changes are lost. </p>
     */
    protected void finalize()
        throws Throwable
    {
        if (dirty)
            log.error("finalize: about to destroy a User with dirty flag set, CHANGES WILL BE LOST.  Current state: "+toString());
        super.finalize();
    }

    /**
     * <p>toString</p>
     *
     * @return a {@link java.lang.String} object.
     */
    public String toString()
    {
        String rs = roles == null ? "{null}" :
            Arrays.deepToString(roles.toArray(new URI[roles.size()]));
        String uu = uri == null ? "(not set)" : uri.toString();

        return "<#User: uri="+uu+
            ", username="+username+", firstName="+firstName+
            ", lastName="+lastName+", mbox="+mbox+", roles="+rs+">";
    }

    // tuple query result handler to gather user contents
    private static class oneUserHandler extends TupleQueryResultHandlerBase
    {
        private User result = null;
        protected HttpServletRequest request = null;

        public oneUserHandler(HttpServletRequest request)
        {
            super();
            oneUserHandler.this.request = request;
        }

        public void endQueryResult()
        {
            // never got any results
            if (result == null)
                log.debug("Failed to get any query results, ");
        }
        public void handleSolution(BindingSet bs)
            throws TupleQueryResultHandlerException
        {
            if (result == null)
                result = populateUser(request, bs, null);
            else
                populateUser(request, bs, result);
        }

        // read the result record into a User - shared with "all" processor
        // if u == null, create a new one
        protected User populateUser(HttpServletRequest request, BindingSet bs, User u)
            throws TupleQueryResultHandlerException
        {
            User ret = u;
            if (ret == null) {
                Value puri = bs.getValue("uri");
                if (puri == null || !(puri instanceof URI))
                    throw new TupleQueryResultHandlerException("Missing valid result for required column 'uri'");
                Value pn = bs.getValue("hasPrincipalName");
                if (pn == null)
                    throw new TupleQueryResultHandlerException("Missing valid result for required column 'hasPrincipalName'");
                ret = new User((URI)puri, (pn instanceof Literal) ? ((Literal)pn).getLabel() : pn.stringValue());

                Value firstName =  bs.getValue("firstName");
                if (firstName != null && firstName instanceof Literal)
                    ret.firstName = ((Literal)firstName).getLabel();
                Value sur = bs.getValue("surname");
                if (sur != null && sur instanceof Literal)
                    ret.lastName = ((Literal)sur).getLabel();
                Value mbox = bs.getValue("mbox");
                if (mbox != null && mbox instanceof Literal)
                    ret.mbox = ((Literal)mbox).getLabel();
            }
            Value role = bs.getValue("hasRole");
            // Value roleLabel = bs.getValue("roleLabel");
            if (role != null) {
                log.debug("Adding role to single User, username="+ret.username+", role="+role.stringValue());
                ret.roles.add((URI)role);
            }
            return ret;
        }
    }

    // tuple query result handler to gather user contents
    private static class allUserHandler extends oneUserHandler
    {
        private List<User> result = null;
        private User curUser = null;
        private URI lastURI = null;

        public allUserHandler(HttpServletRequest request, List<User> u)
        {
            super(request);
            allUserHandler.this.result = u;
            // allUserHandler.this.request = request;
        }

        public void endQueryResult()
        {
            finishCurrent();
        }
        public void handleSolution(BindingSet bs)
            throws TupleQueryResultHandlerException
        {
            Value newURI = bs.getValue("uri");
            if (newURI == null || !(newURI instanceof URI))
                log.error("Should not get null or non-URI result in allUserHandler: "+newURI);

            // start a new one
            else if (lastURI == null || !newURI.equals(lastURI)) {
                finishCurrent();
                lastURI = (URI)newURI;
                curUser = populateUser(request, bs, null);
                          new User(lastURI, null);
            } else {
                populateUser(request, bs, curUser);
            }
        }

        // close off current User, if any, and push onto result list
        private void finishCurrent()
        {
            if (curUser != null) {
                result.add(curUser);
                curUser = null;
            }
        }
    }

    /** {@inheritDoc} */
    public boolean equals(Object o)
    {
        return o instanceof User && uri != null && uri.equals(((User)o).uri);
    }

    /** {@inheritDoc} */
    public int hashCode()
    {
        return uri.hashCode();
    }

    /* ------------------- Support for Import / Export service -----------*/

    /**
     * Import description of user accounts from serialized RDF quads.
     */
    public static void doImportUsers(HttpServletRequest request, HttpServletResponse response,
            RepositoryConnection content,
            Set<String> includes, Set<String> excludes,
            DuplicateArg duplicate,
            boolean transform)

        throws ServletException, IOException
    {
        try {
                RepositoryConnection rc = WithRepositoryConnection.get(request);
                // read document into memory repo so we can query it without polluting main repo
                try {
                    TupleQuery q = content.prepareTupleQuery(QueryLanguage.SPARQL, importUserGetAuthUsers);
                    if (log.isDebugEnabled())
                        log.debug("SPARQL query PASS1 against internal memory repo = "+importUserGetAuthUsers);
                    q.setDataset(User.getUserDataset());
                    q.setIncludeInferred(false);
                    // c is a single transaction for all AuthUser changes.
                    Connection c = AuthUser.startTransaction();
                    authUserPass1Handler ah = new authUserPass1Handler(c, rc, transform, duplicate, includes, excludes);
                    try {
                        // pass 1 - AuthUsers, basic description
                        q.evaluate(ah);

                        // pass 2 - gather all roles and user metadata
                        q = content.prepareTupleQuery(QueryLanguage.SPARQL, importUserGetStatements);
                        if (log.isDebugEnabled())
                            log.debug("SPARQL query PASS2 against internal memory repo = "+importUserGetStatements);
                        q.setDataset(User.getUserDataset());
                        q.setIncludeInferred(false);
                        q.evaluate(new authUserPass2Handler(ah));

                        // XXX NOTE:  sort of a race condition here, but
                        // it can't be helped.  If the Sesame commit
                        // goes south, the RDBMS data will be
                        // inconsitent but we can live with that.
                        AuthUser.commitTransaction(c);
                        rc.commit();
                        c = null;
                    } finally {
                        if (c != null) {
                            AuthUser.abortTransaction(c);
                            c = null;
                        }
                    }

                } catch (HandlerBadRequest e) {
                    throw new BadRequestException(e.getMessage(), e);
                } catch (NamingException e) {
                    log.error("Failed in IMPORT USER: ",e);
                    throw new InternalServerErrorException(e);
                } catch (SQLException e) {
                    log.error("Failed in IMPORT USER: ",e);
                    throw new InternalServerErrorException(e);
                }

        } catch (OpenRDFException e) {
            throw new InternalServerErrorException(e);
        }
    }

    // Pass 1 of user import:
    //  - filter users to import
    //  - create RDBMS entry for selected users
    //  - prepare map of URI to username for second pass
    private static class authUserPass1Handler extends TupleQueryResultHandlerBase
    {
        // map of user URI to principal username
        private RepositoryConnection rc;
        private Connection rdbms;
        private boolean transform;
        private DuplicateArg duplicate;
        private Set<String> includes;
        private Set<String> excludes;
        private int count = 0;
        // usernames which have been allowed to be imported
        private Set<String> allow = new HashSet<String>();

        public authUserPass1Handler(Connection c, RepositoryConnection r,
                boolean t, DuplicateArg d, Set<String> inc, Set<String> exc)
        {
            super();
            rdbms = c;
            rc = r;
            transform = t;
            duplicate = d;
            includes = inc;
            excludes = exc;
        }
        public void handleSolution(BindingSet bs)
            throws TupleQueryResultHandlerException
        {
            Value vsubject = bs.getValue("authUser");
            Value vusername = bs.getValue("username");
            Value vpassword = bs.getValue("password");
            boolean su = bs.getValue("su") != null;

            if (log.isDebugEnabled())
                log.debug("authUserPass1Handler: Got subj="+vsubject+", username="+
                  vusername+", password="+(vpassword==null?null:"[password]")+", su="+su);
            // sanity check
            if (vsubject == null || vusername == null || vpassword == null)
                throw new HandlerBadRequest("Bad export file content: missing one of the required data: "+
                        "subject="+vsubject+", username="+vusername+", password="+(vpassword==null?null:"[password]"));
            try {

                // filter: check include, exclude, duplicates
                String username = vusername.stringValue();
                URI subject = vsubject instanceof URI ? (URI)vsubject : null;
                String ssubject = vsubject.stringValue();
                if (excludes.contains(username) || excludes.contains(ssubject)) {
                    log.debug("PASS1: SKIP USER import because of exclude: username="+username+", or subject="+ssubject);
                    return;
                } else if (!(includes.isEmpty() ||
                           includes.contains(username) ||
                           includes.contains(ssubject))) {
                    log.debug("PASS1: SKIP USER import because of include: username="+username+", or subject="+ssubject);
                    return;
                }

                // is this a duplicate username (principal)?
                AuthUser au = AuthUser.find(username);
                if (au != null) {
                    log.debug("PASS1: FOUND DUPLICATE PRINCIPAL, user principal="+username);
                    if (duplicate == DuplicateArg.ignore)
                        return;
                    else if (duplicate == DuplicateArg.abort)
                        throw new HandlerBadRequest("Import contains a duplicate user for username="+username);
                }

                // is this a duplicate of the user URI?
                if (subject != null && rc.hasStatement(subject, REPO.HAS_PRINCIPAL_NAME, null, false, REPO.NG_INTERNAL)) {
                    log.debug("PASS1: FOUND DUPLICATE SUBJECT, user subject="+subject);
                    if (duplicate == DuplicateArg.ignore)
                        return;
                    else if (duplicate == DuplicateArg.abort)
                        throw new HandlerBadRequest("Import contains a duplicate user for URI="+subject);
                }

                // finally, create or replace user:
                if (au != null) {
                    au.setPassword(vpassword.stringValue());
                    au.setIsSuperuser(su);
                    au.update(rdbms);
                } else {
                    AuthUser.create(rdbms, username, vpassword.stringValue(), su);
                }
                allow.add(username);
                ++count;
                log.debug("Added new user in PASS1: username="+username+", URI="+subject);
            } catch (NamingException e) {
                log.error("Failed in PASS1: ",e);
                throw new TupleQueryResultHandlerException(e);
            } catch (SQLException e) {
                log.error("Failed in PASS1: ",e);
                throw new TupleQueryResultHandlerException(e);
            } catch (RepositoryException e) {
                log.error("Failed in PASS1: ",e);
                throw new TupleQueryResultHandlerException(e);
            }
        }
    }

    // pass 2 - install all the user instance and metadata
    private static class authUserPass2Handler extends TupleQueryResultHandlerBase
    {
        // map of user URI to principal username
        private authUserPass1Handler pass1;

        // imported subject to transformed subject - also used as flag that
        // we've seen this subject before and it is not to be filtered out.
        private Map<URI,URI> subject2uri = new HashMap<URI,URI>();

        public authUserPass2Handler(authUserPass1Handler p1)
        {
            super();
            pass1 = p1;
        }

        public void handleSolution(BindingSet bs)
            throws TupleQueryResultHandlerException
        {
            Value vgraph = bs.getValue("g");
            Value vsubject = bs.getValue("s");
            Value vpredicate = bs.getValue("p");
            Value vobject = bs.getValue("v");
            Value vusername = bs.getValue("username");
            ValueFactory vf = pass1.rc.getValueFactory();

            // sanity check
            if (vsubject == null || vgraph == null || vpredicate == null || vobject == null)
                throw new HandlerBadRequest("Bad export file content: missing one of the required data: "+
                            "subj="+vsubject+", pred="+vpredicate+", obj="+vobject+", graph="+vgraph);
            if (log.isDebugEnabled())
                log.debug("authUserPass2Handler: Got subj="+vsubject+", pred="+vpredicate+", obj="+vobject.stringValue()+", graph="+vgraph);
            try {

                // filter: check include, exclude, if not already passed
                URI subject = vsubject instanceof URI ? (URI)vsubject : null;
                String username = vusername == null ? null : vusername.stringValue();

                // skip if already denied in pass1
                if (username != null && !pass1.allow.contains(username))
                    return;

                if (!subject2uri.containsKey(subject)) {
                    String ssubject = vsubject.stringValue();
                    if (pass1.excludes.contains(username) || pass1.excludes.contains(ssubject)) {
                        log.debug("PASS2: SKIP USER import because of exclude: username="+username+", or subject="+ssubject);
                        return;
                    } else if (!(pass1.includes.isEmpty() ||
                               pass1.includes.contains(username) ||
                               pass1.includes.contains(ssubject))) {
                        log.debug("PASS2: SKIP USER import because of include: username="+username+", or subject="+ssubject);
                        return;
                    }

                    // Is there a duplicate of the user URI?
                    // If replacing, and NOT transforming,remove all old statements.
                    if (subject != null && pass1.rc.hasStatement(subject, REPO.HAS_PRINCIPAL_NAME, null, false, REPO.NG_INTERNAL)) {
                        log.debug("PASS2: FOUND DUPLICATE SUBJECT, user subject="+subject);
                        if (pass1.duplicate == DuplicateArg.ignore)
                            return;
                        else if (pass1.duplicate == DuplicateArg.abort)
                            throw new HandlerBadRequest("Import contains a duplicate user for URI="+subject);
                        else if (pass1.duplicate == DuplicateArg.replace) {
                            if (!pass1.transform) {
                                pass1.rc.remove(subject, null, null, REPO.NG_INTERNAL, REPO.NG_USERS);
                                log.debug("Removing all statements about user URI="+subject+" in NG_Users");
                            }
                        } else
                            throw new HandlerBadRequest("Unknown state of 'duplicate' arg, duplicate="+pass1.duplicate);
                    }

                    // Was there already a Person object for this
                    // username?  If so, remove it:
                    RepositoryResult<Statement> rr =
                        pass1.rc.getStatements(null, REPO.HAS_PRINCIPAL_NAME, vusername, false, REPO.NG_INTERNAL);
                    try {
                        while (rr.hasNext()) {
                            Resource uu = rr.next().getSubject();
                            log.debug("PASS2: Clearing out existing Person with principal="+username+", subject="+uu);
                            pass1.rc.remove(uu, null, null, REPO.NG_INTERNAL, REPO.NG_USERS);
                        }
                    } finally {
                        rr.close();
                    }

                    // record that we've been here so next results for this
                    // subject skip the filter test, and use the same transform
                    if (pass1.transform)
                        subject2uri.put(subject, vf.createURI(DataRepository.getInstance().getDefaultNamespace(),
                                              RIDGenerator.getInstance().newID().toString()));
                    else
                        subject2uri.put(subject, subject);
                }
                URI newsubj = subject2uri.get(subject);
                pass1.rc.add(newsubj, (URI)vpredicate, vobject, (URI)vgraph);
                log.debug("PASS2: Added statement ("+newsubj+", "+vpredicate+", "+vobject.stringValue()+", "+vgraph+")");
            } catch (RepositoryException e) {
                log.error("Failed in PASS1: ",e);
                throw new TupleQueryResultHandlerException(e);
            }
        }

        // when done, add implicit roles to all new users
        public void endQueryResult()
            throws TupleQueryResultHandlerException
        {
            try {
                for (URI subj : subject2uri.values()) {
                    for (URI ruri : implicitRoles) {
                        pass1.rc.add(subj, REPO.HAS_ROLE, ruri, REPO.NG_INTERNAL);
                    }
                }
            } catch (RepositoryException e) {
                log.error("Failed import Pass2, adding implicit roes: ",e);
                throw new TupleQueryResultHandlerException(e);
            }

            log.info("SUMMARY: Added and/or replaced: "+pass1.count+" RDBMS entries, and "+subject2uri.size()+" User Descriptions.");
        }
    }

    /**
     * Export description of user accounts to serialized RDF quads.
     */
    public static void doExportUsers(HttpServletRequest request, HttpServletResponse response,
            RDFFormat format, Set<String> includes, Set<String> excludes)
        throws ServletException, IOException
    {
        try {
            RDFHandler realOut = Rio.createWriter(format, new OutputStreamWriter(response.getOutputStream(), "UTF-8"));
            RDFHandler out = new mergeHandler(realOut);
            realOut.startRDF();
            RepositoryConnection rc = WithRepositoryConnection.get(request);
            ValueFactory vf = rc.getValueFactory();

            // pass 1 - users with RDF presnece
            Map<String,AuthUser> authUsers = AuthUser.findAllAsMap();
            for (User u : User.findAll(request)) {
                URI uu = u.getURI();
                String uus = uu.toString();
                String un = u.getUsername();
                AuthUser au = authUsers.remove(un);
                if (excludes.contains(un) || excludes.contains(uus))
                    log.debug("SKIP USER because of exclude: "+u);
                else if (includes.isEmpty() ||
                         includes.contains(un) || includes.contains(uus)) {
                    log.debug("EXPORT USER: "+uu);
                    // XXX maybe add NG_METADATA graph here? nothing there YET..
                    rc.exportStatements(uu, null, null, false, out,
                        REPO.NG_USERS, REPO.NG_INTERNAL);

                    if (au == null) {
                        log.warn("User is NOT IN RDBMS, so no password: username="+un);
                    } else {
                        // synthetic statement: identify user type
                        out.handleStatement(new ContextStatementImpl(
                            uu, EXPORT_AUTH_TYPE, EXPORT_AUTH_TYPE_BUILTIN, REPO.NG_INTERNAL));

                        // synthetic statement: add the password from RDBMS
                        out.handleStatement(new ContextStatementImpl(
                            uu, EXPORT_AUTH_PASSWORD, new LiteralImpl(au.getPassword(), XMLSchema.STRING), REPO.NG_INTERNAL));
                    }
                } else
                    log.debug("SKIP USER because of include: "+u);
            }

            // pass 2: RDBMS-only users
            // create blank node for each with essential properties
            for (AuthUser au : authUsers.values()) {
                String un = au.getUsername();
                if (excludes.contains(un))
                    log.debug("SKIP UNDOCUMENTED USER because of exclude: \""+un+"\"");
                else if (includes.isEmpty() || includes.contains(un)) {
                    log.debug("EXPORT UNDOCUMENTED USER: \""+un+"\"");
                    BNode subject = vf.createBNode();
                    out.handleStatement(new ContextStatementImpl(subject,
                        EXPORT_AUTH_TYPE,
                        EXPORT_AUTH_TYPE_BUILTIN,
                        REPO.NG_INTERNAL));
                    out.handleStatement(new ContextStatementImpl(subject,
                        REPO.HAS_PRINCIPAL_NAME,
                        new LiteralImpl(au.getUsername(), XMLSchema.STRING),
                        REPO.NG_INTERNAL));
                    out.handleStatement(new ContextStatementImpl(subject,
                        EXPORT_AUTH_PASSWORD,
                        new LiteralImpl(au.getPassword(), XMLSchema.STRING),
                        REPO.NG_INTERNAL));
                } else
                    log.debug("SKIP UNDOCUMENTED USER because of include: \""+un+"\"");
            }

            realOut.endRDF();

        } catch (NamingException e) {
            throw new InternalServerErrorException(e);
        } catch (SQLException e) {
            throw new InternalServerErrorException(e);
        } catch (OpenRDFException e) {
            throw new InternalServerErrorException(e);
        }
    }

    // merge several "streams" of output statements into one output
    private static class mergeHandler extends RDFHandlerWrapper
    {
        private mergeHandler(RDFHandler defer)
        {
            super(defer);
        }
        public void startRDF()
        {
            // do nothing
        }
        public void endRDF()
        {
            // do nothing
        }
    }
}
