package org.eaglei.repository.inferencer;

import java.util.Arrays;
import java.util.Set;
import java.util.HashSet;
import java.io.IOException;
import java.io.OutputStreamWriter;
import java.io.InputStream;
import java.io.PipedOutputStream;
import java.io.PipedInputStream;

import org.apache.log4j.Logger;
import org.apache.log4j.LogManager;

import info.aduna.iteration.CloseableIteration;

import org.openrdf.model.Statement;
import org.openrdf.model.URI;
import org.openrdf.model.impl.URIImpl;
import org.openrdf.model.vocabulary.RDF;
import org.openrdf.model.vocabulary.RDFS;
import org.openrdf.rio.Rio;
import org.openrdf.rio.RDFWriter;
import org.openrdf.rio.RDFFormat;
import org.openrdf.rio.RDFHandlerException;
import org.openrdf.sail.SailException;
import org.openrdf.sail.inferencer.InferencerConnection;
import com.clarkparsia.pellet.owlapiv3.PelletReasonerFactory;
import org.semanticweb.owlapi.util.Version;
import org.semanticweb.owlapi.model.OWLOntologyIRIMapper;
import org.semanticweb.owlapi.apibinding.OWLManager;
import org.semanticweb.owlapi.model.IRI;
import org.semanticweb.owlapi.model.OWLClass;
import org.semanticweb.owlapi.model.OWLClassExpression;
import org.semanticweb.owlapi.model.OWLDataProperty;
import org.semanticweb.owlapi.model.OWLNamedIndividual;
import org.semanticweb.owlapi.model.OWLObjectProperty;
import org.semanticweb.owlapi.model.OWLOntology;
import org.semanticweb.owlapi.model.OWLOntologyManager;
import org.semanticweb.owlapi.model.OWLTransitiveObjectPropertyAxiom;
import org.semanticweb.owlapi.reasoner.Node;
import org.semanticweb.owlapi.reasoner.NodeSet;
import org.semanticweb.owlapi.reasoner.OWLReasoner;
import org.semanticweb.owlapi.model.OWLOntologyCreationException;

import org.eaglei.repository.vocabulary.REPO;
import org.eaglei.repository.DataRepository;
import org.eaglei.repository.vocabulary.DATAMODEL;

/**
 * Ontology (TBox) inference to support
 * Custom minimalist inferencing SAIL layer.
 * @see MinimalInferencerConnection
 * (see comments in that class for all the implementation details)
 *
 * This is a singleton utility class.  Its job is to create all the inferred
 * statements for an ontology graph by feeding everything in the graph to
 * an OWLAPI reasoner (Pellet 2.1, for now) and record the useful results
 * directly in the graph as "inferred" statements.  This is a very expensive
 * operation but it is only done very rarely.
 *
 * Configuration Properties:
 *
 *  eaglei.repository.tbox.graphs = (optional) comma-separated list of URIs of
 *              graphs making up the "TBox" for the inferencer.
 *
 * @author Larry Stone
 * Started May 30 2010
 */
public class TBoxInferencer
{
    private static Logger log = LogManager.getLogger(TBoxInferencer.class);

    private static TBoxInferencer singleton = null;

    /** Configuration: Named graphs with TBox content */
    private Set<URI> tboxGraphs = new HashSet<URI>();

    private static String defaultTBoxGraphs[] = { DATAMODEL.GRAPH_NAME_URI.stringValue() };

    private OWLOntologyManager manager = null;

    private int inferredCount = 0;

    private TBoxInferencer()
    {
        super();

        // turn off loading of imports since we bundle everything in the graph
        manager = OWLManager.createOWLOntologyManager();
        manager.setSilentMissingImportsHandling(true);
        manager.clearIRIMappers();
        manager.addIRIMapper(new OWLOntologyIRIMapper() {
            public IRI getDocumentIRI(IRI in) { return null; }});

        /**
         * Initialize the TBox graph set from configuration, or if
         * not configured, the default set of ontologies:
         *   1. Repository's internal ontology
         *   2. Data model ontology graph
         * NOTE that #1 is ALWAYS configured since the repo won't work right
         * without inference on its own ontology graph.
         */
        String tbgs[] = DataRepository.getInstance().getConfigurationPropertyArray("eaglei.repository.tbox.graphs", defaultTBoxGraphs);
        tboxGraphs.add(REPO.NG_REPO_ONTOLOGY);
        for (String tb : tbgs) {
            try {
                tboxGraphs.add(new URIImpl(tb));
            } catch (IllegalArgumentException e) {
                log.error("Malformed URI in configuration \"eaglei.repository.tbox.graphs\", skipping: "+tb);
            }
        }
        if (log.isDebugEnabled())
            log.debug("Initialzed TBox graphs = "+Arrays.deepToString(tboxGraphs.toArray()));
    }

    /**
     * Get the singleton instance.
     *
     * @return the singleton, a {@link org.eaglei.repository.inferencer.TBoxInferencer} object.
     */
    public static TBoxInferencer getInstance()
    {
        if (singleton == null)
            singleton = new TBoxInferencer();
        return singleton;
    }

    /** Accessor for TBox graph set */
    public Set<URI> getTBoxGraphs()
    {
        return tboxGraphs;
    }

    /**
     * Check if this ontology graph is "stale" and (re) do the
     * inferencing if necessary.
     * Returns count of statements added.
     *
     * @param rc a {@link org.openrdf.sail.inferencer.InferencerConnection} object.
     * @param graphName name of graph (context) on which to reason, a {@link org.openrdf.model.URI} object.
     * @return count of statements added.
     * @throws java$io$IOException if any.
     * @throws org.openrdf.rio.RDFHandlerException if any.
     * @throws org.openrdf.sail.SailException if any.
     */
    int doTBoxInference(InferencerConnection rc, URI graphName)
        throws IOException, RDFHandlerException, SailException
    {
        inferredCount = 0;

        OWLOntology ontology = null;
        OWLReasoner reasoner = null;
        try {
            rc.clearInferred(graphName);
            // pipe the ontology contents into an OWLAPI reasoner
            PipedOutputStream out = new PipedOutputStream();
            PipedInputStream in = new PipedInputStream(out);
            long startMs = System.currentTimeMillis();
            InferenceRunner ir = new InferenceRunner(in);
            ir.start();
            //XXX log.debug("Starting to write ontology from graph="+graphName+"...");
            // XXX fixme? punt namespaces..
            RDFWriter w = Rio.createWriter(RDFFormat.RDFXML, new OutputStreamWriter(out, "UTF-8"));
            w.startRDF();
            CloseableIteration<? extends Statement,SailException> si =
              rc.getStatements(null, null, null, false, graphName);
            try {
                while (si.hasNext()) {
                    w.handleStatement(si.next());
                }
            } finally {
                si.close();
            }
            w.endRDF();
            out.close();
            //XXX log.debug("...Done writing ontology.");
             
            // wait for spawned thread to catch up
            try {
                ir.join(30000);
            } catch (InterruptedException e) {
                log.error("Ontology reading thread was unexpectedly interrupted!",e);
            }
            //XXX log.debug("Ontology reader thread finished.");
             
            ontology = ir.getOntology();
            if (ontology == null)
                throw new SailException("Failed to load TBox ontology from graph = "+graphName);
            log.debug("Loaded Ontology: "+ontology.getAxiomCount()+" axioms, ID="+ontology.getOntologyID()+
                     ", elapsed time: "+
              String.valueOf(System.currentTimeMillis()-startMs)+" mSec");
            reasoner = PelletReasonerFactory.getInstance().createReasoner( ontology );
            Version v = reasoner.getReasonerVersion();
            if (log.isDebugEnabled())
                log.debug(String.format("Created OWLAPI reasoner: \"%s\" version %d.%d.%d.%d, preparing...",
                                             reasoner.getReasonerName(), v.getMajor(), v.getMinor(), v.getPatch(), v.getBuild()));
            long reasonerStartMs = System.currentTimeMillis();
            reasoner.flush();
            reasoner.prepareReasoner();
            log.debug("..done preparing reasoner, spent "+
              String.valueOf(System.currentTimeMillis()-reasonerStartMs)+" mSec");
             
            crawlTBox(rc, graphName, reasoner);
            log.info("Rebuilt inferred statements on TBox graph = "+graphName+", total time = "+
              String.valueOf(System.currentTimeMillis()-startMs)+" mSec");

        // cleanup
        } finally {
            if (reasoner != null)
                reasoner.dispose();
            if (ontology != null)
                manager.removeOntology(ontology);
        }
        // XXX not using this now
        // finally, update provenance MD:
        //rc.addStatement(graphName, REPO.MODIFIED_AT_LAST_INFERENCE, lastMod, REPO.NG_INTERNAL);
        return inferredCount;
    }

    // crawl named classes, adding inferred subclasses
    private void crawlTBox(InferencerConnection rc, URI graphName, OWLReasoner reasoner)
        throws IOException, RDFHandlerException, SailException
    {
        OWLOntology ontology = reasoner.getRootOntology();

        Set<OWLClass> ac = ontology.getClassesInSignature(true);
        for (OWLClass c : ac) {
            URI cls = new URIImpl(c.getIRI().toString());

            NodeSet<OWLClass> nsups = reasoner.getSuperClasses(c, false);
            for (Node<OWLClass> nsup : nsups) {
                if (!nsup.isSingleton())
                    log.error("Superclass Node is not singleton: "+nsup);
                else {
                    OWLClass sup = nsup.getRepresentativeElement();
                    if (!sup.isAnonymous()) {

                        // XXX FIXME: maybe final test to skip owl:Thing or
                        //  other "top" superclass if we want??

                        URI obj = new URIImpl(sup.asOWLClass().getIRI().toString());
                        if (rc.addInferredStatement(cls, RDFS.SUBCLASSOF, obj, graphName))
                            ++inferredCount;
                        //XXX DEBUG log.debug("Adding inferred subclass ("+cls+", "+RDFS.SUBCLASSOF+", "+ obj+")");
                    }
                }
            }
        }

        // Crawl Inferred Subproperties:
        OWLObjectProperty bottomObjectProp = reasoner.getBottomObjectPropertyNode().getRepresentativeElement();
        OWLDataProperty bottomDataProp = reasoner.getBottomDataPropertyNode().getRepresentativeElement();
        //XXX DEBUG log.debug("bottom object property = "+bottomObjectProp);
        //XXX DEBUG log.debug("bottom data property = "+bottomDataProp);

        Set<OWLObjectProperty> ap = ontology.getObjectPropertiesInSignature(true);
        for (OWLObjectProperty p : ap) {
            URI puri = new URIImpl(p.getIRI().toString());
            Set<OWLTransitiveObjectPropertyAxiom> ta = ontology.getTransitiveObjectPropertyAxioms(p);
            boolean isTransitive = !(ta == null || ta.isEmpty());
          //**** debug:
          //  if (isTransitive) {
          //     System.err.println(pstr+": is transitive, "+ta.size()+" axioms = ");
          //     for (OWLTransitiveObjectPropertyAxiom a : ta) {
          //       System.err.println("  "+a);
          //     }
          //  }
          //*****
            NodeSet<OWLObjectProperty> nsubs = reasoner.getSubObjectProperties(p, false);
            for (Node<OWLObjectProperty> nsub : nsubs) {
                if (!nsub.isSingleton())
                    log.error("SubObjectProperty Node is not singleton: "+nsub);
                else {
                    if (!nsub.contains(bottomObjectProp)) {
                        OWLObjectProperty sub = nsub.getRepresentativeElement();
                        if (!sub.isAnonymous()) {
                            URI obj = new URIImpl(sub.getIRI().toString());
                            if (rc.addInferredStatement(obj, RDFS.SUBPROPERTYOF, puri, graphName))
                                ++inferredCount;
                            //XXX DEBUG log.debug("Adding inferred subproperty ("+obj+", "+RDFS.SUBPROPERTYOF+", "+puri+")");
                            if (isTransitive)
                                log.debug("NOT ADDING: <"+obj+"> a owl:TransitiveProperty .");
                        }
                    }
                }
            }
        }

        // Add inferred direct subproperties of data properties
        Set<OWLDataProperty> dp = ontology.getDataPropertiesInSignature(true);
        for (OWLDataProperty p : dp) {
            URI puri = new URIImpl(p.getIRI().toString());
            NodeSet<OWLDataProperty> nsubs = reasoner.getSubDataProperties(p, false);
            for (Node<OWLDataProperty> nsub : nsubs) {
                if (!nsub.isSingleton())
                    log.error("SubDataProperty Node is not singleton: "+nsub);
                else {
                    if (!nsub.contains(bottomDataProp)) {
                        OWLDataProperty sub = nsub.getRepresentativeElement();
                        if (!sub.isAnonymous()) {
                            URI obj = new URIImpl(sub.getIRI().toString());
                            if (rc.addInferredStatement(obj, RDFS.SUBPROPERTYOF, puri, graphName))
                                ++inferredCount;
                            //XXX DEBUG log.debug("Adding inferred subproperty ("+obj+", "+RDFS.SUBPROPERTYOF+", "+puri+")");
                        }
                    }
                }
            }
        }

        // materialize all types of individuals in ontology, as we do in ABox
        for (OWLNamedIndividual ni : ontology.getIndividualsInSignature(true)) {
            URI suri = new URIImpl(ni.getIRI().toString());
            Set<OWLClassExpression> assertedTypes = ni.getTypes(ontology);
            NodeSet<OWLClass> nclasses = reasoner.getTypes(ni, false);
            for (Node<OWLClass> nclass: nclasses) {
                if (!nclass.isSingleton())
                    log.error("OWLClass Node is not singleton: "+nclass);
                else {
                    OWLClass cls = nclass.getRepresentativeElement();
                    if (!assertedTypes.contains(cls)) {
                        if (!cls.isAnonymous()) {
                            URI turi = new URIImpl(cls.getIRI().toString());
                            if (rc.addInferredStatement(suri, RDF.TYPE, turi, graphName))
                                ++inferredCount;
                            //XXX DEBUG
                            //XXX log.debug("Adding inferred type "+suri+" a "+turi);
                        }
                    }
                  /**** XXX TEST DEBUG ONLY
                    else
                        log.debug("Skipping asserted type: "+suri+" a "+cls);
                   ****/
                }
            }
        }
    }

    // daemon thread to read seralized RDF into ontology.
    private class InferenceRunner extends Thread
    {
        private InputStream in = null;
        private OWLOntology ontology = null;

        private InferenceRunner(InputStream in)
        {
            super("InferenceRunner");
            setDaemon(true);
            InferenceRunner.this.in = in;
        }

        private OWLOntology getOntology()
        {
            return ontology;
        }

        public void run()
        {
            //XXX log.debug("Starting to read ontology...");
            try {
                ontology = manager.loadOntologyFromOntologyDocument(in);
                //in.close();
            } catch (OWLOntologyCreationException e) {
                log.error("Failed reading ontology in OWLAPI: ",e);
            }
            log.debug("Finished reading ontology in sub-thread.");
        }
    }
}
