package org.eaglei.repository.servlet;

import java.io.ByteArrayOutputStream;
import java.net.URL;
import java.net.URLEncoder;
import java.nio.charset.Charset;

import javax.servlet.http.HttpServletRequest;
import javax.servlet.http.HttpServletResponse;
import javax.servlet.ServletException;

import org.apache.commons.mail.SimpleEmail;
import org.apache.commons.mail.Email;
import org.apache.commons.mail.EmailException;

import org.apache.log4j.Logger;
import org.apache.log4j.LogManager;

import org.openrdf.query.BindingSet;
import org.openrdf.query.impl.DatasetImpl;
import org.openrdf.query.QueryLanguage;
import org.openrdf.query.TupleQueryResultHandlerBase;
import org.openrdf.query.TupleQueryResultHandlerException;
import org.openrdf.query.TupleQuery;
import org.openrdf.model.URI;
import org.openrdf.model.Value;
import org.openrdf.model.vocabulary.RDFS;
import org.openrdf.model.vocabulary.RDF;
import org.openrdf.repository.RepositoryConnection;
import org.openrdf.OpenRDFException;


import org.eaglei.repository.DataRepository;
import org.eaglei.repository.View;
import org.eaglei.repository.status.BadRequestException;
import org.eaglei.repository.status.NotFoundException;
import org.eaglei.repository.status.InternalServerErrorException;
import org.eaglei.repository.util.Utils;
import org.eaglei.repository.vocabulary.DATAMODEL;

/**
 * Method: POST only.
 *
 * Query Args:
 *   uri (req) - instance about which to contact
 *   client_ip (req) - client ip addr
 *   from_name (req)
 *   from_email (req)
 *   subject
 *   message
 *   test_mode - boolean, if arg is present, redirect mail to postmaster.
 *
 * @author Larry Stone
 * @version $Id: $
 * Started June 2010
 */
public class EmailContact extends RepositoryServlet
{
    private static Logger log = LogManager.getLogger(EmailContact.class);

    // configuration property name of "postmaster" address
    private static final String CONFIG_POSTMASTER = "eaglei.repository.postmaster";
    // contact info for email
    private static final String CONFIG_MTA_HOST = "eaglei.repository.mail.host";
    private static final String CONFIG_MTA_PORT = "eaglei.repository.mail.port";
    private static final String CONFIG_MTA_SSL = "eaglei.repository.mail.ssl";
    private static final String CONFIG_MTA_USERNAME = "eaglei.repository.mail.username";
    private static final String CONFIG_MTA_PASSWORD = "eaglei.repository.mail.password";

    // Bind ?uri
    // Gets label, and email contact addr (if any) for instance.
    private static final String emailQuery =
        "SELECT ?label ?type ?email WHERE { "+
        "?uri <"+RDF.TYPE+"> ?type; <"+RDFS.LABEL+"> ?label . \n"+
        "  OPTIONAL { ?uri ?p ?email . ?p <"+DATAMODEL.IN_PROPERTY_GROUP+"> <"+
            DATAMODEL.PROPERTY_GROUP_EMAIL_CONTACT+"> }}";

    /** {@inheritDoc} */
    protected void doPost(HttpServletRequest request, HttpServletResponse response)
        throws ServletException, java.io.IOException
    {
        URI uri = Utils.parseURI(request.getParameter("uri"), "uri", true);
        String clientIP = request.getParameter("client_ip");
        String fromName = request.getParameter("from_name");
        String fromEmail = request.getParameter("from_email");
        String subject = request.getParameter("subject");
        String message = request.getParameter("message");
        boolean testMode = request.getParameter("test_mode") != null;

        // sanity checks -  note uri is already done.
        if (clientIP == null || clientIP.trim().length() == 0)
            throw new BadRequestException("No value for the required arg 'client_ip'.");
        if (fromName == null || fromName.trim().length() == 0)
            throw new BadRequestException("No value for the required arg 'from_name'.");
        if (fromEmail == null || fromEmail.trim().length() == 0)
            throw new BadRequestException("No value for the required arg 'from_email'.");

        // *rough* sanity check, from email has something like @foo.bar at least..
        int at = fromEmail.indexOf('@');
        if (at < 0 || fromEmail.indexOf('.', at) < 0)
            throw new BadRequestException("Bad value for the required arg 'from_email', must be a valid email address.");

        // must have postmaster for bounces, default mailto
        DataRepository dr = DataRepository.getInstance();
        String postmaster = dr.getConfigurationProperty(CONFIG_POSTMASTER);
        String mtaHost = dr.getConfigurationProperty(CONFIG_MTA_HOST, "localhost");
        String mtaPort = dr.getConfigurationProperty(CONFIG_MTA_PORT);
        String mtaUsername = dr.getConfigurationProperty(CONFIG_MTA_USERNAME);
        String mtaPassword = dr.getConfigurationProperty(CONFIG_MTA_PASSWORD);
        boolean mtaSSL = Boolean.parseBoolean(dr.getConfigurationProperty(CONFIG_MTA_SSL));

        if (postmaster == null)
            throw new InternalServerErrorException("Repository configuration is missing the required property: "+CONFIG_POSTMASTER);

        // need the referrer header to generate a link back to the JSP
        // page to report success.
        String referrer = request.getHeader("Referer");
        if (referrer == null)
            throw new BadRequestException("This service may only be called from another page (to set the referer).");
        URL backLink = new URL(referrer);
        log.debug("Got Referer = "+referrer);

        // collect label, direct email addr if any
        try {
            RepositoryConnection rc = WithRepositoryConnection.get(request);

            emailHandler eh = doQuery(request, rc, uri);
            if (eh.type == null)
                throw new NotFoundException("Resource instance not found in this repository: "+uri);

            // message content
            String myHost = request.getServerName();
            StringBuilder body = new StringBuilder();
            body.append("This message was generated by the eagle-i server on ").
                 append(myHost).append("\n");
            body.append("An eagle-i visitor, ").append(fromName).
              append(", wishes to contact the owner of this resource:\n").
              append("  Label: ").append(eh.label == null ? "" : eh.label).
              append("\n    URI: ").append(uri.stringValue()).append("\n");
            body.append("\nClient remote IP address = ").append(clientIP).append("\n");
            body.append("\nMessage from ").append(fromName).
                append(":\n--------------------------------------------------\n").
                append(message == null ? "(no message)" : message);
            body.append("\n--------------------------------------------------\n");

            // set up message
            Email msg = new SimpleEmail();
            msg.setCharset("UTF-8");
            msg.setMsg(body.toString());
            msg.setFrom(fromEmail, fromName);
            msg.addReplyTo(fromEmail, fromName);
            // this shows up as the envelope from address
            msg.setBounceAddress(postmaster);

            String sendTo = eh.email == null ? postmaster : eh.email;
            if (testMode) {
                msg.addHeader("X-TESTING-Would-Have-Gone-To", sendTo);
                msg.addTo(postmaster, "Repository Postmaster");
            } else {
                msg.addTo(sendTo);
            }
            msg.setSubject(subject == null ? "Inquiry from eagle-i" : subject);

            // setup MTA params
            msg.setHostName(mtaHost);
            if (mtaSSL) {
                msg.setSSL(true);
                if ((mtaUsername != null && mtaPassword == null) ||
                    (mtaUsername == null && mtaPassword != null)) {
                        throw new InternalServerErrorException("Bad configuraiton, must have BOTH username and password for mail server.");
                }
                if (mtaUsername != null &&  mtaPassword != null)
                    msg.setAuthentication(mtaUsername, mtaPassword);
                if (mtaPort != null)
                    msg.setSslSmtpPort(mtaPort);
            } else if (mtaPort != null) {
                try {
                    int p = Integer.parseInt(mtaPort);
                    msg.setSmtpPort(p);
                } catch (NumberFormatException e) {
                    throw new BadRequestException(e.toString(), e);
                }
            }

            if (log.isDebugEnabled()) {
                ByteArrayOutputStream baos = new ByteArrayOutputStream();
                msg.buildMimeMessage();
                javax.mail.internet.MimeMessage mm = msg.getMimeMessage();
                if (mm == null)
                    log.error("Failed getMimeMessage!");
                else {
                    mm.writeTo(baos);
                    log.debug("Msg = "+baos.toString());
                }
            }
            msg.send();

            String redir = backLink.getProtocol()+"://"+backLink.getHost()+
                           (backLink.getPort() < 0 ? "" : ":"+String.valueOf(backLink.getPort()))+
                           backLink.getPath()+"?sent=true&uri="+
              URLEncoder.encode(uri.stringValue(), Charset.defaultCharset().name())+
              "&test_mode="+String.valueOf(testMode);

            log.debug("Redirecting to: "+redir);
            response.sendRedirect(redir);
        } catch (javax.mail.MessagingException e) {
            log.error(e);
            throw new ServletException(e);
        } catch (EmailException e) {
            log.error(e);
            throw new ServletException(e);
        } catch (OpenRDFException e) {
            log.error(e);
            throw new ServletException(e);
        }
    }

    private static emailHandler doQuery(HttpServletRequest request,
                                 RepositoryConnection rc, URI uri)
        throws OpenRDFException
    {
            // Construct a dataset for objects to report on from
            // workspace or view, published resource graphs by default.
            DatasetImpl ds = new DatasetImpl();
            View.addGraphs(request, ds, View.USER_RESOURCES);

            //  Use prettyPrint because ds.toString() breaks when a default graph is null
            //  which is possible for the 'null' view.  don't ask, it's ugly.
            if (log.isDebugEnabled()) {
                log.debug("Dataset for SPARQL query = "+Utils.prettyPrint(ds));
                log.debug("EMAIL QUERY = \n"+emailQuery);
            }
            TupleQuery q = rc.prepareTupleQuery(QueryLanguage.SPARQL, emailQuery);
            q.setDataset(ds);
            q.setIncludeInferred(false);
            q.clearBindings();
            q.setBinding("uri", uri);
            emailHandler eh = new emailHandler();
            q.evaluate(eh);
            return eh;
    }

    public static String getContactEmail(HttpServletRequest request,  RepositoryConnection rc, URI uri)
        throws OpenRDFException
    {
        return doQuery(request, rc, uri).email;
    }

    // pick label and email out of query results
    private static class emailHandler extends TupleQueryResultHandlerBase
    {
        private String label = null;
        private String email = null;
        private Value type = null;

        public emailHandler()
        {
            super();
        }

        public void handleSolution(BindingSet bs)
            throws TupleQueryResultHandlerException
        {
            Value vl = bs.getValue("label");
            Value ve = bs.getValue("email");
            Value vt = bs.getValue("type");
            if (vt != null)
                type = vt;
            if (vl != null)
                label = vl.stringValue();
            if (ve != null)
                email = ve.stringValue();
        }
    }
}
