package org.eaglei.repository.servlet;

import java.io.File;
import java.util.List;
import javax.servlet.http.HttpServletRequest;
import javax.servlet.http.HttpServletResponse;
import javax.servlet.ServletException;

import org.apache.log4j.Logger;
import org.apache.log4j.LogManager;
import org.apache.commons.fileupload.servlet.ServletFileUpload;
import org.apache.commons.fileupload.FileUploadException;
import org.apache.commons.fileupload.disk.DiskFileItemFactory;
import org.apache.commons.fileupload.disk.DiskFileItem;

import org.openrdf.repository.RepositoryException;
import org.openrdf.model.URI;

import org.eaglei.repository.Access;
import org.eaglei.repository.auth.Authentication;
import org.eaglei.repository.workflow.Workflow;
import org.eaglei.repository.util.Utils;
import org.eaglei.repository.status.BadRequestException;
import org.eaglei.repository.status.InternalServerErrorException;
import org.eaglei.repository.status.ForbiddenException;

/**
 *
 * Args:
 *  uri - URI of the resource instance to claim
 *  user - URI of the user
 *
 * @author Larry Stone
 * @version $Id: $
 * Started October 21, 2010
 */
public class WorkflowClaim extends RepositoryServlet
{
    private static Logger log = LogManager.getLogger(WorkflowClaim.class);

    /** {@inheritDoc} */
    protected void doPost(HttpServletRequest request, HttpServletResponse response)
        throws ServletException, java.io.IOException
    {
        request.setCharacterEncoding("UTF-8");

        String rawURI = null;
        String rawUser = null;

        // if we got here through POST with multipart, grovel through args
        if (ServletFileUpload.isMultipartContent(request)) {
            try {
                ServletFileUpload upload = new ServletFileUpload();
                File tmp = (File)getServletConfig().getServletContext().getAttribute("javax.servlet.context.tempdir");
                if (tmp == null)
                    throw new InternalServerErrorException("Cannot find servlet context attr = \"javax.servlet.context.tempdir\"");
                upload.setFileItemFactory(new DiskFileItemFactory(100000, tmp));
                for (DiskFileItem item : (List<DiskFileItem>)upload.parseRequest(request)) {
                    String ifn = item.getFieldName();
                    if (ifn.equals("user"))
                        rawUser = item.getString();
                    else if (ifn.equals("uri"))
                        rawURI = item.getString();
                    else
                        log.warn("Unrecoginized request argument: "+ifn);
                }
            } catch  (FileUploadException e) {
                log.error(e);
                throw new BadRequestException("failed parsing multipart request", e);
            }

        // gather args from input params instead
        } else {
            rawUser = request.getParameter("user");
            rawURI = request.getParameter("uri");
        }

        // defaulting and sanity-checking
        URI uri = Utils.parseURI(rawURI, "uri", true);
        URI userURI = Utils.parseURI(rawUser, "user", false);
        URI myself = Authentication.getPrincipalURI(request);
        if (userURI == null)
            userURI = myself;
        else if (!(myself.equals(userURI) || Authentication.isSuperuser(request)))
            throw new ForbiddenException("Claiming on behalf of another user requires administrator access.");

        // XXX sanity: test taht userURI is actually a User? maybe later.

        // this will fail if resource isn't visible to current user
        Workflow wf = Workflow.find(request, uri);
        if (log.isDebugEnabled())
            log.debug("CLAIM PARAMETERS:\nresource="+uri+"\nowner="+userURI+"\nstate="+wf.getState()+"\nhome graph="+wf.getHomeGraph());

        // this tests for no existing claim, and access to a transition
        wf.assertClaim(request, userURI);

        try {
            WithRepositoryConnection.get(request).commit();
        } catch (RepositoryException e) {
            throw new ServletException(e);
        }
    }
}
