package org.eaglei.repository;

import java.util.Collection;
import java.util.Set;
import java.util.HashSet;
import java.util.List;
import java.util.ArrayList;
import java.util.Arrays;
import java.util.Collections;

import javax.servlet.http.HttpServletRequest;

import org.apache.log4j.Logger;
import org.apache.log4j.LogManager;

import org.openrdf.rio.RDFFormat;
import org.openrdf.rio.RDFWriterFactory;
import org.openrdf.rio.RDFWriterRegistry;
import org.openrdf.query.resultio.BooleanQueryResultFormat;
import org.openrdf.query.resultio.TupleQueryResultFormat;
import org.openrdf.query.resultio.TupleQueryResultWriterFactory;
import org.openrdf.query.resultio.TupleQueryResultWriterRegistry;

import org.eaglei.repository.format.SPARQLTextWriterFactory;
import org.eaglei.repository.format.SPARQLHTMLWriterFactory;
import org.eaglei.repository.format.RDFNQuadsWriterFactory;
import org.eaglei.repository.format.RDFHTMLWriterFactory;
import org.eaglei.repository.status.NotAcceptableException;
import org.eaglei.repository.util.Utils;

/**
 * This is a static utility class containing
 * constants and utility methods related to ALL serialization formats:
 * includes RDF serialization as well as Boolean and Tuple query results.
 *
 * @author Larry Stone
 * @version $Id: $
 */
public class Formats
{
    private static Logger log = LogManager.getLogger(Formats.class);

    private Formats()
    {
    }

    // preferred RDF serialization formats
    private static final String rdfPreferred[] = { "application/rdf+xml", "text/plain" };

    // preferred HTML *or* RDF serialization formats
    private static final String htmlPreferred[] = { "text/html", "application/xhtml+xml" };

    // preferred boolean query result serialization formats
    private static final String booleanPreferred[] = { "application/sparql-results+xml", "text/boolean" };

    // preferred tuple query result serialization formats
    private static final String tuplePreferred[] = { "application/sparql-results+xml", "text/plain" };

    /** media (MIME) types of acceptable RDF formats (for content negotiation) */
    private static Set<String> mtRDF = new HashSet<String>();
    private static RDFWriterRegistry RDFWregistry = RDFWriterRegistry.getInstance();

    static {

        // Add entries for our new RDF formats, NQuads and HTML
        RDFWregistry.add(new RDFNQuadsWriterFactory());
        RDFWregistry.add(new RDFHTMLWriterFactory());

        for (RDFWriterFactory w : RDFWregistry.getAll()) {
            RDFFormat rf = w.getRDFFormat();
            for (String mt : rf.getMIMETypes()) {
                mtRDF.add(mt);
                log.debug("Adding RDF Serialization: mime="+mt+", format="+rf);
            }
        }
    }

    /** media types of Tuple query result formats */
    private static Set<String> mtTuple = new HashSet<String>();
    static {
        // Add entries for our text and HTML formats:
        TupleQueryResultWriterRegistry registry = TupleQueryResultWriterRegistry.getInstance();
        registry.add(new SPARQLTextWriterFactory());
        registry.add(new SPARQLHTMLWriterFactory());

        for (TupleQueryResultWriterFactory w : registry.getAll()) {
            TupleQueryResultFormat tf = w.getTupleQueryResultFormat();
            for (String mt : tf.getMIMETypes()) {
                mtTuple.add(mt);
                log.debug("Adding Tuple serialization: mime="+mt+", format="+tf);
            }
        }
    }

    /** media types of Boolean query result formats */
    private static Set<String> mtBoolean = new HashSet<String>();
    static {
        for (BooleanQueryResultFormat bff : BooleanQueryResultFormat.SPARQL.values()) {
            for (String mt : bff.getMIMETypes()) {
                mtBoolean.add(mt);
                log.debug("Adding Boolean serialization: mime="+mt+", format="+bff);
            }
        }
    }

    /**
     * Get negotiated MIME type for RDF serialization format
     *
     * @param request a {@link javax.servlet.http.HttpServletRequest} object.
     * @param format a {@link java.lang.String} object.
     * @return mime type of negotated content, never null
     */
    public static String negotiateRDFContent(HttpServletRequest request,
                                             String format)
    {
        return negotiateContent(request, format, null, mtRDF, rdfPreferred);
    }

    /**
     * Get negotiated MIME type for Tuple serialization format
     *
     * @param request a {@link javax.servlet.http.HttpServletRequest} object.
     * @param format a {@link java.lang.String} object.
     * @return mime type of negotated content, never null
     */
    public static String negotiateTupleContent(HttpServletRequest request,
                                             String format)
    {
        return negotiateContent(request, format, null, mtTuple, tuplePreferred);
    }

    /**
     * Get negotiated MIME type for Boolean serialization format
     *
     * @param request a {@link javax.servlet.http.HttpServletRequest} object.
     * @param format a {@link java.lang.String} object.
     * @return mime type of negotated content, never null
     */
    public static String negotiateBooleanContent(HttpServletRequest request,
                                             String format)
    {
        return negotiateContent(request, format, null, mtBoolean, booleanPreferred);
    }

    /**
     * Get negotiated MIME type for HTML *or* RDF serialization format
     * Only for Dissemination servlet which prefers to return HTML.
     * This is an EVIL kludge but it is necessary because of broken
     * browsers like Safari (Webkit) and of course MSIE..
     *
     * @param request a {@link javax.servlet.http.HttpServletRequest} object.
     * @param format a {@link java.lang.String} object.
     * @return mime type of negotated content, never null
     */
    public static String negotiateHTMLorRDFContent(HttpServletRequest request,
                                             String format)
    {
        return negotiateContent(request, format, htmlPreferred, mtRDF, rdfPreferred);
    }


    /**
     * Manage HTTP content negotiation for all services.
     * Inputs that determine format are (in order of precedence):
     *  1. Value of 'format' query arg, must match an available format.
     *  2. Accept: header in request, highest-Q matching format.
     *  3. Default format(s).
     *
     * ONLY ONE of these inputs is selected, to remove ambiguity.
     * For example, if you supply a 'format' arg, it does NOT fall back
     * to the accept header.  If there is an accept header, only a format
     * matching its choices is returned.
     *
     * When testing formats against the input matching criteria, first
     * try for one of the "priority" formats -- this lets Dissemination
     * favor HTML if it is acceptable.  Then, try to match the formats
     * on the Preferred list.  Finally, try the whole formats list.  If
     * a Priority format was chosen AND there is no other chosen format
     * with a higher Q value, take the Priority format.  Otherwise, take
     * take the format chosen from preferred and "formats".  This lets
     * a "text/plain" with a Q of .9 from the "formats" set, take precedence
     * over a Priority "application/pdf" with a lower Q of .5.
     *
     * If no format on the formats list is in the "acceptable" set,
     * throw NotAcceptableException to return the appropriate HTTP response.
     *
     * Note that we try the formats in preferred first when matching against
     * the "Accept" header as a way of picking "preferable" formats to match
     * any wildcards in the Accept clauses.
     *
     * @param request a {@link javax.servlet.http.HttpServletRequest} object.
     * @param format the value of 'format' query arg if any
     * @param priority separate list of formats to be given priority
     * @param formats Set of available formats for the returned data.
     * @param preferred Subset of these formats which are preferred over rest
     * @return mime type of negotated content, never null
     */
    private static String negotiateContent(HttpServletRequest request,
                            String format,
                            String priority[],
                            Collection<String> formats,
                            String preferred[])
    {
        String accept = request.getHeader("Accept");

        // try explicitly chosen format first
        if (format != null) {
            if (formats != null && formats.contains(format))
                return format;
            throw new NotAcceptableException("The requested response format is not available: "+format);

        // process Accept: header - see RFC2616, sec. 14.1 for rules
        // break each clause down into media-type, subtype, quality-factor for sorting
        } else if (accept != null) {
            List<acceptClause> al = new ArrayList<acceptClause>();
            for (String a : accept.trim().split("\\s*,\\s*")) {
                try {
                    al.add(new acceptClause(a.trim()));
                } catch (IllegalArgumentException e) {
                    log.warn("Ignoring illegal format in Accept: header, section=\""+a+"\": "+e);
                }
            }
            Collections.sort(al);
            if (log.isDebugEnabled()) {
                log.debug("Accept sorted = "+Utils.collectionDeepToString(al));
                log.debug("Offers preferred = "+Arrays.deepToString(preferred)+", all = "+
                          (formats==null?"null":Utils.collectionDeepToString(formats)));
            }
            // First, get a priority result if possible.
            acceptClause priorityResult = null;
            String priorityResultMIMEType = null;
            if (priority != null) {
                priorityLoop:
                for (acceptClause ac : al) {
                    for (String pf : priority) {
                        if (ac.match(pf)) {
                            if (log.isDebugEnabled())
                                log.debug("Matched PRIORITY format="+pf+" for accept clause="+ac);
                            priorityResult = ac;
                            priorityResultMIMEType = pf;
                            break priorityLoop;
                        }
                    }
                }
            }
            // Look for normal result not overridden by priority
            for (acceptClause ac : al) {
                if (priorityResult != null && ac.q <= priorityResult.q)
                    return priorityResultMIMEType;
                for (String pref : preferred) {
                    if (ac.match(pref)) {
                        if (log.isDebugEnabled())
                            log.debug("Matched PREFERRED format="+pref+" for accept clause="+ac);
                        return pref;
                    }
                }
                if (formats != null) {
                    for (String f : formats) {
                        if (ac.match(f)) {
                            log.debug("Matched format="+f+" for accept clause="+ac);
                            return f;
                        }
                    }
                }
            }
            throw new NotAcceptableException("None of the available response formats is acceptable to the request.");

        // finally, try default - first of either priority or preferred lists.
        } else if (priority != null)
            return priority[0];
        else if (preferred != null)
            return preferred[0];
        throw new NotAcceptableException("The request did not specify any acceptable response formats and there is no default.");
    }

    /**
     * Find RDFFormat which has a writer that generates the given MIME
     * type.  This is distinct (and different) from simply looking up
     * the RDFFormat.forMIMEType() call since we install extra, custom,
     * writers.  Note that a parser is NOT necessarily available for the
     * formats returned by this call.
     *
     * @param mime RDF serialization mime type to look up
     * @return the corresponding {@link org.openrdf.rio.RDFFormat} object, or null if not found
     */
    public static RDFFormat RDFOutputFormatForMIMEType(String mime)
    {
        return RDFWregistry.getFileFormatForMIMEType(mime);
    }

    // Record class to manage a parsed-out clause from Accept: header
    private static class acceptClause implements Comparable
    {
        private String mediaType = null;
        private String mediaSubtype = null;
        private float q = (float)1.0;

        acceptClause(String source)
        {
            super();
            // parse  "type/subtype;param1;param2.."
            String params[] = source.split("\\s*;\\s*");

            // param 0 is the type/subtype
            String types[] = params[0].trim().split("\\s*/\\s*", 2);
            if (types.length != 2)
                throw new IllegalArgumentException("Illegal Accept clause: No media type/subtype in \""+source+"\"");
            mediaType = types[0];
            mediaSubtype = types[1];

            for (int i = 1; i < params.length; ++i) {
                if (params[i].startsWith("q=")) {
                    try {
                        if (params[i].length() > 2)
                            q = Float.parseFloat(params[i].substring(2));
                    } catch (NumberFormatException e) {
                        log.debug("Skipping badly formed q param ("+params[i]+") in Accept header: "+e);
                    }
                } else
                    mediaSubtype += ";"+params[i];
            }
            log.debug("Parsed accept clause: \""+source+"\" => "+this.stringValue());
        }

        boolean match(String mt)
        {
            if (mt == null)
                return false;
            String types[] = mt.split("\\s*/\\s*", 2);
            if (types.length != 2)
                throw new IllegalArgumentException("Illegal media type, no \"/\" divider: "+mt);
            return (mediaType.equals(types[0]) || mediaType.equals("*")) &&
                   (mediaSubtype.equals(types[1]) || mediaSubtype.equals("*"));
        }

        public int compareTo(Object other)
        {
            // note: invert, because higher "q" numbers sort first
            if (((acceptClause)other).q != q)
                return -(Float.compare(q, ((acceptClause)other).q));

            // more-specific type sorts first
            boolean mediaTypeStar = mediaType.equals("*");
            boolean mediaSubtypeStar = mediaSubtype.equals("*");
            boolean oMediaTypeStar = ((acceptClause)other).mediaType.equals("*");
            boolean oMediaSubtypeStar = ((acceptClause)other).mediaSubtype.equals("*");
            if (oMediaTypeStar && !mediaTypeStar)
                return -1;
            else if (mediaTypeStar && !oMediaTypeStar)
                return 1;
            if (oMediaSubtypeStar && !mediaSubtypeStar)
                return -1;
            else if (mediaSubtypeStar && !oMediaSubtypeStar)
                return 1;
            return 0;
        }

        private String getMIMEType()
        {
            return mediaType+"/"+mediaSubtype;
        }
        public String toString()
        {
            return stringValue();
        }
        private String stringValue()
        {
            return mediaType+"/"+mediaSubtype+";q="+String.valueOf(q);
        }
    }
}
