package org.eaglei.repository.auth;

import java.security.Principal;

import javax.servlet.ServletException;
import javax.servlet.http.HttpServletRequest;
import javax.servlet.http.HttpSession;

import org.apache.log4j.Logger;
import org.apache.log4j.LogManager;

import org.openrdf.model.URI;

import org.eaglei.repository.DataRepository;
import org.eaglei.repository.User;
import org.eaglei.repository.vocabulary.REPO;
import org.eaglei.repository.status.InternalServerErrorException;
import org.eaglei.repository.status.BadRequestException;

/**
 * Authentication services: answer the question, "who am I, really?" and
 * "do I have the Superuser role?".  Note that other roles are enforced by
 * the Access module, but since Superuser is special and is enforced by the
 * authentication DB for the purpose of secure bootstrap, it is manged here.
 *
 * Started April, 2010
 *
 * @author Larry Stone
 * @version $Id: $
 */
public class Authentication
{
    private static Logger log = LogManager.getLogger(Authentication.class);

    // Session Attribute to cache User object for authenticated principal
    private static final String S_USER = "org.eaglei.repository.authentication.User";

    // Session Attribute to save original client IP for later checks
    private static final String S_REMOTE_IP = "org.eaglei.repository.authentication.REMOTE_IP";

    /**
     * Find the URI of the :Person object
     * for the current authenticated user, if there is any.
     * If there is no URI (i.e. no :Person) for the current user then
     * return the URI of a plausible Role - either the Superuser role
     * if we have Superuser rights, or Anonymous otherwise.
     *
     * @param request a {@link javax.servlet.http.HttpServletRequest} object.
     * @return URI of the :Person object of current authenticated user,
     *   or if user is undocumented, the URI of their highest Role.
     */
    public static URI getPrincipalURI(HttpServletRequest request)
    {
        User u = getPrincipalUser(request);
        if (u == null) {
            URI result = request.isUserInRole(AuthUser.SUPERUSER_ROLE_NAME) ?
                   REPO.ROLE_SUPERUSER : REPO.ROLE_ANONYMOUS;
            log.debug("getPrincipalURI: Undocumented user, returning Role: "+result);
            return result;
        }
        return u.getURI();
    }

    /**
     * Invalidate the cached authenticated User if it matches the one
     * that was modified by the User API.
     *
     * @param request a {@link javax.servlet.http.HttpServletRequest} object.
     * @param u user to decache, a {@link org.eaglei.repository.User} object.
     */
    public static void decacheAuthentication(HttpServletRequest request, User u)
    {
        HttpSession session = getValidatedSession(request);
        User cu = (User)session.getAttribute(S_USER);
        if (cu != null && cu.equals(u)) {
            session.removeAttribute(S_USER);
            log.debug("Decached authenticated user from Session.");
            logout(request);
        }
    }

    /**
     * Get the current authenticated username from container or other auth'n.
     * @return the username or null if not auth'nd
     */
    public static String getAuthenticatedUsername(HttpServletRequest request)
    {
        Principal p = request.getUserPrincipal();
        return p == null ? null :  p.getName();
    }

    /**
     * Find a User object for the current
     * authenticated user, if one is available.  Note that there MIGHT NOT
     * be a User object if the authencated user has no RDF metadata; in
     * this case it returns null.  It also returns null when there is no
     * authenticated user so this is not a good test for auth'n.
     *
     * @param request a {@link javax.servlet.http.HttpServletRequest} object.
     * @return a User object or null if not auth'n or no there is no RDF metadata for auth'n user
     */
    public static User getPrincipalUser(HttpServletRequest request)
    {
        // 0. Check for cache in the Request - this overrides the Session
        //    cache for both speed and consistency (when Session is decached
        //    because we modified the current user)
        User result = (User)request.getAttribute(S_USER);
        if (result != null) {
            if (log.isDebugEnabled())
                log.debug("Returning Request-cached user = "+result);
            return result;
        }

        // 1. are we authenticated?
        Principal p = request.getUserPrincipal();
        if (p == null)
            return null;
        String pname = p.getName();

        // 2. XXX KLUDGE: if an "anonymous" user is configured, short-circuit
        //  here if it matches.
        String anonUser = DataRepository.getInstance().getConfigurationProperty("eaglei.repository.anonymous.user");
        if (anonUser != null && anonUser.equals(pname)) {
            log.info("Anonymous login because of configured anonymous user: "+pname);
            return null;
        }

        // 3. check for cached answer first:
        //    make sure session isn't invalidated by (a) Remote IP addr check,
        //    and (b) generation count staleness
        // XXX FIXME: consider setting session timeout interval.. can do it in web.xml
        HttpSession session = getValidatedSession(request);
        if (DataRepository.getInstance().isSessionStale(session)) {
            session.removeAttribute(S_USER);
            log.debug("Invalidating cached user URI in stale session, principal="+pname);
        } else {
            result = (User)session.getAttribute(S_USER);
            if (result != null) {
                if (log.isDebugEnabled())
                    log.debug("Returning cached user for principal="+pname+" from session context: "+result);
                request.setAttribute(S_USER, result);
                return result;
            }
        }

        try {
            // 4. Look for a :Person for this Principal
            User u = User.findByUsername(request, pname);
            if (u != null) {

                // check for Superuser granted by container auth'n which is
                // NOT in the RDF metadata, and override:
                // (Note we CANNOT correct the RDF since we cannot write to Sesame here).
                if (request.isUserInRole(AuthUser.SUPERUSER_ROLE_NAME) && !u.isSuperuser()) {
                    u.setIsSuperuser(true);
                }
                session.setAttribute(S_USER, u);
                request.setAttribute(S_USER, u);
            }
            return u;
        } catch (ServletException e) {
            throw new InternalServerErrorException(e.getMessage(), e);
        }
    }

    /**
     * <p>isSuperuser</p>
     * Predicate, true if current auth'd user has superuser role either in RDF
     * assertion or in the container's authz.
     *
     * @param request a {@link javax.servlet.http.HttpServletRequest} object.
     * @return a boolean, true if current authenticated user has Superuser (Admin) privilege.
     */
    public static boolean isSuperuser(HttpServletRequest request)
    {
        User u = getPrincipalUser(request);
        if (u == null)
            return request.isUserInRole(AuthUser.SUPERUSER_ROLE_NAME);
        else
            return u.isSuperuser();
    }

    /**
     * Destroy current session and credentials (if possible)
     * most web browsers cache the HTTP Basic creds so user needs to
     * trash those explicitly right after running this.
     *
     * @param request a {@link javax.servlet.http.HttpServletRequest} object.
     */
    public static void logout(HttpServletRequest request)
    {
        HttpSession s = request.getSession(false);
        if (s == null)
            log.debug("Logout finds no session to destroy!");
        else {
            s.invalidate();
            log.debug("Logout is destroying session ID="+s.getId());
        }
    }

    /**
     * Utility to get a session and check that it came from the client
     * IP that originally created it, to prevent session hijacking
     * through cookie stealing (note..  this probably wouldn't help much
     * for attacker and victim behind the same proxy or NAT, but there's
     * only so much you can do.)
     *
     */
    private static HttpSession getValidatedSession(HttpServletRequest request)
    {
        HttpSession result = request.getSession(false);
        String remoteIP = request.getRemoteAddr();

        // need to create a new session, save a record of the creator's IP
        if (result == null) {
            result = request.getSession(true);
            result.setAttribute(S_REMOTE_IP, remoteIP);

        // confirm that accessor has same IP as creator.
        } else {
            String testIP = (String)result.getAttribute(S_REMOTE_IP);
            if (testIP == null) {
                log.debug("Initializing session's record of remote address, IP="+remoteIP);
                result.setAttribute(S_REMOTE_IP, remoteIP);
            } else if (!testIP.equals(remoteIP)) {
                // XXX FIXME: maybe call logout to wipe session now?
                log.error("POSSIBLE SESSION HIJACKING: session created by IP="+testIP+", but is being accessed by IP="+remoteIP);
                throw new BadRequestException("Authentication denied, this session may only be accessed from the address that created it. Please login again.");
            }
        }
        return result;
    }
}
