package org.eaglei.repository.servlet;

import java.io.IOException;
import java.io.OutputStreamWriter;
import java.util.ArrayList;

import javax.servlet.http.HttpServletRequest;
import javax.servlet.http.HttpServletResponse;
import javax.servlet.ServletException;

import org.apache.log4j.Logger;
import org.apache.log4j.LogManager;

import org.openrdf.OpenRDFException;
import org.openrdf.query.impl.MapBindingSet;
import org.openrdf.query.BindingSet;
import org.openrdf.query.GraphQuery;
import org.openrdf.query.QueryLanguage;
import org.openrdf.model.URI;
import org.openrdf.model.ValueFactory;
import org.openrdf.rio.RDFFormat;
import org.openrdf.rio.RDFParseException;
import org.openrdf.rio.RDFHandlerException;
import org.openrdf.rio.Rio;
import org.openrdf.repository.Repository;
import org.openrdf.repository.RepositoryConnection;
import org.openrdf.repository.RepositoryException;
import org.openrdf.repository.sail.SailRepository;
import org.openrdf.sail.memory.MemoryStore;

import org.eaglei.repository.Access;
import org.eaglei.repository.DataRepository;
import org.eaglei.repository.Formats;
import org.eaglei.repository.auth.Authentication;
import org.eaglei.repository.status.BadRequestException;
import org.eaglei.repository.status.ForbiddenException;
import org.eaglei.repository.util.SPARQL;
import org.eaglei.repository.util.Utils;

/**
 * Report on and optionally reload the internally-maintaned graphs.
 *
 * Arguments:
 *  GET
 *      format=mimetype  the usual
 *      name=<uri>  - names the graph about which to return information
 *                    optional, default is all.
 *
 *  POST
 *      name=<uri>  - the graph to laod, required for action=load
 *      action=read|load|query|decache - required arg (to differentiate from get)
 *          read - returns the contents of initailizer for given graph
 *          query - return result of CONSTRUCT query in 'query' arg
 *          load - reinitializes graph in repo from init file
 *          decache - force decache of ALL in-memory data structres that cache RDF contents
 *      query - required when action=query, text of SPARQL CONSTRUCT query
 *
 * @author Larry Stone
 * Started December, 2010
 * @version $Id: $
 */
public class Internal extends RepositoryServlet
{
    private static Logger log = LogManager.getLogger(Internal.class);

    // values for 'action' arg (POST only), must be public for parser in Utils.
    public enum Action { read, load, query, decache };

    // output MIME type
    private static final String MEDIA_TYPE = "text/plain";

    /** {@inheritDoc} */
    protected void doGet(HttpServletRequest request, HttpServletResponse response)
        throws ServletException, java.io.IOException
    {
        request.setCharacterEncoding("UTF-8");

        // sanity check
        if (request.getParameter("action") != null)
            throw new BadRequestException("The 'action' argument is not allowed in the GET method, use POST.");

        URI name = null;
        String rawName = request.getParameter("name");
        if (rawName != null)
            name = Utils.parseURI(rawName, "name", false);
        String format = request.getParameter("format");
        try {
            RepositoryConnection rc = WithRepositoryConnection.get(request);
            ArrayList<BindingSet> result = new ArrayList<BindingSet>(1);

            if (name == null) {
                for (URI n : DataRepository.UPGRADEABLE_GRAPHS) {
                    result.add(makeRow(rc, n));
                }
            } else {
                result.add(makeRow(rc, name));
            }
            SPARQL.sendTupleQueryResults(request, response, format, result);
        } catch (OpenRDFException e) {
            log.error(e);
            throw new ServletException(e);
        }
    }

    // make one result row for given graph name
    private BindingSet makeRow(RepositoryConnection rc, URI name)
        throws IOException, RepositoryException, RDFParseException, RDFHandlerException
    {
        String available = DataRepository.getInitFileVersion(name);
        String loaded = Utils.getVersionInfo(rc, name, name);

        MapBindingSet bs = new MapBindingSet(3);
        ValueFactory vf = rc.getValueFactory();
        bs.addBinding("name", name);
        bs.addBinding("loaded", loaded == null ? null : vf.createLiteral(loaded));
        bs.addBinding("available", available == null ? null : vf.createLiteral(available));
        return bs;
    }

    /** {@inheritDoc} */
    protected void doPost(HttpServletRequest request, HttpServletResponse response)
        throws ServletException, java.io.IOException
    {
        request.setCharacterEncoding("UTF-8");
        Action action = (Action)Utils.parseKeywordArg(Action.class, request.getParameter("action"), "action", true, null);
        String query = request.getParameter("query");

        if (!Authentication.isSuperuser(request)) {
            throw new ForbiddenException("Administrator privileges required.");

        // decache all in-memory data - requires Superuser.
        } else if (action == Action.decache) {
            DataRepository.decacheAll();

        // read the graph's init file into the repsonse
        } else if (action == Action.read) {
            URI name = Utils.parseURI(request.getParameter("name"), "name", true);
            String path = DataRepository.getGraphInitFile(name);
            if (path == null)
                throw new BadRequestException("There is no initializer for graph named: "+name);
            response.setContentType(MEDIA_TYPE);
            response.setHeader("Content-Disposition", "attachment; filename=\""+path+"\"");
            Utils.copyStream(this.getClass().getClassLoader().getResourceAsStream(path),
                             response.getOutputStream());

        // read the initializer(s) into memory repo and run SPARQL CONSTRUCT
        // query on it, outptu the resutls.
        } else if (action == Action.query) {
            if (query == null)
                throw new BadRequestException("Missing required argument 'query'");
            String rawNames[] = request.getParameterValues("name");
            if (rawNames == null)
                throw new BadRequestException("Missing required argument 'name'");
            Repository mr = null;
            RepositoryConnection mrc = null;
            try {
                mr = new SailRepository(new MemoryStore());
                mr.initialize();
                mrc = mr. getConnection();
                for (String rawName : rawNames) {
                    URI name = Utils.parseURI(rawName, "name", true);
                    String path = DataRepository.getGraphInitFile(name);
                        mrc.add(this.getClass().getClassLoader().getResourceAsStream(path),
                                "", RDFFormat.N3);
                }
                RDFFormat rf = Formats.RDFOutputFormatForMIMEType(MEDIA_TYPE);
                response.setContentType(MEDIA_TYPE);
                GraphQuery q = mrc.prepareGraphQuery(QueryLanguage.SPARQL, query);
                q.evaluate(Rio.createWriter(rf, new OutputStreamWriter(response.getOutputStream(), "UTF-8")));

            } catch (OpenRDFException e) {
                log.error("Failed loading initializer or in query: ",e);
                throw new ServletException(e);
            } finally {
                try {
                    if (mrc != null && mrc.isOpen()) {
                        mrc.close();
                    }
                    if (mr != null)
                        mr.shutDown();
                } catch (RepositoryException e) {
                    log.error("Failed while closing temporary repo of import content: ",e);
                    throw new ServletException(e);
                }
            }
             
        // reload
        } else if (action == Action.load) {
            String rawNames[] = request.getParameterValues("name");
            if (rawNames == null)
                throw new BadRequestException("Missing required argument 'name'");
            RepositoryConnection rc = WithRepositoryConnection.get(request);
            try {
                for (String rawName : rawNames) {
                    URI name = Utils.parseURI(rawName, "name", true);
                    log.debug("Updating internal graph, name="+name);
                    rc.clear(name);
                    DataRepository.getInstance().loadGraphFromInit(rc, name, false);
                }
                rc.commit();
                response.setStatus(HttpServletResponse.SC_CREATED);
            } catch (OpenRDFException e) {
                log.error(e);
                throw new ServletException(e);
            }
        } else {
            throw new BadRequestException("Unrecognized action value: "+action);
        }
    }
}
