package org.eaglei.repository.servlet;

import java.util.ArrayList;
import java.util.Collection;
import java.util.Collections;
import java.util.List;
import javax.servlet.http.HttpServletRequest;
import javax.servlet.http.HttpServletResponse;
import javax.servlet.ServletException;

import org.apache.log4j.Logger;
import org.apache.log4j.LogManager;

import org.openrdf.query.impl.MapBindingSet;
import org.openrdf.query.BindingSet;
import org.openrdf.model.URI;
import org.openrdf.model.Literal;
import org.openrdf.model.impl.BooleanLiteralImpl;
import org.openrdf.model.impl.LiteralImpl;
import org.openrdf.model.impl.NumericLiteralImpl;
import org.openrdf.repository.RepositoryConnection;
import org.openrdf.repository.RepositoryException;

import org.eaglei.repository.Access;
import org.eaglei.repository.AccessGrant;
import org.eaglei.repository.NamedGraph;
import org.eaglei.repository.auth.Authentication;
import org.eaglei.repository.vocabulary.REPO;
import org.eaglei.repository.util.SPARQL;
import org.eaglei.repository.util.Utils;

/**
 * Returns a listing of available named graphs.
 * Optional type param narrows it down to a given Named Graph Type.
 *
 * @author Larry Stone
 * @version $Id: $
 */
public class ListGraphs extends RepositoryServlet
{
    private static Logger log = LogManager.getLogger(ListGraphs.class);

    /** {@inheritDoc} */
    protected void doPost(HttpServletRequest request, HttpServletResponse response)
        throws ServletException, java.io.IOException
    {
        doGet(request, response);
    }

    /** {@inheritDoc} */
    protected void doGet(HttpServletRequest request, HttpServletResponse response)
        throws ServletException, java.io.IOException
    {
        request.setCharacterEncoding("UTF-8");
        String format = request.getParameter("format");
        NamedGraph.Type type = (NamedGraph.Type)Utils.parseKeywordArg(NamedGraph.Type.class,
                    request.getParameter("type"), "type", false, null);
        log.debug("Got graph type filter = "+type);

        List<NamedGraph> ngs = NamedGraph.findAll(request);
        Collections.sort(ngs);
        ArrayList<BindingSet> result = new ArrayList<BindingSet>();
        boolean superuser = Authentication.isSuperuser(request);
        RepositoryConnection rc = WithRepositoryConnection.get(request);
        try {
            for (NamedGraph ng : ngs) {
                NamedGraph.Type ngt = ng.getType();
                if (type != null && !type.equals(ngt))
                    continue;
                MapBindingSet nbs = new MapBindingSet(9);
                URI uri = ng.getName();
                nbs.addBinding("namedGraphURI", uri);
                nbs.addBinding("namedGraphLabel", new LiteralImpl(ng.getLabel()));
                if (ngt != null) {
                    nbs.addBinding("typeURI", ngt.getURI());
                    nbs.addBinding("typeLabel", new LiteralImpl(ngt.getTitle()));
                }
                String loadedVersion = Utils.getVersionInfo(rc, uri, uri);
                nbs.addBinding("version", loadedVersion == null ? null : new LiteralImpl(loadedVersion));
                nbs.addBinding("size", new NumericLiteralImpl(rc.size(uri)));
                if (superuser) {
                    nbs.addBinding("read", BooleanLiteralImpl.TRUE);
                    nbs.addBinding("add", BooleanLiteralImpl.TRUE);
                    nbs.addBinding("remove", BooleanLiteralImpl.TRUE);
                } else {
                    Literal hasRead = BooleanLiteralImpl.FALSE;
                    Literal hasRemove = BooleanLiteralImpl.FALSE;
                    Literal hasAdd = BooleanLiteralImpl.FALSE;
                    for (AccessGrant g : Access.getMyGrants(request, uri)) {
                        if (g.access.uri.equals(REPO.HAS_READ_ACCESS))
                            hasRead = BooleanLiteralImpl.TRUE;
                        else if (g.access.uri.equals(REPO.HAS_ADD_ACCESS))
                            hasAdd = BooleanLiteralImpl.TRUE;
                        else if (g.access.uri.equals(REPO.HAS_REMOVE_ACCESS))
                            hasRemove = BooleanLiteralImpl.TRUE;
                    }
                    nbs.addBinding("read", hasRead);
                    nbs.addBinding("add", hasAdd);
                    nbs.addBinding("remove", hasRemove);
                }
                result.add(nbs);
            }
        } catch (RepositoryException e) {
            log.error(e);
            throw new ServletException(e);
        }
        SPARQL.sendTupleQueryResults(request, response, format, result);
    }
}
