package org.eaglei.repository.servlet;

import java.io.File;
import java.util.List;
import javax.servlet.http.HttpServletRequest;
import javax.servlet.http.HttpServletResponse;
import javax.servlet.ServletException;

import org.apache.log4j.Logger;
import org.apache.log4j.LogManager;
import org.apache.commons.fileupload.servlet.ServletFileUpload;
import org.apache.commons.fileupload.FileUploadException;
import org.apache.commons.fileupload.disk.DiskFileItemFactory;
import org.apache.commons.fileupload.disk.DiskFileItem;

import org.openrdf.repository.RepositoryException;
import org.openrdf.model.URI;

import org.eaglei.repository.workflow.Workflow;
import org.eaglei.repository.util.Utils;
import org.eaglei.repository.status.BadRequestException;
import org.eaglei.repository.status.InternalServerErrorException;

/**
 * Servlet to invoke ("push") a workflow transition, on a specific instance.
 * Implementation logic is in the Workflow class.
 *
 * Args:
 *  uri - URI of the resource instance to push
 *  transition - URI of workflow transition to take
 *
 * @author Larry Stone
 * @version $Id: $
 * Started October 21, 2010
 */
public class WorkflowPush extends RepositoryServlet
{
    private static Logger log = LogManager.getLogger(WorkflowPush.class);

    /** {@inheritDoc} */
    protected void doPost(HttpServletRequest request, HttpServletResponse response)
        throws ServletException, java.io.IOException
    {
        request.setCharacterEncoding("UTF-8");

        String rawURI = null;
        String rawTransition = null;

        // if we got here through POST with multipart, grovel through args
        if (ServletFileUpload.isMultipartContent(request)) {
            try {
                ServletFileUpload upload = new ServletFileUpload();
                File tmp = (File)getServletConfig().getServletContext().getAttribute("javax.servlet.context.tempdir");
                if (tmp == null)
                    throw new InternalServerErrorException("Cannot find servlet context attr = \"javax.servlet.context.tempdir\"");
                upload.setFileItemFactory(new DiskFileItemFactory(100000, tmp));
                for (DiskFileItem item : (List<DiskFileItem>)upload.parseRequest(request)) {
                    String ifn = item.getFieldName();
                    if (ifn.equals("transition"))
                        rawTransition = item.getString();
                    else if (ifn.equals("uri"))
                        rawURI = item.getString();
                    else
                        log.warn("Unrecoginized request argument: "+ifn);
                }
            } catch  (FileUploadException e) {
                log.error(e);
                throw new BadRequestException("failed parsing multipart request", e);
            }

        // gather args from input params instead
        } else {
            rawTransition = request.getParameter("transition");
            rawURI = request.getParameter("uri");
        }

        // defaulting and sanity-checking
        URI uri = Utils.parseURI(rawURI, "uri", true);
        URI transURI = Utils.parseURI(rawTransition, "transition", true);

        // this will fail if resource isn't visible to current user
        Workflow wf = Workflow.find(request, uri);
        if (log.isDebugEnabled())
            log.debug("PUSH PARAMETERS:\nresource="+uri+"\ntransition="+transURI);

        // this tests for existing claim, and access to the transition, etc.
        wf.invokeTransition(request, transURI);
        try {
            WithRepositoryConnection.get(request).commit();
        } catch (RepositoryException e) {
            throw new ServletException(e);
        }
    }
}
