package org.eaglei.repository;

import java.io.File;
import java.io.FileOutputStream;
import java.io.IOException;
import java.net.InetAddress;
import java.util.Date;
import java.util.Enumeration;
import java.util.Iterator;
import java.util.Properties;
import java.util.regex.Pattern;
import java.util.regex.Matcher;

import javax.servlet.ServletContext;

import org.apache.log4j.AppenderSkeleton;
import org.apache.log4j.Logger;
import org.apache.log4j.LogManager;
import org.apache.log4j.PatternLayout;
import org.apache.log4j.PropertyConfigurator;
import org.apache.log4j.RollingFileAppender;

import org.apache.commons.configuration.PropertiesConfiguration;

import org.openrdf.repository.RepositoryConnection;

import org.eaglei.repository.util.Utils;

/**
 * This is a singleton class that manages configuration info.
 * It is created by the Lifecycle singleton.
 *
 * Important configuration variables:
 *  - Sesame repository object (not the connection, but what makes connections)
 *  - Default namespace
 *  - Title (in configuration)
 *
 * @author Larry Stone
 * Started April 2010
 * @version $Id: $
 */
public class Configuration
{
    private static Logger log = LogManager.getLogger(Configuration.class);

    // root of repository log4j hierarchy
    private static final String LOG_REPO_ROOT = "org.eaglei.repository";

    // root of Sesame's log4j hierarchy
    private static final String LOG_OPENRDF_ROOT = "org.openrdf";

    // subversion conventions for branch and tag paths
    private static final String SVN_BRANCHES_PREFIX = "branches/";
    private static final String SVN_TAGS_PREFIX = "tags/";

    /**  name of java system property where home directory is specified (MUST be set) */
    private static final String REPO_HOME = "org.eaglei.repository.home";

    /**--------------------- Configuration property keys */

    /** namespace URI prefix for local resources */
    public static final String NAMESPACE = "eaglei.repository.namespace";

    /** optional alternate log directory */
    public static final String LOGDIR = "eaglei.repository.log.dir";

    /** required subdirectory for sesame nativestore files */
    public static final String SESAMEDIR = "eaglei.repository.sesame.dir";

    /** optional sesame nativestore index configuration */
    public static final String SESAMEINDEXES = "eaglei.repository.sesame.indexes";

    /** title of repository for UI labels */
    public static final String TITLE = "eaglei.repository.title";

    /** stylesheet for transforming HTML dissemination output, if any */
    public static final String INSTANCE_XSLT = "eaglei.repository.instance.xslt";

    /** CSS stylesheet for HTML dissemination output, if any */
    public static final String INSTANCE_CSS = "eaglei.repository.instance.css";

    /** List of JavaScript URLs to be included in HTML dissemination output */
    public static final String INSTANCE_JS = "eaglei.repository.instance.js";

    /** Contains URL of datamodel source within this webapp */
    public static final String DATAMODEL_SOURCE = "eaglei.repository.datamodel.source";

    //---------------------- instance variables:

    // the singleton instance
    private static Configuration instance = null;

    // cache of default namespace prefix
    private String defaultNamespace = null;

    /** Configuration properties */
    private org.apache.commons.configuration.Configuration configuration = null;

    // cache of properties describing how this software was built.
    private Properties buildProperties = null;

    /**
     * <p>Constructor for Configuration.</p>
     *
     * @param sc a {@link javax.servlet.ServletContext} object.
     */
    private Configuration()
    {
        super();
    }

    /**
     * Get the singleton instance.  Will return null unless the initialization has run.
     *
     * @return the singleton, a {@link org.eaglei.repository.Configuration} object.
     */
    public static Configuration getInstance()
    {
        if (instance == null)
            instance = new Configuration();
        return instance;
    }

    // handle error before logging is working
    private static void printError(String msg)
    {
        System.err.println("***ERROR*** (at "+(new Date().toString())+
            ") eagle-i Configuration: "+msg);
    }

    // handle info message before logging is working
    private static void printInfo(String msg)
    {
        System.err.println("***INFO*** (at "+(new Date().toString())+
            ") eagle-i Configuration: "+msg);
    }

    /**
     * Explicit initialization, invoked by bootstrap sequence in Lifecycle.
     */
    public void initialize()
        throws IOException
    {
        // 1. Set up home directory, creating it if necessary
        File home = getHomeDirectory();
        printInfo("Configuring with Eagle-I Repository Home dir = "+home.toString());

        if (!home.exists()) {
            if (!home.mkdirs()) {
                printError("Failed to create repository home directory, configured path = "+home);
                printError("THIS WILL CAUSE MANY OTHER ERRORS, DO NOT EXPECT THE REPOSITORY TO WORK!");
            }

        // If homedir path points to something that is not a directory
        // there isn't much we can do but complain about certain doom:
        } else if (!home.isDirectory()) {
            printError("Path configured as repository home is not a directory: "+home);
            printError("THIS WILL CAUSE MANY OTHER ERRORS, DO NOT EXPECT THE REPOSITORY TO WORK!");
        }

        // 2. Setup configuration properties file:
        // If file does not exist, copy in the default template.
        File pf = new File(home,"configuration.properties");
        if (!pf.exists()) {
            try {
                Utils.copyStream(this.getClass().getClassLoader().getResourceAsStream("default.configuration.properties"),
                          new FileOutputStream(pf));

            } catch (IOException e) {
                try {
                    pf.delete();
                } catch (Exception ee) { }
                printError("Failed to copy default configuration file to repository home dir: "+e);
            }
        }

        // 3. load the configuration properties into TWO separate objects,
        //    since they are for separate purposes:
        //    1. configuration - the normal repo configuration
        //    2. literalConfiguration - literal reading (no delimiter for multiple
        //       values) used to populate log4j's config..
        PropertiesConfiguration literalConfiguration = null;
        try {
            log.debug("Loading configuration from file="+pf);
            configuration = new PropertiesConfiguration(pf);
            // literal == do not break up any property value into multiple values, default=','!
            literalConfiguration = new PropertiesConfiguration();
            literalConfiguration.setListDelimiter((char)0);
            literalConfiguration.load(pf);
        } catch (Exception e) {
            printError("Failed to read configuration, file="+pf+": "+e.toString());
        }

        // 4. set up logging configuration: if it looks like this is
        // the bootstrap sequence or there is no log4j config,
        // set up default logging and then save it out
        // to the (presumably new) properties file for next time
        // and so the admin can edit it.

        // copy over just the log4j properties -- allow these to be
        // set in the main properties because:
        // (1) simpler to have only one config file
        // (2) apache config lets you interpolate variables, would
        //   be confusing to have that in main config but not log4j's..
        Properties lp = new Properties();
        for (Iterator pi = literalConfiguration.getKeys("log4j"); pi.hasNext();) {
            String k = (String)pi.next();
            lp.setProperty(k, literalConfiguration.getString(k));
        }
        setupDefaultLogging(home, lp);
    }

    // add property if user override not in place already
    private void setPropertyUnlessOverride(Properties lp, String key, String value)
    {
        if (!lp.containsKey(key))
            lp.setProperty(key, value);
    }

    /**
     * Initialize log4j:
     *  - All our Loggers are under org.eaglei.repository
     *  - Turn off additivity so events DO NOT go up to root, logger e.g in case
     *    tehre's a console appender, prevent cluttering up the container (Tomcat) log.
     *  - NOTE that programmatic setup of log4j does NOT get along with
     *    property-based configuration, e.g. prop file ONLY sees its own
     *    appenders.. so, we have to insert default properties here.
     *  - Default configuration in absence of overrides:
     *    - log at the INFO level, to "repository" appender, NOT additive.
     *    - the "repository" appender writes to "repository.log" in log dir.
     *  - Any log4j configs on the "org.eaglei.repository" logger will
     *    override these defaults.
     *
     * XXX NOTE: If buffered or non-immediate-flush is used in FileAppender,
     * you MUST call LogManager.shutdown(), see RepositoryContextListener
     */
    private void setupDefaultLogging(File home, Properties lp)
        throws IOException
    {
        // configure log directory and log4j
        String logPath = configuration.getString(LOGDIR);
        File logDir =  (logPath == null) ? new File(home, "logs") : new File(logPath);
        printInfo("Set up default log4j config: logfile directory: "+logDir.toString());
        if (!logDir.exists())
            logDir.mkdirs();
        Logger repoRoot = LogManager.getLogger(LOG_REPO_ROOT);

        // add default configuration unless properties already present:
        printInfo("Got "+String.valueOf(lp.size())+" initial log4j properties from repo configuration.");
        setPropertyUnlessOverride(lp, "log4j.logger."+LOG_REPO_ROOT, "INFO, repository");
        setPropertyUnlessOverride(lp, "log4j.additivity."+LOG_REPO_ROOT, "false");
        setPropertyUnlessOverride(lp, "log4j.appender.repository", RollingFileAppender.class.getName());
        setPropertyUnlessOverride(lp, "log4j.appender.repository.File", new File(logDir, "repository.log").toString());
        setPropertyUnlessOverride(lp, "log4j.appender.repository.ImmediateFlush", "false");
        setPropertyUnlessOverride(lp, "log4j.appender.repository.BufferedIO", "true");
        setPropertyUnlessOverride(lp, "log4j.appender.repository.Append", "true");
        setPropertyUnlessOverride(lp, "log4j.appender.repository.Encoding", "UTF-8");
        setPropertyUnlessOverride(lp, "log4j.appender.repository.layout", PatternLayout.class.getName());
        setPropertyUnlessOverride(lp, "log4j.appender.repository.layout.ConversionPattern", "%d{ISO8601} T=%t %p %c - %m%n");
        setPropertyUnlessOverride(lp, "log4j.logger."+LOG_OPENRDF_ROOT, "WARN, repository");

        printInfo("Configuring log4j with properties: "+lp.toString());
        PropertyConfigurator.configure(lp);

        // XXX NOTE: log4j's PropertyConfigurator does NOT call activateOptions
        // on all the Appenders that get created and/or modified by the
        // the properties.  We have to do it explicitly, but only for
        // the repository root logger.  If you configure any appenders on its
        // descendents, they won't work.

        // Need to explicitly activate any appenders the set up by the
        // config properties. log4j is such a crock.
        Enumeration rae = repoRoot.getAllAppenders();
        if (!rae.hasMoreElements())
            printError("The "+LOG_REPO_ROOT+" logger has no appenders!  This may result in no log records. Check your log4j configuration properties.");
        while (rae.hasMoreElements()) {
            AppenderSkeleton ra = (AppenderSkeleton)rae.nextElement();
            printInfo("Activating log4j appender \""+ra.getName()+"\" for logger "+LOG_REPO_ROOT);
            ra.activateOptions();
        }
        log.debug("logger "+repoRoot.getName()+" additivity="+repoRoot.getAdditivity());
    }

    /**
     * Get value of default namespace prefix for URIs created by the
     * repository.  This is a critically important value that ought to be
     * configured by the administrator, so that URIs are resolved correctly
     * by the server.  Since it is so essential, we have to provide a
     * usable default.
     *
     * Also, sanity-check the configured value.  It must be a legal absolute
     * URI.
     *
     * @return the default namespace prefix as a {@link java.lang.String} object.
     */
    public String getDefaultNamespace()
    {
        if (defaultNamespace == null) {
            String cfg = configuration.getString(NAMESPACE);
            if (cfg != null) {
                log.debug("Got raw configured default namespace="+cfg);
                try {
                    java.net.URI juri = new java.net.URI(cfg);
                    if (!juri.isAbsolute())
                        log.error("The configured namespace prefix must be an absolute URI: "+NAMESPACE+" = "+cfg);
                    else if (juri.getPath() == null)
                        log.error("The configured namespace prefix must include a path ending with a slash ('/'): "+NAMESPACE+" = "+cfg);
                    else if (!juri.getPath().endsWith("/"))
                        log.error("The configured namespace prefix must end with a slash ('/'): "+NAMESPACE+" = "+cfg);
                    else {
                        String scheme = juri.getScheme();
                        if (scheme.equalsIgnoreCase("http") || scheme.equalsIgnoreCase("https"))
                            defaultNamespace = cfg;
                        else
                            log.error("The configured namespace prefix must be an absolute URI: "+NAMESPACE+" = "+cfg);
                    }
                } catch (java.net.URISyntaxException e) {
                    log.error("The configured namespace prefix has illegal URI syntax: "+NAMESPACE+" = "+cfg+": "+e);
                }
            }

            // Fallback default: construct URI out of hostname.  This is
            // unlikely to be correct since JVM doesn't know the full,
            // actual, domain name for the webserver..
            if (defaultNamespace == null) {
                try {
                    defaultNamespace = "http://"+InetAddress.getLocalHost().getHostName()+"/i/";
                } catch (java.net.UnknownHostException e) {
                    log.error("Failed to get hostname for default URI prefix",e);
                    defaultNamespace = "http://failed-to-get-local-host-name/i/";
                }
                if (cfg == null)
                    log.warn("Default Namespace Prefix URI not configured.  You really should set a configuration value for "+NAMESPACE);
                log.warn("Using emergency default for namespace prefix URI (probably wrong) = "+defaultNamespace);
            }
        }
        return defaultNamespace;
    }

    /**
     * <p>getConfigurationProperty</p>
     *
     * @param key a {@link java.lang.String} object.
     * @return property value or null if not set.
     */
    public String getConfigurationProperty(String key)
    {
        return configuration.getString(key);
    }

    /**
     * return default if there is no value for the indicated property.
     *
     * @param key a {@link java.lang.String} object.
     * @param dflt default value to return if not set a {@link java.lang.String} object.
     * @return the property value or dflt if not set.
     */
    public String getConfigurationProperty(String key, String dflt)
    {
        String result = configuration.getString(key);
        return (result == null) ? dflt : result;
    }

    /**
     * Gets a configuration property with multiple values, returned in an array.
     * The values are expected to be separated by comma and optional whitespace.
     * Returns the default value (which may be null) when there is no such key
     *
     * @param key a {@link java.lang.String} object.
     * @return property value as an array of {@link java.lang.String} objects, or null if not set.
     */
    public String[] getConfigurationPropertyArray(String key, String[] dfault)
    {
        return configuration.containsKey(key) ? configuration.getStringArray(key) : dfault;
    }

    /**
     * return boolean value of a config property
     *
     * @param key a {@link java.lang.String} object.
     * @param dflt default value to return if no config found.
     * @return the property value parsed as boolean or dflt if not set.
     */
    public boolean getConfigurationPropertyAsBoolean(String key, boolean dflt)
    {
        String result = configuration.getString(key);
        return (result == null) ? dflt : Boolean.parseBoolean(result);
    }

    /**
     * Get the Maven project version
     *
     * @return version as an opaque a {@link java.lang.String} object.
     */
    public String getProjectVersion()
    {
        return getBuildProperties().getProperty("version");
    }

    /**
     * Get the source revision, e.g. the subversion revision number.
     * Depends on the build properties file left in place by maven.
     *
     * @return revision as an opaque a {@link java.lang.String} object.
     */
    public String getRevision()
    {
        return getBuildProperties().getProperty("revision");
    }

    /**
     * Get the source SCM branch, if any.
     * Depends on the build properties file left in place by maven.
     *
     * @return branch or tag name, or null if on the trunk.
     */
    public String getBranch()
    {
        String branch = getBuildProperties().getProperty("branch");
        if (branch == null) {
            log.warn("Cannot find build property \"branch\".");
            return null;
        } else if (branch.startsWith(SVN_BRANCHES_PREFIX)) {
            int nextSlash = branch.indexOf('/', SVN_BRANCHES_PREFIX.length());
            return nextSlash < 0 ? branch.substring(9) : branch.substring(9, nextSlash);
        } else if (branch.startsWith(SVN_TAGS_PREFIX)) {
            int nextSlash = branch.indexOf('/', SVN_TAGS_PREFIX.length());
            return nextSlash < 0 ? branch.substring(5) : branch.substring(5, nextSlash);

        // maven buildnumber plugin sometimes leaves scmBranch unset on
        // trunk, so value is literally "${scmBranch}".  Ugh.
        } else if (!(branch.equals("trunk") || branch.startsWith("${"/*}*/))) {
            log.warn("Build property \"branch\" has unexpected value: \""+branch+"\"");
        }
        return null;
    }

    /**
     * Get the time when this software version was built
     * Depends on the build properties file left in place by maven.
     *
     * @return time as an opaque a {@link java.lang.String} object.
     */
    public String getTimestamp()
    {
        return getBuildProperties().getProperty("timestamp");
    }

    // Read the "build.properties" resource into a cached Properties.
    // It is used to obtain the version and timestamp info
    private Properties getBuildProperties()
    {
        if (buildProperties == null) {
            buildProperties = new Properties();
            try {
                buildProperties.load(this.getClass().getClassLoader().getResourceAsStream("build.properties"));
            } catch (IOException e) {
                log.error("Failed loading build.properties: ",e);
            }
        }
        return buildProperties;
    }

    /**
     * <p>getHomeDirectory - get configured repository home directory.</p>
     *
     *  Find the root of the repository configuration and data hierarchy.
     *  It will be the first one of these that exists:
     *     a. value of system property "org.eaglei.repository.home"
     *     b. otherwise, default is "$HOME/eaglei/repository/"
     *       (where $HOME == System.getProperty("user.home") in Java-land)
     *  ALSO, this sets the system property to the default if it
     *   was unset before, so e.g. Apache config interpolation will work.
     *
     * @return repository home directory as a {@link java.io.File} object.
     * @throws java$io$IOException if any.
     */
    public File getHomeDirectory()
        throws IOException
    {
        String home = System.getProperty(REPO_HOME);
        File result = null;
        boolean haveSysProp = (home != null);
        if (!haveSysProp)
            result = new File(System.getProperty("user.home"),
                        "eaglei"+File.separator+"repository");
        else
            result = new File(home);
        if (!result.exists() && !result.mkdirs()) {
            log.fatal("The configured home directory does not exist: "+result.toString());
            throw new IOException("The configured home directory does not exist: "+result.toString());
        }
        if (!result.isDirectory()) {
            log.fatal("The configured home directory is not a directory or is protected: "+result.toString());
            throw new IOException("The configured home directory is not a directory or is protected: "+result.toString());
        }
        log.debug("repository home directory = "+result.toString());
        if (!haveSysProp)
            System.setProperty(REPO_HOME, result.toString());
        return result;
    }

    /**
     * Get version of Sesame triplestore libraries
     * @return textual version string of Sesame
     */
    public String getSesameVersion()
        throws IOException
    {
        // easy way -- if hte package has it
        String result = null;
        Package rcp = RepositoryConnection.class.getPackage();
        if (rcp == null) {
            log.debug("Cannot get Sesame version from package, no package for class "+RepositoryConnection.class.getName());
        } else {
            result = rcp.getImplementationVersion();
            if (result == null) {
                log.debug("Cannot get Sesame version, package has no ImplementationVersion: "+rcp.getName());
            } else {
                return result;
            }
        }

        // Try the hard way -- find Jar that this was loaded from.
        // The rest of this kludge assumes we have a jar-scheme URL, which
        // has the form:   jar:<url>!/<entry>
        // e.g.  jar:file:/..../lib/eagle-i-model-owl-0.4.3.jar!/ero.owl
        // so we can extract the URL of the jar itself:
        String classRes = "/"+RepositoryConnection.class.getName().replace('.','/')+".class";
        java.net.URL rurl = RepositoryConnection.class.getResource(classRes);
        if (rurl == null) {
            log.warn("Failed to find Sesame class resource path="+classRes);
        } else {
            log.debug("Got sesame class resource URL ="+rurl);
            String surl = rurl.toString();
            if (surl.startsWith("jar:")) {
                Pattern ep = Pattern.compile("jar:([^!]+)!.*");
                log.debug("Groveling jar URL with pattern=\""+ep.pattern()+"\"");
                Matcher em = ep.matcher(surl);
                if (em.matches()) {
                    String jarURLspec = em.group(1);
                    log.debug("Got jar URL = "+jarURLspec);
                    result = Utils.getMavenVersionFromJar(jarURLspec);
                    if (result != null)
                        return result;
                } else {
                    log.warn("Failed matching jar URL regex on Sesame class URL; pattern=\""+ep.pattern()+"\" against owl resource URL="+surl);
                }
            } else {
                log.warn("Sesame Class Resource URL is NOT in Jar scheme: "+surl);
            }
        }
        return "(unknown)";
    }

    /**
     * Return the directory (creating it if necessary) for Sesame
     * NativeStore files.
     */
    public File getSesameDirectory()
        throws IOException
    {
        String sesPath = getConfigurationProperty(SESAMEDIR);
        File sesDir =  (sesPath == null) ? new File(getHomeDirectory(), "sesame") : new File(sesPath);
        if (!sesDir.exists())
            sesDir.mkdirs();
        return sesDir;
    }
}
