package org.eaglei.repository.servlet;

import java.io.IOException;
import java.io.Reader;
import java.util.ArrayList;
import java.util.Date;
import java.util.Iterator;
import java.util.List;
import java.util.HashSet;
import java.util.Set;
import javax.servlet.http.HttpServletRequest;
import javax.servlet.http.HttpServletResponse;
import javax.servlet.ServletException;

import org.apache.log4j.Logger;
import org.apache.log4j.LogManager;

import org.openrdf.query.Dataset;
import org.openrdf.query.impl.DatasetImpl;
import org.openrdf.model.URI;
import org.openrdf.model.Graph;
import org.openrdf.model.Statement;
import org.openrdf.model.Value;
import org.openrdf.model.ValueFactory;
import org.openrdf.model.vocabulary.RDF;
import org.openrdf.model.vocabulary.XMLSchema;
import org.openrdf.model.impl.BooleanLiteralImpl;
import org.openrdf.model.impl.GraphImpl;
import org.openrdf.model.impl.LiteralImpl;
import org.openrdf.query.impl.MapBindingSet;
import org.openrdf.query.BindingSet;
import org.openrdf.query.QueryLanguage;
import org.openrdf.query.BooleanQuery;
import org.openrdf.query.TupleQuery;
import org.openrdf.query.TupleQueryResult;
import org.openrdf.query.QueryEvaluationException;
import org.openrdf.query.MalformedQueryException;
import org.openrdf.repository.RepositoryConnection;
import org.openrdf.repository.RepositoryException;
import org.openrdf.repository.RepositoryResult;
import org.openrdf.OpenRDFException;
import org.openrdf.rio.RDFFormat;
import org.openrdf.rio.Rio;
import org.openrdf.rio.RDFParser;
import org.openrdf.rio.helpers.RDFHandlerBase;
import org.openrdf.rio.RDFHandlerException;
import org.openrdf.rio.RDFParseException;

import org.eaglei.repository.model.Access;
import org.eaglei.repository.auth.Authentication;
import org.eaglei.repository.model.DataModel;
import org.eaglei.repository.model.EditToken;
import org.eaglei.repository.model.workflow.Workflow;
import org.eaglei.repository.model.NamedGraph;
import org.eaglei.repository.model.NamedGraphType;
import org.eaglei.repository.model.View;
import org.eaglei.repository.model.Provenance;
import org.eaglei.repository.status.BadRequestException;
import org.eaglei.repository.status.ConflictException;
import org.eaglei.repository.status.ForbiddenException;
import org.eaglei.repository.status.NotFoundException;
import org.eaglei.repository.status.HttpStatusException;
import org.eaglei.repository.util.SPARQL;
import org.eaglei.repository.util.Utils;
import org.eaglei.repository.vocabulary.DCTERMS;
import org.eaglei.repository.vocabulary.REPO;

/**
 * Create or update a "single" resource instance.  As of 1.1MS5, this
 * has been redefined to mean the base instance (i.e. direct properties of
 * the given subject URI) *and* all instances of "embedded" classes which
 * are object properties of the instance.  The set of these instances is
 * created and modified in a single transaction.
 *
 * @author Larry Stone
 * @version $Id: $
 */
public class Update extends RepositoryServlet
{
    private static Logger log = LogManager.getLogger(Update.class);

    // allowable values for Action arg
    public enum Action { create, update, gettoken };

    // query to findOrCreate embedded instances, bind ?subject first
    private static final String embeddedInstanceQuery =
        "SELECT DISTINCT ?ei WHERE { ?subject ?p ?ei . ?ei a ?eit . "+
        "?eit <"+DataModel.EMBEDDED_INSTANCE_PREDICATE.toString()+"> <"+DataModel.EMBEDDED_INSTANCE_OBJECT.toString()+"> }";

    // ASK query to verify an embedded instance, bind ?subject first
    private static final String isEmbeddedInstanceQuery =
        "ASK  WHERE { ?subject a ?eit . "+
        "?eit <"+DataModel.EMBEDDED_INSTANCE_PREDICATE.toString()+"> <"+DataModel.EMBEDDED_INSTANCE_OBJECT.toString()+"> }";

    private static final String poachedEmbeddedInstanceQuery =
        "ASK  WHERE { ?subject ?p ?ei . \n"+
        " ?ei a ?eit . \n"+
        " ?eit <"+DataModel.EMBEDDED_INSTANCE_PREDICATE.toString()+"> <"+DataModel.EMBEDDED_INSTANCE_OBJECT.toString()+"> .\n"+
        " ?s2 ?p2 ?ei FILTER(?s2 != ?subject)}";

    /** {@inheritDoc} */
    @Override
    protected void doPost(HttpServletRequest request, HttpServletResponse response)
        throws ServletException, java.io.IOException
    {
        request.setCharacterEncoding("UTF-8");
        URI uri = getParameterAsURI(request, "uri", true);
        Action action = (Action)getParameterAsKeyword(request, "action", Action.class, null, true);
        URI tokenURI = getParameterAsURI(request, "token", false);
        URI wsURI = getParameterAsURI(request, "workspace", false);
        String format = getParameter(request, "format", false);
        boolean bypassSanity = isParameterPresent(request, "bypassSanity");
        Reader deleteReader = getParameterAsReader(request, "delete", false);
        String deleteContentType = getParameterContentType(request, "delete");
        Reader insertReader = getParameterAsReader(request, "insert", false);
        String insertContentType = getParameterContentType(request, "insert");

        // argument sanity checks:
        // - uri is required, must be valid.
        // - action keyword is required.
        // - workspace required (must be valid URI) IF creating new resource.
        // - if neither insert nor delete supplied, it's a bad request.
        // - token required for action=update
        // - delete graph not allowed on create

        // Sanity-check workspace UNLESS action == getToken
        // - workspace is required if action == create
        // - if workspace is specified, make sure it is a valid URI
        if (action == Action.create && wsURI  == null) {
            throw new BadRequestException("The 'workspace' argument is required when creating a new instance.");
        }

        // more sanity and security checking:
        // - for Update, check that URI refers to a resource instance, i.e.
        //   there is an rdf:type statement about it in a graph of type Public or WOrkspace
        // - for Create, there must NOT be any type statements about subject URI alraedy.
        try {
            RepositoryConnection rc = WithRepositoryConnection.get(request);
            ValueFactory vf = rc.getValueFactory();

            // get User view mostly for access control:
            DatasetImpl ds = new DatasetImpl();
            View.addGraphs(request, ds, View.USER);

            //  Use prettyPrint because ds.toString() breaks when a default graph is null
            //  which is possible for the 'null' view.  don't ask, it's ugly.
            if (log.isDebugEnabled())
                log.debug("Dataset for SPARQL query = "+Utils.prettyPrint(ds));

            // when creating, check for resource in ANY context:
            if (action == Action.create) {
                if (rc.hasStatement(uri, RDF.TYPE, null, false))
                    throw new BadRequestException("This resource already exists so 'create' is not allowed: "+uri);

            // Choose home graph and test that it's a workspace:
            // if workspace arg was specified, it must agree with home graph
            // ..otherwise, use home graph *as* workspace.
            } else {
                URI homeGraph = getHomeGraph(rc, uri, ds);
                if (homeGraph == null)
                    throw new NotFoundException("Resource not found in this repository: "+uri.toString());
                if (wsURI == null)
                    wsURI = homeGraph;
                else if (!wsURI.equals(homeGraph))
                    throw new BadRequestException("Resource not found in specified workspace, its home graph is: "+homeGraph.toString());
            }
            log.debug("Final workspace URI = "+wsURI.toString());

            // another workspace sanity check:
            // - workspace must be a named graph of type published or workspace
            NamedGraph wng = NamedGraph.find(request, (URI)wsURI);
            if (wng == null)
                throw new BadRequestException("Invalid workspace, this is not a named graph: "+wsURI);
            NamedGraphType wngt = wng.getType();
            if (!(wngt == NamedGraphType.published || wngt == NamedGraphType.workspace))
                throw new BadRequestException("Resource is not in an appropriate workspace, ws = "+wsURI);

            if (action == Action.gettoken) {
                doGetToken(request, response, uri, format);

            // action == Update|Create: continue sanity & access checks
            } else {
                // XXX FIXME: maybe make this an instance var, shared w/ inner classes
                Set<URI> embeddedInstances = new HashSet<URI>();

                // workspace dataset for queries about resource instance
                DatasetImpl qds = new DatasetImpl();
                View.addWorkspaceGraphs(request, qds, wsURI);

                // INsanity check, only admins are allowed to do insane things:
                if (bypassSanity && !Authentication.isSuperuser(request))
                    throw new ForbiddenException("Bypassing sanity checks requires Administrator privilege.");

                // sanity check: no delete arg when action is create, and
                // create must include insert
                if (action == Action.create) {
                    if (deleteReader != null)
                        throw new BadRequestException("Delete is not allowed when creating a new resource.");
                    if (insertReader == null)
                        throw new BadRequestException("Request to create an instance must include the insert graph.");

                    // NOTE that Access check for create is handled in Workflow.
                }

                // final update checks:
                EditToken token = null;
                if (action == Action.update) {

                    // access check - ONLY look at instance ACL, ignore the graph's ACL
                    // XXX FIXME? may want to change this logic to allow graph
                    //  permissions to allow insert/remove via /update, since user
                    //  could always do the same thing via /graph..
                    if (insertReader != null && !Access.hasPermission(request, uri, Access.ADD))
                        throw new ForbiddenException("User is not permitted to insert into resource instance: "+uri);
                    if (deleteReader != null && !Access.hasPermission(request, uri, Access.REMOVE))
                        throw new ForbiddenException("User is not permitted to delete from resource instance: "+uri);

                    // Token check: when updating, must present the current token
                    token = EditToken.find(request, uri);
                    if (token == null) {
                        // need to log aborted update in case of user complaint.
                        log.info("Update was not accepted because token was already gone: "+uri);
                        throw new ConflictException("The resource instance you were updating has already been changed, so this update cannot be accepted since it would undo the previous update. If you still wish to make changes to this resource, you must open it for editing again: "+uri);
                    } else if (!token.getURI().equals(tokenURI)) {
                        log.info("Update was not accepted because token did not match: "+uri+", current token="+token);
                        throw new ConflictException(
                            "The resource instance you were updating has already been changed, AND "+
                            "ANOTHER UPDATE IS IN PROGRESS.  This update cannot be accepted since it "+
                            "would undo the previous update.  If you still wish to make changes to "+
                            "this resource, you must open it for editing again, but be warned that an "+
                            "update session was started at "+token.getCreated()+" by the user "+token.getCreatorLabel());
                    }

                    // build Set of URIs of existing embedded instances to
                    // use for sanity-checking
                    TupleQueryResult qr = null;
                    try {
                        TupleQuery eiq = rc.prepareTupleQuery(QueryLanguage.SPARQL, embeddedInstanceQuery);
                        eiq.setDataset(qds);
                        eiq.setBinding("subject", uri);
                        log.debug("SPARQL QUERY to get Embedded Instances, ?subject="+uri+"\n  "+embeddedInstanceQuery);
                        qr = eiq.evaluate();
                        while (qr.hasNext()) {
                            URI ei = (URI)qr.next().getValue("ei");
                            log.debug("  Found embedded instance="+ei);
                            embeddedInstances.add(ei);
                        }
                    } finally {
                        if (qr != null)
                            qr.close();
                    }
                }
                if (log.isDebugEnabled())
                    log.debug("Initial embeddedInstances = "+Utils.collectionDeepToString(embeddedInstances));

                // parse removes and inserts
                Graph removes = null;
                Graph inserts = null;
                insertHandler ih = null;
                if  (deleteReader != null) {
                    removeHandler rh = new removeHandler(rc, uri, embeddedInstances, wsURI);
                    String ct = (deleteContentType == null) ? format : deleteContentType;
                    if (ct == null)
                        throw new BadRequestException("No content-type specified for delete graph");
                    removes = parseGraph(deleteReader, "delete", Utils.contentTypeGetMIMEType(ct), rh, vf);
                }
                if  (insertReader != null) {
                    ih = new insertHandler(rc, action, uri, embeddedInstances, wsURI);
                    String ct = (insertContentType == null) ? format : insertContentType;
                    if (ct == null)
                        throw new BadRequestException("No content-type specified for insert graph");
                    inserts = parseGraph(insertReader, "insert", Utils.contentTypeGetMIMEType(ct), ih, vf);
                }

                doCreateOrUpdate(request, response, action, uri, wsURI, token,
                            removes, inserts,  embeddedInstances, qds, ih, bypassSanity);
            }
        // this "should never happen", means the SAIL or repo does not support generic prepareQuery()
        } catch (UnsupportedOperationException e) {
            log.error(e);
            throw new ServletException(e);
        } catch (OpenRDFException e) {
            log.error(e);
            throw new ServletException(e);
        }
    }

    // all of the actions in the create or update
    private void doCreateOrUpdate(HttpServletRequest request, HttpServletResponse response,
                            Action action, URI subject, URI wsURI, EditToken token,
                            Graph removes, Graph inserts,  Set<URI> embeddedInstances,
                            Dataset qds, insertHandler ih, boolean bypassSanity)
        throws ServletException, OpenRDFException
    {
        RepositoryConnection rc = WithRepositoryConnection.get(request);

        // optimize out any removed statements which are later restored.
        if (removes != null && inserts != null) {
            Iterator<Statement> ri = removes.iterator();
            while (ri.hasNext()) {
                Statement rs = ri.next();
                if (inserts.contains(rs)) {
                    ri.remove();
                    log.debug("Optimized out statement remove+insert: "+rs);
                    if (!inserts.remove(rs))
                        log.debug("Failed to remove statement from inserts too: "+rs);
                }
            }
        }

        // create: start workflow
        if (action == Action.create)
        {
            Workflow wf = Workflow.create(request, subject, wsURI);
            log.debug("Started New WorkFlow on resource inst="+subject+", WF state="+wf.getState());
        }

        // execute removes first since they may include wildcards:
        if (removes != null)
            rc.remove(removes, wsURI);
        if (inserts != null)
            rc.add(inserts, wsURI);
        log.info(action+": Deleted "+
                 String.valueOf(removes == null ? 0 : removes.size())+
                 " statements and added "+
                 String.valueOf(inserts == null ? 0 : inserts.size())+
                 " statements for "+subject+" in workspace "+wsURI.toString());

        // final sanity check on create: must not be EI class
        // XXX FIXME: maybe check this on update too in case they change the type??
        //   how likely is that??
        if (action == Action.create && isEmbeddedInstanceClass(rc, qds, subject)) {
            throw new BadRequestException("You may not create an independent instance of a class that is only expected to be an Embedded Instance.");
        }

        // Final sanity checks on embedded instances.

        // First, create merged list of new and existing EIs, then sort
        // them into existing and deleted EIs, testing for incomplete deletion:
        if (ih != null) {
            embeddedInstances.addAll(ih.getUnknownSubjects());
        }
        if (log.isDebugEnabled())
            log.debug("Final (almost) embeddedInstances = "+Utils.collectionDeepToString(embeddedInstances));
        Set<URI> deletedEmbeddedInstances = new HashSet<URI>();
        for (Iterator<URI> eii = embeddedInstances.iterator(); eii.hasNext();) {
            URI eis = eii.next();
            if (!isEmbeddedInstanceClass(rc, qds, eis)) {
                if (rc.hasStatement(eis, RDF.TYPE, null, false)) {
                    throw new BadRequestException("This update attempts to create an Embedded Instance of an inappropriate class, subject="+eis);
                } else if (rc.hasStatement(eis, null, null, false)) {
                    throw new BadRequestException("This update would leave an incompletely deleted Embedded Instance, subject="+eis);
                } else {
                    log.debug("Recording deleted embedded instance URI="+eis);
                    deletedEmbeddedInstances.add(eis);
                    eii.remove();
                }
            }
        }

        // check that remaining EIs have a referring statement (no orphans):
        for (URI eis : embeddedInstances) {
            if (!rc.hasStatement(subject, null, eis, false, wsURI)) {
                if (bypassSanity)
                    log.warn("Allowing update that leaves an orphaned Embedded Instance: resource="+subject+", EIsubject="+eis);
                else
                    throw new BadRequestException("This update would leave an orphaned Embedded Instance, subject="+eis);
            }
        }

        // check that deleted EIs do NOT have a referring statement (no widows):
        // also that they are fully deleted.
        for (URI eis : deletedEmbeddedInstances) {
            if (rc.hasStatement(subject, null, eis, false, wsURI)) {
                throw new BadRequestException("This update would leave broken (widow) reference to a deleted Embedded Instance, subject was="+eis);
            }
            if (rc.hasStatement(eis, null, null, false, wsURI)) {
                throw new BadRequestException("This update would leave an incompletely deleted (ghost) Embedded Instance, subject was="+eis);
            }
        }

        // detect foreign (poaching) references to our EIs:
        if (!embeddedInstances.isEmpty()) {
            log.debug("SPARQL Query to detect poached EIs:\n"+poachedEmbeddedInstanceQuery);
            BooleanQuery q = rc.prepareBooleanQuery(QueryLanguage.SPARQL, poachedEmbeddedInstanceQuery);
            q.setIncludeInferred(true);
            q.setDataset(qds);
            q.setBinding("subject", subject);
            if (q.evaluate()) {
                if (bypassSanity)
                    log.warn("Allowing update that creates a reference to one or more of another Resource Instance's Embedded Instances, resource="+subject);
                else
                    throw new BadRequestException("This update would create a reference to one or more of another Resource Instance's Embedded Instances");
            }
        }

        // action=update final sanity check & token mgt:
        // if ANY properties are left, must include at least one rdf:type
        if (action == Action.update) {
            if (rc.hasStatement(subject, null, null, false, wsURI)) {
                if (!rc.hasStatement(subject, RDF.TYPE, null, false, wsURI))
                    throw new BadRequestException("This update request would leave the resource instance without any rdf:type statements, which is not allowed unless ALL statements are removed.");
            } else {
                log.info("Deleted resource: "+subject);
            }

            // remove the current token as part of this update.
            token.clear(request);
        }

        // set provenance if we changed anything
        if (inserts != null || removes != null) {
            Date now = new Date();
            Provenance p = new Provenance(subject);
            if (action == Action.create) {
                if (ih.hasDCTERMSCreator) {
                    p.setMediated(request, now);
                } else
                    p.setCreated(request, now);
                response.setStatus(HttpServletResponse.SC_CREATED);
            } else
                p.setModified(request, now);

            // include any provenance sorted out from inserts
            List<Statement> provenanceStms = ih == null ? null : ih.getProvenanceStatements();
            if (provenanceStms != null && !provenanceStms.isEmpty())
                rc.add(provenanceStms, Provenance.PROVENANCE_GRAPH);
             
            rc.commit();
        } else
            throw new BadRequestException("Request must include at least one of the insert or delete graphs.");
    }

    private boolean isEmbeddedInstanceClass(RepositoryConnection rc, Dataset qds, URI subject)
        throws RepositoryException, QueryEvaluationException, MalformedQueryException
    {
        BooleanQuery q = rc.prepareBooleanQuery(QueryLanguage.SPARQL, isEmbeddedInstanceQuery);
        q.setIncludeInferred(true);
        q.setDataset(qds);
        q.setBinding("subject", subject);
        return q.evaluate();
    }

    // execute the getToken action and return tabular results
    //  - findOrCreate or create an edit token
    //  - format the results: columns are:
    //    token, created, creator, new
    private void doGetToken(HttpServletRequest request, HttpServletResponse response,
                            URI uri, String format)
        throws ServletException, IOException, RepositoryException
    {
        boolean created = false;
        EditToken et = EditToken.find(request, uri);
        if (et == null) {
            created = true;
            et = EditToken.create(request, uri);
            // need to commit the statements added.
            WithRepositoryConnection.get(request).commit();
        }

        ArrayList<BindingSet> results = new ArrayList<BindingSet>(1);
        MapBindingSet bs = new MapBindingSet(5);
        bs.addBinding("token", et.getURI());
        bs.addBinding("created", new LiteralImpl(et.getCreated(), XMLSchema.DATETIME));
        bs.addBinding("creator", et.getCreator());
        bs.addBinding("creatorLabel", new LiteralImpl(et.getCreatorLabel()));
        bs.addBinding("new", new BooleanLiteralImpl(created));
        results.add(bs);
        SPARQL.sendTupleQueryResults(request, response, format, results);
    }

    // feed graph stream to handler
    private Graph parseGraph(Reader graphStream, String mode, String rawFormat, graphHandler gh, ValueFactory vf)
        throws ServletException, IOException
    {
        if (rawFormat == null)
            throw new BadRequestException("Content-type (MIME Type) of the insert argument was not specified.");
        RDFFormat rf = RDFFormat.forMIMEType(rawFormat);
        if (rf == null) {
            throw new HttpStatusException(HttpServletResponse.SC_UNSUPPORTED_MEDIA_TYPE, "MIME type of "+mode+" argument is not supported: \""+rawFormat+"\"");
        }
        RDFParser parser = Rio.createParser(rf, vf);
        parser.setRDFHandler(gh);
        try {
            parser.parse(graphStream, "");
        } catch (RDFParseException e) {
            throw new BadRequestException("Error parsing "+mode+" statements: "+e.toString(), e);
        } catch (RDFHandlerException e) {
            throw new ServletException(e);
        }
        return gh.getGraph();
    }

    /**
     * Heuristic to check whether URI is subject of a legitimate
     * resource: does it have a rdf:type statement in a named
     * graph that is expected to contain only resources?
     * Also filter only visible graphs in dataset.
     */
    private URI getHomeGraph(RepositoryConnection rc, URI instance, Dataset ds)
        throws RepositoryException
    {
        URI result = Utils.getHomeGraph(rc, instance);

        // shortcut: test for read access by checking if context
        //  URI is listed in Dataset, instead of slower call
        //  to  Access.hasPermission(request, (URI)ctx, REPO.NAMED_GRAPH, Access.READ)
        if (result != null &&
            (ds.getDefaultGraphs().contains(result) ||
             ds.getNamedGraphs().contains(result))) {
                return result;
        } else {
            return null;
        }
    }

    // superclass for handlers of RDF statement traversal
    private abstract class graphHandler extends RDFHandlerBase
    {
        protected Graph batch = new GraphImpl();

        private Graph getGraph()
        {
            return batch;
        }
    }

    // check and insert statements from submitted graph
    private class insertHandler extends graphHandler
    {
        private RepositoryConnection rc;
        private Action action;
        private URI uri;
        private URI wsURI;
        private boolean hasRDFType = false;
        private boolean hasDCTERMSCreator = false;
        private List<Statement> prov = new ArrayList<Statement>();
        private Set<URI> embeddedInstances = null;
        private Set<URI> unknownSubjects = new HashSet<URI>();

        protected insertHandler(RepositoryConnection rc, Action action, URI uri, Set<URI> embeddedInstances, URI wsURI)
        {
            insertHandler.this.rc = rc;
            insertHandler.this.action = action;
            insertHandler.this.uri = uri;
            insertHandler.this.wsURI = wsURI;
            insertHandler.this.embeddedInstances = embeddedInstances;
        }

        /**
         * Sanity-check that subject is eagle-i resource URI;
         * accumulate statements on a list while also checking:
         * 1. whether predicate is rdf:type (at least one is req'd for create,
         *    or else later the resource will not appear to exist.)
         * 2. whether predicate is dcterms:creator (affects provenance)
         * Throws error upon sanity check failure.
         */
        @Override
        public void handleStatement(Statement s)
            throws RDFHandlerException
        {
            URI ssub = (URI)s.getSubject();
            URI p = s.getPredicate();
            if (uri.equals(ssub) || embeddedInstances.contains(ssub)) {
                if (RDF.TYPE.equals(p) && uri.equals(ssub))
                    hasRDFType = true;
                else if (DCTERMS.CREATOR.equals(p))
                    hasDCTERMSCreator = true;
                else if (Workflow.isWorkflowPredicate(p)) {
                    log.warn("Skipping insert of workflow property: subject="+ssub+", predicate="+p);
                    return;
                }

            // save unknown subjects for later evaluation
            } else {
                unknownSubjects.add(ssub);
            }

            // direct provenance statements to the provenance graph
            if (Provenance.isProvenancePredicate(p))
                prov.add(s);
            else
                batch.add(s);
        }

        /**
         * Sanity check, if creating do we have a type?
         * Then add the accumulated list of statements to the repo.
         */
        @Override
        public void endRDF()
            throws RDFHandlerException
        {
            if (action == Action.create && !hasRDFType)
                throw new BadRequestException("Inserted statements for 'create' must include at least one rdf:type statement.");
        }

        // getter for provenance statements
        private List<Statement> getProvenanceStatements()
            throws RepositoryException
        {
            return prov;
        }

        // getter for unknown subjects (hopefully EIs)
        private Set<URI> getUnknownSubjects()
            throws RepositoryException
        {
            return unknownSubjects;
        }
    }

    // check and remove statements from submitted graph;
    // also implement wildcards.
    private class removeHandler extends graphHandler
    {
        private RepositoryConnection rc;
        private URI uri;
        private URI wsURI;
        private Set<URI> embeddedInstances = null;

        protected removeHandler(RepositoryConnection rc, URI uri, Set<URI> embeddedInstances, URI wsURI)
        {
            removeHandler.this.rc = rc;
            removeHandler.this.uri = uri;
            removeHandler.this.wsURI = wsURI;
            removeHandler.this.embeddedInstances = embeddedInstances;
        }

        /**
         * Sanity-check that subject is eagle-i resource URI;
         * then accumulate statements to remove on a list.
         * ALSO: expand wildcard URIs in predicate and/or object.
         * Throws error upon sanity check failure.
         */
        @Override
        public void handleStatement(Statement s)
            throws RDFHandlerException
        {
            URI ssub = (URI)s.getSubject();
            if (uri.equals(ssub) || embeddedInstances.contains(ssub)) {
                URI p = s.getPredicate();
                Value o = s.getObject();

                // if either predicate or object is wildcard, expand list:
                if (REPO.MATCH_ANYTHING.equals(p) ||
                      REPO.MATCH_ANYTHING.equals(o)) {
                    try {
                        RepositoryResult<Statement> rr = null;
                        try {
                            rr = rc.getStatements(ssub,
                                          REPO.MATCH_ANYTHING.equals(p) ? null : p,
                                          REPO.MATCH_ANYTHING.equals(o) ? null : o,
                                          false, wsURI);
                            while (rr.hasNext()) {
                                Statement rs = rr.next();
                                if (batch.add(rs))
                                    log.debug("removeHandler (wildcard) really removing: "+rs);
                            }
                        } finally {
                            rr.close();
                        }
                    } catch (RepositoryException e) {
                        log.error(e);
                        throw new RDFHandlerException(e);
                    }
                }
                else if (Workflow.isWorkflowPredicate(p))
                    log.info("Skipping delete of workflow property: "+p);
                else
                    if (batch.add(s))
                        log.debug("removeHandler really added: "+s);
            } else {
                throw new BadRequestException("Subject is not the resouce URI nor one of its embedded instances: "+ssub);
            }
        }
    }
}
