package org.eaglei.repository.model;

import java.util.List;
import java.util.ArrayList;

import javax.servlet.ServletException;
import javax.servlet.http.HttpServletRequest;

import org.apache.log4j.Logger;
import org.apache.log4j.LogManager;

import org.openrdf.query.impl.DatasetImpl;
import org.openrdf.OpenRDFException;
import org.openrdf.model.URI;
import org.openrdf.model.Literal;
import org.openrdf.model.Value;
import org.openrdf.repository.RepositoryConnection;
import org.openrdf.model.vocabulary.RDFS;
import org.openrdf.query.TupleQuery;
import org.openrdf.query.TupleQueryResultHandlerBase;
import org.openrdf.query.TupleQueryResultHandlerException;
import org.openrdf.query.BindingSet;
import org.openrdf.query.QueryLanguage;
import org.openrdf.query.MalformedQueryException;

import org.eaglei.repository.servlet.WithRepositoryConnection;
import org.eaglei.repository.vocabulary.REPO;
import org.eaglei.repository.util.Utils;

/**
 * Object Model for query macros, used in the SPARQL Workbench.
 *
 * Query that picks up the macro definitions:
 *  SELECT ?l ?t FROM :NG_Query WHERE { ?s a :QueryMacro; rdfs:label ?l; :query ?t }
 *
 * Example macros defined (in n3):
 *
 * :macros
 *   rdf:_1 [
 *     a :QueryMacro;
 *     rdfs:label "first macro";
 *     :query "text of query macro string here" ];
 *   rdf:_2 [
 *     a :QueryMacro;
 *     rdfs:label "second macro";
 *     :query "more macro here, don't forget to escape quotes" ].
 *
 *
 * @author Larry Stone
 * Started March, 2011
 */
public class QueryMacro extends ImmutableObjectModel implements Comparable
{
    private static Logger log = LogManager.getLogger(QueryMacro.class);

    /** immutable, label of hte macro */
    private final String label;

    /**  immutable, content of the macro */
    private final String macro;

    // sparql query to find macros
    private static final String macroQuery =
      "SELECT DISTINCT ?label ?text WHERE { "+
      "?s a <"+REPO.QUERY_MACRO+">; <"+RDFS.LABEL+"> ?label; <"+REPO.QUERY+"> ?text }";

    /** Constructor */
    private QueryMacro(String l, String m)
    {
        super();
        label = l;
        macro = m;
    }

    /**
     * Get all macros
     *
     * @param request a {@link javax.servlet.http.HttpServletRequest} object.
     * @return List of all macros found in RDF db
     */
    public static List<QueryMacro> findAll(HttpServletRequest request)
        throws ServletException
    {
        List<QueryMacro> result = new ArrayList<QueryMacro>();

        RepositoryConnection rc = WithRepositoryConnection.get(request);
        try {
            TupleQuery q = rc.prepareTupleQuery(QueryLanguage.SPARQL, macroQuery);
            DatasetImpl ds = new DatasetImpl();
            ds.addDefaultGraph(REPO.NG_QUERY);
            q.setDataset(ds);
            q.evaluate(new MacroHandler(rc, result));
            return result;
        } catch (MalformedQueryException e) {
            log.info("Rejecting malformed query: "+e);
            throw new ServletException("Malformed query was generated internally: "+e.toString());
        } catch (OpenRDFException e) {
            log.error(e);
            throw new ServletException(e);
        }
    }

    /** {@inheritDoc} */
    @Override
    public int compareTo(Object o)
    {
        return label.compareTo(((QueryMacro)o).label);
    }

    /**
     * Not implemented, no need to get one macro
     *
     * @param request a {@link javax.servlet.http.HttpServletRequest} object.
     * @param resource subject of the resource instance, a {@link org.openrdf.model.URI} object.
     * @return token as a {@link org.eaglei.repository.EditToken} object or null if none available.
     */
    public static QueryMacro find(HttpServletRequest request, URI resource)
    {
        log.error("Someone should not have called unimplemented find()!");
        return null;
    }

    /** not really implemented */
    @Override
    public URI getURI()
    {
        return null;
    }

    /**
     * <p>Getter for label</p>
     *
     * @return the label
     */
    @Override
    public String getLabel()
    {
        return label;
    }

    /**
     * <p>Getter for macro content</p>
     *
     * @return the macro
     */
    public String getMacro()
    {
        return macro;
    }

    /**
     * <p>toString</p>
     *
     * @return a {@link java.lang.String} object.
     */
    @Override
    public String toString()
    {
        return "<#QueryMacro: label="+label+", macro="+macro+">";
    }

    /** {@inheritDoc} */
    @Override
    public boolean equals(Object o)
    {
        return o instanceof QueryMacro && label != null && label.equals(((QueryMacro)o).label);
    }

    /** {@inheritDoc} */
    @Override
    public int hashCode()
    {
        return label.hashCode();
    }

    /** handler to populate result list */
    private static class MacroHandler extends TupleQueryResultHandlerBase
    {
        private RepositoryConnection rc = null;
        private List<QueryMacro> result = null;

        MacroHandler(RepositoryConnection rc, List<QueryMacro> result)
        {
            super();
            MacroHandler.this.rc = rc;
            MacroHandler.this.result = result;
        }

        // columns: label text
        @Override
        public void handleSolution(BindingSet bs)
            throws TupleQueryResultHandlerException
        {
            Value label = bs.getValue("label");
            Value text = bs.getValue("text");
            if (label == null || text == null ||
                  !((text instanceof Literal) && (label instanceof Literal))) {
                log.warn("Skipping incorrect query definition (label and query are not literals): label="+label+", text="+text);
            } else {
                result.add(new QueryMacro(Utils.valueAsString(label), Utils.valueAsString(text)));
            }
        }
    }
}
