package org.eaglei.repository.model;

import org.eaglei.repository.util.SPARQL;
import org.eaglei.repository.status.InternalServerErrorException;
import org.eaglei.repository.status.ForbiddenException;
import org.eaglei.repository.status.BadRequestException;
import java.util.Map;
import java.util.SortedMap;
import java.util.TreeMap;

import javax.servlet.http.HttpServletRequest;
import org.apache.log4j.Logger;
import org.apache.log4j.LogManager;

import org.openrdf.OpenRDFException;
import org.openrdf.repository.RepositoryConnection;
import org.openrdf.query.TupleQuery;
import org.openrdf.query.BindingSet;
import org.openrdf.query.QueryLanguage;
import org.openrdf.query.TupleQueryResultHandlerBase;
import org.openrdf.query.TupleQueryResultHandlerException;
import org.openrdf.query.impl.DatasetImpl;
import org.openrdf.model.URI;
import org.openrdf.model.Literal;
import org.openrdf.model.Value;
import org.openrdf.model.vocabulary.RDF;
import org.openrdf.model.vocabulary.RDFS;
import org.eaglei.repository.vocabulary.REPO;

import org.eaglei.repository.auth.Authentication;
import org.eaglei.repository.util.WithRepositoryConnection;
import org.eaglei.repository.status.InternalServerErrorException;

/**
 * Views are named datasets to be used in SPARQL queries, which are
 * constructed dynamically according to the current user's access permissions.
 *
 * The View class is two things:
 *  1. An enumerated type that defines the set of implemented views.
 *  2. A set of static utility methods to manipulate datasets.
 *
 * The implemented views are:
 *  public - all graphs visible to the anonymous role
 *  metadata - all accessible metadata graphs
 *  ontology - all accessible ontology graphs
 *  metadata+ontology - all accessible md AND ontology graphs
 *  user - every NG user can see
 *  user-resource - every NG user can see related to ei instance resources
 *  all - every named graph whether permitted or not
 *  null - query entire Sesame repo without regard to named graphs (superuser)
 *
 * @author Larry Stone
 * @version $Id: $
 * Started April 2010
 */
public enum View
{
    ALL                     ("all"),
    METADATA                ("metadata"),
    METADATA_ONTOLOGY       ("metadata+ontology"),
    ONTOLOGY                ("ontology"),
    PUBLIC                  ("public"),
    PUBLISHED               ("published"),
    PUBLISHED_RESOURCES     ("published-resources"),
    USER                    ("user"),
    USER_RESOURCES          ("user-resources"),
    NULL                     ("null");

    private String label = null;

    private View (String l) {
        label = l;
    }

    // dataset of internal graphs plus NG_Users to get user's rdf:type
    private static DatasetImpl viewDataset = SPARQL.copyDataset(SPARQL.InternalGraphs);
    static {
        viewDataset.addDefaultGraph(REPO.NG_USERS);
    }

    /**
     * <p>Getter for the field <code>label</code>.</p>
     *
     * @return a {@link java.lang.String} object.
     */
    public String getLabel()
    {
        return label;
    }

    /**
     * <p>getTitle - get pretty title to e.g. include in a menu </p>
     *
     * @return a {@link java.lang.String} object.
     */
    public String getTitle()
    {
        return String.format("%C%s", label.charAt(0), label.substring(1));
    }

    /**
     * <p>parseView - find View matching name.
     * Implemented with dumb linear search, but good enough for now.
     *  </p>
     *
     * @param v the view name to match.
     * @return a {@link org.eaglei.repository.View} object or null if none found.
     */
    public static View parseView(String v)
    {
        for (View result : values()) {
            if (result.label.equals(v))
                return result;
        }
        return null;
    }

    private static Logger log = LogManager.getLogger(View.class);

    /**
     * Add the named graphs composing a view to the specified dataset.
     * Adding graphs to an existing view allows the caller to compose a
     * dataset of multiple views, or a view and some graphs.
     * Each graph is added to both the default graphs list and the
     * named graphs list to allow the SPARQL GRAPH keyword to match graphs.
     * Any failure that prevents the intent from being completely
     * and successfully achieved will throw an error.
     *
     * @param request a {@link javax.servlet.http.HttpServletRequest} object.
     * @param ds dataset to modify a {@link org.openrdf.query.impl.DatasetImpl} object.
     * @param vw the View whose graphs to add.
     */
    public static void addGraphs(HttpServletRequest request, DatasetImpl ds, View vw)
    {
        boolean filterByPermission = false;
        boolean filterByPublic = false;
        boolean addInferred = false;
        StringBuilder pattern = new StringBuilder("{ ?name <").append(RDF.TYPE).append("> <").append(REPO.NAMED_GRAPH).append("> ");
        switch (vw) {

            // next 3: all NGs of specified type(s) that user can see
            case METADATA:
                pattern.append("; <").append(REPO.NG_TYPE).append("> <").append(REPO.NGTYPE_METADATA).append(">");
                filterByPermission = true;
                break;
            case ONTOLOGY:
                pattern.append("; <").append(REPO.NG_TYPE).append("> <").append(REPO.NGTYPE_ONTOLOGY).append(">");
                filterByPermission = true;
                break;
            case METADATA_ONTOLOGY:
                pattern.append("; <").append(REPO.NG_TYPE).append("> ?ngt . ")
                   .append("filter(?ngt = <").append(REPO.NGTYPE_ONTOLOGY).append("> || ")
                   .append(" ?ngt = <").append(REPO.NGTYPE_METADATA).append(">) ");
                filterByPermission = true;
                break;

            // all NGs of any type that user can see
            case USER:
                filterByPermission = true;
                addInferred = true;
                break;

            // all NGs containing RESOURCES that user can see
            // EXCEPT NG_Users
            case USER_RESOURCES:
                pattern.append("; <").append(REPO.NG_TYPE).append("> ?ngt . ")
                       .append("filter(?name != <").append(REPO.NG_USERS).append("> && ")
                       .append("(?ngt = <").append(REPO.NGTYPE_WORKSPACE).append("> || ")
                       .append("?ngt = <").append(REPO.NGTYPE_PUBLISHED).append("> || ")
                       .append("?ngt = <").append(REPO.NGTYPE_ONTOLOGY).append("> || ")
                       .append("?ngt = <").append(REPO.NGTYPE_METADATA).append(">)) ");
                filterByPermission = true;
                addInferred = true;
                break;

            // graphs of type PUBLISHED plus all ontology and public metadata
            case PUBLISHED:
                pattern.append("; <").append(REPO.NG_TYPE).append("> ?ngt . ")
                       .append("filter( ?ngt = <").append(REPO.NGTYPE_ONTOLOGY).append("> || ")
                       .append("?ngt = <").append(REPO.NGTYPE_METADATA).append("> || ")
                       .append("?ngt = <").append(REPO.NGTYPE_PUBLISHED).append("> )");
                filterByPermission = true;
                addInferred = true;
                break;

            // ONLY graphs of type PUBLISHED which contain RESOURCE INSTANCES,
            //  plus all ontology and public metadata
            case PUBLISHED_RESOURCES:
                pattern.append("; <").append(REPO.NG_TYPE).append("> ?ngt . ")
                       .append("filter(?name != <").append(REPO.NG_USERS).append("> && ")
                       .append("(?ngt = <").append(REPO.NGTYPE_ONTOLOGY).append("> || ")
                       .append(" ?ngt = <").append(REPO.NGTYPE_METADATA).append("> || ")
                       .append(" ?ngt = <").append(REPO.NGTYPE_PUBLISHED).append("> ))");
                filterByPermission = true;
                addInferred = true;
                break;

            // all publically visible graphs - so no permission test needed
            case PUBLIC:
                filterByPublic = true;
                addInferred = true;
                break;

            // All graphs, but only for superuser since this does
            // not check for permission (that would be "user")
            // Although the 'user' view is the same for superuser, this will
            // break if a non-superuser tries it which may be desireable,
            // if a 'user' viwe missing graphs would cause unexpected results.
            case ALL:
                if (!Authentication.isSuperuser(request))
                    throw new ForbiddenException("This view is only available to administrators.");
                break;

            // very special case: since "null" is ultimately powerful with
            // no named-graph restrictions, require superuser up front
            case NULL:
                if (!Authentication.isSuperuser(request))
                    throw new ForbiddenException("This view is only available to administrators.");
                ds.addDefaultGraph(null);
                return;

            default:
                throw new IllegalArgumentException("Unimplemented view: "+vw);
        }

        // get list of applicable graph names where access control not needed.
        // this is just a graph pattern group to go in another query.
        pattern.append(" . }");
        String groupString = pattern.toString();

        if (filterByPublic) {
            log.debug("SPARQL query pattern group to get public (view="+vw+") graphs = "+groupString);
            Access.filterByPermission(request, REPO.ROLE_ANONYMOUS, "name", "?name", groupString, Access.READ,
                viewDataset, null, new DatasetHandler("name", ds));
        } else if (filterByPermission && !Authentication.isSuperuser(request)) {
            log.debug("SPARQL query pattern group to get "+vw+" graphs = "+groupString);
            Access.filterByPermission(request, null, "name", "?name", groupString, Access.READ,
                viewDataset, null, new DatasetHandler("name", ds));
        } else {
            try {
                RepositoryConnection rc = WithRepositoryConnection.get(request);
                String queryString = "SELECT ?name WHERE "+groupString;
                TupleQuery q = rc.prepareTupleQuery(QueryLanguage.SPARQL, queryString);
                q.setIncludeInferred(true);
                q.setDataset(SPARQL.InternalGraphs);
                SPARQL.evaluateTupleQuery(queryString, q, new DatasetHandler("name", ds));
            } catch (OpenRDFException e) {
                log.error(e);
                throw new InternalServerErrorException("Failed in query to generate view: ",e);
            }
        }

        // add the graph for inferred statements about non-ontology instances
        if (addInferred && !ds.getDefaultGraphs().isEmpty())
            SPARQL.addGraph(ds, REPO.NG_INFERRED);
    }

    /**
     * Construct a "workspace" view based on a single named graph
     * of type Workspace or Published which is expected to contain
     * resource instances. Add the metadata, ontology, and inferred
     * graphs.
     * WARNING: This assumes all access control has been done already!
     *
     * @param request a {@link javax.servlet.http.HttpServletRequest} object.
     * @param ds dataset to modify a {@link org.openrdf.query.impl.DatasetImpl} object.
     * @param graphName names the workspace graph, a {@link org.openrdf.model.URI} object.
     */
    public static void addWorkspaceGraphs(HttpServletRequest request, DatasetImpl ds, URI graphName)
    {
        // sanity-check that graph URI names a Named Graph of appropriate type
        try {
            RepositoryConnection rc = WithRepositoryConnection.get(request);
            if (rc.hasStatement(graphName, REPO.NG_TYPE, REPO.NGTYPE_PUBLISHED, false) ||
                  rc.hasStatement(graphName, REPO.NG_TYPE, REPO.NGTYPE_WORKSPACE, false)) {
             
                // check for access
                log.debug("Adding Workspace graphs for WS graph="+graphName);
                if (Access.hasPermission(request, graphName, Access.READ)) {
                    SPARQL.addGraph(ds, graphName);
                    addGraphs(request, ds, METADATA_ONTOLOGY);
                    SPARQL.addGraph(ds, REPO.NG_USERS);
                    SPARQL.addGraph(ds, REPO.NG_INFERRED);
                } else {
                    throw new ForbiddenException("User does not have permission to read this workspace.");
                }
            } else {
                throw new BadRequestException("Not a workspace graph: "+graphName);
            }
        } catch (OpenRDFException e) {
            log.error(e);
            throw new InternalServerErrorException("Failed in query to generate view: ",e);
        }
    }

    // SPARQL query to get graph names of all NGs of type workspace or published.
    private static final String wksQuery =
        "SELECT * WHERE { ?name a <"+REPO.NAMED_GRAPH+"> . \n"+
        " {{ ?name <"+REPO.NG_TYPE+"> <"+REPO.NGTYPE_PUBLISHED+"> } UNION \n"+
        "  { ?name <"+REPO.NG_TYPE+"> <"+REPO.NGTYPE_WORKSPACE+"> }} \n"+
        " OPTIONAL { ?name <"+RDFS.LABEL+"> ?label }}";

    /**
     * Returns map loaded with name and graph-URI of all named graphs of
     * type workspace or pubilshed.  Intended for constructing UI menu.
     *
     * @param request a {@link javax.servlet.http.HttpServletRequest} object.
     * @return all named graphs of type published or workspace in a {@link java.util.Map} object.
     */
    public static Map<String,URI> getAllWorkspaceGraphs(HttpServletRequest request)
    {
        try {
            RepositoryConnection rc = WithRepositoryConnection.get(request);
            TupleQuery q = rc.prepareTupleQuery(QueryLanguage.SPARQL, wksQuery);
            q.setDataset(SPARQL.InternalGraphs);
            WorkspaceHandler h = new WorkspaceHandler();
            SPARQL.evaluateTupleQuery(wksQuery, q, h);
            return h.result;
        } catch (OpenRDFException e) {
            log.error(e);
            throw new InternalServerErrorException("Failed in query to get all workspaces: ",e);
        }
    }

    // tuple query result handler to build a workspace list, of
    // SortedMap<String,URI>  where URI is workspace graph, String is its label
    private static final class WorkspaceHandler extends TupleQueryResultHandlerBase
    {
        private SortedMap<String,URI> result = new TreeMap<String,URI>();

        private WorkspaceHandler()
        {
            super();
        }

        /** {@inheritDoc} */
        public void handleSolution(BindingSet bs)
            throws TupleQueryResultHandlerException
        {
            Value name = bs.getValue("name");
            Value label = bs.getValue("label");
            if (name instanceof URI) {
                String k = (label == null) ? ((URI)name).getLocalName() : ((Literal)label).getLabel();
                result.put(k, (URI)name);
            } else {
                throw new InternalServerErrorException("Internal error, Got unexpected non-URI value in query results:: "+name);
            }
        }
    }

    // tuple query result handler to build a Dataset:  just chuck the
    // "name" column result into the default graph of a Dataset
    private static final class DatasetHandler extends TupleQueryResultHandlerBase
    {
        // resulting Dataset to mung
        private DatasetImpl ds;

        // bind variable to look for in result
        private String name;

        private DatasetHandler(String name, DatasetImpl ds)
        {
            super();
            DatasetHandler.this.name = name;
            DatasetHandler.this.ds = ds;
        }

        /** {@inheritDoc} */
        public void handleSolution(BindingSet bs)
            throws TupleQueryResultHandlerException
        {
            Value graphName = bs.getValue(name);
            if (graphName instanceof URI) {
                    SPARQL.addGraph(ds, (URI)graphName);
            } else {
                throw new InternalServerErrorException("Internal error, Got unexpected non-URI value in query results:: "+graphName.toString());
            }
        }
    }
}
