package org.eaglei.repository.util;

import org.openrdf.model.Value;
import org.openrdf.model.Literal;
import org.openrdf.model.ValueFactory;
import org.openrdf.query.algebra.evaluation.ValueExprEvaluationException;
import org.openrdf.query.algebra.evaluation.function.Function;

import org.eaglei.repository.vocabulary.REPO;

/**
 * Custom SPARQL query function, repo:upperCaseStr.  It returns the
 * toUpperCase() version of of hte string value of an RDF value.  Use it
 * to sort values IGNORING whether (a) the case of characters differs,
 * (b) they are datatyped-literals or untyped literals (or other terms).
 * This implementation may be specifc to Sesame 2.3, it depends on
 * loosely-documented interfaces.
 *
 * To invoke it you must have the repository's URI namespace defined as a
 * prefix, e.g.
 *   PREFIX repo: <http://eagle-i.org/ont/repo/1.0/>
 *    ....
 *   ORDER BY repo:upperCaseStr(?label)
 *
 * NOTE that this Function implementation is connected to the query
 * evaluator passively, by listing its class in the resource file:
 *   META-INF/services/org.openrdf.query.algebra.evaluation.function.Function
 *
 * Larry Stone  -  August 2010
 */
public class UpperCaseStr implements Function
{
    private String uri = REPO.NAMESPACE+"upperCaseStr";

    /**
     * Get name of this function as a URI
     * @return
     */
    public String getURI()
    {
        return uri;
    }

    /**
     * Evaluate the function; returns upper case version of given string.
     * @param vf value factory
     * @param args string to convert
     * @return upper case-ified string value
     * @throws ValueExprEvaluationException
     */
    public Value evaluate(ValueFactory vf, Value... args)
        throws ValueExprEvaluationException
    {
        String mixed = args[0] instanceof Literal ? ((Literal)args[0]).getLabel() : args[0].stringValue();
        return vf.createLiteral(mixed.toUpperCase());
    }
}
