package org.eaglei.repository.model;

import java.net.URL;


import org.apache.commons.configuration.PropertiesConfiguration;
import org.apache.commons.configuration.ConfigurationException;
import org.apache.log4j.Logger;
import org.apache.log4j.LogManager;

import org.openrdf.model.URI;

import org.eaglei.repository.Configuration;
import org.eaglei.repository.util.Utils;

/**
 * Access the configurable data model ontology.
 *
 * Started March, 2011
 * @author lcs
 */
public enum DataModel
{
    /** Data model elements as enum values: */
    GRAPH_URI ("datamodel.graph.uri", URI.class),
    VERSION_INFO_URI ("datamodel.versionInfo.uri", URI.class),
    VERSION_INFO_SOURCE ("datamodel.versionInfo.source", String.class),
    HIDE_PROPERTY_PREDICATE ("datamodel.hideProperty.predicate", URI.class),
    HIDE_PROPERTY_OBJECT ("datamodel.hideProperty.object", URI.class),
    CONTACT_PROPERTY_PREDICATE ("datamodel.contactProperty.predicate", URI.class),
    CONTACT_PROPERTY_OBJECT ("datamodel.contactProperty.object", URI.class),
    EMBEDDED_INSTANCE_PREDICATE ("datamodel.embedded.predicate", URI.class),
    EMBEDDED_INSTANCE_OBJECT ("datamodel.embedded.object", URI.class),
    LABEL_PREDICATE ("datamodel.label.predicate", URI.class),
    CONTACT_EMAIL_BINDINGS ("datamodel.contact.email.bindings", String.class),
    CONTACT_EMAIL_QUERY ("datamodel.contact.email.query", String.class),
    ;

    private static final Logger log = LogManager.getLogger(DataModel.class);

    private static final String DEFAULT_DATAMODEL_SOURCE = "eaglei-datamodel.properties";

    private String configKey = null;
    private Class type = null;
    private Object result = null;
    private boolean initialized = false;

    // data model configuration
    private static org.apache.commons.configuration.Configuration dmConfig = null;

    // constructor for enum with uri
    private DataModel(String key, Class type)
    {
        this.configKey = key;
        this.type = type;
    }

    // get data model source and load properties
    private org.apache.commons.configuration.Configuration getConfiguration()
    {
        if (dmConfig == null) {
            String source = Configuration.getInstance().getConfigurationProperty(Configuration.DATAMODEL_SOURCE, DEFAULT_DATAMODEL_SOURCE);
            URL srcURL = DataModel.class.getClassLoader().getResource(source);
            try {
                if (srcURL == null) {
                    log.error("Cannot find Data Model Configuration resource, source="+source);
                    log.error("This will be BAD, no data model is available.  DO NOT EXPECT ANYTHING TO WORK.");
                } else {
                        log.debug("Loading data model config from source="+srcURL);
                        dmConfig = new PropertiesConfiguration(srcURL);
                }
            } catch (ConfigurationException e) {
                log.error("Failed loading Data Model configuration from URL="+srcURL+": ",e);
             
            // bogus fallback to prevent NPEs
            } finally {
                if (dmConfig == null) {
                    dmConfig = new PropertiesConfiguration();
                }
            }
        }
        return dmConfig;
    }

    // fetch the configuration value as a scalar (or first element of array)
    private void loadScalar()
    {
        if (!initialized) {
            String val = getConfiguration().getString(configKey);
            if (val == null) {
                throw new IllegalStateException("Missing data model configuration value for key: "+configKey);
            } else if (type == URI.class) {
                result = Utils.parseURI(val, configKey, true);
            } else if (type == String.class) {
                result = val;
            } else {
                throw new IllegalStateException("Unrecognized type in data model config: "+type.getName());
            }
        }
    }

    // fetch the configured value as an array of possible values
    // separated by comma in the config file value field.
    private void loadArray()
    {
        if (!initialized) {
            if (!getConfiguration().containsKey(configKey)) {
                throw new IllegalStateException("Missing data model configuration value for key: "+configKey);
            } else {
                String val[] = getConfiguration().getStringArray(configKey);
                if (type == String.class) {
                    result = val;
                } else if (type == URI.class) {
                    URI aresult[] = new URI[val.length];
                    for (int i = 0; i < val.length; ++i) {
                        aresult[i] = Utils.parseURI(val[i], configKey, true);
                    }
                    result = aresult;
                } else {
                    throw new IllegalStateException("Unrecognized type in data model config: "+type.getName());
                }
            }
        }
    }

    /**
     * Getter for configured URI for this value.
     * @return URI, never null.
     */
    public URI getURI()
    {
        loadScalar();
        return (URI)result;
    }

    /**
     * Getter for the string form of a configured scalar value.
     * @return string representation, never null
     */
    public String getString()
    {
        loadScalar();
        return (String)result;
    }

    /**
     * Getter for multiple configured values
     * @return array of all configured values, never null
     */
    public URI[] getArrayOfURI()
    {
        loadArray();
        return (URI [])result;
    }

    /**
     * Getter for multiple configured values, returns as string form
     * @return array of String of all configured values, never null
     */
    public String[] getArrayOfString()
    {
        loadArray();
        return (String [])result;
    }

    /**
     * Render as string.
     * @return String representation of configured scalar value
     */
    @Override
    public String toString()
    {
        loadScalar();
        return result == null ? "null" : result.toString();
    }
}
