package org.eaglei.repository.model;

import java.util.Date;
import javax.servlet.http.HttpServletRequest;

import org.apache.log4j.Logger;
import org.apache.log4j.LogManager;
import org.eaglei.repository.Configuration;

import org.openrdf.OpenRDFException;
import org.openrdf.model.URI;
import org.openrdf.model.Literal;
import org.openrdf.model.Value;
import org.openrdf.repository.RepositoryConnection;
import org.openrdf.repository.RepositoryException;
import org.openrdf.model.ValueFactory;
import org.openrdf.model.vocabulary.RDFS;
import org.openrdf.query.TupleQuery;
import org.openrdf.query.TupleQueryResultHandlerBase;
import org.openrdf.query.TupleQueryResultHandlerException;
import org.openrdf.query.BindingSet;
import org.openrdf.query.QueryLanguage;
import org.openrdf.query.MalformedQueryException;

import org.eaglei.repository.auth.Authentication;
import org.eaglei.repository.util.WithRepositoryConnection;
import org.eaglei.repository.rid.RIDGenerator;
import org.eaglei.repository.vocabulary.REPO;
import org.eaglei.repository.vocabulary.DCTERMS;
import org.eaglei.repository.util.SPARQL;
import org.eaglei.repository.status.InternalServerErrorException;
import org.eaglei.repository.status.ForbiddenException;
import org.eaglei.repository.status.ConflictException;

/**
 * Object Model for "edit token", an RDF construct used to regulate
 * access to updating a resource instance.
 *
 * @author Larry Stone
 * Started May 13, 2010
 */
public final class EditToken extends ImmutableObjectModel
{
    private static Logger log = LogManager.getLogger(EditToken.class);

    /** immutable, eagle-i resource instance to which this token applies */
    private URI resource = null;

    /**  immutable, the subject of this instance, the token */
    private URI uri = null;

    /** the User who first created this token */
    private URI creator = null;

    /** rdfs:label of that User so we don't have to look it up.. */
    private String creatorLabel = null;

    /** timestamp when it was created */
    private String created = null;

    /**  SPARQL query to get "all" (should only be 1) tokens for resource
     *   URI and their contents. Must bind ?instance.
     */
    private static final String etQuery =
        "SELECT * WHERE { \n"+
          "?instance <"+REPO.HAS_EDIT_TOKEN+"> ?editToken . \n"+
          "OPTIONAL { ?editToken <"+DCTERMS.CREATOR+"> ?creator  \n"+
          " OPTIONAL { ?creator <"+RDFS.LABEL+"> ?creatorLabel } } \n"+
          "OPTIONAL { ?editToken <"+DCTERMS.CREATED+"> ?created } }";

    /** Constructor */
    private EditToken(URI resource)
    {
        super();
        this.resource = resource;
    }

    /**
     * Return current edit token if any, or null if not found.
     *
     * @param request a {@link javax.servlet.http.HttpServletRequest} object.
     * @param resource subject of the resource instance, a {@link org.openrdf.model.URI} object.
     * @return token as a {@link org.eaglei.repository.EditToken} object or null if none available.
     */
    public static EditToken find(HttpServletRequest request, URI resource)
    {
        RepositoryConnection rc = WithRepositoryConnection.get(request);
        try {
            TupleQuery q = rc.prepareTupleQuery(QueryLanguage.SPARQL, etQuery);
            q.setBinding("instance", resource);
            q.setDataset(SPARQL.InternalGraphs);
            q.setIncludeInferred(false);
            EditTokenHandler eh = new EditTokenHandler(resource);
            SPARQL.evaluateTupleQuery(etQuery, q, eh);
            EditToken result = eh.getResult();
            if (result != null && result.uri != null) {
                if (log.isDebugEnabled())
                    log.debug("Found edit token = "+result.uri+" for Resource Instance = "+resource);
                return result;
            } else {
                return null;
            }
        } catch (MalformedQueryException e) {
            log.error("Rejecting malformed query:"+e);
            throw new InternalServerErrorException(e);
        } catch (OpenRDFException e) {
            log.error(e);
            throw new InternalServerErrorException(e);
        }
    }

    /**
     * Create and return a new token for given resource.
     * There MUST NOT be a token already, throws a conflict error if so.
     *
     * @param request a {@link javax.servlet.http.HttpServletRequest} object.
     * @param resource a {@link org.openrdf.model.URI} object.
     * @return a {@link org.eaglei.repository.EditToken} object, NEVER null.
     */
    public static EditToken create(HttpServletRequest request, URI resource)
    {
        try {
            RepositoryConnection rc = WithRepositoryConnection.get(request);

            // sanity check: do not add an extra edit token.
            if (rc.hasStatement(resource, REPO.HAS_EDIT_TOKEN, null, false, REPO.NG_INTERNAL))
                throw new ConflictException("There is already an edit token for resource instance: "+resource);
             
            // access check; must have either add OR remove at least..
            if (!(Access.hasPermission(request, resource, Access.ADD) || Access.hasPermission(request, resource, Access.REMOVE)))
                throw new ForbiddenException("No permission to modify this resource instance: "+resource);

            EditToken result = new EditToken(resource);
            result.creator = Authentication.getPrincipalURI(request);
            Literal now = Provenance.makeDateTime(new Date());
            result.created = now.getLabel();
            ValueFactory vf = rc.getValueFactory();

            result.uri = vf.createURI(Configuration.getInstance().getDefaultNamespace(),
                                      RIDGenerator.getInstance().newID().toString());
            rc.add(resource, REPO.HAS_EDIT_TOKEN, result.uri, REPO.NG_INTERNAL);
            rc.add(result.uri, DCTERMS.CREATOR, result.creator, REPO.NG_INTERNAL);
            rc.add(result.uri, DCTERMS.CREATED, now, REPO.NG_INTERNAL);
            if (log.isDebugEnabled())
                log.debug("Created edit token = "+result.uri+" for Resource Instance = "+resource);
            return result;
        } catch (RepositoryException e) {
            log.error("Failed creating edit token: ",e);
            throw new InternalServerErrorException("Failed creating edit token: ",e);
        }
    }

    /**
     * Remove this edit token from the RDF DB.
     *
     * @param request a {@link javax.servlet.http.HttpServletRequest} object.
     */
    public void clear(HttpServletRequest request)
    {
        try {
            RepositoryConnection rc = WithRepositoryConnection.get(request);
            rc.remove(resource, REPO.HAS_EDIT_TOKEN, uri, REPO.NG_INTERNAL);
            rc.remove(uri, null, null, REPO.NG_INTERNAL);
        } catch (RepositoryException e) {
            log.error("Failed clearing edit token: ",e);
            throw new InternalServerErrorException("Failed clearing edit token: ",e);
        }
    }

    /**
     * <p>Gets resource instance of edit token </p>
     *
     * @return a {@link org.openrdf.model.URI} object.
     */
    public URI getResource()
    {
        return resource;
    }

    /**
     * <p>getURI - gets URI of edit token itself</p>
     *
     * @return a {@link org.openrdf.model.URI} object.
     */
    public URI getURI()
    {
        return uri;
    }

    /**
     * <p>Getter for the field <code>creator</code>.</p>
     *
     * @return a {@link org.openrdf.model.URI} object.
     */
    public URI getCreator()
    {
        return creator;
    }

    /**
     * <p>Getter for the field <code>creatorLabel</code>.</p>
     *
     * @return a {@link java.lang.String} object.
     */
    public String getCreatorLabel()
    {
        return creatorLabel == null ? creator.getLocalName() : creatorLabel;
    }

    /**
     * <p>Getter for the field <code>created</code>.</p>
     * XXX FIXME someday this ought to return a Date
     *
     * @return created as a {@link java.lang.String} object.
     */
    public String getCreated()
    {
        return created;
    }

    /**
     * Get human-readable, textual representation of the object.
     *
     * @return label as a {@link java.lang.String} object.
     */
    public String getLabel()
    {
        return "EditToken for "+resource;
    }

    /**
     * <p>toString</p>
     *
     * @return a {@link java.lang.String} object.
     */
    public String toString()
    {
        return "<#EditToken: resource="+resource+", uri="+uri+
            ", creator="+creator+
            ", creatorLabel="+creatorLabel+
            ", created="+created+">";
    }

    /** {@inheritDoc} */
    public boolean equals(Object o)
    {
        return o instanceof EditToken && uri != null && uri.equals(((EditToken)o).uri);
    }

    /** {@inheritDoc} */
    public int hashCode()
    {
        return uri.hashCode();
    }

    /** tuple query result handler to gather contents of (hopefully) one token */
    private static class EditTokenHandler extends TupleQueryResultHandlerBase
    {
        private EditToken result = null;

        EditTokenHandler(URI resource)
        {
            super();
            result = new EditToken(resource);
        }

        EditToken getResult()
        {
            return result;
        }

        /**
         * {@inheritDoc}
         *
         * Query handler to pick up what SHOULD be the single result.
         * Check for multiple results, since it is a sign of something badly wrong.
         */
        public void handleSolution(BindingSet bs)
            throws TupleQueryResultHandlerException
        {
            Value editToken = bs.getValue("editToken");
            Value bcreator = bs.getValue("creator");
            Value bcreatorLabel = bs.getValue("creatorLabel");
            Value bcreated = bs.getValue("created");
            if (editToken instanceof URI) {
                if (result.uri != null)
                    log.error("Unexpected multiple editToken results for resource instance URI ="+result.resource);
                result.uri = (URI)editToken;
                if (bcreator instanceof URI)
                    result.creator = (URI)bcreator;
                if (bcreatorLabel instanceof Literal)
                    result.creatorLabel = ((Literal)bcreatorLabel).getLabel();
                if (bcreated instanceof Literal)
                    result.created = ((Literal)bcreated).getLabel();
            } else
                log.error("Bad query result, editToken="+editToken);
        }
    }
}
