package org.eaglei.repository.model;

import java.io.IOException;
import java.io.OutputStreamWriter;
import java.util.Set;

import javax.servlet.ServletException;
import javax.servlet.http.HttpServletRequest;
import javax.servlet.http.HttpServletResponse;

import org.apache.log4j.Logger;
import org.apache.log4j.LogManager;

import org.openrdf.rio.RDFFormat;
import org.openrdf.rio.Rio;
import org.openrdf.rio.RDFHandler;
import org.openrdf.OpenRDFException;
import org.openrdf.model.URI;
import org.openrdf.model.impl.URIImpl;
import org.openrdf.model.Statement;
import org.openrdf.repository.RepositoryConnection;
import org.openrdf.query.BindingSet;
import org.openrdf.query.MalformedQueryException;
import org.openrdf.query.TupleQuery;
import org.openrdf.query.TupleQueryResult;
import org.openrdf.query.QueryLanguage;

import org.eaglei.repository.auth.Authentication;
import org.eaglei.repository.vocabulary.REPO;
import org.eaglei.repository.util.SPARQL;
import org.eaglei.repository.status.InternalServerErrorException;
import org.eaglei.repository.status.ForbiddenException;
import org.eaglei.repository.status.BadRequestException;
import org.eaglei.repository.util.WithRepositoryConnection;
import org.eaglei.repository.util.DuplicateArg;

/**
 * Export and import of access grants about other object(s).
 *
 * @author Larry Stone
 * Started March, 2011
 */
public class TransportAccessGrant implements Transporter
{
    private static Logger log = LogManager.getLogger(TransportAccessGrant.class);

    // SPARQL query to get subjects out of import graph
    private static final String IMPORT_GRANTS_QUERY =
        "SELECT DISTINCT ?subject WHERE { ?subject ?access ?agent }";

    /**
     * Check that current authenticated user is authorized for this
     * operation; some export requests require Superuser privilege.
     */
    @Override
    public void authorizeExport(HttpServletRequest request)
        throws ServletException
    {
        if (!Authentication.isSuperuser(request))
            throw new ForbiddenException("Export of access grants requires administrator privileges.");
    }

    /**
     * Check that current authenticated user is authorized for this
     * operation; some import requests require Superuser privilege.
     */
    @Override
    public void authorizeImport(HttpServletRequest request)
        throws ServletException
    {
        if (!Authentication.isSuperuser(request))
            throw new ForbiddenException("Import of access grants requires administrator privileges.");
    }

    /**
     * Export description of access grants as serialized RDF quads.
     * Handler for the ImportExport servlet.
     * {@inheritDoc}
     */
    @Override
    public void doExport(HttpServletRequest request, HttpServletResponse response,
                       RDFFormat format, Set<String> includes, Set<String> excludes)
        throws ServletException, IOException
    {
        try {
            RDFHandler out = Rio.createWriter(format, new OutputStreamWriter(response.getOutputStream(), "UTF-8"));
            out.startRDF();

            if (includes.isEmpty())
                throw new BadRequestException("Export of grants requires an include list of URIs");
            if (!excludes.isEmpty())
                throw new BadRequestException("Export of grants does not support an exclude list");

            for (String iu : includes) {
                URI uri = new URIImpl(iu);
                for (Statement s : AccessGrant.exportGrants(uri, AccessGrant.getGrants(request, uri))) {
                    out.handleStatement(s);
                }
            }
            out.endRDF();
        } catch (OpenRDFException e) {
            throw new InternalServerErrorException(e);
        }
    }

    /**
     * Import description of access grants from serialized RDF quads.
     * Handler for the ImportExport servlet.
     * {@inheritDoc}
     */
    @Override
    public void doImport(HttpServletRequest request, HttpServletResponse response,
                       RepositoryConnection content,
                       Set<String> includes, Set<String> excludes,
                       DuplicateArg duplicate,
                       boolean transform, boolean ignoreACL)
        throws ServletException, IOException
    {
        RepositoryConnection rc = WithRepositoryConnection.get(request);
        try {
            TupleQuery q = content.prepareTupleQuery(QueryLanguage.SPARQL, IMPORT_GRANTS_QUERY);
            q.setDataset(SPARQL.InternalGraphs);
            TupleQueryResult qr = null;
            try {
                qr = SPARQL.evaluateTupleQuery(IMPORT_GRANTS_QUERY, q);
                while (qr.hasNext()) {
                    BindingSet bs= qr.next();
                    URI uri = (URI)bs.getValue("subject");
                    String us = uri.stringValue();
                    if (excludes.contains(us)) {
                        log.debug("SKIP IMPORT GRANT because of exclude: uri="+us);
                        continue;
                    } else if (!(includes.isEmpty() || includes.contains(us))) {
                        log.debug("SKIP IMPORT GRANT because of include: uri="+us);
                        continue;
                    }
                    for (Statement s : AccessGrant.importGrants(request, content, uri, uri)) {
                        log.debug("Adding access grant statement: "+s);
                        rc.add(s, REPO.NG_INTERNAL);
                    }
                }
            } finally {
                if (qr != null)
                    qr.close();
            }
        } catch (MalformedQueryException e) {
            log.error("Rejecting malformed query:"+e);
            throw new ServletException(e);
        } catch (OpenRDFException e) {
            log.error(e);
            throw new ServletException(e);
        }
    }
}
