package org.eaglei.repository.servlet.admin;

import javax.servlet.http.HttpServletRequest;
import javax.servlet.http.HttpServletResponse;
import javax.servlet.ServletException;

import org.apache.log4j.Logger;
import org.apache.log4j.LogManager;

import org.openrdf.model.URI;
import org.openrdf.repository.RepositoryConnection;
import org.openrdf.OpenRDFException;

import org.eaglei.repository.auth.Authentication;
import org.eaglei.repository.model.AccessGrant;
import org.eaglei.repository.util.Utils;
import org.eaglei.repository.servlet.RepositoryServlet;
import org.eaglei.repository.util.WithRepositoryConnection;
import org.eaglei.repository.status.BadRequestException;
import org.eaglei.repository.status.ForbiddenException;

/**
 * Add and remove Access Grants.  This servlet exists primarily
 * to serve the Edit Grants page in the admin JSP UI.  Of course
 * it can be used by itself, but beware of weird behavior when its
 * referrer is a .JSP URL - it will indicate success by redirecting back
 * to the JSP.
 *
 * Method: POST only.
 *
 *   REQUIRED ARGS:
 *    uri = URI of the object
 *    action = add | remove
 *    access = URI of hte access type to be added/deleted
 *    agent = URI of the person or Role in the access grant.
 *
 *   These args are passed through to the JSP upon redirect. See
 *   editGrants.jsp for details:
 *    type, label, lock_access, backLink, backLabel
 *
 * @author Larry Stone
 * Started March 2011
 */
public class UpdateGrants extends RepositoryServlet
{
    private static Logger log = LogManager.getLogger(UpdateGrants.class);

    // allowable values for Action arg
    public enum Action { add, remove };

    // args to pass through to JSP
    private static final String PASS_THRU_ARGS[] =
        { "type", "label", "lock_access", "backLink", "backLabel"};

    /** {@inheritDoc} */
    @Override
    protected void doPost(HttpServletRequest request, HttpServletResponse response)
        throws ServletException, java.io.IOException
    {
        request.setCharacterEncoding("UTF-8");
        URI uri = getParameterAsURI(request, "uri", true);
        Action action = (Action)getParameterAsKeyword(request, "action", Action.class, null, true);
        URI access = getParameterAsURI(request, "access", true);

        if (!Authentication.isSuperuser(request))
            throw new ForbiddenException("This service is only for use by administrators.");

        // sanity check: only one 'agent' value..
        // since this gets driven by a form with multiple 'agent' inputs we
        // have to be able to deal with extra empty values
        URI agent = null;
        for (String a : getParameters(request, "agent", true)) {
            if (a.trim().length() > 0) {
                if (agent != null)
                    throw new BadRequestException("Only one value is allowed for the 'agent' argument");
                agent = Utils.parseURI(a, "agent", true);
            }
        }
        if (agent == null)
            throw new BadRequestException("Missing required argument: agent");

        RepositoryConnection rc = WithRepositoryConnection.get(request);
        String message = "";
        try {

            if (action == Action.add) {
                AccessGrant.addGrant(request, uri, agent, access);
                message = "New grant has been added.";
            } else if (action == Action.remove) {
                message = AccessGrant.removeGrant(request, uri, agent, access) ?
                    "Grant has been removed." : "Specified grant was not found.";
            }
            // need to save changes to the triplestore.
            rc.commit();
        } catch (OpenRDFException e) {
            log.error(e);
            throw new ServletException(e);
        }
        redirectToJSP(request, response, message, PASS_THRU_ARGS, "uri", uri.stringValue());
    }
}
