package org.eaglei.repository.auth;

import java.sql.Connection;
import java.sql.PreparedStatement;
import java.sql.ResultSet;
import java.sql.SQLException;
import javax.sql.DataSource;
import javax.naming.Context;
import javax.naming.InitialContext;
import javax.naming.NamingException;
import java.util.HashMap;
import java.util.Map;
import java.util.Collection;

import org.apache.log4j.Logger;
import org.apache.log4j.LogManager;

import javax.servlet.http.HttpServletRequest;
import javax.servlet.ServletException;

import org.openrdf.model.URI;
import org.openrdf.model.impl.URIImpl;
import org.eaglei.repository.vocabulary.REPO;

/**
 * Simple lightweight class to manage RDBMS authorization system users
 * for the purpose of importing and exporting auth. users.
 *
 * @author Larry Stone
 * Started June 24 2010
 */
public final class AuthUserTomcatFactory implements AuthUserFactory
{
    private static Logger log = LogManager.getLogger(AuthUserTomcatFactory.class);

    // This must match the Datasource defined in webapp context, see context.xml
    private static final String AUTH_DATASOURCE = "jdbc/eaglei";

    // Special object of EXPORT_AUTH_TYPE only used in export and import
    // of users to indicate the builtin authentication (RDBMS) type.
    // This is really a misnomer since it actually means "Tomcat
    // container authentication database" as implemented by AuthUser Tomcat.
    // XXX move the URI to authuser factory??
    private static final URI EXPORT_AUTH_TYPE_BUILTIN = new URIImpl(REPO.NAMESPACE + "exportAuthType_Builtin");

    // the singleton instance
    private static AuthUserTomcatFactory instance = new AuthUserTomcatFactory();

    /** Getter for singleton. */
    public static AuthUserTomcatFactory getInstance()
    {
        return instance;
    }

    private AuthUserTomcatFactory()
    {
        super();
    }

    /**
     * Get the auth-db description of the indicated user, if present.
     * @param targetUsername the username to look up
     * @return a new AuthUserTomcat object or null if not found.
     */
    @Override
    public AuthUserTomcat find(String targetUsername)
        throws ServletException
    {
        Connection c = null;
        PreparedStatement s = null;
        try {
            try {
                c = getConnection();
                s = c.prepareStatement(
                    "SELECT Users.Username,Users.Password,Roles.Rolename "+
                    " FROM Users LEFT OUTER JOIN Roles "+
                    " ON Users.Username = Roles.Username "+
                    "    AND Roles.Rolename = '"+Authentication.SUPERUSER_ROLE_NAME+"'"+
                    " WHERE Users.Username = ?");
                s.setString(1, targetUsername);
                ResultSet r = s.executeQuery();
                int count = 0;
                AuthUserTomcat result = null;
                while (r.next()) {
                    String dusername = r.getString(1);
                    String dpassword = r.getString(2);
                    String role = r.getString(3);
                    boolean isAdmin = role != null && Authentication.SUPERUSER_ROLE_NAME.equalsIgnoreCase(role);
                    if (log.isDebugEnabled())
                        log.debug("Got row, user="+dusername+", passwd="+dpassword+", role="+role);
                    ++count;
                    if (result == null) {
                        result = new AuthUserTomcat(dusername, dpassword, isAdmin);
                    } else {
                        log.error("Got extra results from single user query: user="+dusername+", pw="+dpassword+", role="+role);
                    }
                }
                if (log.isDebugEnabled())
                    log.debug("Processed "+String.valueOf(count)+" rows of results.");
                return result;
            } finally {
                if (s != null)
                    s.close();
                if (c != null)
                    c.close();
            }
        } catch (NamingException e) {
            log.error("Failed in query for find(): ",e);
            throw new ServletException(e);
        } catch (SQLException e) {
            log.error("Failed in query for find(): ",e);
            throw new ServletException(e);
        }
    }

    /**
     * Get a Map of each username key to AuthUserTomcat object.
     * @return Map from username to AuthUser object, never null but maybe empty
     */
    @Override
    public Map<String,AuthUserTomcat> findAllAsMap()
        throws ServletException
    {
        Map<String,AuthUserTomcat> result = new HashMap<String,AuthUserTomcat>();
        Connection c = null;
        PreparedStatement s = null;
        try {
            try {
                c = getConnection();
                s = c.prepareStatement(
                    "SELECT Users.Username,Users.Password,Roles.Rolename "+
                    " FROM Users LEFT OUTER JOIN Roles "+
                    " ON Users.Username = Roles.Username"+
                    " AND Roles.Rolename = '"+Authentication.SUPERUSER_ROLE_NAME+"'");
                ResultSet r = s.executeQuery();
                while (r.next()) {
                    String dusername = r.getString(1);
                    String role = r.getString(3);
                    boolean isAdmin = role != null && Authentication.SUPERUSER_ROLE_NAME.equalsIgnoreCase(role);
                    if (result.containsKey(dusername)) {
                        log.error("Got duplicate SQL result for user="+dusername+", role="+role);
                    } else {
                        String dpassword = r.getString(2);
                        result.put(dusername, new AuthUserTomcat(dusername, dpassword, isAdmin));
                        if (log.isDebugEnabled())
                            log.debug("Got new user, user="+dusername+", passwd="+dpassword+", role="+role);
                    }
                }
                return result;
            } finally {
                if (s != null)
                    s.close();
                if (c != null)
                    c.close();
            }
        } catch (NamingException e) {
            log.error("Failed in query for findAllAsMap(): ",e);
            throw new ServletException(e);
        } catch (SQLException e) {
            log.error("Failed in query for findAllAsMap(): ",e);
            throw new ServletException(e);
        }
    }

    /**
     * Create a new auth user in the DB with indicated username.
     * See implementation in AuthUserTomcat
     * @param username the username to look up
     * @return the new AuthUser that was created
     */
    @Override
    public AuthUser create(String username)
    {
        AuthUser result = new AuthUserTomcat(username, null, false);
        result.create();
        return result;
    }

    /**
     * Get a jdbc Connection to the authentication RDBMS
     * Has to be public so AuthUserTomcat can use it.
     */
    public static Connection getConnection()
        throws NamingException, SQLException
    {
        Context initContext = new InitialContext();
        Context envContext  = (Context)initContext.lookup("java:/comp/env");
        DataSource ds = (DataSource)envContext.lookup(AUTH_DATASOURCE);
        return ds.getConnection();
    }

    /**
     * Commit a group of modified (or created) entries, more
     * efficient for use by import.
     * @param request the servlet request
     * @param authUsers the users to commit
     */
    public void commitMultiple(HttpServletRequest request, Collection<? extends AuthUser> authUsers)
        throws ServletException
    {
        try {
            Connection c = null;
            try {
                c = getConnection();
                c.setAutoCommit(false);
                for (AuthUserTomcat au : (Collection<AuthUserTomcat>)authUsers) {
                    au.flush(c);
                }
                c.commit();
            } finally {
                if (c != null)
                    c.close();
            }
        } catch (NamingException e) {
            log.error("Failed in multiple commit(): ",e);
            throw new ServletException(e);
        } catch (SQLException e) {
            log.error("Failed in multiple commit(): ",e);
            throw new ServletException(e);
        }
    }

    /**
     * get the URI identifying this AuthUser implementation.
     * @return the identifying URI
     */
    public URI getType()
    {
        return EXPORT_AUTH_TYPE_BUILTIN;
    }
}
